/**
 * Copyright (c) 2008, 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *  
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

using System;
using System.IO;
using System.Reflection;
using System.Threading;
using System.Windows.Forms;
using System.Collections;
using System.Collections.Generic;

using Microsoft.Win32;

using MySQL.Forms;
using MySQL.Grt;
using MySQL.Utilities;
using MySQL.Utilities.SysUtils;
using MySQL.Workbench;

namespace MySQL.GUI.Workbench
{
  static class Program
  {
    #region Static Variables and Enums

    // The types of application metadata information
    public enum ApplicationMetaInfo { Company, Copyright, Version, Revision, Configuration, ReleaseType };

    // The Workbench Context
    private static WbContext wbContext = null;

    // The GRT Manager
    private static GrtManager grtManager = null;

    // Timer to guarantee idle handler is called periodically when nothing happens
    private static System.Windows.Forms.Timer timer = null;

    // Flag to collect garbage
    private static bool gcRequested = false;
    public static void CollectGarbageOnIdle()
    {
      gcRequested = true;
    }

    // Used to synchronize with for messages triggered by background threads.
    private static MainForm mainForm = null;

    #endregion
    
    /// <summary>
    /// The main entry point for the application.
    /// </summary>
    
    [STAThread]
    static void Main(string[] Args)
    {
      // Start with command line parsing.
      Arguments CommandLine = new Arguments();

      CommandLine.AddOptionName("-verbose", true);
      CommandLine.AddOptionName("-v", true);
      CommandLine.AddOptionName("-help", true);
      CommandLine.AddOptionName("-h", true);
      CommandLine.AddOptionName("-open", false);
      CommandLine.AddOptionName("-nologo", true);
      CommandLine.AddOptionName("-version", true);
      CommandLine.AddOptionName("-grtversion", true);
      CommandLine.AddOptionName("-swrendering", false);
      CommandLine.AddOptionName("-log", true);
      CommandLine.AddOptionName("-query", false);
      CommandLine.AddOptionName("-model", false);
      CommandLine.AddOptionName("-admin", false);
      CommandLine.AddOptionName("-script", false);
      CommandLine.AddOptionName("-run", false);
      CommandLine.AddOptionName("-run-python", false);
      CommandLine.AddOptionName("-run-lua", false);
      CommandLine.AddOptionName("-quit-when-done", true);
      CommandLine.AddOptionName("-log-level", false);

      CommandLine.Parse(Args);

      // Command line handling
      bool exitApplication = ProcessCommandLineArguments(CommandLine);

      // Initialize the logger. Enabled log levels are by default error, warning and info
      // (+ debug1 and 2 in debug builds).
      string userDir = System.IO.Path.Combine(System.IO.Path.Combine(
        Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData),
        "MySQL"), "Workbench");
      Logger.InitLogger(userDir);
#if DEBUG
      string logLevel = "debug2";
#else
      string logLevel = "info";
#endif
      string environmentLevel = Environment.GetEnvironmentVariable("WB_LOG_LEVEL");
      if (environmentLevel != null)
        logLevel = environmentLevel;
      if (CommandLine["log-level"] != null)
        logLevel = CommandLine["log-level"];

      Logger.ActiveLevel = logLevel;
      PrintInitialLogInfo();

      if (exitApplication)
      {
        Logger.LogInfo("Workbench", "Command line params told us to shut down.\n");
        return;
      }

      Application.EnableVisualStyles();
      Application.SetCompatibleTextRenderingDefault(false);

      // Hook into the exception handling to establish our own handling.
      AppDomain currentDomain = AppDomain.CurrentDomain; // CLR
      currentDomain.UnhandledException += new UnhandledExceptionEventHandler(OnUnhandledException);

      Application.ThreadException += // Windows Forms
         new System.Threading.ThreadExceptionEventHandler(OnGuiUnhandledException);

      // Read some early values which cannot be stored in the preferences (since they are loaded
      // later) from registry.
      bool singleInstance = false;
      string lastVersion = "";
      string currentVersion = GetApplicationMetaInfo(ApplicationMetaInfo.Version);
      Logger.LogInfo("Workbench", "Current version given by meta info is: " + currentVersion + '\n');
      RegistryKey wbKey = Registry.CurrentUser;
      try
      {
        wbKey = wbKey.OpenSubKey(@"Software\Oracle\MySQL Workbench", false);
        if (wbKey != null)
        {
          singleInstance = wbKey.GetValue("SingleInstance", 0).ToString() == "1";
          lastVersion = wbKey.GetValue("LastStartedAs", "").ToString();
        }
        else
          Registry.CurrentUser.CreateSubKey(@"Software\Oracle\MySQL Workbench");
      }
      catch (Exception e)
      {
        Logger.LogError("Workbench", "Error while checking single instance reg key: " + e.Message + '\n');
      }
      finally
      {
        if (wbKey != null)
          wbKey.Close();
      }

      // First check if this is the first instance of Workbench (if enabled).
      // The setting for single-instance is stored in the registry as it is Windows-only
      // and loading of the application settings happens later.
      if (singleInstance)
      {
        if (!ApplicationInstanceManager.CreateSingleInstance(
          Assembly.GetExecutingAssembly().GetName().Name, Args, SingleInstanceCallback))
        {
          Logger.LogInfo("Workbench", "Exiting as another instance of WB is already running.\n");
          return;
        }
      }

      // Give the main thread a proper name, so we can later check for it when needed.
      Thread.CurrentThread.Name = "mainthread";

      // Get the base directory of the executable and change the working dir to it
      // TODO: the working dir should be untouched and the app should work
      // from any basedir
      System.Reflection.Assembly asm = System.Reflection.Assembly.GetEntryAssembly();
      string baseDir = System.IO.Path.GetDirectoryName(asm.Location);
      string workdir = System.IO.Directory.GetCurrentDirectory();
      System.IO.Directory.SetCurrentDirectory(baseDir);

      // Next check if this is the first start of a new version of WB. In this case remove all
      // compiled python files. They will be automatically recreated and can produce problems
      // under certain circumstances.
      if (currentVersion != lastVersion)
      {
        Logger.LogInfo("Workbench", "This is the first start of a new version. Doing some clean up.\n");
        List<string> failed = new List<string>();
        RemoveCompiledPythonFiles(baseDir, failed);

        // TODO: decide if we wanna ask the user to remove those files manually or just ignore them.
      }

      // Some people don't have c:\windows\system32 in PATH, so we need to set it here
      // for WBA to find the needed commands
      String systemFolder = Environment.GetFolderPath(Environment.SpecialFolder.System);
      String cleanedPath = Environment.GetEnvironmentVariable("PATH");

      String []paths= cleanedPath.Split(new char[]{';'});
      cleanedPath = "";
      // Strip all python related dirs from PATH to avoid conflicts with other Python installations.
      foreach (String path in paths)
      {
        if (!path.ToLower().Contains("python"))
          cleanedPath = cleanedPath+";"+path;
      }
      Environment.SetEnvironmentVariable("PATH", systemFolder + cleanedPath);
      Logger.LogInfo("Workbench", "Setting PATH to: " + systemFolder + cleanedPath + '\n');

      // Clear PYTHONPATH environment variable, as we do not need it but our python impl
      // seriously gets confused with it.
      Environment.SetEnvironmentVariable("PYTHONPATH", workdir + "\\python\\Lib;" + workdir + "\\python\\DLLs;" + workdir + "\\python");
      Environment.SetEnvironmentVariable("PYTHONHOME", workdir + "\\python");

      // Initialize forms stuff.
      MySQL.Forms.Manager formsManager = new MySQL.Forms.Manager();

      // Connect application to console to have proper output there if requested.
      Win32.RedirectConsole();

      #region Runtime path check

      // Currently WB has trouble running from a path containing non-ASCII characters.
      // Actually, our third party libraries have (namely lua, python, ctemplate), 
      // as they don't consider Unicode file names (encoded as UTF-8) which leads to file-not-found
      // errors. Refuse to work in such a path for now.
      foreach (Char c in baseDir)
        if (c > 0x7f)
        {
          MessageBox.Show("MySQL Workbench cannot be executed from a path that contains non-ASCII characters.\n"+
            "This problem is imposed by used third-party libraries.\n" +
            "Please run this application from the default installation path or at least a path which is all ASCII characters.",
            "MySQL Workbench Execution Problem", MessageBoxButtons.OK, MessageBoxIcon.Error);
          return;
        }

      #endregion

      #region Release check (outdated beta or rc version)

      // check the date of the executable and suggest to install a new version if this is a beta or rc
      if (GetApplicationMetaInfo(ApplicationMetaInfo.Configuration).ToUpper().IndexOf("BETA") >= 0 ||
        GetApplicationMetaInfo(ApplicationMetaInfo.Configuration).ToUpper().IndexOf("RC") >= 0)
      {
        DateTime fileDate = System.IO.File.GetCreationTime(Application.ExecutablePath);

        if (DateTime.Now.Subtract(fileDate).TotalDays > 45)
        {
          Logger.LogInfo("Workbench", "Found an old WB pre release. Showing warning.\n");
          if (MessageBox.Show("This version of MySQL Workbench is older than 45 days and most probably outdated. "
            + Environment.NewLine
            + "It is recommended to upgrade to a newer version if available. "
            + Environment.NewLine
            + "Press [OK] to check for a new version and exit the application. "
            + "Press [Cancel] to continue using this version.",
            "MySQL Worbench Version Outdated", MessageBoxButtons.OKCancel,
            MessageBoxIcon.Warning, MessageBoxDefaultButton.Button1) == DialogResult.OK)
          {
            CheckForNewVersion();
            return;
          }
        }
      }

      #endregion

      #region Variables and Splashscreen

      // Show splash screen if not turned off
      if (CommandLine["nologo"] == null)
        SplashScreen.Show(typeof(WorkbenchSplashScreen));

      #endregion

      #region Initialize GRT

      // Try to instantiate the Workbench context and the GRT Manager and catch exceptions
      try
      {
        // Create Workbench Context
        wbContext = new WbContext((CommandLine["verbose"] != null || CommandLine["v"] != null));

        if (wbContext != null)
        {
          // Create the GRT Manager instance
          grtManager = wbContext.get_grt_manager();
        }
      }
      catch (Exception ex)
      {
        HandleException(ex);
      }

      #endregion

      #region Initialize Callbacks and Mainform

      // If the Workbench Context and GRT Manager were successfully created, 
      // initialize the application
      if (wbContext != null && grtManager != null)
      {
        mainForm = new MainForm(wbContext);

        // Initialize the Workbench context
        AppImpl formsApplication = new AppImpl(
          formsManager,
          new AppImpl.StrStrStrStrDelegate(mainForm.ShowFileDialog),
          new AppImpl.VoidStrDelegate(mainForm.ShowStatusText),
          new AppImpl.BoolStrStrFloatDelegate(mainForm.ShowProgress),
          new AppImpl.BoolStrIntStrPtrDelegate(mainForm.RequestInput),
          new AppImpl.CanvasViewStringStringIntPtrDelegate(mainForm.CreateNewDiagram),
          new AppImpl.VoidCanvasViewDelegate(mainForm.DestroyView),
          new AppImpl.VoidCanvasViewDelegate(mainForm.SwitchedView),
          new AppImpl.VoidCanvasViewDelegate(mainForm.ToolChanged),
          new AppImpl.IntPtrGRTManagerModuleStrStrGrtListFlagsDelegate(mainForm.OpenPlugin),
          new AppImpl.VoidIntPtrDelegate(mainForm.ShowPlugin),
          new AppImpl.VoidIntPtrDelegate(mainForm.HidePlugin),
          new AppImpl.VoidRefreshTypeStringIntPtrDelegate(mainForm.RefreshGUI),
          new AppImpl.VoidBoolDelegate(mainForm.LockGUI),
          new AppImpl.VoidStrDelegate(mainForm.PerformCommand),
          new AppImpl.VoidDelegate(mainForm.QuitApplication),
          new AppImpl.StringAppCommandAppViewStringDelegate(mainForm.ApplicationCommand));


        string openAtStartupType = "";
        string openAtStartup = null;

        if (CommandLine["open"] != null)
        {
          openAtStartup = CommandLine["open"];
          openAtStartupType = "";
        }
        else if (CommandLine["model"] != null)
        {
          openAtStartup = CommandLine["model"];
          openAtStartupType = "model";
        }
        else if (CommandLine["query"] != null)
        {
          openAtStartup = CommandLine["query"];
          openAtStartupType = "query";
        }
        else if (CommandLine["admin"] != null)
        {
          openAtStartup = CommandLine["admin"];
          openAtStartupType = "admin";
        }
        else if (CommandLine["script"] != null)
        {
          openAtStartup = CommandLine["script"];
          openAtStartupType = "script";
        }

        if (openAtStartup == null)
        {
          openAtStartup = "";
          if (CommandLine.Leftovers.Count > 0)
            openAtStartup = CommandLine.Leftovers[CommandLine.Leftovers.Count-1];
        }
        if (!string.IsNullOrEmpty(openAtStartup) && (openAtStartupType == "" || openAtStartupType == "model" || openAtStartupType == "script"))
        {
          if (!System.IO.Path.IsPathRooted(openAtStartup))
            openAtStartup = System.IO.Path.Combine(workdir, openAtStartup);

          // Since manually assigning an application to a file type does not automatically add
          // our self defined keywords (like script) to the command line we better check what
          // file type we got to open, based on the extension.
          if (string.IsNullOrEmpty(openAtStartupType))
          {
            switch (Path.GetExtension(openAtStartup).ToLowerInvariant())
            {
              case ".sql":
              case ".txt":
              case ".bak":
                openAtStartupType = "query";
                break;
            }
          }
        }

        String runScript = "";
        String runScriptLanguage = "";
        if (CommandLine["run"] != null)
          runScript = CommandLine["run"];
        else if (CommandLine["run-python"] != null)
        {
          runScript = CommandLine["run-python"];
          runScriptLanguage = "python";
        }
        else if (CommandLine["run-lua"] != null)
        {
          runScript = CommandLine["run-lua"];
          runScriptLanguage = "lua";
        }

        WbOptions wbOptions = new WbOptions(baseDir, // basedir
          baseDir, // plugin
          "",       // struct
          baseDir + "/modules", // modules
          baseDir, // library
          userDir,
          openAtStartup,
          openAtStartupType,
          runScript, runScriptLanguage,
          (CommandLine["swrendering"] != null),
          (CommandLine["openglrendering"] != null),
          (CommandLine["quit-when-done"] != null || CommandLine["quitwhendone"] != null));

        // TODO: check return value and show error message.
        // Currently the return value is always true. In case of an error an exception is raised.
        // That should change.
        wbContext.init(formsApplication, wbOptions,
          new WbContext.VoidStrUIFormDelegate(mainForm.CreateMainFormView)
        );

        // Set the Application.Idle event handler
        Application.Idle += new EventHandler(OnApplicationIdle);

        // Don't call the idle handler too often.
        timer = new System.Windows.Forms.Timer();
        timer.Interval = 100;
        timer.Tick += new EventHandler(timer_Tick);
        timer.Start();

        // Trigger GRT idle tasks
        grtManager.perform_idle_tasks();
        
        // Setup Menus
        Logger.LogInfo("Workbench", "Setting up main UI\n");
        mainForm.SetupUI();
        mainForm.Show();
        Logger.LogInfo("Workbench", "UI is up\n");

        // Tell the backend our main UI is ready. This will also load a model if it was given via command line
        // and opens the overview form for it.
        wbContext.finished_loading(wbOptions);

        // Right before we go to work and everything was loaded write the current version to registry
        // to allow us later to find out if we ran a new version the first time.
        try
        {
          wbKey = Registry.CurrentUser.OpenSubKey(@"Software\Oracle\MySQL Workbench", true);
          if (wbKey != null)
            wbKey.SetValue("LastStartedAs", currentVersion);
        }
        catch (Exception e)
        {
          Logger.LogError("Workbench", "Couldn't write regkey LastStartedAs: " + e.Message + '\n');
        }
        finally
        {
          if (wbKey != null)
            wbKey.Close();
        }

        // Start the Application if we are not already shutting down.
        if (!wbContext.is_quitting())
        {
          try
          {
            Logger.LogInfo("Workbench", "Running the application\n");
            Application.Run(new MySQLApplicationContext(mainForm));
          }
          catch (Exception e)
          {
            HandleException(e);
          }
        }

        Logger.LogInfo("Workbench", "Shutting down Workbench\n");

        // shutdown wb context
        if (wbContext != null)
        {
          while (wbContext.is_busy())
            wbContext.flush_idle_tasks();

          wbContext.finalize();
          wbContext.Dispose();
        }
        formsApplication.Dispose();
        GC.Collect();
      }
      
      #endregion

      Win32.ReleaseConsole();

      Logger.LogInfo("Workbench", "Done\n");
    }

    /// <summary>
    /// Prints some general info to the log file.
    /// </summary>
    private static void PrintInitialLogInfo()
    {
      Logger.LogInfo("Workbench", "Starting up Workbench\n");
      Logger.LogInfo("Workbench", string.Format("Current environment:\n\tCommand line: {0}\n\tCurrentDirectory: {1}\n" + 
        "\tHasShutdownStarted: {2}\n\tOSVersion: {3}\n\tSystemDirectory: {4}\n" +
        "\tTickCount: {5}\n\tUserInteractive: {6}\n\tVersion: {7}\n\tWorkingSet: {8}\n",
        Environment.CommandLine, Environment.CurrentDirectory, Environment.HasShutdownStarted,
        Environment.OSVersion.ToString(), Environment.SystemDirectory,
        Environment.TickCount, Environment.UserInteractive,
        Environment.Version.ToString(), Environment.WorkingSet));
      
      IDictionary environmentVariables = Environment.GetEnvironmentVariables();
      string variables = "";
      foreach (DictionaryEntry entry in environmentVariables)
        variables += string.Format("\t{0} = {1}\n", entry.Key, entry.Value);
      Logger.LogInfo("Workbench", "Environment variables:\n" + variables);
    }
    
    #region Application handlers

    static void timer_Tick(object sender, EventArgs e)
    {
      System.Windows.Forms.Timer timer = sender as System.Windows.Forms.Timer;

      // flush_idle_tasks needs to be reentrant so that functions
      // that crosses threads have their own idle tasks executed
      if (wbContext != null)
      {
        try
        {
          wbContext.flush_idle_tasks();
        }
        catch (Exception exception)
        {
          // temporarily stop the timer so that we don't get into a loop from showing the msgbox
          timer.Stop();
          HandleException(exception);
          timer.Start();
        }
      }
    }

    private static void OnApplicationIdle(object sender, EventArgs e)
    {
      if (gcRequested)
      {
        gcRequested = false;
        GC.Collect();
      }
    }

    /// <summary>
    /// Handler that gets called if this another instance of the application was started. It passes
    /// the command line arguments from the other instance on so we can act accordingly here.
    /// </summary>
    /// <param name="sender">The sender.</param>
    /// <param name="args">The instance containing the event data.</param>
    private static void SingleInstanceCallback(object sender, InstanceCallbackEventArgs args)
    {
      if (args == null || mainForm == null)
        return;

      Action<bool> d = (bool x) =>
      {
        mainForm.Activate(x);

        // Parse command line and extract the model file name if there is one.
        Arguments CommandLine = new Arguments();

        CommandLine.AddOptionName("-verbose", true);
        CommandLine.AddOptionName("-v", true);
        CommandLine.AddOptionName("-help", true);
        CommandLine.AddOptionName("-h", true);
        CommandLine.AddOptionName("-open", false);
        CommandLine.AddOptionName("-nologo", true);
        CommandLine.AddOptionName("-version", true);
        CommandLine.AddOptionName("-grtversion", true);
        CommandLine.AddOptionName("-swrendering", false);
        CommandLine.AddOptionName("-log", true);
        CommandLine.AddOptionName("-query", false);
        CommandLine.AddOptionName("-model", false);
        CommandLine.AddOptionName("-admin", false);
        CommandLine.AddOptionName("-script", false);
        CommandLine.AddOptionName("-run", false);
        CommandLine.AddOptionName("-run-python", false);
        CommandLine.AddOptionName("-run-lua", false);
        CommandLine.AddOptionName("-quit-when-done", true);

        CommandLine.Parse(args.CommandLineArgs);

        string fileType = "";
        string fileToOpen = null;

        if (CommandLine["open"] != null)
        {
          fileToOpen = CommandLine["open"];
          fileType = "";
        }
        else if (CommandLine["model"] != null)
        {
          fileToOpen = CommandLine["model"];
          fileType = "model";
        }

        if (fileToOpen == null)
        {
          fileToOpen = "";
          if (CommandLine.Leftovers.Count > 0)
            fileToOpen = CommandLine.Leftovers[CommandLine.Leftovers.Count - 1];
        }
        if (fileToOpen != null && fileToOpen != "" && (fileType == "" || fileType == "model"))
        {
          if (!System.IO.Path.IsPathRooted(fileToOpen))
            fileToOpen = System.IO.Path.Combine(System.IO.Directory.GetCurrentDirectory(), fileToOpen);
        }

        if (!string.IsNullOrEmpty(fileToOpen) && !wbContext.get_filename().Equals(fileToOpen))
          wbContext.open_document(fileToOpen);
      };
      mainForm.Invoke(d, true);
    }

    // CLR unhandled exception.
    private static void OnUnhandledException(Object sender, UnhandledExceptionEventArgs e)
    {
      HandleException(e.ExceptionObject);
    }

    // Windows Forms unhandled exception.
    private static void OnGuiUnhandledException(Object sender, ThreadExceptionEventArgs e)
    {
      HandleException(e.Exception);
    }

    #endregion

    #region Message Callbacks

    private static int ShowStringInputDialog(String Title, String Prompt,
      ref String InputString)
    {
      DialogResult result;
      result = StringInputForm.ShowModal(Title, "", Prompt, ref InputString);
      if (result == DialogResult.OK)
        return 1;
      else
        return 0;
    }

    #endregion
    
    #region Various Static Functions

    /// <summary>
    /// Retrieves application metadata and returns it as a string
    /// </summary>
    /// <param name="kind">The type of metadata information to return</param>
    /// <returns></returns>
    static public string GetApplicationMetaInfo(ApplicationMetaInfo kind)
    {
      object[] attributes;
      Version version;

      Assembly assembly = Assembly.GetExecutingAssembly();

      string value = "";

      switch (kind)
      {
        case ApplicationMetaInfo.Company:
          attributes = assembly.GetCustomAttributes(typeof(AssemblyCompanyAttribute), false);
          if (attributes.Length > 0)
            value = (attributes[0] as AssemblyCompanyAttribute).Company;
          break;

        case ApplicationMetaInfo.Copyright:
          attributes = assembly.GetCustomAttributes(typeof(AssemblyCopyrightAttribute), false);
          if (attributes.Length > 0)
            value = (attributes[0] as AssemblyCopyrightAttribute).Copyright;
          break;

        case ApplicationMetaInfo.Configuration:
          attributes = assembly.GetCustomAttributes(typeof(AssemblyConfigurationAttribute), false);
          if (attributes.Length > 0)
            value = (attributes[0] as AssemblyConfigurationAttribute).Configuration;
          break;

        case ApplicationMetaInfo.Version:
          version = new Version(Application.ProductVersion);
          value = string.Format("{0}.{1}.{2}",
            version.Major, version.Minor, version.Build);
          break;

        case ApplicationMetaInfo.Revision:
          version = new Version(Application.ProductVersion);
          value = string.Format("{0}", version.MinorRevision);
          break;
        case ApplicationMetaInfo.ReleaseType:
          attributes = assembly.GetCustomAttributes(typeof(AssemblyReleaseTypeAttribute), false);
          if (attributes.Length > 0)
            value = (attributes[0] as AssemblyReleaseTypeAttribute).ReleaseType;
          break;
      }

      return value;
    }

    /// <summary>
    /// Deals with the given command line arguments
    /// </summary>
    /// <param name="CommandLine">The command line arguments passed to the application</param>
    /// <returns></returns>
    static bool ProcessCommandLineArguments(Arguments CommandLine)
    {
      if (CommandLine["help"] != null || CommandLine["h"] != null)
      {
        Console.WriteLine("MySQL Workbench {0} {1}. " 
          + "(C) {2} by {3}. All rights reserved." + Environment.NewLine,
          GetApplicationMetaInfo(ApplicationMetaInfo.Version), 
          GetApplicationMetaInfo(ApplicationMetaInfo.Configuration),
          GetApplicationMetaInfo(ApplicationMetaInfo.Copyright),
          GetApplicationMetaInfo(ApplicationMetaInfo.Company));
        Console.WriteLine("Usage: MySQLWorkbench [options] [model file]" + Environment.NewLine);
        CommandLine.AddOptionName("-query", false);
        CommandLine.AddOptionName("-model", false);
        CommandLine.AddOptionName("-admin", false);
        CommandLine.AddOptionName("-script", false);
        CommandLine.AddOptionName("-run", false);
        CommandLine.AddOptionName("-run-python", false);
        CommandLine.AddOptionName("-run-lua", false);
        CommandLine.AddOptionName("-quit-when-done", true);

        Console.WriteLine("Options" + Environment.NewLine
          + "  -admin instance .... Open an admin tab to the named server instance at startup" + Environment.NewLine
          + "  -open filename ..... Open the given filename at startup" + Environment.NewLine
          + "  -query server ...... Open a DB query tab to the named server connection at startup" + Environment.NewLine
          + "  -run script ........ Executes the given Workbench script at startup"  + Environment.NewLine
          + "  -run-python script . Executes the given Workbench Python script at startup"  + Environment.NewLine
          + "  -run-lua script .... Executes the given Workbench Lua script at startup"  + Environment.NewLine
          + "  -script scriptfile . Executes the given Workbench script file at startup"  + Environment.NewLine
          + "  -quit-when-done .... Quits Workbench once the given script finishes executing"  + Environment.NewLine
          + "  -swrendering ....... Force the canvas to use software rendering instead of OpenGL" + Environment.NewLine
          + "  -nologo ............ Do not display the splash screen" + Environment.NewLine
          + "  -log-level ......... Determines the amount of logging (error, warning, info, debug1/2/3)" + Environment.NewLine
          + "  -verbose (-v) ...... Print verbose output in the GRT Shell" + Environment.NewLine
          + "  -version ........... Print the version information" + Environment.NewLine
          + "  -grtversion ........ Print the GRT version information" + Environment.NewLine
          + "  -help (-h) ......... Print this output" + Environment.NewLine
        );

        return true;
      }
      else
        if (CommandLine["version"] != null)
        {
          Console.WriteLine("MySQL Workbench {0}.{1} {2}",
            GetApplicationMetaInfo(ApplicationMetaInfo.Version), 
            GetApplicationMetaInfo(ApplicationMetaInfo.Revision),
            GetApplicationMetaInfo(ApplicationMetaInfo.Configuration));
          return true;
        }
        else
          if (CommandLine["grtversion"] != null)
          {
            Console.WriteLine("GRT Environment {0}", GRT.version());
            return true;
          }

      return false;
    }

    static public bool CheckForNewVersion()
    {
      try
      {
        System.Diagnostics.Process.Start("http://wb.mysql.com/workbench/version-check.php?config="
          + GetApplicationMetaInfo(ApplicationMetaInfo.Configuration).Replace(' ', '_')
          + "&version="
          + GetApplicationMetaInfo(ApplicationMetaInfo.Version)
          + "&revision="
          + GetApplicationMetaInfo(ApplicationMetaInfo.Revision));
      }
      catch (Exception e)
      {
        MessageBox.Show(e.Message.ToString(), "Error Opening Browser", 
          MessageBoxButtons.OK, MessageBoxIcon.Error, MessageBoxDefaultButton.Button1);
      }

      return true;
    }

    public static void HandleException(Object o)
    {
      Exception e = o as Exception;
      String message;
      String info;

      if (e != null)
      { // Report System.Exception info
        message = e.Message;
        info = "Exception = " + e.GetType() + "\n";
        info += "Message = " + e.Message + "\n";
        info += "FullText = " + e.ToString();
      }
      else
      { // Report exception Object info
        message = "Exception = " + o.GetType();
        info = "Exception = " + o.GetType() + "\n";
        info += "FullText = " + o.ToString();
      }

      Logger.LogError("Workbench", message + "\n" + info + '\n');

      // Check for blocked files (Windows "security" feature).
      if (info.Contains("0x80131515"))
      {
        MessageBox.Show("A blocked library could not be loaded. You probably downloaded MySQL Workbench " +
          "as no-installation zip package. Windows has locked this, preventing so its full operation.\n\n" +
          "When you click \"OK\" MySQL Workbench will try to automatically unblock all its files. Restart the application " +
          "for this change to take effect!\n\nThis operation might fail " +
          "e.g. on read-only volumes or for other reasons. If that is the case then manually unblock the zip package " +
          "(see its Properties in Explorer) and unzip it again.",
          "Blocked DLL Detected", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);

        Win32Api.UnblockWorkbenchFiles(System.IO.Directory.GetCurrentDirectory());
      }
      else
        ExceptionDialog.Show(message, info, wbContext);
    }

    /// <summary>
    /// Removes all pyc files from the given dir and its sub dirs.
    /// </summary>
    private static void RemoveCompiledPythonFiles(string folder, List<string> failed)
    {
      foreach (string file in Directory.GetFiles(folder, "*.pyc"))
        try
        {
          File.Delete(file);
        }
        catch (Exception)
        {
          failed.Add(file);
        }

      foreach (string subfolder in Directory.GetDirectories(folder))
        RemoveCompiledPythonFiles(subfolder, failed);
    }

    #endregion

  }
}
