/* 
 * Copyright (c) 2009, 2012, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#import "MFPanel.h"
#import "NSColor_extras.h"

#import "MFMForms.h"

#import "MFRadioButton.h" // for hadnling radio groups
#import "MFContainerBase.h"

@interface MFPanelContent : NSView
{
  MFPanelImpl *panel;
  float mLeftPadding;
  float mRightPadding;
  float mTopPadding;
  float mBottomPadding;
  float mBasePadding;
}
@end


@implementation MFPanelContent

- (id)initWithPanel:(MFPanelImpl*)aPanel
{
  NSRect frame= [aPanel frame];
  frame.origin= NSMakePoint(0, 0);
  self= [super initWithFrame:frame];
  if (self)
  {
    panel= aPanel;
    mBasePadding = 4;
  }
  return self;
}

/*
- (void)drawRect:(NSRect)frame{
  [[NSColor magentaColor] set];
  NSFrameRect(NSInsetRect([self frame], 2, 1));
}
*/

- (BOOL)isFlipped
{
  return YES;
}


- (void)setPaddingLeft:(float)lpad right:(float)rpad top:(float)tpad bottom:(float)bpad
{
  mLeftPadding = lpad;
  mRightPadding = rpad;
  mTopPadding = tpad;
  mBottomPadding = bpad;
}

- (void)setBasePadding:(float)pad
{
  mBasePadding= pad;
}


- (void)subviewMinimumSizeChanged
{
  NSSize minSize= [[[self subviews] lastObject] minimumSize];
  NSSize size= [self frame].size;
  
  // size of some subview has changed, we check if our current size is enough
  // to fit it and if not, request forward the size change notification to superview
  
  if (minSize.width != size.width || minSize.height != size.height)
    [panel subviewMinimumSizeChanged];
  else
    [[[self subviews] lastObject] setFrameSize: size];    
}


- (void)resizeSubviewsWithOldSize:(NSSize)oldBoundsSize
{
  id content= [[self subviews] lastObject];
  NSSize size= [self frame].size;
  size.width -= 2*mBasePadding + (mLeftPadding + mRightPadding);
  size.height -= 2*mBasePadding + (mTopPadding + mBottomPadding);
  // size.height= [content minimumSize].height;
  [content setFrame: NSMakeRect(mLeftPadding+mBasePadding, mTopPadding+mBasePadding, size.width, size.height)];
}


- (NSSize)minimumSize
{
  NSSize size= [[[self subviews] lastObject] minimumSize];
  size.width+= 2 * mBasePadding + mLeftPadding + mRightPadding;
  size.height+= 2 * mBasePadding + mTopPadding + mBottomPadding;
  return size;
}


@end




@implementation MFPanelImpl

- (id)initWithObject:(::mforms::Panel*)aPanel type:(::mforms::PanelType)type
{
  self= [super initWithFrame:NSMakeRect(10, 10, 10, 10)];
  if (self)
  {
    NSRect frame;
    NSRect content= NSMakeRect(10, 10, 10, 10);
    float basePadding = 0;

    mOwner= aPanel;
    mOwner->set_data(self);
    mType = type;
    switch (type)
    {
      case mforms::TransparentPanel: // just a container with no background
        [self setTransparent: YES];
        [self setTitlePosition: NSNoTitle];
        break;
      case mforms::FilledPanel:      // just a container with color filled background
        [self setTransparent: NO];
        [self setBorderType: NSNoBorder];
        [self setTitlePosition: NSNoTitle];
        [self setBoxType: NSBoxCustom];
        break;
      case mforms::BorderedPanel:    // container with native border
        [self setBorderType: NSBezelBorder];
        [self setTitlePosition: NSNoTitle];
        basePadding = 4;
        break;
      case mforms::LineBorderPanel:  // container with a solid line border
        [self setBorderType: NSLineBorder];
        [self setTitlePosition: NSNoTitle];
        [self setBoxType: NSBoxCustom];
        basePadding = 2;
        break;
      case mforms::TitledBoxPanel:   // native grouping box with a title with border
        [self setBorderType: NSBezelBorder];
        basePadding = 4;
        break;
      case mforms::TitledGroupPanel: // native grouping container with a title (may have no border) 
        [self setBorderType: NSNoBorder];
        basePadding = 4;
        break;
      case mforms::CheckBoxTitlePanel:// container with checkbox
        [self setBorderType: NSBezelBorder];        
        mCheckButton= [[NSButtonCell alloc] initTextCell:@""];
        [mCheckButton setControlSize: NSSmallControlSize];
        [mCheckButton setFont: [NSFont boldSystemFontOfSize: [NSFont systemFontSizeForControlSize: NSSmallControlSize]]];
        [mCheckButton setButtonType: NSSwitchButton];
        [mCheckButton setAction: @selector(toggleCheck:)];
        [mCheckButton setTarget: self];
        [super setTitle: @""];
        basePadding = 4;
        break;
      case mforms::StyledHeaderPanel: 
        [self setBorderType: NSNoBorder];
        [self setTitlePosition: NSNoTitle];
        [self setBoxType: NSBoxCustom];
        [self setTransparent: YES];
        break;
    }
    
    [self setContentViewMargins: NSMakeSize(0, 0)];
    [self setFrameFromContentFrame: content];
    frame= [self frame];
    // calculate the offsets the NSBox adds to the contentView
    mTopLeftOffset.x= NSMinX(content) - NSMinX(frame);
    mTopLeftOffset.y= NSMinY(content) - NSMinY(frame);
    mBottomRightOffset.x= NSMaxX(frame) - NSMaxX(content);
    mBottomRightOffset.y= MAX(NSMaxY(frame) - NSMaxY(content), [mCheckButton cellSize].height);

    [super setContentView: [[[MFPanelContent alloc] initWithPanel: self] autorelease]];
    [[super contentView] setBasePadding: basePadding];
  }
  return self;
}


- (void) dealloc
{
  [mCheckButton release];
  [super dealloc];
}

- (NSRect)titleRect
{
  NSRect rect;
  rect= [super titleRect];
  if (mCheckButton)
  {
    rect.origin.y-= 3;
    rect.size= [mCheckButton cellSize];
    rect.size.width+= 4;
  }
  return rect;
}


- (void)toggleCheck:(id)sender
{
  mOwner->check_toggled();
}


- (void)mouseDown:(NSEvent*)event
{
  if (mCheckButton)
  {
    [mCheckButton trackMouse:event inRect:[self titleRect]
                      ofView:self
                untilMouseUp:NO];
    [self setNeedsDisplay:YES];
  }
}


- (void)mouseUp:(NSEvent*)event
{
  if (mCheckButton)
  {
    [self setNeedsDisplay:YES];
  }
}


- (void)drawRect:(NSRect)rect
{
  if (mType == mforms::StyledHeaderPanel)
  {
    NSRect bounds = [self bounds];
//    [[NSColor whiteColor] set];
//    NSRectFill(bounds);
    NSGradient *grad = [[NSGradient alloc] initWithStartingColor: [NSColor colorWithDeviceWhite:222/255.0 alpha:1.0]
                                                     endingColor: [NSColor colorWithDeviceWhite:1.0 alpha:1.0]];
    [grad drawInRect: NSMakeRect(NSMinX(bounds), NSMinY(bounds), NSWidth(bounds), NSHeight(bounds)) angle: 270];
    [[NSColor darkGrayColor] set];
    [NSBezierPath strokeLineFromPoint: NSMakePoint(NSMinX(bounds), NSMinY(bounds)) toPoint: NSMakePoint(NSMaxX(bounds), NSMinY(bounds))];
//    [NSBezierPath strokeLineFromPoint: NSMakePoint(NSMinX(bounds), NSMaxY(bounds)) toPoint: NSMakePoint(NSMaxX(bounds), NSMaxY(bounds))];
  }
  [super drawRect: rect];

  [mCheckButton drawWithFrame:[self titleRect] inView:self];
}


- (void)resizeSubviewsWithOldSize:(NSSize)oldBoundsSize
{
  NSRect bounds = [self bounds];
  bounds.size.width -= mTopLeftOffset.x + mBottomRightOffset.x;
  bounds.size.height -= mTopLeftOffset.y + mBottomRightOffset.y;
//  bounds.origin.x = mTopLeftOffset.x;
//  bounds.origin.y = mTopLeftOffset.y;
  [[self contentView] setFrame: bounds];
}

- (mforms::Object*)mformsObject
{
  return mOwner;
}


- (NSSize)minimumSize
{
  NSSize size= [[self contentView] minimumSize];

  size.width += mTopLeftOffset.x + mBottomRightOffset.x;
  size.height += mTopLeftOffset.y + mBottomRightOffset.y;
    
  return size;
}


- (void)subviewMinimumSizeChanged
{
  NSSize minSize= [self minimumSize];
  NSSize size= [self frame].size;
  
  // size of some subview has changed, we check if our current size is enough
  // to fit it and if not, request forward the size change notification to superview
  
  if (minSize.width != size.width || minSize.height != size.height)
  {
    if ([self superview])
    {
      [[self superview] subviewMinimumSizeChanged];
      return;
    }
    else
      [self setFrameSize: minSize];
  }
  [self resizeSubviewsWithOldSize:size];
}


- (void)setTag:(NSInteger)tag
{
  mTag= tag;
}


- (NSInteger)tag
{
  return mTag;
}


- (void)setPaddingLeft:(float)lpad right:(float)rpad top:(float)tpad bottom:(float)bpad
{
  [[self contentView] setPaddingLeft:lpad right:rpad top:tpad bottom:bpad];
}


- (void)setContentView:(NSView*)content
{
  if (content)
    [[self contentView] addSubview: content];
  else
    [[[[self contentView] subviews] lastObject] removeFromSuperview];
  [self subviewMinimumSizeChanged];
}


- (void)setTitle:(NSString*)title
{
  if (mCheckButton)
    [mCheckButton setTitle: title];
  else
    [super setTitle: title];
}

- (void)setEnabled:(BOOL)flag
{
  for (id view in [[self contentView] subviews])
  {
    if ([view respondsToSelector:@selector(setEnabled:)])
      [view setEnabled: flag];
  }
}


static bool panel_create(::mforms::Panel *self, ::mforms::PanelType type)
{
  [[[MFPanelImpl alloc] initWithObject:self type:type] autorelease];
    
  return true;
}


static void panel_set_title(::mforms::Panel *self, const std::string &text)
{
  if ( self )
  {
    MFPanelImpl* panel = self->get_data();
    
    if ( panel )
    {
      [panel setTitle:wrap_nsstring(text)];
    }
  }
}



static void panel_set_back_color(mforms::Panel *self, const std::string &color)
{
  if (self)
  {
    MFPanelImpl* panel = self->get_data();
    
    if ( panel && panel->mType != mforms::StyledHeaderPanel)
    {
      [panel setTransparent: NO];
      [panel setFillColor: [NSColor colorFromHexString: wrap_nsstring(color)]];
    }
  }
}



static void panel_set_active(mforms::Panel *self, bool active)
{
  if (self)
  {
    MFPanelImpl* panel = self->get_data();
    
    if ( panel )
    {
      [panel->mCheckButton setState: active ? NSOnState : NSOffState];
    }
  }
}


static bool panel_get_active(mforms::Panel *self)
{
  if (self)
  {
    MFPanelImpl* panel = self->get_data();
    
    if ( panel )
    {
      return [panel->mCheckButton state] == NSOnState;
    }
  }
  return false;
}

static void panel_add(mforms::Panel *self,mforms::View *view)
{
  if (self)
  {
    MFPanelImpl* panel = self->get_data();
    
    if ( panel )
    {
      [panel setContentView: view->get_data()];
    }
  }
}


static void panel_remove(mforms::Panel *self, mforms::View *child)
{
  if (self)
  {
    MFPanelImpl* panel = self->get_data();
    
    if ( panel )
    {
      [panel setContentView:nil];
    }
  }
}



void cf_panel_init()
{
  ::mforms::ControlFactory *f = ::mforms::ControlFactory::get_instance();
  
  f->_panel_impl.create= &panel_create;
  f->_panel_impl.set_title= &panel_set_title;
  f->_panel_impl.set_back_color= &panel_set_back_color;
  f->_panel_impl.set_title= &panel_set_title;
  
  f->_panel_impl.set_active= &panel_set_active;
  f->_panel_impl.get_active= &panel_get_active;
  
  f->_panel_impl.add= &panel_add;
  f->_panel_impl.remove= &panel_remove;
}


@end



