//
//  WBModelImageEditor.mm
//  MySQLWorkbench
//
//  Created by Alfredo Kojima on 19/Feb/09.
//  Copyright 2009 Sun Microsystems Inc. All rights reserved.
//

#import "WBModelImageEditor.h"
#import "MCPPUtilities.h"

@implementation ImageEditor

static void call_refresh(ImageEditor *self)
{
  [self refresh];
}


- (id)initWithModule:(grt::Module*)module GRTManager:(bec::GRTManager*)grtm arguments:(const grt::BaseListRef&)args
{
  self= [super initWithModule:module GRTManager:grtm arguments:args];
  if (self != nil)
  {
    // load GUI. Top level view in the nib is the NSTabView that will be docked to the main window
    if (![[NSBundle bundleForClass:[self class]] loadNibFile:@"WbModelImageEditor"
                                           externalNameTable:[NSDictionary dictionaryWithObject:self forKey:NSNibOwner]
                                                    withZone:nil])
      NSLog(@"Could not load WbModelImageEditor.xib");
    
    // take the minimum size of the view from the initial size in the nib.
    // Therefore the nib should be designed as small as possible
    // note: the honouring of the min size is not yet implemented
    [self setMinimumSize: [tabView frame].size];
    
    [self reinitWithArguments: args];
  }
  return self;
}


- (void)reinitWithArguments:(const grt::BaseListRef&)args
{
  [super reinitWithArguments: args];
  delete mBackEnd;
  
    // setup the editor backend with the image object (args[0])
  mBackEnd= new ImageEditorBE(_grtm, workbench_model_ImageFigureRef::cast_from(args[0]));
    
    // register a callback that will make [self refresh] get called
    // whenever the backend thinks its needed to refresh the UI from the backend data (ie, the
    // edited object was changed from somewhere else in the application)
    mBackEnd->set_refresh_ui_slot(boost::bind(call_refresh, self));
    
    // update the UI
    [self refresh];
  }


- (void) dealloc
{
  delete mBackEnd;
  [super dealloc];
}


/** Fetches object info from the backend and update the UI
 */
- (void)refresh
{
  if (mBackEnd)
  {
    NSImage *image= [[[NSImage alloc] initWithContentsOfFile: 
                      [NSString stringWithCPPString: mBackEnd->get_attached_image_path()]] autorelease];
    [imageView setImage: image];
    
    int w, h;
    mBackEnd->get_size(w, h);
    
    [widthField setIntegerValue: w];
    [heightField setIntegerValue: h];
    [keepAspectRatio setState: mBackEnd->get_keep_aspect_ratio() ? NSOnState : NSOffState];
  }
}


- (IBAction)setSize:(id)sender
{
  if (sender == widthField)
  {
    int w, h;
    mBackEnd->get_size(w, h);
    if (w != [widthField integerValue])
    {
      mBackEnd->set_width([widthField integerValue]);
      [self refresh];
    }
  }
  else if (sender == heightField)
  {
    int w, h;
    mBackEnd->get_size(w, h);
    if (h != [heightField integerValue])
    {
      mBackEnd->set_height([heightField integerValue]);
      [self refresh];
    }
  }
}


- (IBAction)toggleAspectRatio:(id)sender
{
  mBackEnd->set_keep_aspect_ratio([keepAspectRatio state] == NSOnState);
}


- (IBAction)resetSize:(id)sender
{
  NSSize size= [[imageView image] size];
  
  mBackEnd->set_size(size.width, size.height);
}


- (IBAction)browse:(id)sender
{
  NSOpenPanel *panel= [NSOpenPanel openPanel];
  
  [panel setAllowsMultipleSelection: NO];
  [panel setCanChooseFiles: YES];
  [panel setCanChooseDirectories: NO];
  
  [panel setTitle:@"Open Image"];
  if ([panel runModalForTypes:[NSArray arrayWithObject:@"png"]] == NSOKButton)
  {
    NSString *path= [panel filename];
    NSImage *image= [[[NSImage alloc] initWithContentsOfFile: path] autorelease];
    if (!image)
    {
      NSRunAlertPanel(NSLocalizedString(@"Invalid Image", @"cant load image"),
                      NSLocalizedString(@"Could not load the image.", @""),
                      NSLocalizedString(@"OK", @""), nil, nil, nil);
      return;
    }
    
    mBackEnd->set_filename([path UTF8String]);
    
    [self refresh];
  }
}


- (id)identifier
{
  // an identifier for this editor (just take the object id)
  return [NSString stringWithCPPString:mBackEnd->get_object().id()];
}


- (NSString*)title
{
  // the title for the editor
  return NSLocalizedString(@"Image", @"editor tab title");
}


- (NSView*)dockableView
{
  // the view to be docked to the main window
  return [tabView superview];
}


- (BOOL)matchesIdentifierForClosingEditor:(NSString*)identifier
{
  return mBackEnd->should_close_on_delete_of([identifier UTF8String]);
}

@end
