/* 
 * Copyright (c) 2011, 2012, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

using namespace MySQL;
using namespace MySQL::Forms;

//----------------- FileChooserImpl ----------------------------------------------------------------

FileChooserImpl::FileChooserImpl(mforms::FileChooser* self, mforms::Form* owner)
  : ViewImpl(self)
{
}

//-------------------------------------------------------------------------------------------------

FileChooserImpl::~FileChooserImpl()
{
}

//-------------------------------------------------------------------------------------------------

bool FileChooserImpl::create(mforms::FileChooser *self, mforms::Form *owner,
  mforms::FileChooserType type, bool show_hidden)
{
  // On Windows hiding/showing hidden files is a global switch in Explorer so we can just
  // ignore the show_hidden flag here. It is necessary for Linux/Mac.
  FileChooserImpl ^form= gcnew FileChooserImpl(self, owner);

  if (form != nullptr)
  {
    Windows::Forms::CommonDialog^ dialog;
    switch (type)
    {
    case ::mforms::OpenFile:
      dialog= ViewImpl::create_dialog<OpenFileDialog>(self, form);
      ((FileDialog^) dialog)->RestoreDirectory= true;
      break;
    case ::mforms::SaveFile:
      dialog= ViewImpl::create_dialog<SaveFileDialog>(self, form);
      ((FileDialog^) dialog)->RestoreDirectory= true;
      break;
    case ::mforms::OpenDirectory:
      dialog= ViewImpl::create_dialog<FolderBrowserDialog>(self, form);
      break;
    default:
      return false;
    }
    return true;
  }
  return false;
}

//-------------------------------------------------------------------------------------------------

void FileChooserImpl::set_title(mforms::FileChooser *self, const std::string& title)
{
  FileChooserImpl^ form= (FileChooserImpl^)ObjectImpl::FromUnmanaged(self);

  if (form != nullptr)
  {
    if (form->get_control<FileDialog>())
      form->get_control<FileDialog>()->Title= CppStringToNative(title);
    else
      form->get_control<FolderBrowserDialog>()->Description = CppStringToNative(title);
  }
}

//-------------------------------------------------------------------------------------------------

bool FileChooserImpl::run_modal(mforms::FileChooser *self)
{
  FileChooserImpl^ form= (FileChooserImpl^)ObjectImpl::FromUnmanaged(self);

  if (form != nullptr)
  {
    if (form->get_control<FileDialog>())
    {
      Windows::Forms::FileDialog ^f= form->get_control<FileDialog>();
      return f->ShowDialog() == DialogResult::OK;
    }else
    {
      Windows::Forms::FolderBrowserDialog ^f= form->get_control<FolderBrowserDialog>();
      return f->ShowDialog() == DialogResult::OK;
    }
  }
  return false;
}

//-------------------------------------------------------------------------------------------------

void FileChooserImpl::set_directory(mforms::FileChooser* self, const std::string& path)
{
  FileChooserImpl^ form= (FileChooserImpl^)ObjectImpl::FromUnmanaged(self);

  if (form != nullptr)
  {
    if (form->get_control<FileDialog>())
    {
      Windows::Forms::FileDialog ^f= form->get_control<FileDialog>();
      f->InitialDirectory= CppStringToNative(path);
    }else
    {
      Windows::Forms::FolderBrowserDialog ^f= form->get_control<FolderBrowserDialog>();
      f->SelectedPath = CppStringToNative(path);
    }
  }
}

//-------------------------------------------------------------------------------------------------

std::string FileChooserImpl::get_directory(mforms::FileChooser* self)
{
  FileChooserImpl^ form= (FileChooserImpl^) ObjectImpl::FromUnmanaged(self);

  if (form != nullptr)
  {
    Windows::Forms::FolderBrowserDialog ^f= form->get_control<FolderBrowserDialog>();
    if (f != nullptr)
      return NativeToCppString(f->SelectedPath);
    else
    {
      Windows::Forms::FileDialog ^f= form->get_control<FileDialog>();
      return NativeToCppString(System::IO::Path::GetDirectoryName(f->FileName));
    }
  }
  return "";
}

//-------------------------------------------------------------------------------------------------

std::string FileChooserImpl::get_path(mforms::FileChooser* self)
{
  FileChooserImpl^ form= (FileChooserImpl^)ObjectImpl::FromUnmanaged(self);

  if (form != nullptr)
  {
    if (form->get_control<FileDialog>())
    {
      Windows::Forms::FileDialog ^f= form->get_control<FileDialog>();
      return NativeToCppString(f->FileName);
    }else
    {
      Windows::Forms::FolderBrowserDialog ^f= form->get_control<FolderBrowserDialog>();
      return NativeToCppString(f->SelectedPath);
    }
  }
  return "";
}

//-------------------------------------------------------------------------------------------------

void FileChooserImpl::set_path(mforms::FileChooser* self, const std::string& path)
{
  FileChooserImpl^ form= (FileChooserImpl^)ObjectImpl::FromUnmanaged(self);

  if (form != nullptr)
  {
    if (form->get_control<FileDialog>())
    {
      Windows::Forms::FileDialog ^f= form->get_control<FileDialog>();
      f->InitialDirectory= CppStringToNative(base::dirname(path));
      f->FileName= CppStringToNative(base::basename(path));
    }else
    {
      Windows::Forms::FolderBrowserDialog ^f= form->get_control<FolderBrowserDialog>();
      f->SelectedPath = CppStringToNative(path);
    }
  }
}

//-------------------------------------------------------------------------------------------------

void FileChooserImpl::set_extensions(mforms::FileChooser* self, const std::string& extensions, 
  const std::string& default_extension)
{
  FileChooserImpl^ form= (FileChooserImpl^)ObjectImpl::FromUnmanaged(self);

  if (form != nullptr)
  {
    Windows::Forms::FileDialog ^f= form->get_control<FileDialog>();

    f->AddExtension= true;

    f->DefaultExt= CppStringToNative(default_extension);
    f->Filter= CppStringToNative(extensions);
  }
}

//-------------------------------------------------------------------------------------------------

void FileChooserImpl::add_selector_option(mforms::FileChooser* self, const std::string &name,
  const std::string &label, const mforms::FileChooser::StringPairVector& options)
{
  if (name == "format")
  {
    // The backend split the properly formatted string, we now have to concatenate the parts again
    // (and we lost the default extension).
    std::string extensions;
    for (size_t i = 0; i < options.size(); i++)
    {
      if (extensions.size() > 0)
        extensions += '|';

      extensions += options[i].first + "|*." + options[i].second;
    }
    set_extensions(self, extensions, "");
  }
}

//-------------------------------------------------------------------------------------------------

std::string FileChooserImpl::get_selector_option_value(mforms::FileChooser* self, const std::string& name)
{
  if (name == "format")
  {
    FileChooserImpl^ form= (FileChooserImpl^)ObjectImpl::FromUnmanaged(self);

    if (form != nullptr)
    {
      Windows::Forms::FileDialog ^f= form->get_control<FileDialog>();
      if (f->FilterIndex > 0)
        return self->_selector_options["format"][f->FilterIndex-1];
    }
  }
  return "";
}

//-------------------------------------------------------------------------------------------------
