/* 
 * Copyright (c) 2012, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

using namespace System::IO;

using namespace MySQL;
using namespace MySQL::Forms;
using namespace MySQL::Utilities;

//--------------------------------------------------------------------------------------------------

TreeNodeImpl::TreeNodeImpl(TreeViewAdv ^tree, Node^ node)
  : _tree(tree), _node(node), _refcount(0)
{

}

void TreeNodeImpl::release()
{
  _refcount--;
  if (_refcount == 0)
    delete this;
}

void TreeNodeImpl::retain()
{
  _refcount++;
}

MySQL::Forms::TreeViewNode ^TreeNodeImpl::nodeptr() const
{
  Node ^tmp = _node;
  return dynamic_cast<MySQL::Forms::TreeViewNode^>(tmp);
}

bool TreeNodeImpl::is_root() const
{
  return _node->Parent == nullptr;
}

static TreeNodeAdv ^find_node_adv_for(Node ^node, TreeNodeAdv ^root)
{
  if (!node)
    return nullptr;
  if (node->Parent == nullptr)
    return root;
  TreeNodeAdv ^parent = find_node_adv_for(node->Parent, root);
  if (parent != nullptr)
  {
    int i = node->Index;
    if (i >= 0 && i < parent->Children->Count)
      return parent->Children[i];
  }
  return nullptr;
}

TreeNodeAdv^ TreeNodeImpl::find_node_adv()
{
  // apparently no mapping between Node to TreeNodeAdv, so we need
  // to scan it around for it...

  return find_node_adv_for(_node, _tree->Root);
}


Aga::Controls::Tree::TreeModel ^TreeNodeImpl::model()
{
  return dynamic_cast<Aga::Controls::Tree::TreeModel^>(_tree->Model);
}

bool TreeNodeImpl::equals(const mforms::TreeNode &other)
{
  const TreeNodeImpl *oth = dynamic_cast<const TreeNodeImpl*>(&other);
  if (oth)
    return oth->nodeptr() == nodeptr();
  return false;
}

bool TreeNodeImpl::is_valid() const
{
  if (nodeptr() != nullptr)
    return true;
  return false;
}

void TreeNodeImpl::set_icon_path(int column, const std::string &icon)
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr && !icon.empty())
  {
    bool invalidate = true;
    String^ str_icon = CppStringToNative(icon);
    if (!TreeViewNode::_icon_storage.ContainsKey(str_icon))
    {
      String^ path = CppStringToNative(mforms::App::get()->get_resource_path(icon));
      if (File::Exists(path))
        TreeViewNode::_icon_storage[str_icon] = gcnew Bitmap(path);
      else
        invalidate = false;
    }

    if (invalidate)
    {
      node->Icon[column] = TreeViewNode::_icon_storage[str_icon];
      _tree->Invalidate();
    }
  }
}

void TreeNodeImpl::set_attributes(int column, 
  const mforms::TreeNodeTextAttributes& attrs)
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr)
  {
    node->Attributes[column] = attrs;
    _tree->Invalidate(); // Might lead to flickering, investigate if necessary.
  }
}

void TreeNodeImpl::set_string(int column, const std::string &value)
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr)
  {
    node->Text[column] = CppStringToNative(value);
    _tree->Invalidate(); // Might lead to flickering, investigate if necessary.
  }
}

void TreeNodeImpl::set_int(int column, int value)
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr)
  {
    node->Text[column] = Convert::ToString(value);
    _tree->Invalidate();
  }
}

void TreeNodeImpl::set_long(int column, boost::int64_t value)
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr)
  {
    node->Text[column] = Convert::ToString(value);
    _tree->Invalidate();
  }
}


void TreeNodeImpl::set_bool(int column, bool value)
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr)
  {
    node->Text[column] = value ? "1" : "0";
    _tree->Invalidate();
  }
}

std::string TreeNodeImpl::get_string(int column) const
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr)
    return NativeToCppString(node->Text[column]);
  return "";
}

int TreeNodeImpl::get_int(int column) const
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr)
    return Convert::ToInt32(node->Text[column]);
  return 0;
}

bool TreeNodeImpl::get_bool(int column) const
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr)
    return (node->Text[column] == "Checked") || (node->Text[column] == "1") ? true : false;
  return false;
}


boost::int64_t TreeNodeImpl::get_long(int column) const
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr)
    return Convert::ToInt64(node->Text[column]);
  return 0;
}

int TreeNodeImpl::count() const
{
  Node ^node = _node;
  if (node != nullptr)
    return node->Nodes->Count;
  return 0;
}

std::vector<mforms::TreeNodeRef> TreeNodeImpl::add_child_nodes(int column, const std::vector<std::string> &column_values)
{
  std::vector<mforms::TreeNodeRef> result;
  if ((Node^)_node != nullptr && !column_values.empty())
  {
    int i = 0;
    //array<MySQL::Forms::TreeViewNode^>^ newnodes= gcnew array<MySQL::Forms::TreeViewNode^>(column_values.size());
    for (std::vector<std::string>::const_iterator value = column_values.begin();
      value != column_values.end(); ++i, ++value)
    {
      MySQL::Forms::TreeViewNode^ newnode = gcnew MySQL::Forms::TreeViewNode();
      if (newnode->Tag != nullptr)
        throw std::logic_error("internal bug");
      newnode->Text[column] = CppStringToNative(*value);
      newnode->Tag = _tree; // used by ValuePushed handler
      result.push_back(mforms::TreeNodeRef(new TreeNodeImpl(_tree, newnode)));

      _node->Nodes->Add(newnode);
    }
    //_node->Nodes->AddRange(newnodes);
    return result;
  }
  return result;
}


mforms::TreeNodeRef TreeNodeImpl::insert_child(int index)
{
  if ((Node^)_node != nullptr)
  {
    MySQL::Forms::TreeViewNode^ newnode= gcnew MySQL::Forms::TreeViewNode();
    if (newnode->Tag != nullptr)
      throw std::logic_error("internal bug");
    newnode->Tag = _tree; // used by ValuePushed handler
    if (index < 0)
      _node->Nodes->Add(newnode);
    else
      _node->Nodes->Insert(index, newnode);

    return mforms::TreeNodeRef(new TreeNodeImpl(_tree, newnode));
  }
  return mforms::TreeNodeRef();
}

void TreeNodeImpl::remove_from_parent()
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr && node->Parent != nullptr)
    node->Parent->Nodes->Remove(node);
}


void TreeNodeImpl::clear_children()
{
  Node^ node= _node;
  if (node != nullptr)
    node->Nodes->Clear();
}


mforms::TreeNodeRef TreeNodeImpl::get_child(int index) const
{
  Node^ node= _node;
  if (node != nullptr)
  {
    Node^ child = node->Nodes[index];
    if (child != nullptr)
      return mforms::TreeNodeRef(new TreeNodeImpl(_tree, child));
  }
  return mforms::TreeNodeRef();
}

mforms::TreeNodeRef TreeNodeImpl::get_parent() const
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr && node->Parent != nullptr)
  {
    return mforms::TreeNodeRef(new TreeNodeImpl(_tree, node->Parent));
  }
  return mforms::TreeNodeRef();
}

void TreeNodeImpl::expand()
{
  if (!is_root())
    get_parent()->expand();
  TreeNodeAdv ^nodeadv = find_node_adv();
  if (nodeadv != nullptr)
    nodeadv->Expand();
}

void TreeNodeImpl::collapse()
{
  TreeNodeAdv ^nodeadv = find_node_adv();
  if (nodeadv != nullptr)
    nodeadv->Collapse();
}

bool TreeNodeImpl::is_expanded()
{
  TreeNodeAdv ^nodeadv = find_node_adv();
  if (nodeadv != nullptr)
    return nodeadv->IsExpanded;
  return false;
}

void TreeNodeImpl::set_tag(const std::string &tag)
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr)
    node->MyTag = tag;
}

std::string TreeNodeImpl::get_tag() const
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr)
    return node->MyTag;
  return "";
}


void TreeNodeImpl::set_data(mforms::TreeNodeData *data)
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr)
    node->Data = data;
}

mforms::TreeNodeData *TreeNodeImpl::get_data() const
{
  MySQL::Forms::TreeViewNode^ node= nodeptr();
  if (node != nullptr)
    return node->Data;
  return NULL;
}

//----------------- TreeNode -----------------------------------------------------------------------

TreeViewNode::TreeViewNode()
  : _mytag(NULL), _data(NULL)
{
  _attributes = new std::vector<mforms::TreeNodeTextAttributes>();
}

//--------------------------------------------------------------------------------------------------

TreeViewNode::~TreeViewNode()
{
  delete _attributes;
  delete _mytag;
  if (_data)
    _data->release();
}

void TreeViewNode::destroy_data_recursive()
{
    for (int i = 0; i < Nodes->Count; ++i)
        dynamic_cast<MySQL::Forms::TreeViewNode^> (Nodes[i])->destroy_data_recursive();
    Data = nullptr;
};


//--------------------------------------------------------------------------------------------------

std::string TreeViewNode::MyTag::get()
{
  if (_mytag != NULL)
    return *_mytag;
  return "";
}

//--------------------------------------------------------------------------------------------------

void TreeViewNode::MyTag::set(std::string s)
{
  if (_mytag)
    delete _mytag;
  _mytag = new std::string(s);
}

//--------------------------------------------------------------------------------------------------

mforms::TreeNodeData* TreeViewNode::Data::get()
{
  return _data;
}

//--------------------------------------------------------------------------------------------------

void TreeViewNode::Data::set(mforms::TreeNodeData *d)
{
  if (_data != d)
  {
    if (_data)
      _data->release();
    _data = d;
    if (_data)
      _data->retain();
  }
}

//--------------------------------------------------------------------------------------------------

String^ TreeViewNode::Text::get(int index)
{
  if (index < 0 || index >= _text.Count)
    return "";
  return _text[index] == nullptr ? "" : _text[index];
}

//--------------------------------------------------------------------------------------------------

void TreeViewNode::Text::set(int index, String^ newText)
{
  while (index >= _text.Count)
    _text.Add(nullptr);
  _text[index] = newText;
}

//--------------------------------------------------------------------------------------------------

Bitmap^ TreeViewNode::Icon::get(int index)
{
  if (_icon.ContainsKey(index))
    return _icon[index];
  return nullptr;
}

//--------------------------------------------------------------------------------------------------

void TreeViewNode::Icon::set(int index, Bitmap^ newIcon)
{
  _icon[index] = newIcon;
}

//--------------------------------------------------------------------------------------------------

mforms::TreeNodeTextAttributes TreeViewNode::Attributes::get(int index)
{
  if (index < 0 || index >= (int)_attributes->size())
    return mforms::TreeNodeTextAttributes();
  return (*_attributes)[index];
}

//--------------------------------------------------------------------------------------------------

void TreeViewNode::Attributes::set(int index, mforms::TreeNodeTextAttributes attributes)
{
  while (index >= (int)_attributes->size())
    _attributes->push_back(mforms::TreeNodeTextAttributes());
    
  (*_attributes)[index] = attributes;
}

//--------------------------------------------------------------------------------------------------

