grammar MySQL55;

/*
  This grammar assumes that the lexer is set to ignore casing, so we can simplify keyword definition
  here and make the parser smaller (and faster).
*/

//-------------------------------------------------------------------------------------------------

options {
  language = C;
  output = AST;
  ASTLabelType = pANTLR3_BASE_TREE;
  backtrack = true; // XXX: disable backracking (for faster parser and better error messages)
                    //      Need to resolve many ambiquities then.
  memoize = true;
  k = 1;
}

tokens {
	SELECT_EXPR;
	SUBQUERY;
	TABLE_REFERENCE;
	TABLE_REFERENCES;
	JOIN;
	INDEX_HINT_LIST;
	TABLE_REF_ID;
	FIELD_REF_ID;

	// Tokens for expressions.
	EXPRESSION;
	PAR_EXPRESSION;
	FUNCTION_CALL;
	MISSING_ID;
	MISSING_PAR;
	MISSING_OPERAND;
	MISSING_EXPRESSION;
	MISSING_TABLE_REF;
	
	// Tokens for SELECT subtrees. Need to make them more unique to avoid name clashes with other parser.
	UNION_RULE;
	GROUP_BY_RULE;
	ORDER_BY_RULE;
	HAVING_RULE;
	WHERE_RULE;
}

//-------------------------------------------------------------------------------------------------

@parser::header {

#define ANTLR3_HUGE
#ifndef _WIN32
#pragma GCC diagnostic ignored "-Wunused-variable"
#endif
}

@lexer::header {
#define ANTLR3_HUGE
#ifndef _WIN32
#pragma GCC diagnostic ignored "-Wunused-variable"
#endif
}

@parser::postinclude {
#ifdef __cplusplus
extern "C" { 
#endif

  void on_parse_error(struct ANTLR3_BASE_RECOGNIZER_struct * recognizer, pANTLR3_UINT8 * tokenNames); 

#ifdef __cplusplus
};
#endif
}

@parser::apifuncs
{
	// Install custom error message display
	RECOGNIZER->displayRecognitionError = on_parse_error;
}
 
//-------------------------------------------------------------------------------------------------

script:
	statement EOF
;

statement:
	// DDL
	/*alter_statement
	|*/ create_statement
	//| drop_statement
	//| rename_statement
	//| truncate_statement
	
	// DML
	| call_statement
	//| delete_statement
	//| do_statement
	//| handler_statement
	| insert_statement
	//| load_data_file_statement
	//| load_xml_statement
	//| replace_statement
	| select_statement
	| update_statement
	
	// Transaction and locking
	//| transaction_statement
	
	// Replication
	//| replication_statement
	
	// Prepared statements
	//| prepared_statement
	
	// Management
	//| management_statement
	
	// Miscellaneous
	//| utility_statement
;

alter_statement:
  ALTER_SYM
  (
	// ALTER DATABASE
	// | ALTER EVENT
	// | ALTER LOGFILE GROUP
	// | ALTER FUNCTION
	// | ALTER PROCEDURE
	// | ALTER SERVER
	// | ALTER TABLE
	// | ALTER TABLESPACE
	// | ALTER VIEW
  )
;
	
create_statement:
	CREATE_SYM 
	(
		view_or_trigger_or_sp_or_event
		// | CREATE DATABASE
		// | CREATE EVENT
		// | CREATE FUNCTION
		// | CREATE INDEX
		// | CREATE LOGFILE GROUP
		// | CREATE PROCEDURE
		// | CREATE FUNCTION
		// | CREATE SERVER
		// | CREATE TABLE
		// | CREATE TABLESPACE
		// | CREATE TRIGGER
		// | CREATE VIEW
	)
;

drop_statement:
	DROP_SYM
	(
		// DROP DATABASE
		// | DROP EVENT
		// | DROP FUNCTION
		// | DROP INDEX
		// | DROP LOGFILE GROUP
		// | DROP PROCEDURE
		// | DROP FUNCTION
		// | DROP SERVER
		// | DROP TABLE
		// | DROP TABLESPACE
		// | DROP TRIGGER
		// | DROP VIEW
	)
;

rename_statement:
	RENAME_SYM
;
	
truncate_statement:
	TRUNCATE_SYM
;

// --------------- DML statements ------------------------------------------------------------------

call_statement:
	CALL_SYM55 identifier (call_parameters)?
;

call_parameters:
	OPEN_PAR_SYM expression_list closing_parenthesis
	| parentheses
;

delete_statement:
	DELETE_SYM
;

do_statement:
	DO_SYM
;

handler_statement:
	HANDLER_SYM
;

view_or_trigger_or_sp_or_event:	  definer definer_tail
			|  /*no_definer*/ no_definer_tail;

definer_tail:
        /*trigger_tail
        |*/ sp_tail
       ;
        
 no_definer_tail:
	sp_tail
       ;       

sp_tail:
          PROCEDURE_SYM sp_name
          
          OPEN_PAR_SYM
          
          (sp_pdparam_list)?
          closing_parenthesis
          
          sp_c_chistics
          
          sp_proc_stmt
;


sp_name:
          identifier DOT_SYM identifier
          
        | identifier
;

sp_pdparam_list:
          /* Empty -- replaced with ()? on a higher level
        |*/ sp_pdparams
;
 
sp_pdparams:
          sp_pdparam (COMMA_SYM sp_pdparam)*
;        
        
sp_pdparam:
          (sp_opt_inout)? /*sp_init_param*/ identifier type
          
;

sp_opt_inout:
          /* Empty  -- replaced with ()? on a higher level
        |*/ IN_SYM      
        | OUT_SYM     
        | INOUT_SYM   
       ; 
  
definer:
          DEFINER_SYM  user;


user:
          ident_or_text
          
        | ident_or_text AT_SIGN_SYM ident_or_text
          
        | CURRENT_USER_SYM (optional_braces)?
          
       ;
        
ident_or_text:
          identifier           
        |text_string_sys
       ;

identifier:
	IDENTIFIER
	| BACK_TICK_QUOTED_ID
	| ANSI_QUOTED_ID
;

optional_braces:
          /* empty  -- replaced with ()? on a higher level
       | */  OPEN_PAR_SYM closing_parenthesis 
       ;
        
/* Data types */        
type:
          int_type (opt_field_length)? (field_options)? 
        | real_type (opt_precision)? (field_options)? 
        | FLOAT_SYM (float_options)? (field_options)? 
        | BIT_SYM
          
        | BIT_SYM field_length
          
        | BOOL_SYM
          
        | BOOLEAN_SYM
          
        | chr field_length (opt_binary)?
          
        | chr (opt_binary)?
          
        | nchar field_length (opt_bin_mod)?
          
        | nchar (opt_bin_mod)?
          
        | BINARY_SYM field_length
          
        | BINARY_SYM
          
        | varchar field_length (opt_binary)?
          
        | nvarchar field_length (opt_bin_mod)? 
                  
        | VARBINARY_SYM field_length
          
        | YEAR_SYM (opt_field_length)? ( field_options)?
          
        | DATE_SYM
          
        | TIME_SYM
          
        | TIMESTAMP_SYM (opt_field_length)?
          
        | DATETIME_SYM
          
        | TINYBLOB_SYM
          
        | BLOB_SYM (opt_field_length)?
          
        | spatial_type
          
        | MEDIUMBLOB_SYM
          
        | LONGBLOB_SYM
          
        | LONG_SYM VARBINARY_SYM
          
        | LONG_SYM varchar (opt_binary)?
          
        | TINYTEXT_SYM (opt_binary)?
          
        | TEXT_SYM (opt_field_length)? (opt_binary)?
          
        | MEDIUMTEXT_SYM (opt_binary)?
          
        | LONGTEXT_SYM (opt_binary)?
          
        | DECIMAL_SYM ( float_options)? (field_options)?
          
        | NUMERIC_SYM (float_options)? (field_options)?
          
        | FIXED_SYM (float_options)? (field_options)?
          
        | ENUM_SYM
          
          OPEN_PAR_SYM string_list closing_parenthesis (opt_binary)?
          
        | SET_SYM
          
          OPEN_PAR_SYM string_list closing_parenthesis (opt_binary)?
          
        | LONG_SYM (opt_binary)?
          
        | SERIAL_SYM
          
       ;        

int_type:
          INT_SYM   
        | TINYINT_SYM   
        | SMALLINT_SYM  
        | MEDIUMINT_SYM 
        | BIGINT_SYM    
       ;

field_length:
          OPEN_PAR_SYM number closing_parenthesis      
        | OPEN_PAR_SYM NUM closing_parenthesis 
        | OPEN_PAR_SYM DECIMAL_NUM closing_parenthesis   
        | OPEN_PAR_SYM  closing_parenthesis          ;

opt_field_length:
          /* empty -- replaced by ()? and  on a higher level  
        | */ field_length;

field_options:
          /* empty -- replaced by ()? and  on a higher level  
        |*/  field_opt_list 
       ;

field_opt_list:          field_option (field_option)* 
       ;

field_option:
          SIGNED_SYM 
        | UNSIGNED_SYM 
        | ZEROFILL_SYM 
       ;
        
real_type:
          REAL_SYM
          
        | DOUBLE_SYM
          
        | DOUBLE_SYM PRECISION_SYM
          
       ;
opt_precision:
          /* empty  -- replaced by ()? and  on a higher level  
        | */ precision 
       ;
        
precision:
          OPEN_PAR_SYM NUM COMMA_SYM NUM closing_parenthesis
;

float_options:
          /* empty  -- replaced by ()? and  on a higher level  
          
        | */ field_length
          
        | precision
          
       ;

/* char is a reserved word in Java */
chr:
          CHAR_SYM 
       ;

opt_binary:
          /* empty   -- replaced by ()? and  on a higher level  
        |*/ ASCII_SYM (opt_bin_mod)? 
        | BYTE_SYM 
        | UNICODE_SYM (opt_bin_mod)?
          
        | charset charset_name (opt_bin_mod)? 
        | BINARY_SYM (opt_bin_charset)? 
       ;
        
opt_bin_mod:
          /* empty  -- replaced by ()? and  on a higher level
        |*/  BINARY_SYM 
       ;

charset:
          CHAR_SYM SET_SYM 
        | CHARSET_SYM 
       ;        

charset_name:
          ident_or_text
          
        | BINARY_SYM 
       ;                
opt_bin_charset:
          /* empty -- replaced by ()? and  on a higher level
        |*/  ASCII_SYM 
        | UNICODE_SYM
          
        | charset charset_name 
       ;

nchar:
          NCHAR_SYM 
        | NATIONAL_SYM CHAR_SYM 
       ;
varchar:
          chr VARYING_SYM 
        | VARCHAR_SYM 
       ;        

nvarchar:
          NATIONAL_SYM VARCHAR_SYM 
        | NVARCHAR_SYM 
        | NCHAR_SYM VARCHAR_SYM 
        | NATIONAL_SYM CHAR_SYM VARYING_SYM 
        | NCHAR_SYM VARYING_SYM 
       ;        

spatial_type:
          GEOMETRY_SYM        
        | GEOMETRYCOLLECTION_SYM  
        | POINT_SYM  
        | MULTIPOINT_SYM          
        | LINESTRING          
        | MULTILINESTRING_SYM     
        | POLYGON_SYM             
        | MULTIPOLYGON_SYM        
       ;
string_list:          text_string (COMMA_SYM text_string)*;

text_string:
          TEXT_STRING/*_literal*/
          
        | HEX_NUM
          
        | BIN_NUM;


sp_c_chistics:		sp_c_chistic*;

sp_c_chistic:
          sp_chistic            
        | DETERMINISTIC_SYM     
        | not_rule DETERMINISTIC_SYM 
       ;

sp_chistic:
          COMMENT_SYM /*TEXT_STRING_sys*/ text_string_sys
          
        | LANGUAGE_SYM SQL_SYM
          
        | NO_SYM SQL_SYM
          
        | CONTAINS_SYM SQL_SYM
          
        | READS_SYM SQL_SYM DATA_SYM
          
        | MODIFIES_SYM SQL_SYM DATA_SYM
          
        | sp_suid
       ;
 
sp_suid:
          SQL_SYM SECURITY_SYM DEFINER_SYM
          
        | SQL_SYM SECURITY_SYM INVOKER_SYM
          
       ;
 
        
sp_proc_stmt:
          sp_proc_stmt_statement
        | sp_proc_stmt_return
        | sp_proc_stmt_if
        | case_stmt_specification
        | sp_labeled_block
        | sp_unlabeled_block
        | sp_labeled_control
        | sp_proc_stmt_unlabeled
        | sp_proc_stmt_leave
        | sp_proc_stmt_iterate
        | sp_proc_stmt_open
        | sp_proc_stmt_fetch
        | sp_proc_stmt_close
       ;

sp_proc_stmt_close:
          CLOSE_SYM identifier
          
       ;

sp_proc_stmt_fetch:
          FETCH_SYM (sp_opt_fetch_noise)? identifier INTO_SYM
          
          sp_fetch_list
          
       ;

sp_fetch_list:	identifier (COMMA_SYM identifier)*;

sp_opt_fetch_noise:
	(NEXT_SYM)? FROM_SYM
;


sp_proc_stmt_open:
          OPEN_SYM identifier
          
       ;

sp_proc_stmt_iterate:
          ITERATE_SYM label_ident
          
       ;

sp_proc_stmt_unlabeled:
          
          sp_unlabeled_control
          
       ;

sp_proc_stmt_leave:
          LEAVE_SYM label_ident
          
       ;

sp_unlabeled_block:
          
          sp_block_content
          
       ;

sp_labeled_control:
          label_ident COLON_SYM
          
          sp_unlabeled_control (sp_opt_label)?
          
       ;

sp_unlabeled_control:
          LOOP_SYM
          sp_proc_stmts1 END_SYM LOOP_SYM
          
        | WHILE_SYM 
          
          expr_old DO_SYM
          
          sp_proc_stmts1 END_SYM WHILE_SYM
          
        | REPEAT_SYM sp_proc_stmts1 UNTIL_SYM 
          
          expr_old END_SYM REPEAT_SYM
          
       ;

sp_labeled_block:
          label_ident COLON_SYM
          
          sp_block_content (sp_opt_label)?
          
       ;

sp_opt_label:
          /* Empty  -- replaced by ()? and  on a higher level
        |*/ label_ident   
       ;


label_ident:
         /* IDENT_sys*/ identifier    
        | keyword
          
       ;


keyword:
	(
	  actual_token = ACTION_SYM
	| actual_token = ADDDATE_SYM
	| actual_token = AFTER_SYM
	| actual_token = AGAINST_SYM
	| actual_token = AGGREGATE_SYM
	| actual_token = ALGORITHM_SYM
	| actual_token = ANY_SYM
	| actual_token = AT_SYM
	| actual_token = AUTHORS_SYM
	| actual_token = AUTO_INC_SYM
	| actual_token = AUTOEXTEND_SIZE_SYM
	| actual_token = AVG_ROW_LENGTH_SYM
	| actual_token = AVG_SYM
	| actual_token = BINLOG_SYM
	| actual_token = BIT_SYM
	| actual_token = BLOCK_SYM
	| actual_token = BOOL_SYM
	| actual_token = BOOLEAN_SYM
	| actual_token = BTREE_SYM
	| actual_token = CASCADED_SYM
	| actual_token = CHAIN_SYM
	| actual_token = CHANGED_SYM
	| actual_token = CIPHER_SYM
	| actual_token = CLIENT_SYM
	| actual_token = COALESCE_SYM
	| actual_token = CODE_SYM
	| actual_token = COLLATION_SYM
	| actual_token = COLUMNS_SYM
	| actual_token = COMMITTED_SYM
	| actual_token = COMPACT_SYM
	| actual_token = COMPLETION_SYM
	| actual_token = COMPRESSED_SYM
	| actual_token = CONCURRENT_SYM
	| actual_token = CONNECTION_SYM
	| actual_token = CONSISTENT_SYM
	| actual_token = CONTEXT_SYM
	| actual_token = CONTRIBUTORS_SYM
	| actual_token = COUNT_SYM
	| actual_token = CPU_SYM
	| actual_token = CUBE_SYM
	| actual_token = DATA_SYM
	| actual_token = DATAFILE_SYM
	| actual_token = DATETIME_SYM
	| actual_token = DATE_SYM
	| actual_token = DAY_SYM
	| actual_token = DEFINER_SYM
	| actual_token = DELAY_KEY_WRITE_SYM
	| actual_token = DES_KEY_FILE_SYM
	| actual_token = DIRECTORY_SYM
	| actual_token = DISABLE_SYM
	| actual_token = DISCARD_SYM
	| actual_token = DISK_SYM
	| actual_token = DUMPFILE_SYM
	| actual_token = DUPLICATE_SYM
	| actual_token = DYNAMIC_SYM
	| actual_token = ENDS_SYM
	| actual_token = ENUM_SYM
	| actual_token = ENGINE_SYM
	| actual_token = ENGINES_SYM
	| actual_token = ERRORS_SYM
	| actual_token = ESCAPE_SYM
	| actual_token = EVENT_SYM
	| actual_token = EVENTS_SYM
	| actual_token = EVERY_SYM
	| actual_token = EXPANSION_SYM
	| actual_token = EXTENDED_SYM
	| actual_token = EXTENT_SIZE_SYM
	| actual_token = FAULTS_SYM
	| actual_token = FAST_SYM
	| actual_token = FOUND_SYM
	| actual_token = ENABLE_SYM
	| actual_token = FULL_SYM
	| actual_token = FILE_SYM
	| actual_token = FIRST_SYM
	| actual_token = FIXED_SYM
	| actual_token = FRAC_SECOND_SYM
	| actual_token = GEOMETRY_SYM
	| actual_token = GEOMETRYCOLLECTION_SYM
	| actual_token = GET_FORMAT_SYM
	| actual_token = GRANTS_SYM
	| actual_token = GLOBAL_SYM
	| actual_token = HASH_SYM
	| actual_token = HOSTS_SYM
	| actual_token = HOUR_SYM
	| actual_token = IDENTIFIED_SYM
	| actual_token = INVOKER_SYM
	| actual_token = IMPORT_SYM
	| actual_token = INDEXES_SYM
	| actual_token = INITIAL_SIZE_SYM
	| actual_token = IO_SYM
	| actual_token = IPC_SYM
	| actual_token = ISOLATION_SYM
	| actual_token = ISSUER_SYM
	| actual_token = INNOBASE_SYM
	| actual_token = INSERT_METHOD_SYM
	| actual_token = KEY_BLOCK_SIZE_SYM
	| actual_token = LAST_SYM
	| actual_token = LEAVES_SYM
	| actual_token = LESS_SYM
	| actual_token = LEVEL_SYM
	| actual_token = LINESTRING
	| actual_token = LIST_SYM
	| actual_token = LOCAL_SYM
	| actual_token = LOCKS_SYM
	| actual_token = LOGFILE_SYM
	| actual_token = LOGS_SYM
	| actual_token = MAX_ROWS_SYM
	| actual_token = MASTER_SYM
	| actual_token = MASTER_HOST_SYM
	| actual_token = MASTER_PORT_SYM
	| actual_token = MASTER_LOG_FILE_SYM
	| actual_token = MASTER_LOG_POS_SYM
	| actual_token = MASTER_USER_SYM
	| actual_token = MASTER_PASSWORD_SYM
	| actual_token = MASTER_SERVER_ID_SYM
	| actual_token = MASTER_CONNECT_RETRY_SYM
	| actual_token = MASTER_SSL_SYM
	| actual_token = MASTER_SSL_CA_SYM
	| actual_token = MASTER_SSL_CAPATH_SYM
	| actual_token = MASTER_SSL_CERT_SYM
	| actual_token = MASTER_SSL_CIPHER_SYM
	| actual_token = MASTER_SSL_KEY_SYM
	| actual_token = MAX_CONNECTIONS_PER_HOUR_SYM
	| actual_token = MAX_QUERIES_PER_HOUR_SYM
	| actual_token = MAX_SIZE_SYM
	| actual_token = MAX_UPDATES_PER_HOUR_SYM
	| actual_token = MAX_USER_CONNECTIONS_SYM
	| actual_token = MAX_VALUE_SYM
	| actual_token = MEDIUM_SYM
	| actual_token = MEMORY_SYM
	| actual_token = MERGE_SYM
	| actual_token = MICROSECOND_SYM
	| actual_token = MIGRATE_SYM
	| actual_token = MINUTE_SYM
	| actual_token = MIN_ROWS_SYM
	| actual_token = MODIFY_SYM
	| actual_token = MODE_SYM
	| actual_token = MONTH_SYM
	| actual_token = MULTILINESTRING_SYM
	| actual_token = MULTIPOINT_SYM
	| actual_token = MULTIPOLYGON_SYM
	| actual_token = MUTEX_SYM
	| actual_token = NAME_SYM
	| actual_token = NAMES_SYM
	| actual_token = NATIONAL_SYM
	| actual_token = NCHAR_SYM
	| actual_token = NDBCLUSTER_SYM
	| actual_token = NEXT_SYM
	| actual_token = NEW_SYM
	| actual_token = NO_WAIT_SYM
	| actual_token = NODEGROUP_SYM
	| actual_token = NONE_SYM
	| actual_token = NVARCHAR_SYM
	| actual_token = OFFSET_SYM
	| actual_token = OLD_PASSWORD_SYM
	| actual_token = ONE_SHOT_SYM
	| actual_token = ONE_SYM
	| actual_token = PACK_KEYS_SYM
	| actual_token = PAGE_SYM
	| actual_token = PARTIAL_SYM
	| actual_token = PARTITIONING_SYM
	| actual_token = PARTITIONS_SYM
	| actual_token = PASSWORD_SYM
	| actual_token = PHASE_SYM
	| actual_token = PLUGIN_SYM
	| actual_token = PLUGINS_SYM
	| actual_token = POINT_SYM
	| actual_token = POLYGON_SYM
	| actual_token = PRESERVE_SYM
	| actual_token = PREV_SYM
	| actual_token = PRIVILEGES_SYM
	| actual_token = PROCESS_SYM
	| actual_token = PROCESSLIST_SYM
	| actual_token = PROFILE_SYM
	| actual_token = PROFILES_SYM
	| actual_token = QUARTER_SYM
	| actual_token = QUERY_SYM
	| actual_token = QUICK_SYM
	| actual_token = READ_ONLY_SYM
	| actual_token = REBUILD_SYM
	| actual_token = RECOVER_SYM
	| actual_token = REDO_BUFFER_SIZE_SYM
	| actual_token = REDOFILE_SYM
	| actual_token = REDUNDANT_SYM
	| actual_token = RELAY_LOG_FILE_SYM
	| actual_token = RELAY_LOG_POS_SYM
	| actual_token = RELAY_THREAD_SYM
	| actual_token = RELOAD_SYM
	| actual_token = REORGANIZE_SYM
	| actual_token = REPEATABLE_SYM
	| actual_token = REPLICATION_SYM
	| actual_token = RESOURCES_SYM
	| actual_token = RESUME_SYM
	| actual_token = RETURNS_SYM
	| actual_token = ROLLUP_SYM
	| actual_token = ROUTINE_SYM
	| actual_token = ROWS_SYM
	| actual_token = ROW_FORMAT_SYM
	| actual_token = ROW_SYM
	| actual_token = RTREE_SYM
	| actual_token = SCHEDULE_SYM
	| actual_token = SECOND_SYM
	| actual_token = SERIAL_SYM
	| actual_token = SERIALIZABLE_SYM
	| actual_token = SESSION_SYM
	| actual_token = SIMPLE_SYM
	| actual_token = SHARE_SYM
	| actual_token = SHUTDOWN_SYM
	| actual_token = SNAPSHOT_SYM
	| actual_token = SOUNDS_SYM
	| actual_token = SOURCE_SYM
	| actual_token = SQL_CACHE_SYM
	| actual_token = SQL_BUFFER_RESULT_SYM
	| actual_token = SQL_NO_CACHE_SYM
	| actual_token = SQL_THREAD_SYM
	| actual_token = STARTS_SYM
	| actual_token = STATUS_SYM
	| actual_token = STORAGE_SYM
	| actual_token = STRING_SYM
	| actual_token = SUBDATE_SYM
	| actual_token = SUBJECT_SYM
	| actual_token = SUBPARTITION_SYM
	| actual_token = SUBPARTITIONS_SYM
	| actual_token = SUPER_SYM
	| actual_token = SUSPEND_SYM
	| actual_token = SWAPS_SYM
	| actual_token = SWITCHES_SYM
	| actual_token = TABLE_CHECKSUM_SYM
	| actual_token = TABLESPACE_SYM
	| actual_token = TEMPORARY_SYM
	| actual_token = TEMPTABLE_SYM
	| actual_token = TEXT_SYM
	| actual_token = THAN_SYM
	| actual_token = TRANSACTION_SYM
	| actual_token = TRIGGERS_SYM
	| actual_token = TIMESTAMP_SYM
	| actual_token = TIMESTAMP_ADD_SYM
	| actual_token = TIMESTAMP_DIFF_SYM
	| actual_token = TIME_SYM
	| actual_token = TYPES_SYM
	| actual_token = TYPE_SYM
	| actual_token = UDF_RETURNS_SYM
	| actual_token = FUNCTION_SYM
	| actual_token = UNCOMMITTED_SYM
	| actual_token = UNDEFINED_SYM
	| actual_token = UNDO_BUFFER_SIZE_SYM
	| actual_token = UNDOFILE_SYM
	| actual_token = UNKNOWN_SYM
	| actual_token = UNTIL_SYM
	| actual_token = USER_SYM
	| actual_token = USE_FRM_SYM
	| actual_token = VARIABLES_SYM
	| actual_token = VIEW_SYM
	| actual_token = VALUE_SYM
	| actual_token = WARNINGS_SYM
	| actual_token = WAIT_SYM
	| actual_token = WEEK_SYM
	| actual_token = WORK_SYM
	| actual_token = X509_SYM
	| actual_token = YEAR_SYM
	) {$actual_token->setType($actual_token, IDENTIFIER); }
;

sp_block_content:
          BEGIN_SYM
          
          sp_decls
          sp_proc_stmts
          END_SYM
       ;

sp_proc_stmts:  (sp_proc_stmt SEMICOLON_SYM)*;

	

sp_decls:  (sp_decl SEMICOLON_SYM)*;

sp_decl:
          DECLARE_SYM sp_decl_idents
          
          type
          (sp_opt_default)?
          
        | DECLARE_SYM identifier CONDITION_SYM FOR_SYM sp_cond
          
        | DECLARE_SYM sp_handler_type HANDLER_SYM FOR_SYM
          
          sp_hcond_list sp_proc_stmt
          
        | DECLARE_SYM identifier CURSOR_SYM FOR_SYM sp_cursor_stmt
          
       ;
  
  sp_cursor_stmt:
          
          
          
       ;
  
  sp_hcond_list:          sp_hcond_element (SEMICOLON_SYM sp_hcond_element)*;
  
  sp_hcond_element:
          sp_hcond
          
       ;
 sp_hcond:
          sp_cond
          
        | identifier /* CONDITION name */
          
        | SQLWARNING_SYM /* SQLSTATEs 01??? */
          
        | not_rule FOUND_SYM /* SQLSTATEs 02??? */
          
        | SQLEXCEPTION_SYM /* All other SQLSTATEs */
          
       ;         
  
   
   sp_handler_type:
          EXIT_SYM      
        | CONTINUE_SYM  
        /*| UNDO_SYM       */
       ;
   
   
 sp_cond:
          number
          
        | SQLSTATE_SYM (opt_value)? /*TEXT_STRING_literal*/ text_string_literal
          
       ;

opt_value:
          /* Empty -- replaced by ()? and  on a higher level
        |*/ VALUE_SYM    
       ;
       
sp_decl_idents:	 identifier  (COMMA_SYM identifier)*;
          

sp_opt_default:
          /* Empty  -- replaced by ()? and  on a higher level
        |*/ DEFAULT_SYM expr_old 
       ;


          
case_stmt_specification:
          simple_case_stmt
        | searched_case_stmt
       ;

simple_case_stmt:
          CASE_SYM
          
          expr_old
          
          simple_when_clause_list
          (else_clause_opt)?
          END_SYM
          CASE_SYM
          
;

else_clause_opt:
          /* empty  -- replaced by ()? and  on a higher level
          
        |*/ ELSE_SYM sp_proc_stmts1
       ;


searched_case_stmt:
          CASE_SYM
          
          searched_when_clause_list
          (else_clause_opt)?
          END_SYM
          CASE_SYM
          
       ;

searched_when_clause_list:	searched_when_clause (searched_when_clause)*;

searched_when_clause:
          WHEN_SYM
          
          expr_old
          
          THEN_SYM
          sp_proc_stmts1
          
       ;

simple_when_clause_list:	 simple_when_clause (simple_when_clause_list)*;

simple_when_clause:
          WHEN_SYM
          
          expr_old
          
          THEN_SYM
          sp_proc_stmts1
          
       ;


sp_proc_stmt_if:
          IF_SYM
          
          sp_if END_SYM IF_SYM
          
       ;

sp_if:
          
          expr_old THEN_SYM
          
          sp_proc_stmts1
          
          (sp_elseifs)?
          
       ;
        
sp_proc_stmts1:    (sp_proc_stmt SEMICOLON_SYM) (sp_proc_stmt SEMICOLON_SYM)*; 

sp_elseifs:
          /* Empty  -- replaced by ()? and  on a higher level
        |*/ ELSEIF_SYM sp_if
        | ELSE_SYM sp_proc_stmts1
       ;

sp_proc_stmt_statement:
        commit
        | delete_clause
        | execute
        | insert_statement
        | rollback
        | select_statement
        | update_statement
       ;
          
sp_proc_stmt_return:
          RETURN_SYM 
          
          expr_old
          
       ;

//--------------------------------------------------------------------------------------------------

// Expression support.

expression:
	conditional_or_expression (EQUAL_SYM55 expression)? -> ^(EXPRESSION conditional_or_expression (EQUAL_SYM55 expression)?)
;

conditional_or_expression:
	conditional_and_expression (or_rule conditional_and_expression)*
;

conditional_and_expression:
	inclusive_or_expression
	(
		and_rule inclusive_or_expression
		| (not_rule)? IN_SYM OPEN_PAR_SYM expression_list closing_parenthesis
		| (not_rule)? IN_SYM par_expression
		| (not_rule)? BETWEEN_SYM inclusive_or_expression AND_SYM inclusive_or_expression
		| SOUNDS_SYM LIKE_SYM inclusive_or_expression
		| (not_rule)? LIKE_SYM primary (escape)?
		| (not_rule)? REGEXP_SYM inclusive_or_expression
	)*
;

escape:
	ESCAPE_SYM primary 
;       

inclusive_or_expression:
	exclusive_or_expression (BITWISE_OR exclusive_or_expression)*
;

exclusive_or_expression:
	and_expression (BITWISE_XOR and_expression)*
;

and_expression:
	comparison_expression (BITWISE_AND comparison_expression)*
;

comparison_expression:
	shift_expression (comparison_operator shift_expression)*
;

comparison_operator:
	EQUAL_SYM55
	| GREATER_OR_EQUAL
	| GREATER_THAN
	| LESS_OR_EQUAL
	| LESS_THAN
	| NOT_EQUAL
	| NOT_EQUAL2
	| NULL_SAFE_EQUAL
;

shift_expression:
	additive_expression (shift_op additive_expression)*
;

shift_op:
	  ('<' '<') => t1='<' t2='<' 
	  { $t1->getLine($t1) == $t2->getLine($t2) && 
	    $t1->getCharPositionInLine($t1) + 1 == $t2->getCharPositionInLine($t2) }?
	| ('>' '>' '>') => t1='>' t2='>' t3='>' 
	  { $t1->getLine($t1) == $t2->getLine($t2) && 
	    $t1->getCharPositionInLine($t1) + 1 == $t2->getCharPositionInLine($t2) &&
	    $t2->getLine($t2) == $t3->getLine($t3) && 
	    $t2->getCharPositionInLine($t2) + 1 == $t3->getCharPositionInLine($t3) }?
	| ('>' '>') => t1='>' t2='>'
	    { $t1->getLine($t1) == $t2->getLine($t2) && 
	    $t1->getCharPositionInLine($t1) + 1 == $t2->getCharPositionInLine($t2) }?
;


additive_expression:
	multiplicative_expression ((PLUS_SYM | MINUS_SYM) multiplicative_expression)*
;

multiplicative_expression:
	unary_expression (( MULT | DIV | DIV_SYM55 | MOD | MOD_SYM55 ) unary_expression)*
;
    
unary_expression:
	PLUS_SYM unary_expression
	| MINUS_SYM unary_expression
	| INC unary_expression
	| DEC unary_expression
	| unary_expression_not_plus_minus
;

unary_expression_not_plus_minus:
	LOGICAL_NOT unary_expression
	|   BITWISE_NOT unary_expression
//	|   castExpression
	|   primary
;

/*
castExpression
    :  OPEN_PAR_SYM primitiveType closing_parenthesis unaryExpression
    |  OPEN_PAR_SYM (type | expression) closing_parenthesis unaryExpressionNotPlusMinus
    ;
*/

primary:
	par_expression
	| function_call -> ^(FUNCTION_CALL function_call)
	| literal
	| user_variable
	| system_variable
	| PARAM_MARKER
	| match_expression -> ^(match_expression)
	| case_expression -> ^(case_expression)
	| field_name (INC | DEC | IS_SYM (NOT_SYM)? NULL_SYM)?
;

literal:
	number
	| (DATE_SYM | TIME_SYM | TIMESTAMP_SYM)? (UNDERLINE_SYM)? TEXT_STRING
	| NULL_SYM
	| FALSE_SYM
	| TRUE_SYM
	| (UNDERLINE_SYM)? HEX_NUM
	| (UNDERLINE_SYM)? BIN_NUM
;

number:
	HEX_NUM
	| NUM
	| DECIMAL_NUM
;

user_variable:
	AT_SIGN_SYM identifier
;

system_variable:
	AT_AT_SIGN_SYM ((LOCAL_SYM | GLOBAL_SYM | SESSION_SYM) DOT_SYM)? identifier
;

par_expression:
	OPEN_PAR_SYM
	(
		select_statement -> ^(SUBQUERY OPEN_PAR_SYM select_statement CLOSE_PAR_SYM)
		| expression -> ^(PAR_EXPRESSION OPEN_PAR_SYM expression CLOSE_PAR_SYM)
		| -> MISSING_EXPRESSION
	)
	closing_parenthesis
;

match_expression:
	MATCH_SYM OPEN_PAR_SYM field_name_list CLOSE_PAR_SYM AGAINST_SYM OPEN_PAR_SYM expression ( (IN_SYM BOOLEAN_SYM MODE_SYM ) | ( WITH_SYM QUERY_SYM EXPANSION_SYM ))? CLOSE_PAR_SYM
		-> ^( MATCH_SYM field_name_list expression ( IN_SYM BOOLEAN_SYM MODE_SYM )? ( WITH_SYM QUERY_SYM EXPANSION_SYM )? )
;

case_expression:
	CASE_SYM (expression)? (when_expression then_expression)+ (else_expression)? END_SYM
;

when_expression:
	WHEN_SYM expression -> ^(WHEN_SYM expression)
;

then_expression:
	THEN_SYM expression -> ^(THEN_SYM expression)
;

else_expression:
	ELSE_SYM expression -> ^(ELSE_SYM expression)
;

not_rule:
	NOT_SYM
	| LOGICAL_NOT
;

or_rule:
	OR_SYM
	| LOGICAL_OR
;

and_rule:
	AND_SYM
	| LOGICAL_AND
;

//--------------------------------------------------------------------------------------------------

expr_old	:     	expr2 | expr_not | expr_paren | expr_paren_list | expr_paren_list_row;


expr_not	:	not_rule expr2;

expr_paren
	:	OPEN_PAR_SYM expr2 closing_parenthesis;

expr_paren_list
	:	OPEN_PAR_SYM expr2 (COMMA_SYM expr2 )+ closing_parenthesis;

	
expr_paren_list_row
	:	 ROW_SYM expr_paren_list;	

expr_cast	
	:	CAST_SYM OPEN_PAR_SYM expr2 AS_SYM cast_type closing_parenthesis;

expr_case
	:	CASE_SYM (expr2)? when_list opt_else END_SYM;

expr_convert
	:	CONVERT_SYM OPEN_PAR_SYM expr2 COMMA_SYM cast_type closing_parenthesis;		

expr_convert2
	:	CONVERT_SYM OPEN_PAR_SYM expr2 USING_SYM charset_name closing_parenthesis;

expr_interval
	:	 INTERVAL_SYM expr2;

expr2
	:	bool_pri_with_opt_suffix		 
		 (
		  or_rule expr2
		 |  XOR_SYM expr2
		 | and_rule expr2
		 |  interval '+' expr_old
		 )*;
		 

bool_pri_with_opt_suffix
	:	bool_pri
		
		( 
		   IS_SYM TRUE_SYM
		 |  not_rule TRUE_SYM
		 | IS_SYM FALSE_SYM
		 | IS_SYM not_rule FALSE_SYM
		 | IS_SYM UNKNOWN_SYM
		 | bool_pri IS_SYM not_rule UNKNOWN_SYM
		 )?
	;

/*
bool_pri:
          bool_pri IS NULL_SYM %prec IS
          
        | bool_pri IS not NULL_SYM %prec IS
          
        | bool_pri EQUAL_SYM predicate %prec EQUAL_SYM
          
        | bool_pri comp_op predicate %prec EQ
          
        | bool_pri comp_op all_or_any OPEN_PAR_SYM subselect closing_parenthesis %prec EQ
          
        | predicate
       ;
*/

bool_pri:
          predicate (IS_SYM NULL_SYM | IS_SYM not_rule NULL_SYM | EQUAL_SYM55 predicate | comparison_operator predicate | comparison_operator all_or_any OPEN_PAR_SYM subselect_deprecated closing_parenthesis)*
       ;


all_or_any:
          ALL_SYM     
        | ANY_SYM 
       ;

predicate:
          bit_expr IN_SYM OPEN_PAR_SYM subselect_deprecated closing_parenthesis
          
        | bit_expr not_rule IN_SYM OPEN_PAR_SYM subselect_deprecated closing_parenthesis
          
        | bit_expr IN_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | bit_expr IN_SYM OPEN_PAR_SYM expr_old COMMA_SYM expression_list closing_parenthesis
          
        | bit_expr not_rule IN_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | bit_expr not_rule IN_SYM OPEN_PAR_SYM expr_old COMMA_SYM expression_list closing_parenthesis
          
        | bit_expr BETWEEN_SYM bit_expr AND_SYM predicate
          
        | bit_expr not_rule BETWEEN_SYM bit_expr AND_SYM predicate
          
        | bit_expr SOUNDS_SYM LIKE_SYM bit_expr
          
        | bit_expr LIKE_SYM simple_expr (escape)?
          
        | bit_expr not_rule LIKE_SYM simple_expr (escape)?
          
        | bit_expr REGEXP_SYM bit_expr
          
        | bit_expr not_rule REGEXP_SYM bit_expr
          
        | bit_expr
       ;
   
        
 
/* The following must be handled here -- but not sure if this is the right way. It could cause that nested calls of this statement do not work.
simple_expr_bit_expr
	:	MATCH ident_list_arg AGAINST OPEN_PAR_SYM bit_expr fulltext_options closing_parenthesis;
*/

bit_expr	:	bit_expr2 | MATCH_SYM ident_list_arg AGAINST_SYM OPEN_PAR_SYM bit_expr2 fulltext_options closing_parenthesis;
	
bit_expr2
	:	simple_expr ( '|' bit_expr 
			| '&' bit_expr 
			| SHIFT_LEFT bit_expr
			| SHIFT_RIGHT bit_expr
			| '+' bit_expr
			|  '*' bit_expr
			|  '/' bit_expr
			| '%' bit_expr
			| '^' bit_expr
			| DIV_SYM55 bit_expr
			| MOD_SYM55 bit_expr 
			| '+' INTERVAL_SYM expr_old interval  //Should work because it is right recursion
			| '-' INTERVAL_SYM expr_old interval  //Should work because it is a right recursion 
			)*; 		
    

simple_expr
	:    (simple_ident
        	    | function_call_keyword
        	    | function_call_nonkeyword
        	    | function_call_generic
        	    | function_call_conflict  
        	    | literal
        	    | PARAM_MARKER
        	    | system_variable
        	    | user_variable
        	    |  OPEN_PAR_SYM subselect_deprecated closing_parenthesis
        	    | EXISTS_SYM OPEN_PAR_SYM subselect_deprecated closing_parenthesis
        	    | DEFAULT_SYM OPEN_PAR_SYM simple_ident closing_parenthesis
        	    |  VALUES_SYM OPEN_PAR_SYM simple_ident_nospvar closing_parenthesis 
        	    | '+'
        	    | '-'
        	    | '~'
        	    | not_rule
        	    | BINARY_SYM)   (LOGICAL_OR simple_expr | COLLATE_SYM ident_or_text)*
        	    //Expected to cause problems because it calls again expr
        	    //| simple_expr_expr
        	    //| simple_expr_bit_expr
        	    | simple_expr_sum_expr     	   
        	    ;

//Causes problems because of left recursion
simple_expr_expr
	: 	
		 OPEN_PAR_SYM expr_old closing_parenthesis
		| OPEN_PAR_SYM expr_old COMMA_SYM expression_list closing_parenthesis
		| ROW_SYM OPEN_PAR_SYM expr_old COMMA_SYM expression_list closing_parenthesis
		| CAST_SYM OPEN_PAR_SYM expr_old AS_SYM cast_type closing_parenthesis
		| CASE_SYM opt_expr when_list opt_else END_SYM
		| CONVERT_SYM OPEN_PAR_SYM expr_old COMMA_SYM cast_type closing_parenthesis
       		| CONVERT_SYM OPEN_PAR_SYM expr_old USING_SYM charset_name closing_parenthesis
       		| INTERVAL_SYM expr_old interval '+' expr_old
       	;
	

//Causes problems because of left recusions
simple_expr_bit_expr
	:	MATCH_SYM ident_list_arg AGAINST_SYM OPEN_PAR_SYM bit_expr fulltext_options closing_parenthesis;


simple_expr_sum_expr
	:	aggregate_function;



simple_ident_nospvar:
          identifier
          
        | simple_ident_q 
       ;

opt_else:
          /* empty  -- replaced by ()? and  on a higher level  
        |*/ ELSE_SYM expr_old    
       ;

opt_expr:
          /* empty  -- replaced by ()? and  on a higher level    
        |*/ expr_old           
       ;

when_list:	( WHEN_SYM expr_old THEN_SYM expr_old) (WHEN_SYM expr_old THEN_SYM expr_old)*;


cast_type:
          BINARY_SYM (opt_field_length)?
          
        | CHAR_SYM (opt_field_length)? opt_binary
          
        | NCHAR_SYM (opt_field_length)?
          
        | SIGNED_SYM
          
        | SIGNED_SYM INT_SYM
          
        | UNSIGNED_SYM
          
        | UNSIGNED_SYM INT_SYM
          
        | DATE_SYM
          
        | TIME_SYM
          
        | DATETIME_SYM
          
        | DECIMAL_SYM (float_options)?
          
       ;

ident_list_arg:
          ident_list          
        | OPEN_PAR_SYM ident_list closing_parenthesis  
       ;

ident_list:	simple_ident (simple_ident)*;

fulltext_options:
          (opt_natural_language_mode)? opt_query_expansion
          
        | IN_SYM BOOLEAN_SYM MODE_SYM
          
       ;
opt_natural_language_mode:
          /* empty -- replaced by ()? and  on a higher level                         
        |*/ IN_SYM NATURAL_SYM LANGUAGE_SYM MODE_SYM  
       ;

opt_query_expansion:
          /* nothing */                         
        | WITH_SYM QUERY_SYM EXPANSION_SYM          
       ;

simple_ident:
          identifier
          
        | simple_ident_q 
       ;


simple_ident_q:
          identifier DOT_SYM identifier
          
        | DOT_SYM identifier DOT_SYM identifier
          
        | identifier DOT_SYM identifier DOT_SYM identifier
          
       ;

function_call_keyword:
          CHAR_SYM OPEN_PAR_SYM expression_list closing_parenthesis
          
        | CHAR_SYM OPEN_PAR_SYM expression_list USING_SYM charset_name closing_parenthesis
          
        | CURRENT_USER_SYM (optional_braces)?
          
        | DATE_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | DAY_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | HOUR_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | INSERT_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old COMMA_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | INTERVAL_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | INTERVAL_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old COMMA_SYM expression_list closing_parenthesis
          
        | LEFT_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | MINUTE_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | MONTH_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | RIGHT_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | SECOND_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | TIME_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | TIMESTAMP_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | TIMESTAMP_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | TRIM_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | TRIM_SYM OPEN_PAR_SYM LEADING_SYM expr_old FROM_SYM expr_old closing_parenthesis
          
        | TRIM_SYM OPEN_PAR_SYM TRAILING_SYM expr_old FROM_SYM expr_old closing_parenthesis
          
        | TRIM_SYM OPEN_PAR_SYM BOTH_SYM expr_old FROM_SYM expr_old closing_parenthesis
          
        | TRIM_SYM OPEN_PAR_SYM LEADING_SYM FROM_SYM expr_old closing_parenthesis
          
        | TRIM_SYM OPEN_PAR_SYM TRAILING_SYM FROM_SYM expr_old closing_parenthesis
          
        | TRIM_SYM OPEN_PAR_SYM BOTH_SYM FROM_SYM expr_old closing_parenthesis
          
        | TRIM_SYM OPEN_PAR_SYM expr_old FROM_SYM expr_old closing_parenthesis
          
        | USER_SYM OPEN_PAR_SYM closing_parenthesis
                  | GLOBAL_SYM DOT_SYM  
        | LOCAL_SYM DOT_SYM   
        | SESSION_SYM DOT_SYM 
        | YEAR_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
       ;

function_call_nonkeyword:
          ADDDATE_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | ADDDATE_SYM OPEN_PAR_SYM expr_old COMMA_SYM INTERVAL_SYM expr_old interval closing_parenthesis
          
        | CURDATE_SYM (optional_braces)?
          
        | CURTIME_SYM (optional_braces)?
          
        | CURTIME_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | DATE_ADD_INTERVAL_SYM OPEN_PAR_SYM expr_old COMMA_SYM INTERVAL_SYM expr_old interval closing_parenthesis

        | DATE_SUB_INTERVAL_SYM OPEN_PAR_SYM expr_old COMMA_SYM INTERVAL_SYM expr_old interval closing_parenthesis
          
        | EXTRACT_SYM OPEN_PAR_SYM interval FROM_SYM expr_old closing_parenthesis
          
        | GET_FORMAT_SYM OPEN_PAR_SYM date_time_type  COMMA_SYM expr_old closing_parenthesis
          
        | NOW_SYM (optional_braces)?
          
        | NOW_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | POSITION_SYM OPEN_PAR_SYM bit_expr IN_SYM expr_old closing_parenthesis
          
        | SUBDATE_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | SUBDATE_SYM OPEN_PAR_SYM expr_old COMMA_SYM INTERVAL_SYM expr_old interval closing_parenthesis
          
        | SUBSTRING_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | SUBSTRING_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | SUBSTRING_SYM OPEN_PAR_SYM expr_old FROM_SYM expr_old FOR_SYM expr_old closing_parenthesis
          
        | SUBSTRING_SYM OPEN_PAR_SYM expr_old FROM_SYM expr_old closing_parenthesis
          
        | SYSDATE_SYM (optional_braces)?
          
        | SYSDATE_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | TIMESTAMP_ADD_SYM OPEN_PAR_SYM interval_time_stamp COMMA_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | TIMESTAMP_DIFF_SYM OPEN_PAR_SYM interval_time_stamp COMMA_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | UTC_DATE_SYM (optional_braces)?
          
        | UTC_TIME_SYM (optional_braces)?
          
        | UTC_TIMESTAMP_SYM (optional_braces)?
          
       ;

function_call_generic:
          IDENTIFIER /*IDENT_sys*/ OPEN_PAR_SYM
          
          (opt_udf_expr_list)? closing_parenthesis
          
        | IDENTIFIER DOT_SYM IDENTIFIER OPEN_PAR_SYM (opt_expr_list)? closing_parenthesis
          
       ;

opt_udf_expr_list:
        /* empty -- replaced by ()? and  on a higher level     
        |*/ udf_expr_list 
       ;

udf_expr_list:   udf_expr (COMMA_SYM udf_expr)*;

udf_expr:
          /*remember_name*/ expr_old /*remember_end*/ (select_alias)?
          
       ;

opt_expr_list:
          /* empty -- replaced by ()? and  on a higher level 
        |*/ expression_list 
       ;


date_time_type:
          DATE_SYM  
        | TIME_SYM  
        | DATETIME_SYM  
        | TIMESTAMP_SYM 
       ;

function_call_conflict:
          ASCII_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | CHARSET_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | COALESCE_SYM OPEN_PAR_SYM expression_list closing_parenthesis
          
        | COLLATION_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | DATABASE_SYM OPEN_PAR_SYM closing_parenthesis
          
        | IF_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | MICROSECOND_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | MOD_SYM55 OPEN_PAR_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | OLD_PASSWORD_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | PASSWORD_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | QUARTER_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | REPEAT_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | REPLACE_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | TRUNCATE_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | WEEK_SYM OPEN_PAR_SYM expr_old closing_parenthesis
          
        | WEEK_SYM OPEN_PAR_SYM expr_old COMMA_SYM expr_old closing_parenthesis
          
        | geometry_function
          
       ;	


interval_time_stamp:
	interval_time_st	
	| FRAC_SECOND_SYM	
	;


interval:
	interval_time_st
	| DAY_HOUR_SYM
	| DAY_MICROSECOND_SYM
	| DAY_MINUTE_SYM
	| DAY_SECOND_SYM
	| HOUR_MICROSECOND_SYM
	| HOUR_MINUTE_SYM
	| HOUR_SECOND_SYM
	| MINUTE_MICROSECOND_SYM
	| MINUTE_SECOND_SYM
	| SECOND_MICROSECOND_SYM
	| YEAR_MONTH_SYM
;

interval_time_st:
	DAY_SYM   
	| WEEK_SYM
	| HOUR_SYM
	| MINUTE_SYM
	| MONTH_SYM
	| QUARTER_SYM
	| SECOND_SYM
	| MICROSECOND_SYM
	| YEAR_SYM
;

commit:
          COMMIT_SYM (opt_work)? (opt_chain)? (opt_release)?
          
       ;

opt_work:
	 WORK_SYM  
       ;

opt_chain:
        AND_SYM NO_SYM CHAIN_SYM 
        | AND_SYM CHAIN_SYM        
       ;

opt_release:
         RELEASE_SYM        
        | NO_SYM RELEASE_SYM 
;

rollback:
          ROLLBACK_SYM (opt_work)? (opt_chain)? (opt_release)?
          
        | ROLLBACK_SYM (opt_work)?
          TO_SYM (opt_savepoint)? identifier
          
       ;

opt_savepoint:
         SAVEPOINT_SYM 
       ;

execute:
          EXECUTE_SYM identifier
          
          execute_using
          
       ;

execute_using:
          /* nothing */
        | USING_SYM execute_var_list
       ;

execute_var_list:
          execute_var_ident (COMMA_SYM execute_var_ident)*
       ;

execute_var_ident:
          AT_SIGN_SYM ident_or_text
          
       ;

delete_clause:
          DELETE_SYM
          
          opt_delete_options single_multi 
       ;

opt_delete_options: opt_delete_option (opt_delete_option)*;

opt_delete_option:
          QUICK_SYM        
        | LOW_PRIORITY_SYM 
        | IGNORE_SYM   
       ;
        
single_multi:
          FROM_SYM 
                    
          (where_clause)? (order_clause)?
          (LIMIT_SYM limit_option)? 
        | table_wild_list
          
          FROM_SYM table_references (where_clause)?
          
        | FROM_SYM table_alias_ref_list
          
          USING_SYM table_references (where_clause)?
          
       ;
        
table_alias_ref_list:	 table_alias_ref (COMMA_SYM table_alias_ref)*;

table_alias_ref:
          table_identifier
          
       ;        

table_wild_list:		 table_wild_one (COMMA_SYM table_wild_one)*;

table_wild_one:
          identifier (opt_wild)? (table_alias)?
          
        | identifier DOT_SYM identifier (opt_wild)?  (table_alias)?
          
       ;

opt_wild:
          /* empty -- replaced by ()? and  on a higher level 
        |*/ DOT_SYM '*' 
       ;
      
//--------------------------------------------------------------------------------------------------

insert_statement:
	INSERT_SYM (insert_option)? (IGNORE_SYM)? (INTO_SYM)? target_table insert_field_spec (duplicate_key_update)?
;

insert_option:
	LOW_PRIORITY_SYM
	| DELAYED_SYM
	| HIGH_PRIORITY_SYM
;

target_table:
	table_identifier -> ^(TABLE_REF_ID table_identifier)
;
	
insert_field_spec:
	(OPEN_PAR_SYM short_field_name_list closing_parenthesis)? insert_values
	| SET_SYM column_assignment_list_with_default
;

insert_values:
	(VALUES_SYM | VALUE_SYM) value_list
	| select_statement
;

value_list:
	OPEN_PAR_SYM (expression | DEFAULT_SYM) (COMMA_SYM (expression | DEFAULT_SYM))* closing_parenthesis
;

column_assignment_list_with_default:
	column_assignment_with_default (COMMA_SYM column_assignment_with_default)*
;

column_assignment_with_default:
	field_name EQUAL_SYM55 (expression | DEFAULT_SYM)
;
	
duplicate_key_update:
	ON_SYM DUPLICATE_SYM KEY_SYM UPDATE_SYM55 insert_update_list
;

insert_update_list:
	column_assignment (COMMA_SYM column_assignment)*
;

column_assignment:
	field_name EQUAL_SYM55 expression
;

//--------------------------------------------------------------------------------------------------

update_statement:
	UPDATE_SYM55 (LOW_PRIORITY_SYM)? (IGNORE_SYM)? update_tail
;

update_tail:
	table_reference SET_SYM column_assignment_list_with_default (where_clause)? (order_clause)? (LIMIT_SYM limit_option)?
	| table_references SET_SYM column_assignment_list_with_default (where_clause)?
;

//--------------------------------------------------------------------------------------------------

select_statement:
	SELECT_SYM55 select_init
	| OPEN_PAR_SYM select_statement closing_parenthesis (union_opt)?
;
  
subquery:
	OPEN_PAR_SYM select_statement closing_parenthesis -> ^(SUBQUERY OPEN_PAR_SYM select_statement closing_parenthesis)
;

subselect_deprecated:
	SELECT_SYM55 select_init
	| OPEN_PAR_SYM select_paren closing_parenthesis
;

select_init:
	select_part1 (union_clause)?
;

union_clause:
	union_list
;

select_part1:
	(select_option)* select_item (COMMA_SYM select_item)* (select_source_and_options)? (select_from_tail)?
;

select_option:
	(ALL_SYM | DISTINCT_SYM | DISTINCTROW_SYM)
	| STRAIGHT_JOIN_SYM
	| HIGH_PRIORITY_SYM
	| SQL_SMALL_RESULT_SYM
	| SQL_BIG_RESULT_SYM
	| SQL_BUFFER_RESULT_SYM
	| SQL_CALC_FOUND_ROWS_SYM
	| (SQL_NO_CACHE_SYM | SQL_CACHE_SYM)
;

select_item:
	select_expr (select_alias)?
;

select_alias:
	AS_SYM? (identifier | TEXT_STRING)
;         

select_expr:
	MULT -> ^(SELECT_EXPR MULT)
	| expression -> ^(SELECT_EXPR expression)
;

field_name_list:
	field_name (COMMA_SYM field_name)*
;
	
field_name:
	{LA(2) != DOT_SYM}? identifier -> ^(FIELD_REF_ID identifier)
	| schema_table_column_id  -> ^(FIELD_REF_ID schema_table_column_id) // Separated to have an error node when the id after the dot is missing.
;

short_field_name_list: // Column without a schema qualifier.
	short_field_name (COMMA_SYM short_field_name)*
;
	
short_field_name:
	{LA(2) != DOT_SYM}? identifier -> ^(FIELD_REF_ID identifier)
	| table_column_identifier -> ^(FIELD_REF_ID table_column_identifier)
;

schema_table_column_id:
	identifier DOT_SYM
	(
		{LA(2) != DOT_SYM}? qualified_identifier
		| table_column_identifier
	)
;
	
table_column_identifier:
	identifier DOT_SYM qualified_identifier
;

qualified_identifier:
	MULT
	| identifier
	| keyword -> keyword // Explicitly include all word tokens as they are automatically excluded
	                     // from IDENTIFIER by ANTLR, but are allowed after a dot.
	| -> MISSING_ID
;

function_call:
	// Function names that are keywords.
	CHAR_SYM OPEN_PAR_SYM expression_list closing_parenthesis
	| CHAR_SYM OPEN_PAR_SYM expression_list USING_SYM charset_name closing_parenthesis
	| CURRENT_USER_SYM (parentheses)?
	| DATE_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| DAY_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| HOUR_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| INSERT_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression COMMA_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| INTERVAL_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| INTERVAL_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| LEFT_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| MINUTE_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| MONTH_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| RIGHT_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| SECOND_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| TIME_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| TIMESTAMP_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| TIMESTAMP_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| TRIM_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| TRIM_SYM OPEN_PAR_SYM LEADING_SYM function_expression FROM_SYM function_expression closing_parenthesis
	| TRIM_SYM OPEN_PAR_SYM TRAILING_SYM function_expression FROM_SYM function_expression closing_parenthesis
	| TRIM_SYM OPEN_PAR_SYM BOTH_SYM function_expression FROM_SYM function_expression closing_parenthesis
	| TRIM_SYM OPEN_PAR_SYM LEADING_SYM FROM_SYM function_expression closing_parenthesis
	| TRIM_SYM OPEN_PAR_SYM TRAILING_SYM FROM_SYM function_expression closing_parenthesis
	| TRIM_SYM OPEN_PAR_SYM BOTH_SYM FROM_SYM function_expression closing_parenthesis
	| TRIM_SYM OPEN_PAR_SYM function_expression FROM_SYM function_expression closing_parenthesis
	| USER_SYM parentheses
	| YEAR_SYM OPEN_PAR_SYM function_expression closing_parenthesis

	// Function names that not keywords.
	| ADDDATE_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| ADDDATE_SYM OPEN_PAR_SYM function_expression COMMA_SYM INTERVAL_SYM function_expression interval closing_parenthesis
	| CURDATE_SYM (parentheses)?
	| CURTIME_SYM (parentheses)?
	| CURTIME_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| DATE_ADD_INTERVAL_SYM OPEN_PAR_SYM function_expression COMMA_SYM INTERVAL_SYM function_expression interval closing_parenthesis
	| DATE_SUB_INTERVAL_SYM OPEN_PAR_SYM function_expression COMMA_SYM INTERVAL_SYM function_expression interval closing_parenthesis
	| EXTRACT_SYM OPEN_PAR_SYM interval FROM_SYM function_expression closing_parenthesis
	| GET_FORMAT_SYM OPEN_PAR_SYM date_time_type  COMMA_SYM function_expression closing_parenthesis
	| NOW_SYM (parentheses)?
	| NOW_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| POSITION_SYM OPEN_PAR_SYM bit_expr IN_SYM function_expression closing_parenthesis
	| SUBDATE_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| SUBDATE_SYM OPEN_PAR_SYM function_expression COMMA_SYM INTERVAL_SYM function_expression interval closing_parenthesis
	| SUBSTRING_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression COMMA_SYM expression closing_parenthesis
	| SUBSTRING_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| SUBSTRING_SYM OPEN_PAR_SYM function_expression FROM_SYM function_expression FOR_SYM expression closing_parenthesis
	| SUBSTRING_SYM OPEN_PAR_SYM function_expression FROM_SYM function_expression closing_parenthesis
	| SYSDATE_SYM (parentheses)?
	| SYSDATE_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| TIMESTAMP_ADD_SYM OPEN_PAR_SYM interval_time_stamp COMMA_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| TIMESTAMP_DIFF_SYM OPEN_PAR_SYM interval_time_stamp COMMA_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| UTC_DATE_SYM (parentheses)?
	| UTC_TIME_SYM (parentheses)?
	| UTC_TIMESTAMP_SYM (parentheses)?

	| ASCII_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| CHARSET_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| COALESCE_SYM OPEN_PAR_SYM expression_list closing_parenthesis
	| COLLATION_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| DATABASE_SYM parentheses
	| IF_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| MICROSECOND_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| MOD_SYM55 OPEN_PAR_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| OLD_PASSWORD_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| PASSWORD_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| QUARTER_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| REPEAT_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| REPLACE_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| TRUNCATE_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| WEEK_SYM OPEN_PAR_SYM function_expression closing_parenthesis
	| WEEK_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| geometry_function
	| aggregate_function

	// Generic function call.
	| IDENTIFIER OPEN_PAR_SYM (aliased_expression_list)? closing_parenthesis
	| IDENTIFIER DOT_SYM IDENTIFIER OPEN_PAR_SYM (expression_list)? closing_parenthesis
;

geometry_function:
	CONTAINS_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| GEOMETRYCOLLECTION_SYM OPEN_PAR_SYM expression_list closing_parenthesis
	| LINESTRING OPEN_PAR_SYM expression_list closing_parenthesis
	| MULTILINESTRING_SYM OPEN_PAR_SYM expression_list closing_parenthesis
	| MULTIPOINT_SYM OPEN_PAR_SYM expression_list closing_parenthesis
	| MULTIPOLYGON_SYM OPEN_PAR_SYM expression_list closing_parenthesis
	| POINT_SYM OPEN_PAR_SYM function_expression COMMA_SYM function_expression closing_parenthesis
	| POLYGON_SYM OPEN_PAR_SYM expression_list closing_parenthesis
;

aliased_expression_list:
	aliased_expression (COMMA_SYM aliased_expression)*
;

aliased_expression:
	function_expression (select_alias)?
;

aggregate_function:
	AVG_SYM OPEN_PAR_SYM aggregate_expression closing_parenthesis
	| AVG_SYM OPEN_PAR_SYM DISTINCT_SYM aggregate_expression closing_parenthesis
	| BIT_AND_SYM  OPEN_PAR_SYM aggregate_expression closing_parenthesis
	| BIT_OR_SYM  OPEN_PAR_SYM aggregate_expression closing_parenthesis
	| BIT_XOR_SYM  OPEN_PAR_SYM aggregate_expression closing_parenthesis
	| COUNT_SYM OPEN_PAR_SYM (ALL_SYM)? '*' closing_parenthesis
	| COUNT_SYM OPEN_PAR_SYM (DISTINCT_SYM)? expression_list closing_parenthesis
	| MIN_SYM OPEN_PAR_SYM (DISTINCT_SYM)? aggregate_expression closing_parenthesis
	| MAX_SYM OPEN_PAR_SYM (DISTINCT_SYM)? aggregate_expression closing_parenthesis
	| STD_SYM OPEN_PAR_SYM aggregate_expression closing_parenthesis
	| VARIANCE_SYM OPEN_PAR_SYM aggregate_expression closing_parenthesis
	| STDDEV_SAMP_SYM OPEN_PAR_SYM aggregate_expression closing_parenthesis
	| VAR_SAMP_SYM OPEN_PAR_SYM aggregate_expression closing_parenthesis
	| SUM_SYM OPEN_PAR_SYM (DISTINCT_SYM)? aggregate_expression closing_parenthesis
	| GROUP_CONCAT_SYM OPEN_PAR_SYM (DISTINCT_SYM)? expression_list order_clause (SEPARATOR_SYM text_string)? closing_parenthesis
;

aggregate_expression:
	(ALL_SYM)? select_expr
;

function_expression:
	function_call -> ^(FUNCTION_CALL function_call)
	| expression
;

//--------------------------------------------------------------------------------------------------

select_source_and_options:
	limit_clause
	| into (select_from)?
	| select_from (into)?
;

limit_clause: // [LIMIT {[offset,] row_count | row_count OFFSET offset}]
	LIMIT_SYM limit_options
;

limit_options:
	limit_option ((COMMA_SYM | OFFSET_SYM) limit_option)?
;

limit_option:
	PARAM_MARKER
	| number
;

into:
	INTO_SYM into_destination
;

into_destination:
	OUTFILE_SYM text_string_filesystem (opt_load_data_charset)? (opt_field_term)? (opt_line_term)?
	| DUMPFILE_SYM text_string_filesystem
	| select_var_list_init
;

opt_load_data_charset:
	charset charset_name_or_default 
;

charset_name_or_default:
          charset_name 
        | DEFAULT_SYM    
       ;

opt_field_term:
          /* empty -- replaced by ()? and  on a higher level
        |*/ COLUMNS_SYM field_term_list
       ;


field_term_list:  field_term (field_term)*;
        
field_term:
          TERMINATED_SYM BY_SYM text_string 
          
        | OPTIONALLY_SYM ENCLOSED_SYM BY_SYM text_string
          
        | ENCLOSED_SYM BY_SYM text_string
          
        | ESCAPED_SYM BY_SYM text_string
          
       ;

opt_line_term:
          /* empty
        |*/ LINES_SYM line_term_list
       ;

line_term_list:	line_term (line_term)*;

line_term:
          TERMINATED_SYM BY_SYM text_string
          
        | STARTING_SYM BY_SYM text_string
;

select_var_list_init:
          
          select_var_list
          
       ;

select_var_list:  select_var_ident (COMMA_SYM select_var_ident)*;

select_var_ident:  
          AT_SIGN_SYM ident_or_text
          
        | ident_or_text
          
       ;

select_from:
	FROM_SYM
	(
		DUAL_SYM (where_clause)? (limit_clause)?-> ^(FROM_SYM DUAL_SYM where_clause)
		| table_references (where_clause)? (group_clause)? (having_clause)? (order_clause)? (limit_clause)? (procedure_clause)?
	    	-> ^(FROM_SYM table_references where_clause group_clause having_clause order_clause limit_clause procedure_clause)
	    | -> MISSING_TABLE_REF
	)
;

procedure_clause:
	PROCEDURE_SYM identifier OPEN_PAR_SYM (expression_list)? closing_parenthesis -> ^(PROCEDURE_SYM identifier expression_list)
;

expression_list:
	expression (COMMA_SYM expression)*
;
	
having_clause:
	HAVING_SYM expression-> ^(HAVING_SYM expression)
;

group_clause:
	GROUP_SYM55 BY_SYM group_list (olap_option)? -> ^(GROUP_SYM55 BY_SYM group_list (olap_option)?)
;

group_list:
	expression (order_dir)? (COMMA_SYM expression (order_dir)?)*
;

olap_option:
	WITH_SYM CUBE_SYM
	| WITH_SYM ROLLUP_SYM
;

order_clause:
	ORDER_SYM55 BY_SYM order_list -> ^(ORDER_SYM55 BY_SYM order_list)
;

order_list:
	expression (order_dir)? (COMMA_SYM expression (order_dir)?)*
;
	
order_dir:
	ASC_SYM
	| DESC_SYM
;
        
where_clause:
	WHERE_SYM expression -> ^(WHERE_SYM expression)
;

table_references:
	 table_reference_list -> ^(TABLE_REFERENCES table_reference_list)
;

table_reference_list:
	table_reference (COMMA_SYM table_reference)*
;

table_reference:
	table_factor (join)* -> ^(TABLE_REFERENCE table_factor (join)*)
;

// The OJ construct is a bit special as it requires a left outer join, so this must be handled separately.
// We still create the same AST as for the general case to ease common parsing.
oj_table_reference:
	table_factor oj_join -> ^(TABLE_REFERENCE table_factor oj_join)
;
	
join:
	join_table -> ^(JOIN join_table)
;

oj_join:
	LEFT_SYM OUTER_SYM JOIN_SYM55 table_reference ON_SYM expression -> ^(JOIN LEFT_SYM OUTER_SYM JOIN_SYM55 table_reference ON_SYM expression)
;
	
table_factor:
	subquery (AS_SYM)? identifier // Alias is mandatory.
	| OPEN_PAR_SYM table_references closing_parenthesis
	| OJ_SYM oj_table_reference
	| table_identifier (table_alias)? (index_hint_list)? -> ^(TABLE_REF_ID table_identifier (table_alias)? (index_hint_list)?)
;

join_table:
	(INNER_SYM55 | CROSS_SYM)? JOIN_SYM55 table_factor (join_condition)?
	| STRAIGHT_JOIN_SYM table_factor (ON_SYM expression)?
	| (LEFT_SYM | RIGHT_SYM)? (OUTER_SYM)? JOIN_SYM55 table_reference join_condition
	| NATURAL_SYM ((LEFT_SYM | RIGHT_SYM) (OUTER_SYM)?)? JOIN_SYM55 table_factor
;

join_condition:
	ON_SYM expression
	| USING_SYM OPEN_PAR_SYM column_list closing_parenthesis
;

column_list:
	identifier (COMMA_SYM identifier)*
;
          

union_opt:
	union_list
	| union_order_or_limit
; 

union_order_or_limit:
          
          order_or_limit
          
       ;
        
order_or_limit:
          order_clause (limit_clause)?
        | limit_clause
       ;        

union_list:
  UNION_SYM (union_option)? select_statement -> ^(UNION_RULE (union_option)? select_statement)
 ;

union_option:
        DISTINCT_SYM  
        | ALL_SYM       
       ;

select_paren:
          SELECT_SYM55 select_part1
          
        | OPEN_PAR_SYM select_paren closing_parenthesis
       ;

        
select_from_tail:
	FOR_SYM UPDATE_SYM55
	| LOCK_SYM IN_SYM SHARE_SYM MODE_SYM
;
        
table_identifier:
 	identifier (DOT_SYM table_identifier2)?
 ;       

table_identifier2:
	identifier
	| keyword -> keyword // Explicitly include all word tokens as they are automatically excluded
	                     // from IDENTIFIER by ANTLR, but are allowed after a dot.
	| -> MISSING_ID
;

table_alias:
	(AS_SYM)? identifier
;

index_hint_list:
	index_hint (COMMA_SYM index_hint)* -> ^(INDEX_HINT_LIST index_hint (COMMA_SYM index_hint)*)
;

index_hint:
	index_hint_type key_or_index (index_hint_clause)? OPEN_PAR_SYM index_list closing_parenthesis
	| USE_SYM key_or_index (index_hint_clause)? OPEN_PAR_SYM (index_list)? closing_parenthesis
;

index_hint_type:
	FORCE_SYM
	| IGNORE_SYM
;

key_or_index:
	KEY_SYM
	| INDEX_SYM
;

index_hint_clause:
	FOR_SYM	(JOIN_SYM55 | ORDER_SYM55 BY_SYM | GROUP_SYM55 BY_SYM)
;

index_list:
	index_list_element (COMMA_SYM index_list_element)*
;

index_list_element:
	identifier
	| PRIMARY_SYM
;

parentheses:
	OPEN_PAR_SYM closing_parenthesis
;

// An own rule only for this, to allow reporting a missing one.
closing_parenthesis:
	CLOSE_PAR_SYM
	| -> MISSING_PAR
;
	
text_string_sys:		TEXT_STRING;
text_string_literal:	TEXT_STRING;
text_string_filesystem:	TEXT_STRING;       

//----- Lexer tokens -------------------------------------------------------------------------------

// Due to name clashes I have to accept some inconsistent naming, unless we want an ugly name scheme for all of the tokens.

// Operators
EQUAL_SYM55:		'=';  // Also assign.
ASSIGN: 			':=';
NULL_SAFE_EQUAL:	'<=>';
GREATER_OR_EQUAL:	'>=';
GREATER_THAN:		'>';
LESS_OR_EQUAL:		'<=';
LESS_THAN:			'<';
NOT_EQUAL:			'!=';
NOT_EQUAL2:			'<>';

PLUS_SYM:			'+';
MINUS_SYM:			'-';
MULT:				'*';
DIV:				'/';
DIV_SYM55:			'DIV';

MOD:				'%';
MOD_SYM55:			'MOD';

INC:				'++';
DEC:				'--';

LOGICAL_NOT:		'!';
NOT_SYM:			'NOT';
BITWISE_NOT:		'~';

SHIFT_LEFT:			'<<';
SHIFT_RIGHT:		'>>';

LOGICAL_AND:		'&&';
BITWISE_AND:		'&';
AND_SYM:			'AND';

BITWISE_XOR:		'^';
XOR_SYM:			'XOR';

LOGICAL_OR:			'||';
BITWISE_OR:			'|';
OR_SYM:				'OR';

DOT_SYM:			'.';
COMMA_SYM:			',';
SEMICOLON_SYM:		';';
COLON_SYM:			':';
OPEN_PAR_SYM:		'(';
CLOSE_PAR_SYM:		')';
UNDERLINE_SYM:		'_';
AT_SIGN_SYM:		'@';
AT_AT_SIGN_SYM:		'@@';

PARAM_MARKER:		'?';
BACK_TICK:			'`';
SINGLE_QUOTE:		'\'';
DOUBLE_QUOTE:		'"';

CREATE_SYM:			'CREATE';
PROCEDURE_SYM:		'PROCEDURE';
DEFINER_SYM:		'DEFINER';
CURRENT_USER_SYM:	'CURRENT''USER';
IN_SYM:				'IN';
OUT_SYM:			'OUT';
INOUT_SYM:			'INOUT';

// Datatypes
FLOAT_SYM:		'FLOAT';
BIT_SYM:		'BIT';
BOOL_SYM:		'BOOL';		
BOOLEAN_SYM:	'BOOLEAN';
BINARY_SYM:		'BINARY';
VARBINARY_SYM:	'VARBINARY';
YEAR_SYM:		'YEAR';
DATE_SYM:		'DATE';		
TIME_SYM:		'TIME';
TIMESTAMP_SYM:	'TIMESTAMP';
DATETIME_SYM:	'DATETIME';
BLOB_SYM:		'BLOB';
TINYBLOB_SYM:	'TINYBLOB';
MEDIUMBLOB_SYM:	'MEDIUMBLOB';
LONGBLOB_SYM:	'LONGBLOB';
LONG_SYM:		'LONG';
TEXT_SYM:		'TEXT';
TINYTEXT_SYM:	'TINYTEXT';
MEDIUMTEXT_SYM:	'MEDIUMTEXT';
LONGTEXT_SYM:	'LONGTEXT';
DECIMAL_SYM:	'DECIMAL';
NUMERIC_SYM:	'NUMERIC';
FIXED_SYM:		'FIXED';
ENUM_SYM:		'ENUM';	
SET_SYM:		'SET';	
SERIAL_SYM:		'SERIAL';		
INT_SYM:		'INT';
TINYINT_SYM:	'TINYINT';
SMALLINT_SYM:	'SMALLINT';
MEDIUMINT_SYM:	'MEDIUMINT';
BIGINT_SYM:		'BIGINT';
SIGNED_SYM:		'SIGNED';
UNSIGNED_SYM:	'UNSIGNED';
ZEROFILL_SYM:	'ZEROFILL';	
DOUBLE_SYM:		'DOUBLE';
REAL_SYM:		'REAL';
PRECISION_SYM:	'PRECISION';
CHAR_SYM:		'CHAR';
CHARSET_SYM:	'CHARSET';
VARYING_SYM:	'VARYING';
VARCHAR_SYM:	'VARCHAR';

ASCII_SYM:		'ASCII';
BYTE_SYM:		'BYTE';
UNICODE_SYM:	'UNICODE';
NCHAR_SYM:		'NCHAR';
NATIONAL_SYM:	'NATIONAL';
NVARCHAR_SYM:	'NVARCHAR';

GEOMETRY_SYM:			'GEOMETRY';
GEOMETRYCOLLECTION_SYM:	'GEOMETRYCOLLECTION';
POINT_SYM:				'POINT';
MULTIPOINT_SYM:			'MULTIPOINT';
LINESTRING:				'LINESTRING';
MULTILINESTRING_SYM:	'MULTILINESTRING';
POLYGON_SYM:			'POLYGON';
MULTIPOLYGON_SYM:		'MULTIPOLYGON';

COMMENT_SYM:		'COMMENT';
LANGUAGE_SYM:		'LANGUAGE';
SQL_SYM:			'SQL';	
NO_SYM:				'NO';	
CONTAINS_SYM:		'CONTAINS';
READS_SYM:			'READS';
DATA_SYM:			'DATA';
MODIFIES_SYM:		'MODIFIES';
DETERMINISTIC_SYM:	'DETERMINISTIC';

SECURITY_SYM:	'SECURITY';
INVOKER_SYM:	'INVOKER';
RETURN_SYM:		'RETURN';

TRUE_SYM:		'TRUE';
IS_SYM:			'IS';
NULL_SYM:		'NULL';
FALSE_SYM:		'FALSE';	

UNKNOWN_SYM:	'UNKNOWN';	
ALL_SYM:		'ALL';
ANY_SYM:		'ANY';
BETWEEN_SYM:	'BETWEEN';
SOUNDS_SYM:		'SOUNDS';
LIKE_SYM:		'LIKE';
REGEXP_SYM:		'REGEXP';

INTERVAL_SYM:			'INTERVAL';
DAY_HOUR_SYM:			'DAY_HOUR';	
DAY_MICROSECOND_SYM:	'DAY_MICROSECOND';
DAY_MINUTE_SYM:			'DAY_MINUTE';
DAY_SECOND_SYM:			'DAY_SECOND';
HOUR_MICROSECOND_SYM:	'HOUR_MICROSECOND';
HOUR_MINUTE_SYM:		'HOUR_MINUTE';
HOUR_SECOND_SYM:		'HOUR_SECOND';
MINUTE_MICROSECOND_SYM:	'MINUTE_MICROSECOND';
MINUTE_SECOND_SYM:		'MINUTE_SECOND';
SECOND_MICROSECOND_SYM:	'SECOND_MICROSECOND';
YEAR_MONTH_SYM:			'YEAR_MONTH';
DAY_SYM	:				'DAY';
WEEK_SYM:				'WEEK';
HOUR_SYM:				'HOUR';
MINUTE_SYM:				'MINUTE';
MONTH_SYM:				'MONTH';
QUARTER_SYM:			'QUARTER';
SECOND_SYM:				'SECOND';
MICROSECOND_SYM:		'MICROSECOND';

SELECT_SYM55:			'SELECT'; // Name clash with other SELECT_SYM.
AS_SYM:					'AS';
DISTINCT_SYM:			'DISTINCT';
DISTINCTROW_SYM:		'DISTINCTROW';
STRAIGHT_JOIN_SYM:		'STRAIGHT_JOIN';
HIGH_PRIORITY_SYM:		'HIGH_PRIORITY';
	
SQL_SMALL_RESULT_SYM:	'SQL_SMALL_RESULT';
SQL_BIG_RESULT_SYM:		'SQL_BIG_RESULT';	
SQL_BUFFER_RESULT_SYM:	'SQL_BUFFER_RESULT';	
SQL_CALC_FOUND_ROWS_SYM:'SQL_CALC_FOUND_ROWS';
SQL_NO_CACHE_SYM:		'SQL_NO_CACHE';	
SQL_CACHE_SYM:			'SQL_CACHE';

ORDER_SYM55:	'ORDER';
BY_SYM:			'BY';		
ASC_SYM:		'ASC';
DESC_SYM:		'DESC';
LIMIT_SYM:		'LIMIT';
OFFSET_SYM:		'OFFSET';
INTO_SYM:		'INTO';			
OUTFILE_SYM:	'OUTFILE';
DUMPFILE_SYM:	'DUMPFILE';

DEFAULT_SYM:	'DEFAULT';
COLUMNS_SYM:	'COLUMNS';
TERMINATED_SYM:	'TERMINATED';
OPTIONALLY_SYM:	'OPTIONALLY';
ENCLOSED_SYM:	'ENCLOSED';
ESCAPED_SYM:	'ESCAPED';
LINES_SYM:		'LINES';
STARTING_SYM:	'STARTING';
FROM_SYM:		'FROM';
DUAL_SYM:		'DUAL';
FORCE_SYM:		'FORCE';
IGNORE_SYM:		'IGNORE';	

KEY_SYM:		'KEY';
INDEX_SYM:		'INDEX';
FOR_SYM:		'FOR';
JOIN_SYM55:		'JOIN';
GROUP_SYM55:	'GROUP';
PRIMARY_SYM:	'PRIMARY';
USE_SYM:		'USE';
UPDATE_SYM55:	'UPDATE';
LOCK_SYM:		'LOCK';
SHARE_SYM:		'SHARE';
MODE_SYM:		'MODE';
UNION_SYM:		'UNION';		
INNER_SYM55:	'INNER';
CROSS_SYM:		'CROSS';
ON_SYM:			'ON';
USING_SYM:		'USING';
NATURAL_SYM:	'NATURAL';
LEFT_SYM:		'LEFT';
RIGHT_SYM:		'RIGHT';
OUTER_SYM:		'OUTER';
WHERE_SYM:		'WHERE';
CUBE_SYM:		'CUBE';
ROLLUP_SYM:		'ROLLUP';
WITH_SYM:		'WITH';
HAVING_SYM:		'HAVING';	
COLLATE_SYM:	'COLLATE';
ROW_SYM:		'ROW';
EXISTS_SYM:		'EXISTS';
MATCH_SYM:		'MATCH';
AGAINST_SYM:	'AGAINST';
CAST_SYM:		'CAST';
END_SYM:		'END';
CONVERT_SYM:	'CONVERT';
VALUES_SYM:		'VALUES';

TRIM_SYM:		'TRIM';
LEADING_SYM:	'LEADING';
TRAILING_SYM:	'TRAILING';
BOTH_SYM:		'BOTH';
USER_SYM:		'USER';
INSERT_SYM:		'INSERT';

ADDDATE_SYM:		'ADDDATE';
CURDATE_SYM:		'CURDATE';
CURTIME_SYM:		'CURTIME';
DATE_ADD_INTERVAL_SYM:	'DATE_ADD_INTERVAL';
DATE_SUB_INTERVAL_SYM:	'DATE_SUB_INTERVAL';
EXTRACT_SYM:		'EXTRACT';
GET_FORMAT_SYM:		'GET''FORMAT';
NOW_SYM:			'NOW';
POSITION_SYM:		'POSITION';
SUBDATE_SYM:		'SUBDATE';
SUBSTRING_SYM:		'SUBSTRING';
SYSDATE_SYM:		'SYSDATE';
TIMESTAMP_ADD_SYM:	'TIMESTAMP_ADD';
TIMESTAMP_DIFF_SYM:	'TIMESTAMP_DIFF';
UTC_DATE_SYM:		'UTC_DATE';
UTC_TIME_SYM:		'UTC_TIME';
UTC_TIMESTAMP_SYM:	'UTC_TIMESTAMP';
FRAC_SECOND_SYM:	'FRAC_SECOND';

COALESCE_SYM:		'COALESCE';
COLLATION_SYM:		'COLLATION';
DATABASE_SYM:		'DATABASE';
IF_SYM:				'IF';
OLD_PASSWORD_SYM:	'OLD_PASSWORD';
PASSWORD_SYM:		'PASSWORD';
REPEAT_SYM:			'REPEAT';
REPLACE_SYM:		'REPLACE';
TRUNCATE_SYM:		'TRUNCATE';

GLOBAL_SYM:			'GLOBAL';
LOCAL_SYM:			'LOCAL';
SESSION_SYM:		'SESSION';
	
BIT_AND_SYM:		'BIT_AND';
BIT_OR_SYM:			'BIT_OR';
BIT_XOR_SYM:		'BIT_XOR';

AVG_SYM:			'AVG';
COUNT_SYM:			'COUNT';
MIN_SYM:			'MIN';
MAX_SYM:			'MAX';
STD_SYM:			'STD';
VARIANCE_SYM:		'VARIANCE';
STDDEV_SAMP_SYM:	'STDDEV_SAMP';
VAR_SAMP_SYM:		'VAR_SAMP';
SUM_SYM:			'SUM';
GROUP_CONCAT_SYM:	'GROUP_CONCAT';

SEPARATOR_SYM:		'SEPARATOR';
QUERY_SYM:			'QUERY';
EXPANSION_SYM:		'EXPANSION';

CASE_SYM:			'CASE';
WHEN_SYM:			'WHEN';
THEN_SYM:			'THEN';
ELSE_SYM:			'ELSE';
ELSEIF_SYM:			'ELSEIF';

ESCAPE_SYM:			'ESCAPE';

ALTER_SYM:				'ALTER';
ACTION_SYM:				'ACTION';
AFTER_SYM:				'AFTER';
AGGREGATE_SYM:			'AGGREGATE';
ALGORITHM_SYM:			'ALGORITHM';
AT_SYM:					'AT';
AUTHORS_SYM:			'AUTHORS';
AUTO_INC_SYM:			'AUTO_INC';
AUTOEXTEND_SIZE_SYM:	'AUTOEXTEND_SIZE';
AVG_ROW_LENGTH_SYM:		'AVG_ROW_LENGTH';
BINLOG_SYM:				'BINLOG';
BLOCK_SYM:				'BLOCK';
BTREE_SYM:				'BTREE';
CALL_SYM55:				'CALL';
CASCADED_SYM:			'CASCADED';
CHAIN_SYM:				'CHAIN';
CHANGED_SYM:			'CHANGED';
CIPHER_SYM:				'CIPHER';
CLIENT_SYM:				'CLIENT';
CODE_SYM:				'CODE';
COMMITTED_SYM:			'COMMITTED';
COMPACT_SYM:			'COMPACT';
COMPLETION_SYM:			'COMPLETION';
COMPRESSED_SYM:			'COMPRESSED';
CONCURRENT_SYM:			'CONCURRENT';
CONNECTION_SYM:			'CONNECTION';
CONSISTENT_SYM:			'CONSISTENT';
CONTEXT_SYM:			'CONTEXT';
CONTRIBUTORS_SYM:		'CONTRIBUTORS';
CPU_SYM:				'CPU';
DATAFILE_SYM:			'DATAFILE';
DELAY_KEY_WRITE_SYM:	'DELAY_KEY_WRITE';
DES_KEY_FILE_SYM:		'DES_KEY_FILE';
DIRECTORY_SYM:			'DIRECTORY';
DISABLE_SYM:			'DISABLE';
DISCARD_SYM:			'DISCARD';
DISK_SYM:				'DISK';
DROP_SYM: 				'DROP';
DUPLICATE_SYM:			'DUPLICATE';
DYNAMIC_SYM:			'DYNAMIC';
ENDS_SYM:				'ENDS';
ENGINE_SYM:				'ENGINE';
ENGINES_SYM:			'ENGINES';
ERRORS_SYM:				'ERRORS';
EVENT_SYM:				'EVENT';
EVENTS_SYM:				'EVENTS';
EVERY_SYM:				'EVERY';
EXTENDED_SYM:			'EXTENDED';
EXTENT_SIZE_SYM:		'EXTENT_SIZE';
FAULTS_SYM:				'FAULTS';
FAST_SYM:				'FAST';
FOUND_SYM:				'FOUND';
ENABLE_SYM:				'ENABLE';
FULL_SYM:				'FULL';
FILE_SYM:				'FILE';
FIRST_SYM:				'FIRST';
GRANTS_SYM:				'GRANTS';
HASH_SYM:				'HASH';
HOSTS_SYM:				'HOSTS';
IDENTIFIED_SYM:			'IDENTIFIED';
IMPORT_SYM:				'IMPORT';
INDEXES_SYM:			'INDEXES';
INITIAL_SIZE_SYM:		'INITIAL_SIZE';
IO_SYM:					'IO';
IPC_SYM:				'IPC';
ISOLATION_SYM:			'ISOLATION';
ISSUER_SYM:				'ISSUER';
INNOBASE_SYM:			'INNOBASE';
INSERT_METHOD_SYM:		'INSERT_METHOD';
KEY_BLOCK_SIZE_SYM:		'KEY_BLOCK_SIZE';
LAST_SYM:				'LAST';
LEAVES_SYM:				'LEAVES';
LEVEL_SYM:				'LEVEL';
LESS_SYM:				'LESS';
LIST_SYM:				'LIST';
LOCKS_SYM:				'LOCKS';
LOGFILE_SYM:			'LOGFILE';
LOGS_SYM:				'LOGS';
MAX_ROWS_SYM:			'MAX_ROWS';
MASTER_SYM:				'MASTER';
MASTER_HOST_SYM:		'MASTER_HOST';
MASTER_PORT_SYM:		'MASTER_PORT';
MASTER_LOG_FILE_SYM:	'MASTER_LOG_FILE';
MASTER_LOG_POS_SYM:		'MASTER_LOG_POS';
MASTER_USER_SYM:		'MASTER_USER';
MASTER_PASSWORD_SYM:	'MASTER_PASSWORD';
MASTER_SERVER_ID_SYM:	'MASTER_SERVER_ID';
MASTER_CONNECT_RETRY_SYM:	'MASTER_CONNECT_RETRY';
MASTER_SSL_SYM:			'MASTER_SSL';
MASTER_SSL_CA_SYM:		'MASTER_SSL_CA';
MASTER_SSL_CAPATH_SYM:	'MASTER_SSL_CAPATH';
MASTER_SSL_CERT_SYM:	'MASTER_SSL_CERT';
MASTER_SSL_CIPHER_SYM:	'MASTER_SSL_CIPHER';
MASTER_SSL_KEY_SYM:		'MASTER_SSL_KEY';
MAX_CONNECTIONS_PER_HOUR_SYM:	'MAX_CONNECTIONS_PER_HOUR';
MAX_QUERIES_PER_HOUR_SYM:	'MAX_QUERIES_PER_HOUR';
MAX_SIZE_SYM:			'MAX_SIZE';
MAX_UPDATES_PER_HOUR_SYM:	'MAX_UPDATES_PER_HOUR';
MAX_USER_CONNECTIONS_SYM:	'MAX_USER_CONNECTIONS';
MAX_VALUE_SYM:			'MAX_VALUE';
MEDIUM_SYM:				'MEDIUM';
MEMORY_SYM:				'MEMORY';
MERGE_SYM:				'MERGE';
MIGRATE_SYM:			'MIGRATE';
MIN_ROWS_SYM:			'MIN_ROWS';
MODIFY_SYM:				'MODIFY';
MUTEX_SYM:				'MUTEX';
NAME_SYM:				'NAME';
NAMES_SYM:				'NAMES';
NDBCLUSTER_SYM:			'NDBCLUSTER';
NEXT_SYM:				'NEXT';
NEW_SYM:				'NEW';
NO_WAIT_SYM:			'NO_WAIT';
NODEGROUP_SYM:			'NODEGROUP';
NONE_SYM:				'NONE';
OJ_SYM:					'OJ';
ONE_SHOT_SYM:			'ONE_SHOT';
ONE_SYM:				'ONE';
PACK_KEYS_SYM:			'PACK_KEYS';
PAGE_SYM:				'PAGE';
PARTIAL_SYM:			'PARTIAL';
PARTITIONING_SYM:		'PARTITIONING';
PARTITIONS_SYM:			'PARTITIONS';
PHASE_SYM:				'PHASE';
PLUGIN_SYM:				'PLUGIN';
PLUGINS_SYM:			'PLUGINS';
PRESERVE_SYM:			'PRESERVE';
PREV_SYM:				'PREV';
PRIVILEGES_SYM:			'PRIVILEGES';
PROCESS_SYM:			'PROCESS';
PROCESSLIST_SYM:		'PROCESSLIST';
PROFILE_SYM:			'PROFILE';
PROFILES_SYM:			'PROFILES';
QUICK_SYM:				'QUICK';
READ_ONLY_SYM:			'READ_ONLY';
REBUILD_SYM:			'REBUILD';
RECOVER_SYM:			'RECOVER';
REDO_BUFFER_SIZE_SYM:	'REDO_BUFFER_SIZE';
REDOFILE_SYM:			'REDOFILE';
REDUNDANT_SYM:			'REDUNDANT';
RELAY_LOG_FILE_SYM:		'RELAY_LOG_FILE';
RELAY_LOG_POS_SYM:		'RELAY_LOG_POS';
RELAY_THREAD_SYM:		'RELAY_THREAD';
RELOAD_SYM:				'RELOAD';
RENAME_SYM:				'RENAME';
REORGANIZE_SYM:			'REORGANIZE';
REPEATABLE_SYM:			'REPEATABLE';
REPLICATION_SYM:		'REPLICATION';
RESOURCES_SYM:			'RESOURCES';
RESUME_SYM:				'RESUME';
RETURNS_SYM:			'RETURNS';
ROUTINE_SYM:			'ROUTINE';
ROWS_SYM:				'ROWS';
ROW_FORMAT_SYM:			'ROW_FORMAT';
RTREE_SYM:				'RTREE';
SCHEDULE_SYM:			'SCHEDULE';
SERIALIZABLE_SYM:		'SERIALIZABLE';
SIMPLE_SYM:				'SIMPLE';
SHUTDOWN_SYM:			'SHUTDOWN';
SNAPSHOT_SYM:			'SNAPSHOT';
SOURCE_SYM:				'SOURCE';
SQL_THREAD_SYM:			'SQL_THREAD';
STARTS_SYM:				'STARTS';
STATUS_SYM:				'STATUS';
STORAGE_SYM:			'STORAGE';
STRING_SYM:				'STRING';
SUBJECT_SYM:			'SUBJECT';
SUBPARTITION_SYM:		'SUBPARTITION';
SUBPARTITIONS_SYM:		'SUBPARTITIONS';
SUPER_SYM:				'SUPER';
SUSPEND_SYM:			'SUSPEND';
SWAPS_SYM:				'SWAPS';
SWITCHES_SYM:			'SWITCHES';
TABLE_CHECKSUM_SYM:		'TABLE_CHECKSUM';
TABLESPACE_SYM:			'TABLESPACE';
TEMPORARY_SYM:			'TEMPORARY';
TEMPTABLE_SYM:			'TEMPTABLE';
THAN_SYM:				'THAN';
TRANSACTION_SYM:		'TRANSACTION';
TRIGGERS_SYM:			'TRIGGERS';
TYPES_SYM:				'TYPES';
TYPE_SYM:				'TYPE';
UDF_RETURNS_SYM:		'UDF_RETURNS';
FUNCTION_SYM:			'FUNCTION';
UNCOMMITTED_SYM:		'UNCOMMITTED';
UNDEFINED_SYM:			'UNDEFINED';
UNDO_BUFFER_SIZE_SYM:	'UNDO_BUFFER_SIZE';
UNDOFILE_SYM:			'UNDOFILE';
UNTIL_SYM:				'UNTIL';
USE_FRM_SYM:			'USE_FRM';
VARIABLES_SYM:			'VARIABLES';
VIEW_SYM:				'VIEW';
VALUE_SYM:				'VALUE';
WARNINGS_SYM:			'WARNINGS';
WAIT_SYM:				'WAIT';
WORK_SYM:				'WORK';
X509_SYM:				'X509';

BEGIN_SYM:				'BEGIN';
DECLARE_SYM:			'DECLARE';
CONDITION_SYM:			'CONDITION';			
SQLSTATE_SYM:			'SQLSTATE';
EXIT_SYM:				'EXIT';
CONTINUE_SYM:			'CONTINUE';
HANDLER_SYM:			'HANDLER';
SQLWARNING_SYM:			'SQLWARNING';
SQLEXCEPTION_SYM:		'SQLEXCEPTION';
CURSOR_SYM:				'CURSOR';

LOOP_SYM: 				'LOOP';
WHILE_SYM:				'WHILE';
DO_SYM:					'DO';
LEAVE_SYM:				'LEAVE';
ITERATE_SYM:			'ITERATE';			

OPEN_SYM:				'OPEN';
FETCH_SYM:				'FETCH';
CLOSE_SYM:				'CLOSE';

DELETE_SYM:				'DELETE';

LOW_PRIORITY_SYM:		'LOW_PRIORITY';
DELAYED_SYM:			'DELAYED';
SET_VAR_SYM:			'SET_VAR';

COMMIT_SYM:				'COMMIT';
RELEASE_SYM:			'RELEASE';

EXECUTE_SYM:			'EXECUTE';

ROLLBACK_SYM:			'ROLLBACK';
TO_SYM:					'TO';
SAVEPOINT_SYM:			'SAVEPOINT';	

// Basic tokens. Tokens used in parser rules must not be fragments!
NUM:					DIGITS;
DECIMAL_NUM:			DIGITSDELIM;
HEX_NUM:				HEXLITERAL;
BIN_NUM:				BINLITERAL;

// Identifiers might start with a digit, even tho it is discouraged.
IDENTIFIER:				(UNICODE_LETTER | UNICODE_DIGIT)+; // All keywords above are automatically excluded.

BACK_TICK_QUOTED_ID:	(
							BACK_TICK
							( options {greedy=false;} :
								ESCAPE_SEQUENCE	
								| .
							)*
							BACK_TICK
						)+; // Includes Unicode text of any form. Several directly concatenated ids count as one.

ANSI_QUOTED_ID:			(
							DOUBLE_QUOTE 
							( options {greedy=false;} :
								ESCAPE_SEQUENCE	
								| .
							)* DOUBLE_QUOTE
						)+; // ditto

TEXT_STRING:			(
							SINGLE_QUOTE 
							( options {greedy=false;} :
								ESCAPE_SEQUENCE	
								| .
							)* SINGLE_QUOTE
						)+; // ditto

fragment ESCAPE_SEQUENCE:	'\\' ('0' | '\\' | 'b' | 'n' | 'r' | 't' | 'Z' | SINGLE_QUOTE | DOUBLE_QUOTE | BACK_TICK | MOD | UNDERLINE_SYM);

// White space handling
WS:						( ' ' | '\t' | '\f' | '\r'| '\n') { $channel = HIDDEN; };  // Ignore whitespaces.

// DASHDASH_COMMENTS are special in the sense that they require the '--' to be followed by whitespace. If there's no whitespace '--' should be lexed as MINUS MINUS
COMMENT_RULE:
	(
		MULTILINE_COMMENT 
		| POUND_COMMENT
		| MINUS_MINUS_COMMENT
		| {LA(3)==' ' || LA(3) == '\t' || LA(3) == '\n' || LA(3) == '\r'}?=> DASHDASH_COMMENT
	)
	{$channel = 98;} // Own channel (not hidden, but also not the normal stream).
	;

fragment MULTILINE_COMMENT:		'/*' ( options {greedy=false;} : . )* '*/';
fragment POUND_COMMENT:			'#' ~('\n'|'\r')* '\r'? '\n';
fragment MINUS_MINUS_COMMENT:	'-''-' ~('\n'|'\r')* '\r'? '\n';
fragment DASHDASH_COMMENT:		'--' (' ' | '\t' | '\n' | '\r') ~('\n'|'\r')* '\r'? '\n';

fragment HEXLITERAL:			'X' TEXT_STRING;
fragment BINLITERAL:			'B' TEXT_STRING;
		
	
// Simple digits.
fragment DIGIT:					'0'..'9';
fragment DIGITS:				DIGIT+;
fragment DIGITSDELIM:			DIGITS '.' DIGITS;

// Unicode ranges taken from JavaCC's grammar, but these two overlap. It works but...
fragment UNICODE_LETTER:
	'\u0024'
	| '\u0041'..'\u005a'
	| '\u005f'
	| '\u0061'..'\u007a'
	| '\u00c0'..'\u00d6'
	| '\u00d8'..'\u00f6'
	| '\u00f8'..'\u00ff'
	| '\u0100'..'\u1fff'
	| '\u3040'..'\u318f'
	| '\u3300'..'\u337f'
	| '\u3400'..'\u3d2d'
	| '\u4e00'..'\u9fff'
	| '\uf900'..'\ufaff'
;

fragment UNICODE_DIGIT:
	'\u0030'..'\u0039'
	| '\u0660'..'\u0669'
	| '\u06f0'..'\u06f9'
	| '\u0966'..'\u096f'
	| '\u09e6'..'\u09ef'
	| '\u0a66'..'\u0a6f'
	| '\u0ae6'..'\u0aef'
	| '\u0b66'..'\u0b6f'
	| '\u0be7'..'\u0bef'
	| '\u0c66'..'\u0c6f'
	| '\u0ce6'..'\u0cef'
	| '\u0d66'..'\u0d6f'
	| '\u0e50'..'\u0e59'
	| '\u0ed0'..'\u0ed9'
	| '\u1040'..'\u1049'
;

