/* 
 * Copyright (c) 2012, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#ifndef _MYSQL55_PARSER_H_
#define _MYSQL55_PARSER_H_

/**
 * C++ interface for the ANTLR based MySQL 5.5 parser.
 */

#ifdef _WIN32
  #ifdef MYSQL55_PARSER_EXPORTS
    #define MYSQL55_PARSER_PUBLIC_FUNC __declspec(dllexport)
  #else
    #define MYSQL55_PARSER_PUBLIC_FUNC __declspec(dllimport)
  #endif
#else
  #define MYSQL55_PARSER_PUBLIC_FUNC
#endif

#include <stack>

#include "MySQL55Lexer.h"  // The generated lexer.
#include "MySQL55Parser.h" // The generated parser.

extern "C" {
void on_parse_error(struct ANTLR3_BASE_RECOGNIZER_struct * recognizer, pANTLR3_UINT8 * tokenNames);
};

struct MySQL55ErrorInfo
{
  std::string message;
  ANTLR3_UINT32 error;
  ANTLR3_UINT32 token_type;
  ANTLR3_UINT32 length;
  ANTLR3_UINT32 line;
  ANTLR3_UINT32 offset;
};

class MySQL55Recognizer;

/**
 * The tree walker allows to easily navigate the AST generated by the recognizer and get all kind
 * of token information (like position, type, content etc.).
 */
class MYSQL55_PARSER_PUBLIC_FUNC MySQL55TreeWalker
{
  friend class MySQL55Recognizer; // For protected c-tor.
public:
  MySQL55Recognizer *recognizer() { return _recognizer; };
  
  bool next();
  bool next_sibling();
  bool previous();
  bool previous_sibling();
  bool up();
  bool advance_to_position(int line, int offset);
  bool advance_to_type(unsigned int type, bool recurse);
  bool skip_token_sequence(unsigned int start_token, ...);
  unsigned int look_ahead(bool recursive);
  
  void reset();
  void push();
  bool pop();
  void remove_tos();
  
  // Properties of the current token.
  bool is_nil();
  bool is_subtree();
  bool is_identifier();

  // Properties of the given token.
  bool is_identifier(unsigned int type);

  std::string token_text();
  unsigned int token_type();
  unsigned int token_line();
  unsigned int token_start();
  int token_length();
  
protected:
  MySQL55TreeWalker(MySQL55Recognizer *recognizer, pANTLR3_BASE_TREE tree);
  
  pANTLR3_BASE_TREE get_next(pANTLR3_BASE_TREE node, bool recurse);
  pANTLR3_BASE_TREE get_previous(pANTLR3_BASE_TREE node, bool recurse);
  void print_token(pANTLR3_BASE_TREE tree);
private:
  pANTLR3_BASE_TREE _origin;
  pANTLR3_BASE_TREE _tree;
  std::stack<pANTLR3_BASE_TREE> _token_stack;
  MySQL55Recognizer *_recognizer;
};

class MYSQL55_PARSER_PUBLIC_FUNC MySQL55Recognizer
{
public:
  MySQL55Recognizer(const char *text, int length);
  ~MySQL55Recognizer();
  
  std::string dump_tree();
  std::string dump_tree(pANTLR3_BASE_TREE tree, const std::string &indentation);
  
  MySQL55TreeWalker tree_walker();
  std::vector<MySQL55ErrorInfo> error_info() { return _error_info; };
  
  // Internal routine, called from the error callback.
  void add_error(const std::string &text, ANTLR3_UINT32 error, ANTLR3_UINT32 token, ANTLR3_UINT32 line,
                 ANTLR3_UINT32 offset, ANTLR3_UINT32 length);
protected:
  void parse();
  
private:
  const char *_text;
  int _text_length;
  std::vector<MySQL55ErrorInfo> _error_info;
  
  pANTLR3_INPUT_STREAM _input;
  pMySQL55Lexer _lexer;
  pANTLR3_COMMON_TOKEN_STREAM _tokens;
  pMySQL55Parser _parser;
  MySQL55Parser_script_return _ast;
};

#endif // _MYSQL55_PARSER_H_
