/* 
 * Copyright (c) 2012 Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "tut_stdafx.h"

#include "connection_helpers.h"
#include "base/file_utilities.h"

#include "grtdb/db_object_helpers.h"
#include "sqlide/wb_sql_editor_form.h"
#include "sqlide/autocomplete_object_name_cache.h"
#include "sqlide/sql_editor_be.h"

using namespace bec;
using namespace wb;

using namespace boost::assign;

// Helper to wait for cache operations to finish.
static GMutex *cache_mutex;
static GCond *cache_condition;

BEGIN_TEST_DATA_CLASS(sql_editor_be_autocomplete_tests)
protected:
  WBTester _tester;
  Sql_editor::Ref _sql_editor;
  GrtVersionRef _version;

  GMutex *_connection_mutex;
  sql::Dbc_connection_handler::Ref _conn;
  AutoCompleteCache *_cache;

public:
TEST_DATA_CONSTRUCTOR(sql_editor_be_autocomplete_tests)
  : _cache(NULL), _conn(new sql::Dbc_connection_handler())
{
  populate_grt(_tester.grt, NULL, _tester);

  // Auto completion needs a cache for object name look up, so we have to set up one
  // with all bells and whistles.
  _connection_mutex = g_mutex_new();

  cache_mutex = g_mutex_new();
  cache_condition = g_cond_new();
}

TEST_DATA_DESTRUCTOR(sql_editor_be_autocomplete_tests)
{
  _cache->shutdown();
  delete _cache;
  g_mutex_free(_connection_mutex);
  g_mutex_free(cache_mutex);
  g_cond_free(cache_condition);
}

base::GMutexLock get_connection(sql::Dbc_connection_handler::Ref &conn)
{
  base::GMutexLock lock(_connection_mutex);
  conn = _conn;
  return lock;
}

END_TEST_DATA_CLASS;

TEST_MODULE(sql_editor_be_autocomplete_tests, "SQL code completion tests");

void cache_callback(bool working)
{
  if (!working)
  {
    g_mutex_lock(cache_mutex);
    g_cond_signal(cache_condition);
    g_mutex_unlock(cache_mutex);
  }
}

/**
 * Setup for the cache.
 */
TEST_FUNCTION(5)
{
  db_mgmt_ConnectionRef connectionProperties(_tester.grt);
  setup_env(_tester.grt, connectionProperties);

  sql::DriverManager *dm = sql::DriverManager::getDriverManager();
  _conn->ref = dm->getConnection(connectionProperties);

  base::remove("testconn.cache");
  _cache = new AutoCompleteCache("testconn", boost::bind(&Test_object_base<sql_editor_be_autocomplete_tests>::get_connection, this, _1),
    ".", cache_callback);

  std::auto_ptr<sql::Statement> stmt(_conn->ref->createStatement());

  sql::ResultSet *res = stmt->executeQuery("SELECT VERSION() as VERSION");
  if (res && res->next())
  {
    std::string version = res->getString("VERSION");
    _version = CatalogHelper::parse_version(_tester.grt, version);
  }
  delete res;

  ensure("Server version is invalid", _version.is_valid());

  _tester.get_rdbms()->version(_version);

  // Copy a current version of the code editor configuration file to the test data folder.
  gchar *contents;
  gsize length;
  GError *error = NULL;
  if (g_file_get_contents("../../res/wbdata/code_editor.xml", &contents, &length, &error))
  {
    ensure("Could not write editor configuration to target file",
      g_file_set_contents("data/code_editor.xml", contents, length, &error) == TRUE);
    g_free(contents);
  }
  else
    fail("Could not copy code editor configuration");

  _sql_editor = Sql_editor::create(_tester.get_rdbms(), _version);
  _sql_editor->set_current_schema("sakila");
  _sql_editor->set_auto_completion_cache(_cache);

  // We don't set up the sakila schema. This is needed in so many places, it should simply exist.
  std::vector<std::string> list = _cache->get_matching_schema_names("sakila");

  // The loops are not necessary, but there can be spurious condition signals,
  // as the glib docs say.
  while (!_cache->is_schema_list_fetch_done())
    g_cond_wait(cache_condition, cache_mutex);

  _cache->get_matching_table_names("sakila"); // Cache all tables and columns.

  while (!_cache->is_schema_table_columns_fetch_done("sakila", "store"))
    g_cond_wait(cache_condition, cache_mutex);
}

/**
 * Another prerequisites test. See that the cache contains needed objects.
 */
TEST_FUNCTION(10)
{
  std::vector<std::string> list = _cache->get_matching_schema_names("sakila");
  int found = 0;
  for (std::vector<std::string>::const_iterator i = list.begin(); i != list.end(); ++i)
  {
    if (*i == "sakila" || *i == "mysql")
      found++;
  }
  ensure_equals("Sakila schema missing. Is the DB set up properly?", found, 1);
}

//--------------------------------------------------------------------------------------------------

/**
 * Checks that exactly the given set of AC parts are set.
 */
void match_included_parts(const std::string msg, const Sql_editor::AutoCompletionContext &context,
  int parts)
{
  ensure_equals(msg, context.wanted_parts, parts);
}

// Checks that exactly the given entries are in the list.
void check_ac_entries(const std::string msg, const std::vector<std::pair<int, std::string> > &ac_list,
  const std::vector<std::string> &entries)
{
  ensure_equals("Auto completion list has invalid size", ac_list.size(), entries.size());
  for (size_t i = 0; i < entries.size(); ++i)
  {
    ensure_equals(msg + ", invalid list entry", ac_list[i].second, entries[i]);
  }
}

struct ac_test_entry
{
  std::string query;
  std::string typed_part;
  int line;
  int offset;

  bool check_entries;
  std::vector<std::string> entries;
  int parts;
};

const ac_test_entry query_typing_test_data[] = {
  {"", "", 1, 0, false,
    list_of (""),
    Sql_editor::CompletionWantMajorKeywords
  },
  {" s", "s", 1, 1, true,
    list_of ("savepoint") ("select") ("set") ("show") ("start") ("stop"),
    Sql_editor::CompletionWantMajorKeywords
  },
  {" se", "se", 1, 2, true,
    list_of ("select") ("set"),
    Sql_editor::CompletionWantMajorKeywords
  },
  {" sel", "sel", 1, 3, true,
    list_of ("select"),
    Sql_editor::CompletionWantMajorKeywords
  },
  {" sele", "sele", 1, 4, true,
    list_of ("select"),
    Sql_editor::CompletionWantMajorKeywords
  },
  {" selec", "selec", 1, 5, true, // 5
    list_of ("select"),
    Sql_editor::CompletionWantMajorKeywords
  },
  {" select", "select", 1, 6, true,
    list_of ("select"),
    Sql_editor::CompletionWantMajorKeywords
  },
  {" select ", "", 1, 7, false,
    list_of (""),
    Sql_editor::CompletionWantFunctions | Sql_editor::CompletionWantSchemas |
    Sql_editor::CompletionWantTables | Sql_editor::CompletionWantColumns
  },
  {" select c", "c", 1, 8, true,
    list_of ("cast()") ("category") ("category_id") ("ceil()") ("ceiling()") ("char_length()")
      ("character_length()") ("city") ("city_id") ("coalesce()") ("concat()") ("concat_ws()")
      ("connection_id()") ("conv()") ("convert()") ("cos()") ("cot()") ("count()") ("country")
      ("country_id") ("create_date") ("curdate()") ("current_user()") ("curtime()") ("customer")
      ("customer_id") ("customer_list"),
    Sql_editor::CompletionWantFunctions | Sql_editor::CompletionWantSchemas |
    Sql_editor::CompletionWantTables | Sql_editor::CompletionWantColumns
  },
  {" select co", "co", 1, 9, true,
    list_of ("coalesce()") ("concat()") ("concat_ws()") ("connection_id()") ("conv()") ("convert()")
    ("cos()") ("cot()") ("count()") ("country") ("country_id"),
    Sql_editor::CompletionWantFunctions | Sql_editor::CompletionWantSchemas |
    Sql_editor::CompletionWantTables | Sql_editor::CompletionWantColumns
  },
  {" select cou", "cou", 1, 10, true,
    list_of ("count()") ("country") ("country_id"),
    Sql_editor::CompletionWantFunctions | Sql_editor::CompletionWantSchemas |
    Sql_editor::CompletionWantTables | Sql_editor::CompletionWantColumns
  },
  {" select coun", "coun", 1, 11, true,
    list_of ("count()") ("country") ("country_id"),
    Sql_editor::CompletionWantFunctions | Sql_editor::CompletionWantSchemas |
    Sql_editor::CompletionWantTables | Sql_editor::CompletionWantColumns
  },
  {" select count", "count", 1, 12, true,
    list_of ("count()") ("country") ("country_id"),
    Sql_editor::CompletionWantFunctions | Sql_editor::CompletionWantSchemas |
    Sql_editor::CompletionWantTables | Sql_editor::CompletionWantColumns
  },
};

/**
 * Collecting AC info for each typed character in a simple query starting with an empty statement.
 */
TEST_FUNCTION(15)
{
  for (size_t i = 0; i < sizeof(query_typing_test_data) / sizeof(query_typing_test_data[0]); i++)
  {
    Sql_editor::AutoCompletionContext context;
    context.statement = query_typing_test_data[i].query;
    context.typed_part = query_typing_test_data[i].typed_part;
    context.line = query_typing_test_data[i].line;
    context.offset = query_typing_test_data[i].offset;

    std::string message = base::strfmt("Step %u", i);
    _sql_editor->create_auto_completion_list(context);
    match_included_parts(message, context, query_typing_test_data[i].parts);
    if (query_typing_test_data[i].check_entries)
    {
      std::vector<std::pair<int, std::string> > entries = _sql_editor->update_auto_completion(context.typed_part);
      check_ac_entries(message, entries, query_typing_test_data[i].entries);
    }
  }

}

END_TESTS