/* 
 * Copyright (c) 2008, 2012, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#import "WBModelSidebarController.h"
#import "GRTTreeDataSource.h"
#import "GRTListDataSource.h"
#import "MCanvasViewer.h"
#import "WBObjectPropertiesController.h"
#import "MOutlineView.h"
#import "MTextImageCell.h"
#import "GRTIconCache.h"

#include "workbench/wb_context_ui.h"
#include "workbench/wb_history_tree.h"
#include "model/wb_model_diagram_form.h"
#include "model/wb_user_datatypes.h"

#import "MCPPUtilities.h"

#include "grtdb/db_object_helpers.h"


@implementation WBModelSidebarController

static void refresh_history(const bec::NodeId &parent, int orow, WBModelSidebarController *self);
static void refresh_user_types(const bec::NodeId &parent, int orow, WBModelSidebarController *self);

- (void)setupWithWBContextUI:(wb::WBContextUI*)wbui
{
  _wbui= wbui;
 
  /*
  [[NSNotificationCenter defaultCenter] addObserver: self
                                           selector: @selector(menuAction:)
                                               name: NSMenuActionNotification
                                             object: nil];
  */
  [catalogOutline setDoubleAction:@selector(activateCatalogItem:)];
  [catalogOutline setTarget:self];
  [historyTable setDoubleAction:@selector(activateHistoryItem:)];
  [historyTable setTarget:self];
  
  // set backend model for catalog DS
  [catalogDS setHidesRootNode:YES];
  [catalogDS setDragDelegate:self];
  
  [catalogDS setTreeModel: _wbui->get_catalog_tree()];
  
  // set backend for usertype ds
  if (userTypeDS)
  {
    [userTypeDS setListModel: _wbui->get_usertypes_tree()];
    _userTypesRefresh = _wbui->get_usertypes_tree()->tree_changed_signal()->connect(boost::bind(refresh_user_types, _1, _2, self));
  }

  // set backend model for history DS
  if (historyDS)
  {
    [historyDS setListModel:_wbui->get_history_tree()];
    _historyRefresh = _wbui->get_history_tree()->tree_changed_signal()->connect(boost::bind(refresh_history, _1, _2, self));
  }
  
  [self refresh];  
}


- (void)reuseShareableDataSourcesFrom:(WBModelSidebarController*)sidebar
{
  if (sidebar)
  {
    catalogDS = sidebar->catalogDS;
    historyDS = sidebar->historyDS;
    userTypeDS = sidebar->userTypeDS;
  }
  else
  {
    catalogDS = nil;
    historyDS = nil;
    userTypeDS = nil;
  }
  
  [catalogOutline setDataSource: catalogDS];
  [catalogOutline setDelegate: self];
  
  [historyTable setDataSource: historyDS];
  [historyTable setDelegate: historyDS];
  
  [userTypeTable setDataSource: userTypeDS];
  [userTypeTable setDelegate: userTypeDS];
}


- (void)dealloc
{ 
  _wbui= 0;
  
  _historyRefresh.disconnect();
  _userTypesRefresh.disconnect();
  [[NSNotificationCenter defaultCenter] removeObserver: self];
  [super dealloc];
}


- (void)invalidate
{
  // since performSelectorOnMainThread could have pending refreshes, set _wbui to nil
  _wbui= 0;
}


- (void)setZoom:(id)x
{
  NSLog(@"here");
}

//--------------------------------------------------------------------------------------------------

/**
 * Triggered if the underlying outline view was told the backend does not handle the specific
 * menu command (usually the case for pure UI related tasks). Thus handle them here.
 */
- (void) menuAction: (NSNotification*) notification
{
  id sender = [notification object];
  NSString* command = [sender representedObject];
  
  if ([command compare: @"refreshCatalog"] == NSOrderedSame)
    [self refreshCatalogTree];
  else
    if ([command compare: @"copyHistory"] == NSOrderedSame)
      [self copyHistoryToPasteboard];
}

//--------------------------------------------------------------------------------------------------

- (void) copyHistoryToPasteboard
{
  NSString* text = @"";
  
  bec::ListModel* model = [historyDS listModel];
  std::vector<bec::NodeId> nodes = [historyTable selectedNodeIds];
  if (nodes.size() > 0)
  {
    // Copy only selected items if there are any.
    for (int i = 0; i < (int)nodes.size(); i++)
    {
      std::string label;
      model->get_field(nodes[i], 0, label);
      text = [NSString stringWithFormat: @"%@%@\n", text, [NSString stringWithUTF8String: label.c_str()]];
    }
  }
  else
  {
    // Nothing selected, copy everything.
    for (int i = 0; i < model->count(); i++)
    {
      bec::NodeId node = model->get_node(i);
      
      std::string label;
      model->get_field(node, 0, label);
      text = [NSString stringWithFormat: @"%@%@\n", text, [NSString stringWithUTF8String: label.c_str()]];
    }
  }    
  
  NSPasteboard* pasteboard= [NSPasteboard generalPasteboard];
  [pasteboard declareTypes: [NSArray arrayWithObject: NSStringPboardType] owner: nil];
  [pasteboard setString: text forType: NSStringPboardType];
}

//--------------------------------------------------------------------------------------------------

- (void) updateForSelectionChange
{
  // XXX
}

//--------------------------------------------------------------------------------------------------

- (void)refresh
{
  [self refreshCatalogTree];
  [self refreshUserTypeList];
  [self refreshHistory];
}

- (void)refreshUserTypeList
{
  if (_wbui)
  {
    NSPoint pos = [[[userTypeTable enclosingScrollView] contentView] documentVisibleRect].origin;
    
    if ([userTypeDS listModel])
      [userTypeDS listModel]->refresh();
  
    [userTypeTable reloadData];
    
    [[[userTypeTable enclosingScrollView] contentView] scrollToPoint: pos];
    [[userTypeTable enclosingScrollView] reflectScrolledClipView: [[userTypeTable enclosingScrollView] contentView]];
  }
}


- (void)refreshCatalogTree
{
  if (_wbui)
  {
    NSPoint pos = [[[catalogOutline enclosingScrollView] contentView] documentVisibleRect].origin;
    NSIndexSet *selectedIndexes = [[catalogOutline selectedRowIndexes] copy];
    
    _wbui->get_catalog_tree()->refresh();
    [catalogOutline reloadData];
  
    [catalogOutline expandItem:[catalogOutline itemAtRow: 0]];
    // expand everything from bottom to top so that only the folders get expanded
    for (int i= [catalogOutline numberOfRows]-1; i >= 1; --i)
    {
      // force expansion of node
      [catalogOutline collapseItem:[catalogOutline itemAtRow:i]];
      [catalogOutline expandItem:[catalogOutline itemAtRow:i]];
    }
    
    [[[catalogOutline enclosingScrollView] contentView] scrollToPoint: pos];
    [[catalogOutline enclosingScrollView] reflectScrolledClipView: [[catalogOutline enclosingScrollView] contentView]];
    
    [catalogOutline selectRowIndexes: selectedIndexes byExtendingSelection: NO];
    [selectedIndexes release];
  }
}


- (BOOL)dataSource:(id)source
        writeItems:(NSArray*)items
      toPasteboard:(NSPasteboard*)pboard
{
  if (source == catalogDS)
  {
    std::list<db_DatabaseObjectRef> objects;
    std::string path;
    
    for (id item in items)
    {
      bec::NodeId node= [source nodeIdForItem:item];
      grt::ValueRef value(_wbui->get_catalog_tree()->get_node_value(node));

      if (value.is_valid() && db_DatabaseObjectRef::can_wrap(value))
      {
        objects.push_back(db_DatabaseObjectRef::cast_from(value));
        
        path= "/wb/doc/physicalModels/0/catalog/schemata" + _wbui->get_catalog_tree()->get_path_for_node(node, true);
      }
    }

    std::string str= bec::CatalogHelper::dbobject_list_to_dragdata(objects);
    if (!str.empty())
    {
      [pboard declareTypes:[NSArray arrayWithObjects:
                            @"x-mysql-wb/db.DatabaseObject", 
                            NSStringPboardType,
                            nil]
                     owner:nil];

      [pboard setString:[NSString stringWithCPPString: str.c_str()]
                forType:@"x-mysql-wb/db.DatabaseObject"];
      [pboard setString:[NSString stringWithCPPString: path.c_str()]
                forType:NSStringPboardType];
      return YES;
    }
  }
  return NO;
}


- (void)outlineView:(NSOutlineView *)outlineView willDisplayCell:(id)cell forTableColumn:(NSTableColumn *)tableColumn item:(id)item
{
  if (outlineView == catalogOutline)
  {
    if ([cell isKindOfClass:[MTextImageCell class]])
    {
      bec::NodeId node_id= [catalogDS nodeIdForItem:item];
      bec::TreeModel *tree= [catalogDS treeModel];
      bec::IconId icon_id= tree->get_field_icon(node_id, [[tableColumn identifier] integerValue], bec::Icon16);
      
      NSImage *image= [[GRTIconCache sharedIconCache] imageForIconId:icon_id];
      
      if (icon_id != 0 && !image && tree->is_expandable(node_id))
      {
        image= [[GRTIconCache sharedIconCache] imageForFolder:bec::Icon16];
      }
      
      if (icon_id != 0)
        [cell setImage:image];
      else
        [cell setImage:nil];
      
      std::string tmp;
      tree->get_field(node_id, 1, tmp);
      if (tmp.empty())
        [cell setFont: [NSFont systemFontOfSize: [NSFont smallSystemFontSize]]];
      else
        [cell setFont: [NSFont boldSystemFontOfSize: [NSFont smallSystemFontSize]]];
    }
  }
}


- (void)outlineViewItemWillExpand:(NSNotification *)notification
{
  id datasource = [notification.object dataSource];
  if ([datasource conformsToProtocol: @protocol(NSOutlineViewDelegate)])
    [datasource outlineViewItemWillExpand: notification];
}


- (void)outlineViewItemDidCollapse:(NSNotification *)notification
{
  id datasource = [notification.object dataSource];
  if ([datasource conformsToProtocol: @protocol(NSOutlineViewDelegate)])
    [datasource outlineViewItemDidCollapse: notification];
}


- (void)refreshHistory
{
  if (_wbui)
  {
    NSPoint pos = [[[historyTable enclosingScrollView] contentView] documentVisibleRect].origin;
    
    _wbui->get_history_tree()->refresh();
    [historyTable reloadData];
    
    [[[historyTable enclosingScrollView] contentView] scrollToPoint: pos];
    [[historyTable enclosingScrollView] reflectScrolledClipView: [[historyTable enclosingScrollView] contentView]];
  }
}


- (IBAction)activateCatalogItem:(id)sender
{
  _wbui->get_catalog_tree()->activate_node([catalogDS nodeIdForItem:[catalogOutline itemAtRow: [catalogOutline clickedRow]]]);
}


- (IBAction)activateHistoryItem:(id)sender
{
  if ([historyTable clickedRow] >= 0)
    _wbui->get_history_tree()->activate_node(bec::NodeId([historyTable clickedRow]));
}

//--------------------------------------------------------------------------------------------------

- (NSOutlineView*) catalogOutline
{
  return catalogOutline;
}

//--------------------------------------------------------------------------------------------------

- (GRTTreeDataSource*) catalogDS
{
  return catalogDS;
}

//--------------------------------------------------------------------------------------------------

- (NSTableView*) historyTable
{
  return historyTable;
}

//--------------------------------------------------------------------------------------------------

- (GRTListDataSource*) historyDS
{
  return historyDS;
}

//--------------------------------------------------------------------------------------------------

- (GRTListDataSource*) userTypeDS
{
  return userTypeDS;
}

//--------------------------------------------------------------------------------------------------

static void refresh_history(const bec::NodeId &parent, int orow, WBModelSidebarController *self)
{
  [self performSelectorOnMainThread:@selector(refreshHistory)
                         withObject:nil
                      waitUntilDone:NO];
}

static void refresh_user_types(const bec::NodeId &parent, int orow, WBModelSidebarController *self)
{
  [self performSelectorOnMainThread:@selector(refreshUserTypeList)
                         withObject:nil
                      waitUntilDone:NO];
}

@end
