/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include "wb_context_model.h"
#include "workbench/wb_context_ui.h"
#include "wb_component.h"
#include "wb_component_basic.h"
#include "wb_component_physical.h"
#include "wb_component_logical.h"
#include "model/wb_model_diagram_form.h"
#include "wb_overview_physical.h"
#include "wb_overview_physical_schema.h"
#include "workbench/wb_model_file.h"
#include "model/wb_overview_physical.h"

#include "wbcanvas/workbench_physical_model_impl.h"

#include "grt/common.h"
#include "string_utilities.h"
#include "mforms/utilities.h"

using namespace bec;
using namespace wb;


WBContextModel::WBContextModel(WBContextUI *wbui)
: _wbui(wbui)
{  
  _overview= new PhysicalOverviewBE(_wbui->get_wb());

  CommandUI *cmdui = wbui->get_command_ui();
  sigc::slot<bool> validate = sigc::mem_fun(this, &WBContextModel::has_selected_schema);
  cmdui->add_builtin_command("addModelDiagram",
                             sigc::mem_fun(this, &WBContextModel::add_model_diagram),
                             sigc::mem_fun(this, &WBContextModel::has_selected_model));
  cmdui->add_builtin_command("addModelSchema",
                             sigc::mem_fun(this, &WBContextModel::add_model_schema),
                             sigc::mem_fun(this, &WBContextModel::has_selected_model));
  cmdui->add_builtin_command("addModelTable",
                             sigc::mem_fun(this, &WBContextModel::add_model_table),
                             validate);
  cmdui->add_builtin_command("addModelView",
                             sigc::mem_fun(this, &WBContextModel::add_model_view),
                             validate);
  cmdui->add_builtin_command("addModelRoutine",
                             sigc::mem_fun(this, &WBContextModel::add_model_rgroup),
                             validate);
  
  // used to update the currentDiagram field
  _wbui->signal_main_form_change().connect(sigc::mem_fun(this, &WBContextModel::update_current_diagram));
}


WBContextModel::~WBContextModel()
{
  delete _overview;
}


void WBContextModel::unrealize()
{
  // unrealize all models
  if (_doc.is_valid() && _doc->physicalModels().is_valid())
  {
    for (size_t c= _doc->physicalModels().count(), i= 0; i < c; i++)
    {
      _doc->physicalModels().get(i)->get_data()->unrealize();
    }
  }  
}


grt::ListRef<app_ToolbarItem> WBContextModel::get_toolbar_items(const std::string &toolbar)
{
  WBContext *wb= _wbui->get_wb();
  grt::ListRef<app_ToolbarItem> items;
  
  if (toolbar == WB_TOOLBAR_MAIN)
  {
    if (!_toolbar.is_valid())
    {
      _toolbar= app_ToolbarRef::cast_from(wb->get_grt()->unserialize(make_path(wb->get_datadir(), 
                                                                               "data/model_toolbar.xml")));
    }
    
    return _toolbar->items();
  }
  else if (toolbar.compare(0, sizeof(WB_TOOLBAR_TOOLS)-1, WB_TOOLBAR_TOOLS) == 0)
  {
    if (!_tools_toolbar.is_valid())
      _tools_toolbar= app_ToolbarRef::cast_from(wb->get_grt()->unserialize(make_path(wb->get_datadir(),
                                                                                     "data/tools_toolbar.xml")));
   
    items= grt::ListRef<app_ToolbarItem>(wb->get_grt());
    grt::append_contents(items, _tools_toolbar->items());
  }
  else if (toolbar == WB_TOOLBAR_OPTIONS)
  {
    items= grt::ListRef<app_ToolbarItem>(wb->get_grt());
  }
  
  // view specific contextual toolbars
  if (_wbui->get_active_context() == WB_CONTEXT_MODEL)
  {
    grt::ListRef<app_ToolbarItem> model_items;
      
    // append model specific items
    wb->foreach_component(sigc::compose(sigc::bind<0>(sigc::ptr_fun(grt::append_contents), items),
                                        sigc::bind(sigc::mem_fun(&WBComponent::get_toolbar_items), toolbar)));
  }
  
  return items;
}


model_DiagramRef WBContextModel::get_active_model_diagram(bool main_form)
{
  bec::UIForm *form= main_form ? get_wbui()->get_active_main_form() : get_wbui()->get_active_form();
  
  if (dynamic_cast<ModelDiagramForm*>(form))
    return dynamic_cast<ModelDiagramForm*>(form)->get_model_diagram();
  
  return model_DiagramRef();
}


model_ModelRef WBContextModel::get_active_model(bool main_form)
{
  bec::UIForm *form= main_form ? get_wbui()->get_active_main_form() : get_wbui()->get_active_form();
  
  if (dynamic_cast<OverviewBE*>(form))
    return dynamic_cast<OverviewBE*>(form)->get_model();
  else if (dynamic_cast<ModelDiagramForm*>(form))
    return dynamic_cast<ModelDiagramForm*>(form)->get_model_diagram()->owner();
  return model_ModelRef();
}



void WBContextModel::model_created(workbench_DocumentRef doc)
{
  _doc= doc;
  
  _wbui->get_wb()->get_component<WBComponentLogical>()->setup_logical_model(doc.get_grt(), _doc);
  _wbui->get_wb()->get_component<WBComponentPhysical>()->setup_physical_model(doc.get_grt(), _doc, "Mysql", "5.1.22");
  
  _wbui->get_wb()->foreach_component(sigc::mem_fun(&WBComponent::reset_document));
  
  _doc->physicalModels().get(0)->get_data()->set_delegate(this);
  
  _doc->physicalModels()[0]->get_data()->realize();
  
  _wbui->get_wb()->request_refresh(RefreshNewModel, "", 0);
}


void WBContextModel::model_loaded(workbench_DocumentRef doc)
{
  _doc= doc;
  
  _wbui->get_wb()->foreach_component(sigc::mem_fun(&WBComponent::reset_document));
  
  _wbui->get_wb()->foreach_component(sigc::mem_fun(&WBComponent::document_loaded));
  
  _doc->physicalModels().get(0)->get_data()->set_delegate(this);
    
  _wbui->get_wb()->request_refresh(RefreshNewModel, "", 0);
}


void WBContextModel::realize()
{
  _doc->physicalModels()[0]->get_data()->realize();
}


cairo_surface_t *WBContextModel::fetch_image(const std::string &file)
{
  return _wbui->get_wb()->get_file()->get_image(file);
}


std::string WBContextModel::attach_image(const std::string &file)
{
  return _wbui->get_wb()->get_file()->add_image_file(file);
}


void WBContextModel::release_image(const std::string &file)
{
  // QQQ
  // _wbui->get_wb()->get_file()->release_image(file);
}


/** 
 ****************************************************************************
 * @brief Delegate method for creating canvas views
 *
 * This is called by the GRT bridge when a new view object is created.
 * It will in turn, call a frontend supplied callback which should create
 * the canvas view (and its viewer) and then return the canvas view.
 *
 * The free_canvas_view() method will be called once the view can be freed.
 * 
 * @param name the name of the canvas. Will be the object-id of the view.
 *****************************************************************************/
mdc::CanvasView *WBContextModel::create_diagram(const model_DiagramRef &view)
{
  return _wbui->get_wb()->execute_in_main_thread<mdc::CanvasView*>("create_diagram", sigc::bind
                                                                                  (sigc::mem_fun(this, &WBContextModel::create_diagram_main), view));
}


void WBContextModel::free_canvas_view(mdc::CanvasView *view)
{
  // this function is expected to be called from the main thread
  notify_diagram_destroyed(get_diagram_form(view));
  
  // forward to frontend
  if (_wbui->get_wb()->get_grt_manager()->in_main_thread())
    _wbui->get_wb()->destroy_view(view);
  else
    _wbui->get_wb()->execute_in_main_thread<void>("destroy view", 
                   sigc::bind<mdc::CanvasView*>(_wbui->get_wb()->destroy_view, view));
}



mdc::CanvasView *WBContextModel::create_diagram_main(const model_DiagramRef &diagram_reference)
{
  ModelDiagramForm *diagram= 0;
  WBContext *wb= _wbui->get_wb();
  
  FOREACH_COMPONENT(wb->_components, iter)
  {
    if (diagram_reference.is_instance((*iter)->get_diagram_class_name())
        && (*iter)->get_diagram_class_name() != model_Diagram::static_class_name())
    {
      diagram= new ModelDiagramForm(*iter, diagram_reference);
      break;
    }
  }
  
  // fallback
  if (!diagram)
    diagram= new ModelDiagramForm(wb->get_component_named("basic"), diagram_reference);
  
  if (!diagram)
  {
    mforms::Utilities::show_error("Internal error adding a new diagram.", "Unknown diagram type.", _("Close"));
    return 0;
  }
  
  diagram_reference->signal_objectActivated().connect(sigc::mem_fun(this, &WBContextModel::activate_canvas_object));
    
  diagram_reference->signal_list_changed().connect(sigc::bind(sigc::mem_fun(this, &WBContextModel::diagram_object_list_changed), diagram));
  
  register_diagram_form(diagram);
  
  // Forward creation to front end.
  mdc::CanvasView *view= wb->create_view(diagram_reference.id(), diagram_reference->name());
  if (view)
  {
    diagram->attach_canvas_view(view);
    
    notify_diagram_created(diagram);
    
    wb->request_refresh(RefreshNewDiagram, diagram_reference.id(), (NativeHandle)view->get_user_data());
  }
  else
  {
    delete diagram;
    mforms::Utilities::show_error("Internal error adding a new diagram.", "Frontend did not return a diagram.", _("Close"));
  }
  
  if (getenv("DEBUG_CANVAS"))
    view->enable_debug(true);
  
  return view;
}



void WBContextModel::activate_canvas_object(const model_ObjectRef &object, int flags)
{
  bool newwindow= flags & 1;
  
  FOREACH_COMPONENT(_wbui->get_wb()->_components, iter)
  {
    if ((*iter)->handles_figure(object))
      (*iter)->activate_canvas_object(object, newwindow);
  }
}


void WBContextModel::register_diagram_form(ModelDiagramForm *view)
{
  _model_forms[view->get_model_diagram().id()]= view;
}


ModelDiagramForm *WBContextModel::get_diagram_form(mdc::CanvasView *view)
{
  for (std::map<std::string,ModelDiagramForm*>::const_iterator iter= _model_forms.begin();
       iter != _model_forms.end(); ++iter)
  {
    if (iter->second->get_view() == view)
      return iter->second;
  }
  return 0;
}


void WBContextModel::notify_diagram_created(ModelDiagramForm *view)
{
  view->get_view()->get_selection()->signal_changed().connect(sigc::bind(sigc::mem_fun(_wbui, &WBContextUI::view_selection_changed), view));
  
  view->get_model_diagram()->signal_changed().
  connect(sigc::bind(sigc::mem_fun(this, &WBContextModel::diagram_object_changed), view));
  
  _wbui->get_physical_overview()->send_refresh_diagram(model_DiagramRef());
}


void WBContextModel::notify_diagram_destroyed(ModelDiagramForm *view)
{
  std::string id= view->get_model_diagram().id();
  delete _model_forms[id];
  _model_forms.erase(id);
  
  _wbui->get_physical_overview()->send_refresh_diagram(model_DiagramRef());
}


void WBContextModel::update_current_diagram(bec::UIForm *form)
{
  ModelDiagramForm* dform= dynamic_cast<ModelDiagramForm*>(form);
  if (dform)
  {
    model_DiagramRef diagram(dform->get_model_diagram());
    if (diagram.is_valid() && diagram->owner().is_valid())
      diagram->owner()->currentDiagram(diagram);
  }
}


void WBContextModel::diagram_object_changed(const std::string &member, const grt::ValueRef &ovalue, ModelDiagramForm *view)
{
  if (member == "name")
  {
    _wbui->get_wb()->request_refresh(RefreshViewName, view->get_model_diagram()->name(), (NativeHandle)view->get_view()->get_user_data());
    _wbui->get_physical_overview()->send_refresh_diagram(view->get_model_diagram());
  }
  else if (member == "zoom")
  {
    _wbui->get_wb()->request_refresh(RefreshZoom, "");
  }
}


void WBContextModel::diagram_object_list_changed(grt::internal::OwnedList *list, bool added, const grt::ValueRef &value, ModelDiagramForm *vform)
{
  if (vform == _wbui->get_active_main_form())
  {
    if (vform->get_model_diagram()->selection().valueptr() == list)
      _wbui->get_wb()->request_refresh(RefreshSelection, "", reinterpret_cast<NativeHandle>(vform->get_frontend_data()));
  }
}


bool WBContextModel::has_selected_schema()
{
  PhysicalOverviewBE *active_form = dynamic_cast<PhysicalOverviewBE*>(_wbui->get_active_main_form());
  if (active_form == _overview && _overview->get_active_schema_node())
    return true;
  
  return false;
}

bool WBContextModel::has_selected_model()
{
  if (_wbui->get_active_main_form() == _overview)
    return true;
  return false;
}

void WBContextModel::add_model_schema()
{
  _wbui->get_wb()->get_component<WBComponentPhysical>()->add_new_db_schema(workbench_physical_ModelRef::cast_from(get_active_model(true)));
}

void WBContextModel::add_model_diagram()
{
  _wbui->get_wb()->add_new_diagram(get_active_model(true));
}

void WBContextModel::add_model_table()
{
  _overview->get_active_schema_node()->add_new_db_table(_wbui->get_wb());
}


void WBContextModel::add_model_view()
{
  _overview->get_active_schema_node()->add_new_db_view(_wbui->get_wb());
}


void WBContextModel::add_model_rgroup()
{
  _overview->get_active_schema_node()->add_new_db_routine(_wbui->get_wb());
}

