#include "stdafx.h"

#include "db_sql_editor_history_be.h"
#include "sqlide/recordset_data_storage.h"
#include "string_utilities.h"
#include <boost/foreach.hpp>


using namespace bec;
using namespace grt;


DbSqlEditorHistory::DbSqlEditorHistory(GRTManager *grtm)
:
_grtm(grtm), _current_entry_index(-1)
{
  _entries_model= EntriesModel::create(_grtm);
  _details_model= DetailsModel::create(_grtm);
}


void DbSqlEditorHistory::reset()
{
  _details_model->reset();
  _entries_model->reset();
  _current_entry_index= -1;
}


void DbSqlEditorHistory::add_entry(const std::list<std::string> &statements)
{
  _entries_model->add_entry(statements);
  if (1 == _entries_model->row_count())
    current_entry(_entries_model->row_count()-1);
}


void DbSqlEditorHistory::current_entry(int index)
{
  DetailsModel::Ref details_model= _entries_model->details_model(index);
  _details_model->load_from(details_model);
  _current_entry_index= index;
}


std::string DbSqlEditorHistory::restore_sql_from_history(int entry_index, std::list<int> &detail_indexes)
{
  std::string sql;
  if (entry_index >= 0)
  {
    DetailsModel::Ref details_model= _entries_model->details_model(entry_index);
    NodeId node;
    node.index->push_back(int());
    std::string statement;
    BOOST_FOREACH (int row, detail_indexes)
    {
      node[0]= row;
      details_model->get_field(node, 0, statement);
      sql+= statement + ";\n";
    }
  }
  return sql;
}


void DbSqlEditorHistory::EntriesModel::reset()
{
  VarGridModel::reset();

  reinit(_details_models);

  _readonly= true;

  add_column("Time", std::string());

  boost::shared_ptr<sqlite::connection> data_swap_db= this->data_swap_db();
  Recordset_data_storage::create_data_swap_tables(data_swap_db.get(), _column_names, _column_types);

  refresh_ui();
}


void DbSqlEditorHistory::EntriesModel::add_entry(const std::list<std::string> &statements)
{
  if (statements.empty())
    return;

  {
    GMutexLock data_mutex(_data_mutex);

    _data.reserve(_data.size() + _column_count);

    try
    {
      _data.push_back(current_time());
      DetailsModel::Ref details_model= DetailsModel::create(_grtm);
      _details_models.push_back(details_model);
      details_model->add_entries(statements);
    }
    catch(...)
    {
      _data.resize(_row_count * _column_count);
      _details_models.resize(_row_count);
      throw;
    }

    ++_row_count;
    ++_data_frame_end;
  }

  refresh_ui();
}


void DbSqlEditorHistory::DetailsModel::reset()
{
  VarGridModel::reset();

  _readonly= true;

  add_column("SQL", std::string());

  boost::shared_ptr<sqlite::connection> data_swap_db= this->data_swap_db();
  Recordset_data_storage::create_data_swap_tables(data_swap_db.get(), _column_names, _column_types);

  refresh_ui();
}


void DbSqlEditorHistory::DetailsModel::add_entries(const std::list<std::string> &statements)
{
  if (statements.empty())
    return;

  {
    GMutexLock data_mutex(_data_mutex);

    _data.reserve(_data.size() + _column_count);

    try
    {
      BOOST_FOREACH (std::string statement, statements)
        _data.push_back(base::strip_text(statement));
    }
    catch(...)
    {
      _data.resize(_row_count * _column_count);
      throw;
    }

    _row_count+= statements.size();
    _data_frame_end= _row_count;
  }

  refresh_ui();
}


void DbSqlEditorHistory::DetailsModel::load_from(DbSqlEditorHistory::DetailsModel::Ref details_model)
{
  {
    GMutexLock data_mutex(_data_mutex);

    _data= details_model->data();
    _row_count= details_model->row_count();
    _data_frame_end= _row_count;
  }

  refresh_ui();
}
