/* 
 * (c) 2009-2010 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "stdafx.h"

#include "workbench/wb_context_ui.h"
#include "sqlide/wb_context_sqlide.h"
#include "sqlide/wb_sql_editor_form.h"
#include "sqlide/wb_sqlide_module.h"
#include "sqlide/wb_sql_editor_snippets.h"
#include "recordset_export_form.h"
#include "grt/common.h"
#include "string_utilities.h"

using namespace wb;
using namespace bec;
using namespace base;

static void call_commit(wb::WBContextSQLIDE *sqlide)
{
  if (sqlide->get_active_sql_editor())
    sqlide->get_active_sql_editor()->commit();
}

static bool validate_commit(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (!db_sql_editor || db_sql_editor->is_running_query())
    return false;
  return !db_sql_editor->auto_commit();
}


static void call_rollback(wb::WBContextSQLIDE *sqlide)
{
  if (sqlide->get_active_sql_editor())
    sqlide->get_active_sql_editor()->rollback();  
}

static bool validate_rollback(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (!db_sql_editor || db_sql_editor->is_running_query())
    return false;
  return !db_sql_editor->auto_commit();
}


static void call_autocommit(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->auto_commit(!db_sql_editor->auto_commit());
}

static wb::CommandItemValidationState validate_autocommit(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (!db_sql_editor || db_sql_editor->is_running_query())
    return wb::ItemDisabled;
  // Note: meaning has been switched here. Auto commit on means the button is NOT checked.
  return db_sql_editor->auto_commit() ? wb::ItemEnabled : wb::ItemEnabledAndChecked;
}


static void call_export(wb::WBContextSQLIDE *sqlide)
{
  if (sqlide->get_active_sql_editor())
  {
    RecordsetExportForm exporter(sqlide->get_active_sql_editor());
    
    exporter.run();
  }
}

static bool validate_export(wb::WBContextSQLIDE *sqlide)
{
  if (sqlide->get_active_sql_editor())
    return sqlide->get_active_sql_editor()->recordset_count() > 0;

  return false;
}



static void call_cancel(wb::WBContextSQLIDE *sqlide)
{
  sqlide->get_active_sql_editor()->cancel_query();
}

static bool validate_cancel(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (!db_sql_editor || !db_sql_editor->is_running_query())
    return false;
  return true;
}


static void call_new_file(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *editor= sqlide->get_active_sql_editor();
  if (editor)
    editor->new_sql_script_file();
}

static bool validate_new_file(wb::WBContextSQLIDE *sqlide)
{
  return sqlide->get_active_sql_editor() != NULL;
}


static void call_open_file(wb::WBContextSQLIDE *sqlide)
{
  std::string path= sqlide->get_wbui()->get_wb()->show_file_dialog("open", _("Open SQL Script"), "SQL Files (*.sql)|*.sql");
  if (!path.empty())
    sqlide->open_document(path);
}

static bool validate_open_file(wb::WBContextSQLIDE *sqlide)
{
  return sqlide->get_active_sql_editor() != NULL;
}

static void call_save_file(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *editor = sqlide->get_active_sql_editor();
  if (editor)
  {
    int i = editor->active_sql_editor_index();
    if (i >= 0)
    {
      editor->save_sql_script_file(editor->sql_editor_path(i), i);
    }
  }
}

static void call_save_file_as(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *editor = sqlide->get_active_sql_editor();
  if (editor)
  {
    int i = editor->active_sql_editor_index();
    if (i >= 0)
    {
      editor->save_sql_script_file("", i);
    }
  }
}

static bool validate_save_file(wb::WBContextSQLIDE *sqlide)
{
  return sqlide->get_active_sql_editor() != NULL;
}

static void call_save_snippet(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (db_sql_editor)
  {
    int start, end;
    std::string text = db_sql_editor->sql_editor()->sql();
    if (db_sql_editor->sql_editor()->selected_range(start, end))
      text = text.substr(start, end - start);
    if (db_sql_editor->save_snippet(text))
      sqlide->get_grt_manager()->replace_status_text("SQL saved to snippets list.");
  }
}


static bool validate_save_snippet(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  
  // don't check contents as the toolbar is not updated when these change atm
  if (!db_sql_editor/* || db_sql_editor->sql_editor()->sql().empty()*/)
    return false;
  return true;
}


static void call_continue_on_error(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->continue_on_error(!db_sql_editor->continue_on_error());
}

static wb::CommandItemValidationState validate_continue_on_error(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (!db_sql_editor)
    return wb::ItemDisabled;
  return db_sql_editor->continue_on_error() ? wb::ItemEnabled : wb::ItemEnabledAndChecked;
}


static void call_recordsets_are_pinned_by_default(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->recordsets_are_pinned_by_default(!db_sql_editor->recordsets_are_pinned_by_default());
}

static wb::CommandItemValidationState validate_recordsets_are_pinned_by_default(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (!db_sql_editor)
    return wb::ItemDisabled;
  
  return db_sql_editor->recordsets_are_pinned_by_default() ? wb::ItemEnabledAndChecked : wb::ItemEnabled;
}


static void call_explain_sql(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->explain_sql(false);
}

static bool validate_explain_sql(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (!db_sql_editor || db_sql_editor->is_running_query())
    return false;
  return true;
}


static void call_explain_current_statement(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (db_sql_editor)
    db_sql_editor->explain_current_statement();
}

static bool validate_explain_current_statement(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (!db_sql_editor || db_sql_editor->is_running_query())
    return false;
  return true;
}


static void call_reconnect(wb::WBContextSQLIDE *sqlide)
{
  if (sqlide->get_active_sql_editor())
    sqlide->get_active_sql_editor()->connect();
}

static bool validate_reconnect(wb::WBContextSQLIDE *sqlide)
{
  return sqlide->get_active_sql_editor() != NULL;
}


static void call_refresh_physical_overview(wb::WBContextSQLIDE *sqlide)
{
  if (sqlide->get_active_sql_editor())
    sqlide->get_active_sql_editor()->refresh_physical_overview();
}

static bool validate_refresh_physical_overview(wb::WBContextSQLIDE *sqlide)
{
  return sqlide->get_active_sql_editor() != NULL;
}


static void call_new_tab(wb::WBContextSQLIDE *sqlide)
{
  if (sqlide->get_active_sql_editor())
  {
    db_mgmt_ConnectionRef conn(sqlide->get_active_sql_editor()->connection_descriptor());
    
    sqlide->get_wbui()->get_wb()->add_new_query_window(conn);
  }
}

static bool validate_new_tab(wb::WBContextSQLIDE *sqlide)
{
  return sqlide->get_active_sql_editor() != NULL;
}


#ifdef _WIN32
static void call_exec_sql(wb::WBContextSQLIDE *sqlide)
{
  if (sqlide->get_active_sql_editor())
    sqlide->get_active_sql_editor()->call_exec_sql();
}

static bool validate_exec_sql(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (!db_sql_editor || db_sql_editor->is_running_query())
    return false;
  return db_sql_editor->validate_exec_sql();
}


static void call_exec_current_sql_statement(wb::WBContextSQLIDE *sqlide)
{
  if (sqlide->get_active_sql_editor())
    sqlide->get_active_sql_editor()->call_exec_current_sql_statement();
}

static bool validate_exec_current_sql_statement(wb::WBContextSQLIDE *sqlide)
{
  Db_sql_editor *db_sql_editor = sqlide->get_active_sql_editor();
  if (!db_sql_editor || db_sql_editor->is_running_query())
    return false;
  return db_sql_editor->validate_exec_current_sql_statement();
}


static void call_save_edits(wb::WBContextSQLIDE *sqlide)
{
  if (sqlide->get_active_sql_editor())
    sqlide->get_active_sql_editor()->call_save_edits();
}

static bool validate_save_edits(wb::WBContextSQLIDE *sqlide)
{
  if (sqlide->get_active_sql_editor())
    return sqlide->get_active_sql_editor()->validate_save_edits();
  return false;
}


static void call_discard_edits(wb::WBContextSQLIDE *sqlide)
{
  if (sqlide->get_active_sql_editor())
    sqlide->get_active_sql_editor()->call_discard_edits();
}

static bool validate_discard_edits(wb::WBContextSQLIDE *sqlide)
{
  if (sqlide->get_active_sql_editor())
    return sqlide->get_active_sql_editor()->validate_discard_edits();
  return false;
}

#endif // _WIN32

//--------------------------------------------------------------------------------------------------

WBContextSQLIDE::WBContextSQLIDE(WBContextUI *wbui)
: _wbui(wbui), _search_replace_dialog(NULL)
{
}

//--------------------------------------------------------------------------------------------------

WBContextSQLIDE::~WBContextSQLIDE()
{
  delete _search_replace_dialog;
}

//--------------------------------------------------------------------------------------------------

bec::GRTManager *WBContextSQLIDE::get_grt_manager()
{
  return _wbui->get_wb()->get_grt_manager();
}


void WBContextSQLIDE::init()
{
  DbSqlEditorSnippets::setup(this, bec::make_path(get_grt_manager()->get_user_datadir(), "snippets"));
  
  // register the builtin module
  /*WbSqlIdeImpl *module= */_wbui->get_wb()->get_grt()->get_native_module<WbSqlIdeImpl>();
  _wbui->signal_main_form_change().connect(sigc::mem_fun(this, &WBContextSQLIDE::active_form_changed));
  
  // setup some builtin commands handled by ourselves for the SQL IDE
  _wbui->get_command_ui()->add_builtin_command("query.commit", sigc::bind(sigc::ptr_fun(call_commit), this), 
                                              sigc::bind(sigc::ptr_fun(validate_commit), this));
  _wbui->get_command_ui()->add_builtin_command("query.rollback", sigc::bind(sigc::ptr_fun(call_rollback), this), 
                                              sigc::bind(sigc::ptr_fun(validate_rollback), this));
  _wbui->get_command_ui()->add_builtin_toggle_command("query.autocommit",
                                               sigc::bind(sigc::ptr_fun(call_autocommit), this),
                                               sigc::bind(sigc::ptr_fun(validate_autocommit), this));

  _wbui->get_command_ui()->add_builtin_command("query.savesnippet", sigc::bind(sigc::ptr_fun(call_save_snippet), this), 
                                               sigc::bind(sigc::ptr_fun(validate_save_snippet), this));

  _wbui->get_command_ui()->add_builtin_command("query.newtab", sigc::bind(sigc::ptr_fun(call_new_tab), this), 
                                               sigc::bind(sigc::ptr_fun(validate_new_tab), this));

  _wbui->get_command_ui()->add_builtin_command("query.newFile", sigc::bind(sigc::ptr_fun(call_new_file), this), 
                                               sigc::bind(sigc::ptr_fun(validate_new_file), this));
  _wbui->get_command_ui()->add_builtin_command("query.openFile", sigc::bind(sigc::ptr_fun(call_open_file), this), 
                                               sigc::bind(sigc::ptr_fun(validate_open_file), this));
  _wbui->get_command_ui()->add_builtin_command("query.saveFile", sigc::bind(sigc::ptr_fun(call_save_file), this), 
                                               sigc::bind(sigc::ptr_fun(validate_save_file), this));
  _wbui->get_command_ui()->add_builtin_command("query.saveFileAs", sigc::bind(sigc::ptr_fun(call_save_file_as), this), 
                                               sigc::bind(sigc::ptr_fun(validate_save_file), this));

  _wbui->get_command_ui()->add_builtin_command("query.export", sigc::bind(sigc::ptr_fun(call_export), this), 
                                              sigc::bind(sigc::ptr_fun(validate_export), this));
  
  _wbui->get_command_ui()->add_builtin_command("query.cancel", sigc::bind(sigc::ptr_fun(call_cancel), this), 
                                               sigc::bind(sigc::ptr_fun(validate_cancel), this));

  _wbui->get_command_ui()->add_builtin_command("query.reconnect", sigc::bind(sigc::ptr_fun(call_reconnect), this), 
                                               sigc::bind(sigc::ptr_fun(validate_reconnect), this));

  _wbui->get_command_ui()->add_builtin_command("query.refresh_physical_overview", sigc::bind(sigc::ptr_fun(call_refresh_physical_overview), this), 
                                               sigc::bind(sigc::ptr_fun(validate_refresh_physical_overview), this));

  _wbui->get_command_ui()->add_builtin_toggle_command("query.stopOnError",
                                               sigc::bind(sigc::ptr_fun(call_continue_on_error), this),
                                               sigc::bind(sigc::ptr_fun(validate_continue_on_error), this));
  _wbui->get_command_ui()->add_builtin_toggle_command("query.recordsets_are_pinned_by_default",
                                               sigc::bind(sigc::ptr_fun(call_recordsets_are_pinned_by_default), this), 
                                               sigc::bind(sigc::ptr_fun(validate_recordsets_are_pinned_by_default), this));

  _wbui->get_command_ui()->add_builtin_command("search_replace", sigc::mem_fun(this, &WBContextSQLIDE::start_search_replace));

  _wbui->get_command_ui()->add_builtin_command("query.explain",
                                               sigc::bind(sigc::ptr_fun(call_explain_sql), this),
                                               sigc::bind(sigc::ptr_fun(validate_explain_sql), this));
  _wbui->get_command_ui()->add_builtin_command("query.explain_current_statement",
                                               sigc::bind(sigc::ptr_fun(call_explain_current_statement), this),
                                               sigc::bind(sigc::ptr_fun(validate_explain_current_statement), this));

  // other commands declared in frontend code
#ifdef _WIN32
  _wbui->get_command_ui()->add_builtin_command("query.execute",
    sigc::bind(sigc::ptr_fun(call_exec_sql), this), sigc::bind(sigc::ptr_fun(validate_exec_sql), this));
  _wbui->get_command_ui()->add_builtin_command("query.execute_current_statement",
    sigc::bind(sigc::ptr_fun(call_exec_current_sql_statement), this), sigc::bind(sigc::ptr_fun(validate_exec_current_sql_statement), this));
  _wbui->get_command_ui()->add_builtin_command("query.save_edits",
    sigc::bind(sigc::ptr_fun(call_save_edits), this), sigc::bind(sigc::ptr_fun(validate_save_edits), this));
  _wbui->get_command_ui()->add_builtin_command("query.discard_edits",
    sigc::bind(sigc::ptr_fun(call_discard_edits), this), sigc::bind(sigc::ptr_fun(validate_discard_edits), this));
#endif // _WIN32
}


grt::ListRef<app_ToolbarItem> WBContextSQLIDE::get_toolbar_items(const std::string &name)
{
  WBContext *wb= _wbui->get_wb();
 
  if (name == "main")
    return app_ToolbarRef::cast_from(wb->get_grt()->unserialize(make_path(wb->get_datadir(), "data/dbquery_toolbar.xml")))->items();
  
  return grt::ListRef<app_ToolbarItem>();
}


Db_sql_editor* WBContextSQLIDE::create_connected_editor(const db_mgmt_ConnectionRef &conn)
{
  Db_sql_editor::Ref editor= Db_sql_editor::create(this, conn);
  return editor.get();
}


Db_sql_editor* WBContextSQLIDE::get_active_sql_editor()
{
  bec::UIForm *form= _wbui->get_active_main_form();
  if (form)
    return dynamic_cast<Db_sql_editor*>(form);
  return 0;
}


void WBContextSQLIDE::active_form_changed(bec::UIForm *form)
{
  Db_sql_editor *editor= get_active_sql_editor();
  if (editor)
  {
    // tell frontend to update the sidebar
    editor->do_partial_ui_refresh(Db_sql_editor::RefreshSidebar);
  }
}


bool WBContextSQLIDE::activate_live_object(GrtObjectRef object)
{
  Db_sql_editor *editor= get_active_sql_editor();
  if (!editor)
    return false;
  return editor->activate_live_object(object);
}


bool WBContextSQLIDE::create_live_object(GrtObjectRef object_type, std::string owner_name, std::string obj_name)
{
  Db_sql_editor *editor= get_active_sql_editor();
  if (!editor)
    return false;
  return editor->create_live_object(object_type, owner_name, obj_name);
}


bool WBContextSQLIDE::drop_live_object(GrtObjectRef object_type, std::string owner_name, std::string obj_name)
{
  Db_sql_editor *editor= get_active_sql_editor();
  if (!editor)
    return false;
  return editor->drop_live_object(object_type, owner_name, obj_name);
}

//--------------------------------------------------------------------------------------------------

void WBContextSQLIDE::start_search_replace()
{
  // We cannot re-use the dialog, as it automatically gets disposed on Windows when closed.
  // Needs investigation if that can be prevented. 
  // However, we *have* to delete the mforms part of that dialog.
  delete _search_replace_dialog;
  _search_replace_dialog= new mforms::SearchReplace();
  
  Db_sql_editor *editor= get_active_sql_editor();
  if (editor)
  {
    _search_replace_dialog->set_callback(editor->sql_editor()->do_search_slot);
    _search_replace_dialog->show(false, mforms::SearchNone, true);
  }
}

//--------------------------------------------------------------------------------------------------

void WBContextSQLIDE::open_document(const std::string &path)
{
  Db_sql_editor *editor= get_active_sql_editor();
  if (editor)
    editor->open_sql_script_file(path, false);
  else
    mforms::Utilities::show_error("Open SQL Script",
                                  "Please select a connected SQL Editor tab to open a script file.",
                                  "OK", "", "");
}

