/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "stdafx.h"

#include "wb_sql_editor_form.h"
#include "sqlide/recordset_be.h"
#include "sqlide/recordset_cdbc_storage.h"
#include "grtsqlparser/sql_facade.h"
#include "grtui/grtdb_connection_editor.h"
#include "sqlide/wb_sql_editor_snippets.h"

#include "workbench/wb_context_ui.h"

#include "grtdb/editor_dbobject.h"
#include "grtdb/editor_table.h"

#include "diff/diffchange.h"
#include "grtdb/diff_dbobjectmatch.h"
#include "db.mysql/src/module_db_mysql.h"

#include "sqlide/sql_script_run_wizard.h"
#include "grtui/file_charset_dialog.h"
#include "sqlide/recordset_export_form.h"

#include <boost/foreach.hpp>
#include <locale>
#include <algorithm>
#include <cctype>
#include <sstream>


using namespace bec;
using namespace grt;
using namespace wb;
using namespace base;


#define CATCH_EXCEPTION_AND_DISPATCH(statement) \
catch (sql::SQLException &e)\
{\
  add_log_message(grt::ErrorMsg, strfmt("Error Code: %i\n%s", e.getErrorCode(), e.what()), statement);\
}\
catch (std::exception &e)\
{\
  add_log_message(grt::ErrorMsg, strfmt("Error: %s", e.what()), statement);\
}


#define CATCH_EXCEPTION_AND_DISPATCH_TO_DEFAULT_LOG(statement) \
catch (sql::SQLException &e)\
{\
  _grtm->get_grt()->send_error(strfmt("Error Code: %i\n%s", e.getErrorCode(), e.what()), statement);\
}\
catch (std::exception &e)\
{\
  _grtm->get_grt()->send_error(strfmt("Error: %s", e.what()), statement);\
}


class Timer
{
public:
  Timer(bool run_immediately) : _is_running(false), _start_timestamp(0), _duration(0)
  {
    if (run_immediately)
      run();
  }
  void reset()
  {
    _is_running= false;
    _start_timestamp= 0;
    _duration= 0;
  }
  void run()
  {
    if (_is_running)
      return;
    _is_running= true;
    _start_timestamp= timestamp();
  }
  void stop()
  {
    if (!_is_running)
      return;
    _is_running= false;
    _duration+= timestamp() - _start_timestamp;
  }
  time_t duration()
  {
    return _is_running ? (_duration + timestamp() - _start_timestamp) : _duration;
  }
  std::string duration_formatted()
  {
    std::div_t s;
    s= std::div((int)duration(), 1000);
    return strfmt(_("%i.%.3i sec"), s.quot, s.rem);
  }
private:
  bool _is_running;
  time_t _start_timestamp;
  time_t _duration;
};


std::string & strip_sql_ident(std::string &text)
{
  std::locale loc;

  size_t size= text.size();

  size_t start= 0;
  for (; start < size; ++start)
    if (!std::isspace(text[start], loc) && (text[start] != '`') && (text[start] != '\'') && (text[start] != '"'))
      break;

  size_t end= size;
  for (; end > 0; --end)
    if (!std::isspace(text[end-1], loc) && (text[end-1] != '`') && (text[end-1] != '\'') && (text[end-1] != '"'))
      break;

  return text= text.substr(start, end-start);
}


class ScopeExitCaller
{
public:
  typedef sigc::slot<void> Slot;
  ScopeExitCaller() {}
  ScopeExitCaller(const Slot &cb) : slot(cb) {}
  ~ScopeExitCaller() { slot(); }
  ScopeExitCaller & operator=(const Slot &cb) { slot= cb; return *this; }
  Slot slot;
};


Db_sql_editor::Ref Db_sql_editor::create(wb::WBContextSQLIDE *wbsql, const db_mgmt_ConnectionRef &conn)
{
  Ref editor(new Db_sql_editor(wbsql, conn)); 

  // self-reference must be set first of all, right after construction
  editor->_self= editor;

  try
  {
    if (conn.is_valid())
      editor->connect();
  }
  catch (...)
  {
    // object was not created properly, hence release self-reference to let it die
    editor->_self.reset();

    throw;
  }
  return editor;
}


Db_sql_editor::Db_sql_editor(wb::WBContextSQLIDE *wbsql, const db_mgmt_ConnectionRef &conn)
:
_wbsql(wbsql),
_context_ui(_wbsql->get_wbui()),
_grtm(wbsql->get_grt_manager()),
_active_sql_editor_index(0),
_keep_alive_thread(NULL),
_connection(conn),
_aux_dbc_conn(new sql::Dbc_connection_handler()),
_usr_dbc_conn(new sql::Dbc_connection_handler()),
exec_sql_task(GrtThreadedTask::create(wbsql->get_grt_manager())),
_is_running_query(false),
_recordsets_are_pinned_by_default(false),
_rs_sequence(0),
_schema_tree(wbsql->get_grt_manager()),
live_schema_fetch_task(GrtThreadedTask::create(wbsql->get_grt_manager())),
_live_physical_overview(NULL),
_log(DbSqlEditorLog::create(wbsql->get_grt_manager())),
_history(DbSqlEditorHistory::create(wbsql->get_grt_manager()))
{
  _sql_editors_mutex= g_mutex_new();
  _aux_dbc_conn_mutex= g_mutex_new();
  _usr_dbc_conn_mutex= g_mutex_new();
  _schema_contents_mutex= g_mutex_new();
  _recordsets_mutex= g_mutex_new();

  _schema_tree.refresh_signal.connect(sigc::mem_fun(this, &Db_sql_editor::refresh_physical_overview));
  _schema_tree.details_fetch_slot= sigc::mem_fun(this, &Db_sql_editor::fetch_object_details);
  _schema_tree.list_fetch_slot= sigc::mem_fun(this, &Db_sql_editor::fetch_live_schema_list);
  _schema_tree.fetch_slot= sigc::mem_fun(this, &Db_sql_editor::fetch_live_schema_contents);
  _schema_tree.sql_editor_text_insert_signal.connect(sql_editor_text_insert_signal.make_slot());
  _schema_tree.activate_object_signal.connect(sigc::mem_fun(this, &Db_sql_editor::schema_object_activated));
  _schema_tree.alter_live_object_signal.connect(sigc::mem_fun(this, &Db_sql_editor::do_alter_live_object));
  _schema_tree.create_live_object_signal.connect(sigc::mem_fun(this, &Db_sql_editor::do_create_live_object));
  _schema_tree.drop_live_object_signal.connect(sigc::mem_fun(this, &Db_sql_editor::do_drop_live_object));

  _live_physical_overview= new LivePhysicalOverviewBE(_context_ui->get_wb());
  _live_physical_overview->refresh_signal.connect(sigc::mem_fun(this, &Db_sql_editor::refresh_physical_overview));
  //!_live_physical_overview->schema_list_fetch_slot= sigc::mem_fun(this, &Db_sql_editor::fetch_live_schema_list);
  _live_physical_overview->schema_fetch_slot= sigc::mem_fun(this, &Db_sql_editor::fetch_live_schema_contents);
  _live_physical_overview->activate_object_signal.connect(sigc::mem_fun(this, &Db_sql_editor::schema_object_activated));
  _live_physical_overview->alter_live_object_signal.connect(sigc::mem_fun(this, &Db_sql_editor::do_alter_live_object));
  //!_live_physical_overview->create_live_object_signal.connect(sigc::mem_fun(this, &Db_sql_editor::do_create_live_object));
  _live_physical_overview->drop_live_object_signal.connect(sigc::mem_fun(this, &Db_sql_editor::do_drop_live_object));
  _live_physical_overview->active_schema_index= sigc::mem_fun(this, &Db_sql_editor::active_schema_index);

  exec_sql_task->send_task_res_msg(false);
  exec_sql_task->msg_cb(sigc::mem_fun(this, &Db_sql_editor::add_log_message));

  live_schema_fetch_task->send_task_res_msg(false);
  live_schema_fetch_task->msg_cb(sigc::mem_fun(this, &Db_sql_editor::add_log_message));

  _continue_on_error= false;
  {
    DictRef options= DictRef::cast_from(_grtm->get_grt()->get("/wb/options/options"));
    int keep_alive_interval= options.get_int("DbSqlEditor:KeepAliveInterval", 600);
    if (keep_alive_interval != 0)
      _keep_alive_thread= TimerActionThread::create(sigc::mem_fun(this, &Db_sql_editor::send_message_keep_alive), keep_alive_interval*1000);
    _continue_on_error= (options.get_int("DbSqlEditor:ContinueOnError", 0) != 0);
  }

  add_sql_editor(); // start with 1 text-buffer for editing sql
}


Db_sql_editor::~Db_sql_editor()
{
  reset();

  if (_keep_alive_thread)
    _keep_alive_thread->stop();

  delete _live_physical_overview;

  g_mutex_free(_recordsets_mutex);
  g_mutex_free(_schema_contents_mutex);
  g_mutex_free(_usr_dbc_conn_mutex);
  g_mutex_free(_aux_dbc_conn_mutex);
  g_mutex_free(_sql_editors_mutex);
}


Db_sql_editor::Ref Db_sql_editor::self_ref()
{
  return _self;
}


void Db_sql_editor::dispose()
{
  exec_sql_task->dispose();
  live_schema_fetch_task->dispose();

  close_recordset_ui= sigc::signal<int, long long>();
  sql_editor_text_insert_signal= SqlEditorTextInsertSignal();
  sql_editor_new_ui= sigc::signal<int, int>();
  sql_editor_caption_ui= sigc::signal<int, std::string>();

  on_sql_script_run_error= Error_cb();
  on_sql_script_run_progress= Batch_exec_progress_cb();
  on_sql_script_run_statistics= Batch_exec_stat_cb();

  set_refresh_ui_slot(RefreshSlot());
  set_partial_refresh_ui_slot(RefreshUI::PartialRefreshSlot());

  BOOST_FOREACH (Recordset::Ref rs, _recordsets)
    rs->dispose();

  if (_live_physical_overview)
    _live_physical_overview->dispose();

  _schema_tree.dispose();

  // release retained self-reference to allow destruction of the object
  // this will also prevent initiation of new threaded tasks
  // running threaded tasks will retain reference and hold the object from destruction until they are finished
  _self.reset();
}


bool Db_sql_editor::close()
{
  bool res= bec::UIForm::close();
  if (res)
    dispose();
  return res;
}


std::string Db_sql_editor::get_form_context_name() const
{
  return WB_CONTEXT_QUERY;
}


int Db_sql_editor::add_sql_editor()
{
  Sql_editor::Ref sql_editor= Sql_editor::create(rdbms());

  {
    DictRef options= DictRef::cast_from(_grtm->get_grt()->get("/wb/options/options"));
    int progress_status_update_interval= options.get_int("DbSqlEditor:ProgressStatusUpdateInterval", 500);
    sql_editor->sql_check_progress_msg_throttle(progress_status_update_interval);
  }
  sql_editor->text_change_signal.connect(sigc::mem_fun(this, &Db_sql_editor::on_sql_editor_text_change));
  //sql_editor->text_selection_change_signal.connect(sigc::mem_fun(this, &Db_sql_editor::on_sql_editor_text_selection_change));

  int sql_editor_index;
  {
    GMutexLock sql_editors_mutex(_sql_editors_mutex);
    Sql_editor_info info;
    info.filename = "";
    info.editor = sql_editor;
    info.dirty = false;
    _sql_editors.push_back(info);
    
    sql_editor_index= _sql_editors.size() - 1;
  }

  return sql_editor_index;
}


void Db_sql_editor::remove_sql_editor(int index)
{
  GMutexLock sql_editors_mutex(_sql_editors_mutex);
  if ((index < 0) || (index >= (int)_sql_editors.size()))
    return;
  _sql_editors.erase(_sql_editors.begin() + index);
}


int Db_sql_editor::sql_editor_count()
{
  GMutexLock sql_editors_mutex(_sql_editors_mutex);
  return _sql_editors.size();
}


Sql_editor::Ref Db_sql_editor::sql_editor()
{
  return _sql_editors.at(_active_sql_editor_index).editor;
}


Sql_editor::Ref Db_sql_editor::sql_editor(int index)
{
  GMutexLock sql_editors_mutex(_sql_editors_mutex);
  return _sql_editors.at(index).editor;
}


void Db_sql_editor::split_sql(const std::string &sql, std::list<std::string> &statements)
{
  SqlFacade::Ref sql_facade= SqlFacade::instance_for_rdbms(rdbms());
  sql_facade->splitSqlScript(sql, statements);
}


void Db_sql_editor::reset()
{
  //_log->reset();
  _rs_sequence= 0;
}


RowId Db_sql_editor::add_log_message(int msg_type, const std::string &msg, const std::string &context)
{
  return add_log_message_ex(msg_type, msg, context, "");
}


RowId Db_sql_editor::add_log_message_ex(int msg_type, const std::string &msg, const std::string &context, const std::string &duration)
{
  RowId new_log_message_index= _log->add_message(msg_type, context, msg, duration);
  _has_pending_progress_messages= true;
  return new_log_message_index;
}


void Db_sql_editor::set_log_message_ex(RowId log_message_index, int msg_type, const std::string &msg, const std::string &context, const std::string &duration)
{
  _log->set_message(log_message_index, msg_type, context, msg, duration);
  _has_pending_progress_messages= true;
}


void Db_sql_editor::advance_exec_sql_task_progress()
{
  if (_has_pending_progress_messages)
  {
    exec_sql_task->send_progress(0.f, std::string(), std::string());
    _has_pending_progress_messages= false;
  }
}


void Db_sql_editor::get_log_event_details(int log_event_no, int &log_event_type_code, std::string &log_event_time, std::string &log_event_action, std::string &log_event_message)
{
  NodeId node(log_event_no);
  _log->get_field(node, 0, log_event_type_code);
  _log->get_field(node, 2, log_event_time);
  _log->get_field(node, 3, log_event_action);
  _log->get_field(node, 4, log_event_message);
}


int Db_sql_editor::recordset_count()
{
  return _recordsets.size();
}


Recordset::Ref Db_sql_editor::recordset(int index)
{
  return _recordsets.at(index);
}


void Db_sql_editor::init_connection(sql::Connection* dbc_conn_ref, const db_mgmt_ConnectionRef& connectionProperties, sql::Dbc_connection_handler::Ref& dbc_conn)
{
  db_mgmt_RdbmsRef rdbms= db_mgmt_RdbmsRef::cast_from(_connection->driver()->owner());
  SqlFacade::Ref sql_facade= SqlFacade::instance_for_rdbms(rdbms);
  Sql_specifics::Ref sql_specifics= sql_facade->sqlSpecifics();

  // connection startup script
  {
    std::list<std::string> sql_script;
    {
      sql_specifics->get_connection_startup_script(sql_script);
    }
    std::auto_ptr<sql::Statement> stmt(dbc_conn_ref->createStatement());
    sql::SqlBatchExec sql_batch_exec;
    sql_batch_exec(stmt.get(), sql_script);
  }

  // remember connection id
  {
    std::string query_connection_id= sql_specifics->query_connection_id();
    if (!query_connection_id.empty())
    {
      std::auto_ptr<sql::Statement> stmt(dbc_conn_ref->createStatement());
      stmt->execute(query_connection_id);
      boost::shared_ptr<sql::ResultSet> rs(stmt->getResultSet());
      rs->next();
      dbc_conn->id= rs->getInt(1);
    }
  }
}


void Db_sql_editor::create_connection(sql::Dbc_connection_handler::Ref dbc_conn, db_mgmt_ConnectionRef db_mgmt_conn, bool autocommit_mode)
{
  sql::DriverManager *dbc_drv_man= sql::DriverManager::getDriverManager();
  dbc_conn->ref= dbc_drv_man->getConnection(db_mgmt_conn,
    sigc::bind(sigc::mem_fun(this, &Db_sql_editor::init_connection), sigc::ref(dbc_conn)));
  
  if (dbc_conn->ref->getMetaData()->getDatabaseMajorVersion() < 5)
  {
    throw std::runtime_error("MySQL Server version is older than 5.0, which is not supported");
  }
  
  if (dbc_conn->active_schema.empty())
    dbc_conn->active_schema= dbc_conn->ref->getSchema();
  else
    dbc_conn->ref->setSchema(dbc_conn->active_schema);
  dbc_conn->ref->setAutoCommit(autocommit_mode);
  dbc_conn->autocommit_mode= dbc_conn->ref->getAutoCommit();
}


void Db_sql_editor::connect()
{
  Ref self= _self;
  if (!self)
    return;

  {
    GMutexLock aux_dbc_conn_mutex(_aux_dbc_conn_mutex);
    GMutexLock usr_dbc_conn_mutex(_usr_dbc_conn_mutex);

    reset();

    _aux_dbc_conn->ref.reset();
    _usr_dbc_conn->ref.reset();
    create_connection(_aux_dbc_conn, _connection, _aux_dbc_conn->autocommit_mode);
    create_connection(_usr_dbc_conn, _connection, _usr_dbc_conn->autocommit_mode);

    // connection info
    try
    {
      _connection_details["name"] = _connection->name();
      _connection_details["hostName"] = _connection->parameterValues().get_string("hostName");
      _connection_details["port"] = strfmt("%li\n", _connection->parameterValues().get_int("port"));
      _connection_details["socket"] = _connection->parameterValues().get_string("socket");
      _connection_details["driverName"] = _connection->driver()->name();
      _connection_details["dbmsProductName"] = _usr_dbc_conn->ref->getMetaData()->getDatabaseProductName();
      _connection_details["dbmsProductVersion"] = _usr_dbc_conn->ref->getMetaData()->getDatabaseProductVersion();
      _connection_details["userName"] = _connection->parameterValues().get_string("userName");
  
      // Connection:
      _connection_info=       "Connection Information\n";
      _connection_info.append("======================\n");
      _connection_info.append("Name:    ").append(_connection->name()).append("\n");
      // Host:
      if (_connection->driver()->name() == "MysqlNativeSocket")
      {
#ifdef _WIN32
        std::string name = _connection->parameterValues().get_string("socket", "pipe");
#else
        std::string name = _connection->parameterValues().get_string("socket", "UNIX domain socket");
#endif
        _connection_info.append("Localhost through ").append(name);
      }
      else
        _connection_info.append("Host:    ").append(strfmt("%s:%i\n",
                                                           _connection->parameterValues().get_string("hostName").c_str(),
                                                           _connection->parameterValues().get_int("port")));
      // Server:
      _connection_info.append("Server:  ").append(_usr_dbc_conn->ref->getMetaData()->getDatabaseProductName()).append("\n");
      _connection_info.append("Version: ").append(_usr_dbc_conn->ref->getMetaData()->getDatabaseProductVersion()).append("\n");
      // User:
      _connection_info.append("User:    ").append(_connection->parameterValues().get_string("userName")).append("\n");
    }
    CATCH_EXCEPTION_AND_DISPATCH("Get connection information");
  }

  db_mgmt_RdbmsRef rdbms= db_mgmt_RdbmsRef::cast_from(grt::copy_object(_grtm->get_grt(), this->rdbms()));
  std::string database_package= *rdbms->databaseObjectPackage();

  workbench_physical_ModelRef model(_grtm->get_grt());
  rdbms->owner(model);
  model->rdbms(rdbms);

  db_CatalogRef catalog= _grtm->get_grt()->create_object<db_Catalog>(database_package + ".Catalog");
  {
    catalog->owner(model);
    catalog->name("default");
    catalog->oldName("default");
    catalog->version(rdbms->version());
    grt::replace_contents(catalog->simpleDatatypes(), rdbms->simpleDatatypes());
    model->catalog(catalog);
  }

  _live_physical_overview->set_model(model);

  refresh_physical_overview();

  _context_ui->refresh_toolbar(WB_TOOLBAR_MAIN);
}


bool Db_sql_editor::connected() const
{
  return (_usr_dbc_conn && _usr_dbc_conn->ref.get_ptr() && !_usr_dbc_conn->ref->isClosed());
}


GMutexLock Db_sql_editor::ensure_valid_aux_connection()
{
  return ensure_valid_dbc_connection(_aux_dbc_conn, _aux_dbc_conn_mutex);
}


GMutexLock Db_sql_editor::ensure_valid_usr_connection()
{
  return ensure_valid_dbc_connection(_usr_dbc_conn, _usr_dbc_conn_mutex);
}


GMutexLock Db_sql_editor::ensure_valid_dbc_connection(sql::Dbc_connection_handler::Ref dbc_conn, GMutex *dbc_conn_mutex)
{
  GMutexLock mutex_lock(dbc_conn_mutex);
  bool valid= false;
  if (dbc_conn && dbc_conn->ref.get_ptr())
  {
    if (dbc_conn->ref->isClosed())
    {
      if (dbc_conn->autocommit_mode)
      {
        create_connection(dbc_conn, _connection, dbc_conn->autocommit_mode);
        if (!dbc_conn->ref->isClosed())
          valid= true;
      }
    }
    else
      valid= true;
  }
  if (!valid)
    throw std::runtime_error("DBMS connection is not available");

  return mutex_lock;
}


bool Db_sql_editor::auto_commit()
{
  if (_usr_dbc_conn)
    return _usr_dbc_conn->autocommit_mode;
  return false;
}


void Db_sql_editor::auto_commit(bool value)
{
  if (!_usr_dbc_conn)
    return;
  if (value == _usr_dbc_conn->autocommit_mode)
    return;
  {
    const char *STATEMENT= value ? "AUTOCOMMIT=1" : "AUTOCOMMIT=0";
    try
    {
      GMutexLock usr_dbc_conn_mutex= ensure_valid_usr_connection();
      if (value)
        do_commit();
      _usr_dbc_conn->ref->setAutoCommit(value);
      _usr_dbc_conn->autocommit_mode= _usr_dbc_conn->ref->getAutoCommit();
    }
    CATCH_EXCEPTION_AND_DISPATCH(STATEMENT)
  }
  _context_ui->refresh_toolbar(WB_TOOLBAR_MAIN);
}


void Db_sql_editor::cancel_query()
{
  std::string query_kill_connection;
  {
    db_mgmt_RdbmsRef rdbms= db_mgmt_RdbmsRef::cast_from(_connection->driver()->owner());
    SqlFacade::Ref sql_facade= SqlFacade::instance_for_rdbms(rdbms);
    Sql_specifics::Ref sql_specifics= sql_facade->sqlSpecifics();
    query_kill_connection= sql_specifics->query_kill_connection(_usr_dbc_conn->id);
  }

  try
  {
    if (!query_kill_connection.empty())
    {
      GMutexLock aux_dbc_conn_mutex= ensure_valid_aux_connection();
      std::auto_ptr<sql::Statement> stmt(_aux_dbc_conn->ref->createStatement());
      stmt->execute(query_kill_connection);
      add_log_message(grt::InfoMsg, "OK", query_kill_connection);
    }
  }
  CATCH_EXCEPTION_AND_DISPATCH(query_kill_connection);

  try
  {
    // reconnect but only if in autocommit mode
    if (_usr_dbc_conn->autocommit_mode)
    {
      // this will recreate connection if it was established previously
      ensure_valid_usr_connection();
    }
  }
  catch (sql::SQLException &)
  {
  }
}


void Db_sql_editor::commit()
{
  static const char *STATEMENT= "COMMIT";
  try
  {
    GMutexLock use_dbc_conn_mutex= ensure_valid_usr_connection();
    do_commit();
  }
  CATCH_EXCEPTION_AND_DISPATCH(STATEMENT)
}


void Db_sql_editor::do_commit()
{
  static const char *STATEMENT= "COMMIT";
  std::string duration;
  Timer timer(true);
  _usr_dbc_conn->ref->commit();
  timer.stop();
  add_log_message_ex(grt::InfoMsg, "OK", STATEMENT, timer.duration_formatted());
}


void Db_sql_editor::rollback()
{
  static const char *STATEMENT= "ROLLBACK";
  try
  {
    GMutexLock use_dbc_conn_mutex= ensure_valid_usr_connection();
    std::string duration;
    Timer timer(true);
    _usr_dbc_conn->ref->rollback();
    timer.stop();
    add_log_message_ex(grt::InfoMsg, "OK", STATEMENT, timer.duration_formatted());
  }
  CATCH_EXCEPTION_AND_DISPATCH(STATEMENT)
}


bool Db_sql_editor::explain_sql(bool only_check_if_applicable)
{
  int start, end;
  Sql_editor::Ref sql_editor_= sql_editor();
  sql_editor_->selected_range(start, end);
  std::string sql= sql_editor_->sql();
  if (start != end)
    sql= sql.substr(start, end-start);

  do_explain_sql(sql);
  return true;
}


void Db_sql_editor::explain_current_statement()
{
  Sql_editor::Ref sql_editor_= sql_editor();
  do_explain_sql(sql_editor_->current_statement());
}


void Db_sql_editor::do_explain_sql(const std::string &sql)
{
  Ref self= _self;
  if (!self)
    return;

  SqlFacade::Ref sql_facade= SqlFacade::instance_for_rdbms(rdbms());
  std::list<std::string> statements;
  sql_facade->splitSqlScript(sql, statements);
  Sql_syntax_check::Ref sql_syntax_check= sql_facade->sqlSyntaxCheck();
  std::string explain_script;
  for (std::list<std::string>::iterator i= statements.begin(), i_end= statements.end(); i != i_end; ++i)
    if (Sql_syntax_check::sql_select == sql_syntax_check->determine_statement_type(*i))
      explain_script+= "EXPLAIN " + *i + ";\n";
  exec_sql_task->proc_cb(sigc::bind(sigc::mem_fun(this, &Db_sql_editor::do_exec_sql), self, explain_script, true, false));
  exec_sql_task->exec(false, true);
}


void Db_sql_editor::set_sql_editor_text(const std::string &sql, bool run_sql)
{
  sql_editor()->sql(sql);
  do_partial_ui_refresh(Db_sql_editor::RefreshEditor);
  if (run_sql)
    do_partial_ui_refresh(Db_sql_editor::RunCurrentScript);
}


void Db_sql_editor::on_sql_editor_text_change()
{
//  _context_ui->refresh_menus();
  sql_editor_dirty(_active_sql_editor_index, true);
}


void Db_sql_editor::on_sql_editor_text_selection_change()
{
  _context_ui->refresh_menus();
}


void Db_sql_editor::exec_sql(std::string &sql, bool sync, bool wrap_with_non_std_delimiter)
{
  Ref self= _self;
  if (!self)
    return;

  exec_sql_task->proc_cb(sigc::bind(sigc::mem_fun(this, &Db_sql_editor::do_exec_sql), self, sql, false, wrap_with_non_std_delimiter));
  exec_sql_task->exec(sync, true);
}


grt::StringRef Db_sql_editor::do_exec_sql(grt::GRT *grt, Ref self, std::string &sql, bool retaining, bool wrap_with_non_std_delimiter)
{
  std::string statement;
  try
  {
    if (!retaining)
    {
      reset();
      if (!_recordsets_are_pinned_by_default)
      {
        GMutexLock recordsets_mutex(_recordsets_mutex);
        Recordsets recordsets;
        recordsets.reserve(_recordsets.size());
        BOOST_FOREACH (Recordset::Ref rs, _recordsets)
          if (rs->pinned())
            recordsets.push_back(rs);
        _recordsets.swap(recordsets);
      }
    }

    GMutexLock use_dbc_conn_mutex= ensure_valid_usr_connection();

    bool is_running_query= true;
    AutoSwap<bool> is_running_query_keeper(_is_running_query, is_running_query);
    request_refresh_of_menu_and_toolbar();

    _has_pending_progress_messages= false;
    bec::TimerActionThread *progress_status_update_thread;
    {
      DictRef options= DictRef::cast_from(_grtm->get_grt()->get("/wb/options/options"));
      int progress_status_update_interval= options.get_int("DbSqlEditor:ProgressStatusUpdateInterval", 500);
      progress_status_update_thread= TimerActionThread::create(
        sigc::mem_fun(this, &Db_sql_editor::advance_exec_sql_task_progress),
        progress_status_update_interval);
    }
    ScopeExitCaller progress_status_update_thread_stopper(sigc::mem_fun(progress_status_update_thread, &bec::TimerActionThread::stop));
    ScopeExitCaller final_progress_status_update(sigc::mem_fun(this, &Db_sql_editor::advance_exec_sql_task_progress));

    SqlFacade::Ref sql_facade= SqlFacade::instance_for_rdbms(rdbms());
    Sql_syntax_check::Ref sql_syntax_check= sql_facade->sqlSyntaxCheck();
    Sql_specifics::Ref sql_specifics= sql_facade->sqlSpecifics();

    if (wrap_with_non_std_delimiter)
      sql= sql_specifics->setting_non_std_sql_delimiter() + sql;

    std::list<std::string> statements;
    split_sql(sql, statements);
    _history->add_entry(statements);
    BOOST_FOREACH (statement, statements)
    {
      statement= strip_text(statement, false, true);
      if (statement.empty())
        continue;

      Sql_syntax_check::Statement_type statement_type= sql_syntax_check->determine_statement_type(statement);
      if (Sql_syntax_check::sql_empty == statement_type)
        continue;

      std::string schema_name;
      std::string table_name;
      std::string editable_query_tail;
      if (Sql_syntax_check::sql_edit == statement_type)
      {
        // to open table contents in edit mode the following syntax is used: EDIT [SCHEMA.]TABLE ...
        // this type of query needs special preprocessing: rewriting using 'select' statement
        statement_type= Sql_syntax_check::sql_select;
        sql_syntax_check->parse_edit_statement(statement, schema_name, table_name, editable_query_tail);
        if (schema_name.empty())
          schema_name= active_schema();
        if (schema_name.empty())
          schema_name= _usr_dbc_conn->ref->getSchema();
      }

      Recordset_cdbc_storage::Ref data_storage;//!
      if (Sql_syntax_check::sql_select == statement_type)
      {
        data_storage= Recordset_cdbc_storage::create(_grtm);
        data_storage->rdbms(rdbms());
        data_storage->dbms_conn(_usr_dbc_conn);
        if (table_name.empty())
          data_storage->sql_query(statement);
        data_storage->schema_name(schema_name);
        data_storage->table_name(table_name);
        data_storage->additional_clauses(editable_query_tail);

        if (Sql_syntax_check::sql_select == statement_type)
        {
          DictRef options= DictRef::cast_from(grt->get("/wb/options/options"));

          int limit_rows= options.get_int("SqlEditor:LimitRows");
          data_storage->limit_rows(0 != limit_rows);

          int limit_rows_count= options.get_int("SqlEditor:LimitRowsCount");
          if (0 != limit_rows_count)
            data_storage->limit_rows_count(limit_rows_count);
        }
        else
        {
          data_storage->limit_rows_applicable(false);
        }
        statement= data_storage->decorated_sql_query();
      }

      {
        RowId log_message_index= add_log_message_ex(grt::InfoMsg, "Running...", statement,
          ((Sql_syntax_check::sql_select == statement_type) ? "? / ?" : "?"));

        bool statement_failed= false;
        long long updated_rows_count= -1;
        Timer statement_exec_timer(false);
        Timer statement_fetch_timer(false);
        boost::shared_ptr<sql::Statement> dbc_statement(_usr_dbc_conn->ref->createStatement());
        try
        {
          statement_exec_timer.run();
          dbc_statement->execute(statement);
          statement_exec_timer.stop();
          updated_rows_count= dbc_statement->getUpdateCount();
        }
        catch (sql::SQLException &e)
        {
          if (!_continue_on_error)
            throw;
          add_log_message(grt::ErrorMsg, strfmt("Error Code: %i\n%s", e.getErrorCode(), e.what()), statement);
          statement_failed= true;
        }
        if (statement_failed)
        {
          continue; // goto next statement
        }

        set_log_message_ex(log_message_index, grt::InfoMsg, "Fetching...", statement, statement_exec_timer.duration_formatted() + " / ?");
        statement_fetch_timer.run();
        boost::shared_ptr<sql::ResultSet> dbc_resultset(dbc_statement->getResultSet());
        statement_fetch_timer.stop();
        if (dbc_resultset)
        {
          for (int resultset_count= 0; dbc_resultset; ++resultset_count)
          {
            if (!data_storage)
            {
              data_storage= Recordset_cdbc_storage::create(_grtm);
              data_storage->rdbms(rdbms());
              data_storage->dbms_conn(_usr_dbc_conn);
              if (table_name.empty())
                data_storage->sql_query(statement);
              data_storage->schema_name(schema_name);
              data_storage->table_name(table_name);
            }
            data_storage->dbc_statement(dbc_statement);
            data_storage->dbc_resultset(dbc_resultset);
            data_storage->reloadable(Sql_syntax_check::sql_select == statement_type);

            Recordset::Ref rs= Recordset::create(exec_sql_task);
            rs->is_field_value_truncation_enabled(true);
            rs->apply_changes= sigc::mem_fun(this, &Db_sql_editor::apply_changes_to_recordset);
            rs->on_close.connect(sigc::mem_fun(*this, &Db_sql_editor::on_close_recordset));
            rs->caption(strfmt("%s (%i)",
              (table_name.empty() ? "Result" : table_name.c_str()),
              ++_rs_sequence));
            rs->data_storage(data_storage);
            rs->export_wizard= sigc::bind(sigc::mem_fun(this, &Db_sql_editor::show_export_recordset), rs);
            rs->reset(true);

            if (data_storage->valid()) // query statement
            {
              {
                GMutexLock recordsets_mutex(_recordsets_mutex);
                _recordsets.push_back(rs);
              }
              std::string statement_res_msg= strfmt("%i row(s) returned", rs->row_count());
              std::string exec_and_fetch_durations=
                ((resultset_count) ? std::string("-") : statement_exec_timer.duration_formatted()) + " / " +
                statement_fetch_timer.duration_formatted();
              set_log_message_ex(log_message_index, grt::InfoMsg, statement_res_msg, statement, exec_and_fetch_durations);
            }
            //! else failed to fetch data

            if (dbc_statement->getMoreResults())
            {
              log_message_index= add_log_message_ex(grt::InfoMsg, "Fetching...", statement, "- / ?");
              statement_fetch_timer.reset();
              statement_fetch_timer.run();
              dbc_resultset.reset(dbc_statement->getResultSet());
              statement_fetch_timer.stop();
            }
            else
            {
              dbc_resultset.reset();
            }

            data_storage.reset();
          }
        }
        else
        {
          std::string statement_res_msg= (updated_rows_count < 0) ? "OK" : strfmt("%lli row(s) affected", updated_rows_count);
          set_log_message_ex(log_message_index, grt::InfoMsg, statement_res_msg, statement, statement_exec_timer.duration_formatted());
        }
      }
    }
  }
  CATCH_EXCEPTION_AND_DISPATCH(statement)

  request_refresh_of_menu_and_toolbar();
  
  return grt::StringRef("");
}


void Db_sql_editor::request_refresh_of_menu_and_toolbar()
{
  exec_sql_task->execute_in_main_thread(
    sigc::bind(sigc::mem_fun(_context_ui, &wb::WBContextUI::refresh_toolbar),
      WB_TOOLBAR_MAIN),
    false,
    false);
  exec_sql_task->execute_in_main_thread(
    sigc::mem_fun(_context_ui, &wb::WBContextUI::refresh_menus),
    false,
    false);
}


bool Db_sql_editor::is_running_query()
{
  return _is_running_query;
}


void Db_sql_editor::recordsets_are_pinned_by_default(bool value)
{
  _recordsets_are_pinned_by_default= value; 
  
  _context_ui->refresh_toolbar(WB_TOOLBAR_MAIN);
}


void Db_sql_editor::continue_on_error(bool val)
{
  if (_continue_on_error == val)
    return;

  _continue_on_error= val;

  DictRef options= DictRef::cast_from(_grtm->get_grt()->get("/wb/options/options"));
  options.set("DbSqlEditor:ContinueOnError", grt::IntegerRef((int)_continue_on_error));
  
  _context_ui->refresh_toolbar(WB_TOOLBAR_MAIN);
}


void Db_sql_editor::send_message_keep_alive()
{
  try
  {
    // simple check of dbc connection pings server
    // and resets interactive client timeout counter
    ensure_valid_aux_connection();
    ensure_valid_usr_connection();
  }
  catch (const std::exception &)
  {
  }
}


void Db_sql_editor::on_close_recordset(Recordset::Ref rs)
{
  GMutexLock recordsets_mutex(_recordsets_mutex);

  Recordsets::iterator i= std::find(_recordsets.begin(), _recordsets.end(), rs);
  if (_recordsets.end() == i)
    return;
  close_recordset_ui.emit(rs->key());
  _recordsets.erase(i);
  
  _context_ui->refresh_menus();
}


void Db_sql_editor::apply_changes_to_recordset(Recordset::Ref rs)
{
  GMutexLock usr_dbc_conn_mutex= ensure_valid_usr_connection();

  // we need transaction to enforce atomicity of change set
  // so if autocommit is currently enabled disable it temporarily
  bool auto_commit= _usr_dbc_conn->ref->getAutoCommit();
  ScopeExitCaller autocommit_mode_keeper;
  int res= -2;
  if (!auto_commit)
  {
    int res= mforms::Utilities::show_warning(
      _("Apply Changes to Recordset"),
      _("Autocommit is currently disabled. Do you want to perform a COMMIT before applying the changes?\n"
        "If you do not commit, a failure during the recordset update will result in a rollback of the active transaction, if you have one."),
      _("Commit and Apply")/*1*/,
      _("Apply")/*0*/,
      _("Cancel")/*-1*/);

    if (res == 1)
    {
      _usr_dbc_conn->ref->commit();
    }
  }
  else
  {
    autocommit_mode_keeper.slot= sigc::bind(
      sigc::mem_fun(_usr_dbc_conn->ref.get(), &sql::Connection::setAutoCommit),
      auto_commit);
    _usr_dbc_conn->ref->setAutoCommit(false);
  }

  if (res != -1) // only if not canceled
  {
    DictRef options= DictRef::cast_from(_grtm->get_grt()->get("/wb/options/options"));
    bool is_data_changes_commit_wizard_enabled= (0 != options.get_int("DbSqlEditor:IsDataChangesCommitWizardEnabled", 1));
    if (is_data_changes_commit_wizard_enabled)
    {
      /*bool are_changes_committed= */run_data_changes_commit_wizard(rs);
    }
    else
    {
      rs->apply_changes_(rs);
    }
  }
}


bool Db_sql_editor::run_data_changes_commit_wizard(Recordset::Ref rs)
{
  on_sql_script_run_error.clear();
  on_sql_script_run_progress.clear();
  on_sql_script_run_statistics.clear();

  // set underlying recordset data storage to use sql substitute (potentially modified by user)
  // instead of generating sql based on swap db contents
  Recordset_data_storage::Ref data_storage= rs->data_storage();
  Recordset_sql_storage *sql_storage= dynamic_cast<Recordset_sql_storage *>(data_storage.get());
  if (!sql_storage)
    return false;
  sql_storage->init_sql_script_substitute(rs, true);
  sql_storage->is_sql_script_substitute_enabled(true);
  const Sql_script &sql_script= sql_storage->sql_script_substitute();;
  std::string sql_script_text= Recordset_sql_storage::statements_as_sql_script(sql_script.statements);

  SqlScriptRunWizard wizard(_grtm);
  on_sql_script_run_error.connect(sigc::mem_fun(wizard.apply_page, &SqlScriptApplyPage::on_error));
  on_sql_script_run_progress.connect(sigc::mem_fun(wizard.apply_page, &SqlScriptApplyPage::on_exec_progress));
  on_sql_script_run_statistics.connect(sigc::mem_fun(wizard.apply_page, &SqlScriptApplyPage::on_exec_stat));
  wizard.values().gset("sql_script", sql_script_text);
  wizard.apply_page->apply_sql_script= sigc::bind(sigc::mem_fun(this, &Db_sql_editor::apply_data_changes_commit), rs);
  grt::DictRef dict(_grtm->get_grt());
  wizard.run_modal(dict);

  return !wizard.has_errors();
}


void Db_sql_editor::apply_data_changes_commit(std::string &sql_script_text, Recordset::Ref rs)
{
  // this lock is supposed to be acquired lower in call-stack by Db_sql_editor::apply_changes_to_recordset
  //GMutexLock usr_conn_mutex= ensure_valid_usr_connection();

  Recordset_data_storage::Ref data_storage= rs->data_storage();
  Recordset_sql_storage *sql_storage= dynamic_cast<Recordset_sql_storage *>(data_storage.get());
  if (!sql_storage)
    return;

  Sql_script sql_script= sql_storage->sql_script_substitute();
  sql_script.statements.clear();
  SqlFacade::Ref sql_splitter= SqlFacade::instance_for_rdbms(rdbms());
  sql_splitter->splitSqlScript(sql_script_text, sql_script.statements);

  sigc::connection on_sql_script_run_error_conn= sql_storage->on_sql_script_run_error.connect(on_sql_script_run_error.make_slot());
  sigc::connection on_sql_script_run_progress_conn= sql_storage->on_sql_script_run_progress.connect(on_sql_script_run_progress.make_slot());
  sigc::connection on_sql_script_run_statistics_conn= sql_storage->on_sql_script_run_statistics.connect(on_sql_script_run_statistics.make_slot());

  try
  {
    sql_storage->sql_script_substitute(sql_script);
    rs->do_apply_changes(_grtm->get_grt(), data_storage, rs);
  }
  catch (...)
  {
    on_sql_script_run_error_conn.disconnect();
    on_sql_script_run_progress_conn.disconnect();
    on_sql_script_run_statistics_conn.disconnect();
    throw;
  }
  on_sql_script_run_error_conn.disconnect();
  on_sql_script_run_progress_conn.disconnect();
  on_sql_script_run_statistics_conn.disconnect();

  _history->add_entry(sql_script.statements);
}


void Db_sql_editor::show_export_recordset(Recordset::Ref rs)
{
  RecordsetExportForm exporter(this, rs);

  exporter.run();  
}

void Db_sql_editor::new_sql_script_file()
{
  _active_sql_editor_index= add_sql_editor();
  sql_editor_new_ui.emit(_active_sql_editor_index);
}


void Db_sql_editor::open_sql_script_file(const std::string &file_path, bool run_immediately)
{
  if (file_path.empty())
    return;
  
  if (_active_sql_editor_index < 0 || _active_sql_editor_index >= _sql_editors.size() ||
      !_sql_editors[_active_sql_editor_index].editor->sql().empty())
    new_sql_script_file();

  gchar *contents= NULL;
  gsize length= 0;
  GError *error= NULL;
  
  _grtm->replace_status_text(strfmt("Loading SQL script file '%s'...", file_path.c_str()));
  
  if (!g_file_get_contents(file_path.c_str(), &contents, &length, &error))
  {
    _grtm->replace_status_text(strfmt("Error loading SQL script file '%s'.", file_path.c_str()));
    mforms::Utilities::show_error(strfmt(_("Error opening file %s"), file_path.c_str()), error->message, "", "", "");
    //!_context_ui->get_wb()->show_error(strfmt(_("Error opening file %s"), file_path.c_str()), error->message);
    g_error_free(error);
    return;
  }

  std::string utf8_contents;
  
  if (!FileCharsetDialog::ensure_filedata_utf8(contents, length,
                                               file_path,
                                               utf8_contents))
  {
    g_free(contents);
    _grtm->replace_status_text(_("Cancelled"));
    return;
  }
  g_free(contents);

  replace_string_inplace(utf8_contents, "\r\n", "\n");
  replace_string_inplace(utf8_contents, "\r", "\n");

  set_sql_editor_text(utf8_contents, run_immediately);

  _sql_editors[_active_sql_editor_index].filename = file_path;
  _sql_editors[_active_sql_editor_index].dirty = false;
  _context_ui->get_wb()->add_recent_file(file_path);

  sql_editor_caption_ui.emit(file_path);

  _grtm->replace_status_text(strfmt("Loaded SQL script file '%s'.", file_path.c_str()));
}


// file_name - filename to save to or empty if it should be asked to user
// editor_index - index of the editor which contents should be saved
// Returns true if file was saved or false if it was cancelled or failed.
bool Db_sql_editor::save_sql_script_file(const std::string &file_name, int editor_index)
{  
  std::string path = file_name;
  
  if (path.empty())
  {
    mforms::FileChooser dlg(mforms::SaveFile);
    
    dlg.set_title(_("Save SQL Script"));
    dlg.set_extensions("SQL Files (*.sql)|*.sql", "sql");
    if (!dlg.run_modal())
      return false;
    
    path = dlg.get_path();
  }
  
  if (!path.empty())
  {
    GError *error= NULL;
    
    if (!g_str_has_suffix(path.c_str(), ".sql"))
      path.append(".sql");
    
    // tell UI that editor data changed
    do_partial_ui_refresh(Db_sql_editor::RefreshEditorBackend);
    
    std::string contents= sql_editor(editor_index)->sql();
    
    _grtm->replace_status_text(base::strfmt("Saving SQL script to '%s'...", path.c_str()));
    
    if (!g_file_set_contents(path.c_str(), contents.data(), contents.length(), &error))
    {
      _grtm->replace_status_text(strfmt("Error saving SQL script to '%s'.", path.c_str()));
      
      mforms::Utilities::show_error(strfmt(_("Error writing file %s"), path.c_str()),
                                    error->message, "OK", "", "");
      g_error_free(error);
      return false;
    }
    
    _sql_editors[editor_index].filename = path;
    _sql_editors[editor_index].dirty = false;
    sql_editor_caption_ui.emit(path);
    
    _context_ui->get_wb()->add_recent_file(path);
    
    _grtm->replace_status_text(strfmt("SQL script saved to '%s'.", path.c_str()));
    return true;
  }
  return false;
}


std::string Db_sql_editor::restore_sql_from_history(int entry_index, std::list<int> &detail_indexes)
{
  return _history->restore_sql_from_history(entry_index, detail_indexes);
}


void Db_sql_editor::get_schemata(std::list<std::string> &schemata_names)
{
  try
  {
    workbench_physical_ModelRef model= workbench_physical_ModelRef::cast_from(_live_physical_overview->get_model());
    db_CatalogRef catalog= model->catalog();

    ListRef<db_Schema> schemata= catalog->schemata();
    if (0 == schemata.count())
    {
      GMutexLock aux_dbc_conn_mutex= ensure_valid_aux_connection();

      bool show_metadata_schemata;
      {
        DictRef options= DictRef::cast_from(_grtm->get_grt()->get("/wb/options/options"));
        show_metadata_schemata= (0 != options.get_int("DbSqlEditor:ShowMetadataSchemata", 0));
      }
      db_mgmt_RdbmsRef rdbms= db_mgmt_RdbmsRef::cast_from(grt::copy_object(_grtm->get_grt(), this->rdbms()));
      std::string database_package= *rdbms->databaseObjectPackage();

      std::auto_ptr<sql::ResultSet> rs(_aux_dbc_conn->ref->getMetaData()->getSchemata());
      while (rs->next())
      {
        std::string name= rs->getString(1);
        db_SchemaRef schema= find_named_object_in_list(schemata, name);
        if (!schema.is_valid())
        {
          if (show_metadata_schemata || ((name != "information_schema") && (name != "mysql")))
          {
            schema= _grtm->get_grt()->create_object<db_Schema>(database_package + ".Schema");
            schema->owner(model->catalog());
            schema->modelOnly(1);
            schema->name(name);
            schemata.insert(schema);
          }
        }
      }
    }

    for (int n= 0, count= schemata.count(); n < count; ++n)
    {
      std::string name= *schemata.get(n)->name();
      schemata_names.push_back(name);
    }
  }
  CATCH_EXCEPTION_AND_DISPATCH("Get schemata")
}


std::string Db_sql_editor::active_schema() const
{
  return (_usr_dbc_conn) ? _usr_dbc_conn->active_schema : std::string();
}


int Db_sql_editor::active_schema_index() const
{
  std::string schema_name= active_schema();
  if (schema_name.empty())
  {
    workbench_physical_ModelRef model= workbench_physical_ModelRef::cast_from(_live_physical_overview->get_model());
    db_CatalogRef catalog= model->catalog();
    if (catalog->schemata().count() > 0)
      return 0;
    else
      return -1;
  }
  int res= _schema_tree.get_index_of_schema(schema_name);
  return res;
}


void Db_sql_editor::active_schema(const std::string &value)
{
  try
  {
    if (value.empty())
      return;

    std::string schema= active_schema();
    if (value == schema)
      return;

    _aux_dbc_conn->active_schema= value;
    {
      GMutexLock aux_dbc_conn_mutex= ensure_valid_aux_connection();
      _aux_dbc_conn->ref->setSchema(value);
    }

    _usr_dbc_conn->active_schema= value;
    {
      GMutexLock usr_dbc_conn_mutex= ensure_valid_usr_connection();
      _usr_dbc_conn->ref->setSchema(value);
    }    
    
    grt_manager()->replace_status_text(strfmt("Active schema changed to %s", value.c_str()));
  }
  CATCH_EXCEPTION_AND_DISPATCH("Set active schema")
}


wb::LiveSchemaTree *Db_sql_editor::get_schema_tree()
{
  return &_schema_tree;
}


std::list<std::string> Db_sql_editor::fetch_live_schema_list()
{
  std::list<std::string> schemata;
  get_schemata(schemata);
  return schemata;
}


void Db_sql_editor::notify_of_fetched_schema_contents(
  Ref self,
  bec::NodeId node,
  db_SchemaRef schema,
  bool schemata_tree_needs_refresh,
  bool live_physical_overview_schema_node_needs_refresh,
  bool live_physical_overview_needs_refresh)
{
  if (schemata_tree_needs_refresh)
  {
    _schema_tree.refresh_node_ui.emit(node);
    do_partial_ui_refresh(RefreshSchemaTree);
  }

  if (live_physical_overview_schema_node_needs_refresh)
  {
    _live_physical_overview->refresh_schema_node(schema);
  }

  if (live_physical_overview_needs_refresh)
  {
#if defined(__APPLE__) || !defined(_WIN32)
    // all platforms should refresh on RefreshOverview or some variation of it
    // send_refresh_for_schema() is something that was meant for model overview, not live --alfredo
    do_partial_ui_refresh(RefreshOverview);
#else
    _live_physical_overview->send_refresh_for_schema(schema, false);
#endif
  }
}


bool Db_sql_editor::fetch_live_schema_contents(bec::NodeId node_id, const std::string &schema_name, const wb::LiveSchemaTree::SchemaContentArrivedSlot &arrived_slot)
{
  Ref self= _self;
  if (!self)
    return false;

  // in windows we use TreeViewAdv feature to expand nodes asynchronously
  // that is this function is already called from a separate thread
  // and it must have items loaded when it returns.
  bool sync= !_grtm->in_main_thread();

  live_schema_fetch_task->proc_cb(
    sigc::bind(
      sigc::mem_fun(this, &Db_sql_editor::do_fetch_live_schema_contents),
      self,
      node_id,
      schema_name,
      arrived_slot));

  live_schema_fetch_task->exec(sync, true);

  return true;//!
}


void Db_sql_editor::refresh_live_object_in_overview(wb::LiveSchemaTree::ObjectType type, const std::string schema_name, const std::string old_obj_name, const std::string new_obj_name)
{
  Ref self= _self;
  if (!self)
    return;

  try
  {
    //GMutexLock aux_dbc_conn_mutex= ensure_valid_aux_connection(); // this lock is acquired by caller
    GMutexLock schema_contents_mutex(_schema_contents_mutex);

    bool is_obj_created= old_obj_name.empty();
    bool is_obj_dropped= new_obj_name.empty();
    bool is_obj_renamed= !is_obj_created && !is_obj_dropped && (old_obj_name != new_obj_name);
    if (is_obj_renamed)
      is_obj_created= is_obj_dropped= true;

    if (!is_obj_created && !is_obj_dropped)
      return;

    std::string obj_type;
    switch (type)
    {
      case wb::LiveSchemaTree::Schema:
        obj_type= "schema";
        break;
      case wb::LiveSchemaTree::Table:
        obj_type= "table";
        break;
      case wb::LiveSchemaTree::View:
        obj_type= "view";
        break;
      case wb::LiveSchemaTree::Routine:
        obj_type= "routine";
        break;
    }

    if (is_obj_created)
    {
      is_obj_created= false;
      std::auto_ptr<sql::ResultSet> rs(_aux_dbc_conn->ref->getMetaData()->getSchemaObjects("", schema_name, obj_type, false, new_obj_name));
      while (rs->next())
        is_obj_created= true;
    }
    if (is_obj_dropped)
    {
      std::auto_ptr<sql::ResultSet> rs(_aux_dbc_conn->ref->getMetaData()->getSchemaObjects("", schema_name, obj_type, false, old_obj_name));
      while (rs->next())
        is_obj_dropped= false;
    }

    if (!is_obj_created && !is_obj_dropped)
      return;

    _schema_tree.update_live_object_state(type, schema_name, old_obj_name, new_obj_name);

    workbench_physical_ModelRef model= workbench_physical_ModelRef::cast_from(_live_physical_overview->get_model());
    db_CatalogRef catalog= model->catalog();
    db_SchemaRef schema= find_named_object_in_list(catalog->schemata(), schema_name);
    db_mgmt_RdbmsRef rdbms= this->rdbms();
    std::string database_package= *rdbms->databaseObjectPackage();
    switch (type)
    {
      case wb::LiveSchemaTree::Schema:
        {
          if (is_obj_created)
          {
            db_SchemaRef obj= find_named_object_in_list(catalog->schemata(), new_obj_name);
            if (!obj.is_valid())
            {
              schema= _grtm->get_grt()->create_object<db_Schema>(database_package + ".Schema");
              schema->owner(model->catalog());
              schema->modelOnly(1);
              schema->name(new_obj_name);
              catalog->schemata().insert(schema);
            }
          }
          if (is_obj_dropped)
          {
            db_SchemaRef obj= find_named_object_in_list(catalog->schemata(), old_obj_name);
            if (obj.is_valid())
              catalog->schemata().remove_value(obj);
          }
        }
        break;

      case wb::LiveSchemaTree::Table:
        {
          if (is_obj_created)
          {
            db_TableRef obj= find_named_object_in_list(schema->tables(), new_obj_name);
            if (!obj.is_valid())
            {
              db_TableRef table= _grtm->get_grt()->create_object<db_Table>(database_package + ".Table");
              table->owner(schema);
              table->name(new_obj_name);
              schema->tables().insert(table);
            }
          }
          if (is_obj_dropped)
          {
            db_TableRef obj= find_named_object_in_list(schema->tables(), old_obj_name);
            if (obj.is_valid())
              schema->tables().remove_value(obj);
          }
        }
        break;
      case wb::LiveSchemaTree::View:
        {
          if (is_obj_created)
          {
            db_ViewRef obj= find_named_object_in_list(schema->views(), new_obj_name);
            if (!obj.is_valid())
            {
              db_ViewRef view= _grtm->get_grt()->create_object<db_View>(database_package + ".View");
              view->owner(schema);
              view->name(new_obj_name);
              schema->views().insert(view);
            }
          }
          if (is_obj_dropped)
          {
            db_ViewRef obj= find_named_object_in_list(schema->views(), old_obj_name);
            if (obj.is_valid())
              schema->views().remove_value(obj);
          }
        }
        break;
      case wb::LiveSchemaTree::Routine:
        {
          if (is_obj_created)
          {
            db_RoutineRef obj= find_named_object_in_list(schema->routines(), new_obj_name);
            if (!obj.is_valid())
            {
              db_RoutineRef routine= _grtm->get_grt()->create_object<db_Routine>(database_package + ".Routine");
              routine->owner(schema);
              routine->name(new_obj_name);
              schema->routines().insert(routine);
            }
          }
          if (is_obj_dropped)
          {
            db_RoutineRef obj= find_named_object_in_list(schema->routines(), old_obj_name);
            if (obj.is_valid())
              schema->routines().remove_value(obj);
          }
        }
        break;
    }

    if (wb::LiveSchemaTree::Schema == type)
    {
      do_refresh_physical_overview(false);
    }
    else
    {
      NodeId schema_node_id= _schema_tree.get_node_for_schema(schema_name);
      live_schema_fetch_task->execute_in_main_thread(
        sigc::bind(sigc::mem_fun(this, &Db_sql_editor::notify_of_fetched_schema_contents),
          self, schema_node_id, schema, true, true, true),
        true,
        false);
    }
  }
  CATCH_EXCEPTION_AND_DISPATCH_TO_DEFAULT_LOG("Refresh live schema object")
}


grt::StringRef Db_sql_editor::do_fetch_live_schema_contents(grt::GRT *grt, Ref self, bec::NodeId node_id, std::string schema_name, wb::LiveSchemaTree::SchemaContentArrivedSlot arrived_slot)
{
  try
  {
    GMutexLock schema_contents_mutex(_schema_contents_mutex);

    workbench_physical_ModelRef model= workbench_physical_ModelRef::cast_from(_live_physical_overview->get_model());
    db_CatalogRef catalog= model->catalog();
    db_SchemaRef schema= find_named_object_in_list(catalog->schemata(), schema_name);

    if (!schema.is_valid())
      return grt::StringRef("");

    if (!schema->modelOnly() && arrived_slot.empty())
      return grt::StringRef("");

    bool live_physical_overview_schema_node_needs_refresh= false;
#ifdef _WIN32
    bool live_physical_overview_needs_refresh= schema->modelOnly() && arrived_slot.empty();
#else
    bool live_physical_overview_needs_refresh= true;
#endif

    LiveSchemaTree::SchemaNode node;
    node.fetched= true;
    node.name= schema_name;

    if (schema->modelOnly())
    {
      schema->modelOnly(0);

      db_mgmt_RdbmsRef rdbms= this->rdbms();
      std::string database_package= *rdbms->databaseObjectPackage();

      GMutexLock aux_dbc_conn_mutex= ensure_valid_aux_connection();

      {
        std::auto_ptr<sql::ResultSet> rs(_aux_dbc_conn->ref->getMetaData()->getSchemaObjects("", schema_name, "table", false));
        while (rs->next())
        {
          wb::LiveSchemaTree::ObjectNode obj_node;
          obj_node.name= rs->getString(4);
          node.tables.push_back(obj_node);

          db_TableRef table= _grtm->get_grt()->create_object<db_Table>(database_package + ".Table");
          table->owner(schema);
          table->name(obj_node.name);
          schema->tables().insert(table);
        }
      }
      
      {
        std::auto_ptr<sql::ResultSet> rs(_aux_dbc_conn->ref->getMetaData()->getSchemaObjects("", schema_name, "view", false));
        while (rs->next())
        {
          wb::LiveSchemaTree::ObjectNode obj_node;
          obj_node.name= rs->getString(4);
          node.views.push_back(obj_node);

          db_ViewRef view= _grtm->get_grt()->create_object<db_View>(database_package + ".View");
          view->owner(schema);
          view->name(obj_node.name);
          schema->views().insert(view);
        }
      }
      
      {
        std::auto_ptr<sql::ResultSet> rs(_aux_dbc_conn->ref->getMetaData()->getSchemaObjects("", schema_name, "routine", false));
        while (rs->next())
        {
          wb::LiveSchemaTree::ObjectNode obj_node;
          obj_node.name= rs->getString(4);
          obj_node.details= " "; // means no details
          node.routines.push_back(obj_node);

          db_RoutineRef routine= _grtm->get_grt()->create_object<db_Routine>(database_package + ".Routine");
          routine->owner(schema);
          routine->name(obj_node.name);
          schema->routines().insert(routine);
        }
      }

      live_physical_overview_schema_node_needs_refresh= true;
    }
    else
    {
      for (size_t n= 0, count= schema->tables().count(); n < count; ++n)
      {
        wb::LiveSchemaTree::ObjectNode obj_node;
        obj_node.name= *schema->tables().get(n)->name();
        node.tables.push_back(obj_node);
      }
      for (size_t n= 0, count= schema->views().count(); n < count; ++n)
      {
        wb::LiveSchemaTree::ObjectNode obj_node;
        obj_node.name= *schema->views().get(n)->name();
        node.views.push_back(obj_node);
      }
      for (size_t n= 0, count= schema->routines().count(); n < count; ++n)
      {
        wb::LiveSchemaTree::ObjectNode obj_node;
        obj_node.name= *schema->routines().get(n)->name();
        obj_node.details= " "; // means no details
        node.routines.push_back(obj_node);
      }
    }

    arrived_slot(node);

    live_schema_fetch_task->execute_in_main_thread(
      sigc::bind(sigc::mem_fun(this, &Db_sql_editor::notify_of_fetched_schema_contents),
        self, node_id, schema, !arrived_slot.empty(), live_physical_overview_schema_node_needs_refresh, live_physical_overview_needs_refresh),
      false,
      false);
  }
  CATCH_EXCEPTION_AND_DISPATCH_TO_DEFAULT_LOG("Get schema contents")

  return grt::StringRef("");
}


bool Db_sql_editor::fetch_object_details(wb::LiveSchemaTree::ObjectType obj_type, const std::string &schema_name, const std::string &obj_name, const wb::LiveSchemaTree::ObjectDetailsArrivedSlot &arrived_slot)
{
  if ((obj_type == wb::LiveSchemaTree::Table) || (obj_type == wb::LiveSchemaTree::View))
  {
    LiveSchemaTree::SchemaNode schema_node;
    schema_node.fetched= true;
    schema_node.name= schema_name;

    LiveSchemaTree::ObjectNode obj_node;
    obj_node.fetched= true;
    obj_node.name= obj_name;

    std::string col_brief_list;
    std::string col_detail_list;
    try
    {
      GMutexLock aux_dbc_conn_mutex= ensure_valid_aux_connection();
      std::auto_ptr<sql::ResultSet> rs(_aux_dbc_conn->ref->getMetaData()->getColumns("", schema_name, obj_name, "%"));
      while (rs->next())
      {
        std::string column_name= rs->getString(4);
        obj_node.column_names.push_back(column_name);
        col_brief_list+= (col_brief_list.empty() ? "" : ", ") + column_name;
        col_detail_list+= strfmt("%-12s", column_name.c_str()) + " " + rs->getString(6) + "\n";
      }
    }
    catch (const sql::SQLException&)
    {
    }

    obj_node.details= strfmt("Table %s\n", obj_name.c_str());
    size_t header_length= obj_node.details.size()-1;
    obj_node.details+= std::string(header_length, '=') + "\n" +
      col_brief_list + "\n" +
      std::string(header_length, '-').append("\n") +
      col_detail_list;

    arrived_slot(schema_node, obj_node, obj_type);

    return true;
  }
  return false;
}


void Db_sql_editor::schema_object_activated(const std::string &action, wb::LiveSchemaTree::ObjectType type, const std::string &schema, const std::string &name)
{
  std::string action_modifier; // action can contain prefix denoting action modifier
  std::string real_action= action; // if action modifier is present real_action will store part after w/o modifier prefix
  const char *action_modifier_clipboard= "clipboard_"; // denotes copy of the action result to clipboard
  const char *action_modifier_editor= "editor_"; // denotes insertion of the action result into editor at cursor position
  {
    const char* known_modifiers[]= { action_modifier_clipboard, action_modifier_editor };
    for (size_t n= 0, size= sizeof(known_modifiers)/sizeof(known_modifiers[0]); n < size; ++n)
    {
      if (0 == action.find(known_modifiers[n]))
      {
        action_modifier= known_modifiers[n];
        real_action= action.substr(action_modifier.size());
      }
    }
  }

  if (real_action == "object_name_short" || real_action == "object_name_long")
  {
    std::string text;
    if (type == wb::LiveSchemaTree::Schema)
      text= schema;
    else if (real_action == "object_name_short")
      text= name;
    else if (real_action == "object_name_long")
      text= strfmt("`%s`.`%s`", schema.c_str(), name.c_str());
    if (action_modifier == action_modifier_clipboard)
      mforms::Utilities::set_clipboard_text(text);
    else if (action_modifier == action_modifier_editor)
      sql_editor()->insert_text(text);
    return;
  }
  else if (real_action == "table_create_statement")
  {
    std::string text= get_object_ddl_script(type, schema, name);
    if (action_modifier == action_modifier_clipboard)
      mforms::Utilities::set_clipboard_text(text);
    else if (action_modifier == action_modifier_editor)
      sql_editor()->insert_text(text);
  }
  else if (real_action == "table_delete_statement")
  {
    std::string text= strfmt("DELETE FROM `%s`.`%s`\nWHERE <where_condition>;\n", schema.c_str(), name.c_str());
    if (action_modifier == action_modifier_clipboard)
      mforms::Utilities::set_clipboard_text(text);
    else if (action_modifier == action_modifier_editor)
      sql_editor()->insert_text(text);
    return;
  }
  else if (real_action == "table_select_statement" || real_action == "table_insert_statement" ||
    real_action == "table_update_statement" || real_action == "table_column_names")
  {
    typedef std::list<std::pair<std::string, std::string> > ColumnDefinitions;
    ColumnDefinitions column_defs;
    try
    {
      GMutexLock aux_dbc_conn_mutex= ensure_valid_aux_connection();
      std::auto_ptr<sql::ResultSet> rs(_aux_dbc_conn->ref->getMetaData()->getColumns("", schema, name, "%"));
      while (rs->next())
        column_defs.push_back(std::make_pair(rs->getString(4), rs->getString(6)));
    }
    catch (const sql::SQLException&)
    {
    }

    std::string text;

    if (real_action == "table_select_statement")
    {
      std::string columns;
      BOOST_FOREACH (const ColumnDefinitions::value_type &col_def, column_defs)
        columns+= (columns.empty() ? "" : ",\n") + strfmt("`%s`.`%s`", name.c_str(), col_def.first.c_str());
      text= strfmt("SELECT\n%s\nFROM `%s`.`%s`;\n", columns.c_str(), schema.c_str(), name.c_str());
    }

    if (real_action == "table_insert_statement")
    {
      std::string columns;
      std::string values;
      BOOST_FOREACH (const ColumnDefinitions::value_type &col_def, column_defs)
      {
        columns+= (columns.empty() ? "" : ",\n") + strfmt("`%s`", col_def.first.c_str());
        values+= (values.empty() ? "" : ",\n") + strfmt("{%s: %s}", col_def.first.c_str(), col_def.second.c_str());
      }
      text= strfmt("INSERT INTO `%s`.`%s`\n(%s)\nVALUES\n(\n%s\n);\n", schema.c_str(), name.c_str(), columns.c_str(), values.c_str());
    }

    if (real_action == "table_update_statement")
    {
      std::string columns;
      BOOST_FOREACH (const ColumnDefinitions::value_type &col_def, column_defs)
        columns+= (columns.empty() ? "" : ",\n") + strfmt("`%s` = {%s: %s}", col_def.first.c_str(), col_def.first.c_str(), col_def.second.c_str());
      text= strfmt("UPDATE `%s`.`%s`\nSET\n%s\nWHERE <where_condition>;\n", schema.c_str(), name.c_str(), columns.c_str());
    }

    if (real_action == "table_column_names")
    {
      BOOST_FOREACH (const ColumnDefinitions::value_type &col_def, column_defs)
        text+= (text.empty() ? "" : ", ") + strfmt("`%s`.`%s`", name.c_str(), col_def.first.c_str());
    }

    if (action_modifier == action_modifier_clipboard)
      mforms::Utilities::set_clipboard_text(text);
    else if (action_modifier == action_modifier_editor)
      sql_editor()->insert_text(text);

    return;
  }
  
  {
    std::string sql;
    switch (type)
    {
    case wb::LiveSchemaTree::Table:
      if (real_action == "activate" || real_action == "edit_data")
        sql= strfmt("EDIT `%s`.`%s`;", schema.c_str(), name.c_str());
      else if (real_action == "select_data")
        sql= strfmt("SELECT * FROM `%s`.`%s`;", schema.c_str(), name.c_str());
      break;
    case wb::LiveSchemaTree::View:
      if (real_action == "activate" || real_action == "select_data")
        sql= strfmt("SELECT * FROM `%s`.`%s`;", schema.c_str(), name.c_str());
      break;
    default:
      break;
    }
    if (!sql.empty())
      set_sql_editor_text(sql, true);
  }
}


void Db_sql_editor::do_alter_live_object(wb::LiveSchemaTree::ObjectType type, const std::string &schema_name, const std::string &obj_name)
{
  try
  {
    db_mgmt_RdbmsRef rdbms= this->rdbms();
    std::string database_package= *rdbms->databaseObjectPackage();

    db_CatalogRef client_state_catalog= _grtm->get_grt()->create_object<db_Catalog>(database_package + ".Catalog");
    client_state_catalog->name("default");
    client_state_catalog->oldName("default");
    client_state_catalog->version(rdbms->version());
    grt::replace_contents(client_state_catalog->simpleDatatypes(), rdbms->simpleDatatypes());
    //apply_user_datatypes(client_state_catalog, rdbms);

    db_SchemaRef schema;
    if (wb::LiveSchemaTree::Schema != type)
    {
      schema= _grtm->get_grt()->create_object<db_Schema>(database_package + ".Schema");
      schema->owner(client_state_catalog);
      schema->name(schema_name);
      schema->oldName(schema_name);
      client_state_catalog->schemata().insert(schema);
      client_state_catalog->defaultSchema(schema);
    }

    bool is_object_new= obj_name.empty();

    std::string ddl_script;
    if (!is_object_new)
    {
      // parse selected object DDL into auxiliary catalog
      ddl_script= get_object_ddl_script(type, schema_name, obj_name);
      if (ddl_script.empty())
        return;
      SqlFacade::Ref sql_facade= SqlFacade::instance_for_rdbms(rdbms);
      Sql_parser::Ref sql_parser= sql_facade->sqlParser();
      sql_parser->messages_enabled(false);
      grt::DictRef options(_grtm->get_grt());
      sql_parser->parse_sql_script(client_state_catalog, ddl_script, options);
    }

    db_CatalogRef server_state_catalog(db_CatalogRef::cast_from(grt::copy_object(_grtm->get_grt(), client_state_catalog)));

    db_DatabaseObjectRef db_object;
    switch (type)
    {
      case wb::LiveSchemaTree::Schema:
        db_object= is_object_new ?
          create_new_schema(client_state_catalog) :
          find_named_object_in_list(client_state_catalog->schemata(), obj_name);
        break;
      case wb::LiveSchemaTree::Table:
        db_object= is_object_new ?
          create_new_table(schema) :
          find_named_object_in_list(schema->tables(), obj_name);
        break;
      case wb::LiveSchemaTree::View:
        db_object= is_object_new ?
          create_new_view(schema) :
          find_named_object_in_list(schema->views(), obj_name);
        break;
      case wb::LiveSchemaTree::Routine:
        db_object= is_object_new ?
          create_new_routine(schema) :
          find_named_object_in_list(schema->routines(), obj_name);
        break;
    }
    if (!db_object.is_valid())
      return;

    db_object->customData().set("liveRdbms", rdbms);
    db_object->customData().set("clientStateCatalog", client_state_catalog);
    db_object->customData().set("serverStateCatalog", server_state_catalog);
    db_object->customData().set("originalObjectDDL", grt::StringRef(ddl_script));
    {
      std::ostringstream oss;
      oss << this;
      db_object->customData().set("contextDbSqlEditor", grt::StringRef(oss.str()));
    }

    _context_ui->get_wb()->open_object_editor(db_object, bec::StandaloneWindowFlag);
  }
  catch (const std::exception & e)
  {
    mforms::Utilities::show_error(strfmt(_("Failed to create/alter `%s`.`%s`"), schema_name.c_str(), obj_name.c_str()), e.what(), "", "", "");
    //!_grtm->show_error(strfmt(_("Failed to alter `%s`.`%s`"), schema_name.c_str(), obj_name.c_str()), e.what());
  }
}


db_SchemaRef Db_sql_editor::create_new_schema(db_CatalogRef owner)
{
  db_SchemaRef object= _grtm->get_grt()->create_object<db_Schema>(owner->schemata()->content_type_spec().object_class);
  object->owner(owner);
  object->name("new_schema");
  owner->schemata().insert(object);
  owner->defaultSchema(object);
  return object;
}


db_TableRef Db_sql_editor::create_new_table(db_SchemaRef owner)
{
  db_TableRef object= _grtm->get_grt()->create_object<db_Table>(owner->tables()->content_type_spec().object_class);
  object->owner(owner);
  object->name("new_table");
  owner->tables().insert(object);
  return object;
}


db_ViewRef Db_sql_editor::create_new_view(db_SchemaRef owner)
{
  db_ViewRef object= _grtm->get_grt()->create_object<db_View>(owner->views()->content_type_spec().object_class);
  object->owner(owner);
  object->name("new_view");
  owner->views().insert(object);
  return object;
}


db_RoutineRef Db_sql_editor::create_new_routine(db_SchemaRef owner)
{
  db_RoutineRef object= _grtm->get_grt()->create_object<db_Routine>(owner->routines()->content_type_spec().object_class);
  object->owner(owner);
  object->name("new_routine");
  owner->routines().insert(object);
  return object;
}


void Db_sql_editor::do_create_live_object(wb::LiveSchemaTree::ObjectType type, const std::string &schema_name, const std::string &obj_name)
{
  do_alter_live_object(type, schema_name, obj_name);
}


void Db_sql_editor::do_drop_live_object(wb::LiveSchemaTree::ObjectType type, const std::string &schema_name, const std::string &obj_name)
{
  std::string full_obj_identifier;
  std::string obj_type;

  try
  {
    GMutexLock aux_dbc_conn_mutex= ensure_valid_aux_connection();

    std::string sql;
    std::string title;
    switch (type)
    {
      case wb::LiveSchemaTree::Schema:
        full_obj_identifier= strfmt("`%s`", obj_name.c_str());
        obj_type= "schema";
        title = "DROP DATABASE";
        break;
      case wb::LiveSchemaTree::Table:
        full_obj_identifier= strfmt("`%s`.`%s`", schema_name.c_str(), obj_name.c_str());
        obj_type= "table";
        title = "DROP TABLE";
        break;
      case wb::LiveSchemaTree::View:
        full_obj_identifier= strfmt("`%s`.`%s`", schema_name.c_str(), obj_name.c_str());
        obj_type= "view";
        title = "DROP VIEW";
        break;
      case wb::LiveSchemaTree::Routine:
        {
          int routine_type= 0;
          title = "DROP ROUTINE";
          {
            std::string query_sql= strfmt("select routine_type from information_schema.routines where routine_schema='%s' and routine_name='%s'", schema_name.c_str(), obj_name.c_str());
            std::auto_ptr<sql::Statement> stmt(_aux_dbc_conn->ref->createStatement());
            boost::shared_ptr<sql::ResultSet> rs(stmt->executeQuery(query_sql));
            if (rs->next())
              routine_type= (rs->getString(1) == "PROCEDURE") ? 1 : -1;
          }
          if (routine_type)
          {
            full_obj_identifier= strfmt("`%s`.`%s`", schema_name.c_str(), obj_name.c_str());
            obj_type= (routine_type == 1) ? "procedure" : "function";
            title = (routine_type == 1) ? "DROP PROCEDURE" : "DROP FUNCTION";
          }
        }
        break;
    }

    if (!obj_type.empty())
    {
      std::string server_name = strfmt("%s:%i",
                                       _connection->parameterValues().get_string("hostName").c_str(),
                                       (int) _connection->parameterValues().get_int("port"));
      std::string message;
      
      if (wb::LiveSchemaTree::Schema == type)
        message = strfmt(_("Do you want to DROP DATABASE %s from DB server %s?\nAll its tables, views, routines and stored data will be permanently deleted."), 
                         full_obj_identifier.c_str(), server_name.c_str());
      else if (wb::LiveSchemaTree::Table == type)
        message = strfmt(_("Do you want to DROP TABLE %s from DB server %s?\nALL its data will be permanently deleted."), 
                         full_obj_identifier.c_str(), server_name.c_str());
      else
        message = strfmt(_("Do you want to DROP %s %s from DB server %s?"), 
                         obj_type.c_str(), full_obj_identifier.c_str(), server_name.c_str());

      if (mforms::Utilities::show_warning(title, message, 
                                          strfmt(_("DROP %s"), full_obj_identifier.c_str()), 
                                          _("Cancel"),
                                          "") != mforms::ResultCancel)
      {
        sql= strfmt("drop %s %s", obj_type.c_str(), full_obj_identifier.c_str());
        std::auto_ptr<sql::Statement> stmt(_aux_dbc_conn->ref->createStatement());
        stmt->execute(sql);
      }

      refresh_live_object_in_overview(type, schema_name, obj_name, "");
    }
  }
  catch (const std::exception &e)
  {
    mforms::Utilities::show_error(strfmt(_("Failed to drop %s %s"), obj_type.c_str(), full_obj_identifier.c_str()), e.what(), "", "", "");
    //!_grtm->show_error(e.what(), "");
  }
}


std::string Db_sql_editor::get_object_ddl_script(wb::LiveSchemaTree::ObjectType type, const std::string &schema_name, const std::string &obj_name)
{
  //! to make this function dbms-agnostic cppconnector requires extension to fetch DDL for single objects

  std::string ddl_script;
  std::string query_sql;
  size_t ddl_column= 2;
  std::string delimiter= ";";
  std::string non_std_sql_delimiter;
  {
    SqlFacade::Ref sql_facade= SqlFacade::instance_for_rdbms(rdbms());
    Sql_specifics::Ref sql_specifics= sql_facade->sqlSpecifics();
    non_std_sql_delimiter= sql_specifics->non_std_sql_delimiter();
  }

  GMutexLock aux_dbc_conn_mutex= ensure_valid_aux_connection();

  try
  {
    switch (type)
    {
      case wb::LiveSchemaTree::Schema:
        query_sql= strfmt("show create schema `%s`", obj_name.c_str());
        break;

      case wb::LiveSchemaTree::Table:
        {
          std::string query_sql= strfmt("select trigger_schema, trigger_name from information_schema.triggers where event_object_schema='%s' and event_object_table='%s'", schema_name.c_str(), obj_name.c_str());
          std::auto_ptr<sql::Statement> stmt(_aux_dbc_conn->ref->createStatement());
          boost::shared_ptr<sql::ResultSet> rs(stmt->executeQuery(query_sql));
          while (rs->next())
          {
            std::string trigger_schema_name= rs->getString(1);
            std::string trigger_name= rs->getString(2);
            try
            {
              query_sql= strfmt("show create trigger `%s`.`%s`", trigger_schema_name.c_str(), trigger_name.c_str());
              std::auto_ptr<sql::Statement> stmt(_aux_dbc_conn->ref->createStatement());
              boost::shared_ptr<sql::ResultSet> rs(stmt->executeQuery(query_sql));
              ddl_script+= strfmt("delimiter %s\n", non_std_sql_delimiter.c_str());
              if (rs->next())
              {
                ddl_script+= rs->getString(3);
                ddl_script+= strfmt("%s\n\n", non_std_sql_delimiter.c_str());
              }
            }
            catch (const sql::SQLException &)
            {
              // 'show create trigger' is not supported by server versions below 5.1.21
            }
          }
        }
        query_sql= strfmt("show create table `%s`.`%s`", schema_name.c_str(), obj_name.c_str());
        break;

      case wb::LiveSchemaTree::View:
        query_sql= strfmt("show create view `%s`.`%s`", schema_name.c_str(), obj_name.c_str());
        break;

      case wb::LiveSchemaTree::Routine:
        // first need to determine whether routine is procedure or function
        int routine_type= 0;
        {
          std::string query_sql= strfmt("select routine_type from information_schema.routines where routine_schema='%s' and routine_name='%s'", schema_name.c_str(), obj_name.c_str());
          std::auto_ptr<sql::Statement> stmt(_aux_dbc_conn->ref->createStatement());
          boost::shared_ptr<sql::ResultSet> rs(stmt->executeQuery(query_sql));
          if (rs->next())
            routine_type= (rs->getString(1) == "PROCEDURE") ? 1 : -1;
        }
        if (routine_type)
          query_sql= strfmt("show create %s `%s`.`%s`", (routine_type == 1) ? "procedure" : "function", schema_name.c_str(), obj_name.c_str());
        ddl_column= 3;
        delimiter= non_std_sql_delimiter;
        break;
    }

    if (!query_sql.empty())
    {
      std::auto_ptr<sql::Statement> stmt(_aux_dbc_conn->ref->createStatement());
      boost::shared_ptr<sql::ResultSet> rs(stmt->executeQuery(query_sql));
      if (rs->next())
      {
        ddl_script+= strfmt("delimiter %s\n", delimiter.c_str());
        ddl_script+= rs->getString(ddl_column);
        ddl_script+= delimiter + "\n\n";
      }
    }
  }
  catch (const sql::SQLException &)
  {
    ddl_script.clear();
  }
  return ddl_script;
}


void Db_sql_editor::refresh_live_object_in_editor(bec::DBObjectEditorBE* obj_editor, bool using_old_name)
{
  db_DatabaseObjectRef db_object= obj_editor->get_dbobject();

  db_CatalogRef client_state_catalog= db_CatalogRef::cast_from(db_object->customData().get("clientStateCatalog"));

  std::string obj_name= using_old_name ? db_object->oldName() : db_object->name();
  // don't refresh new objects that where not applied yet
  if (obj_name.empty())
    return;
  db_object->name(obj_name);
  db_object->oldName("");

  std::string schema_name= db_SchemaRef::can_wrap(db_object) ? std::string() : *db_object->owner()->name();
  db_SchemaRef schema;
  if (!schema_name.empty())
    schema= db_SchemaRef::cast_from(db_object->owner());

  wb::LiveSchemaTree::ObjectType db_object_type;

  if (db_SchemaRef::can_wrap(db_object))
  {
    db_object_type= wb::LiveSchemaTree::Schema;
  }
  else
  {
    if (db_TableRef::can_wrap(db_object))
    {
      db_object_type= wb::LiveSchemaTree::Table;

      // reset selection of fkeys/indices to avoid warnings
      bec::TableEditorBE *table_editor= dynamic_cast <bec::TableEditorBE *> (obj_editor);
      table_editor->get_fks()->select_fk(NodeId());
      table_editor->get_indexes()->select_index(NodeId());

      db_TableRef table= db_TableRef::cast_from(db_object);
      table->isStub(1);
      table->triggers().remove_all();
      table->foreignKeys().remove_all();
      table->indices().remove_all();
      table->columns().remove_all();
    }
    else if (db_ViewRef::can_wrap(db_object))
    {
      db_object_type= wb::LiveSchemaTree::View;
    }
    else if (db_RoutineRef::can_wrap(db_object))
    {
      db_object_type= wb::LiveSchemaTree::Routine;
    }
  }

  // reparse object's DDL
  std::string ddl_script;
  {
    ddl_script= get_object_ddl_script(db_object_type, schema_name, obj_name);
    if (!ddl_script.empty())
    {
      db_object->oldName(obj_name);

      SqlFacade::Ref sql_facade= SqlFacade::instance_for_rdbms(obj_editor->get_rdbms());
      Sql_parser::Ref sql_parser= sql_facade->sqlParser();
      sql_parser->messages_enabled(false);
      grt::DictRef options(_grtm->get_grt());
      options.set("reuse_existing_objects", grt::IntegerRef(1));
      sql_parser->parse_sql_script(client_state_catalog, ddl_script, options);
    }
  }

  {
    db_CatalogRef server_state_catalog(db_CatalogRef::cast_from(grt::copy_object(_grtm->get_grt(), client_state_catalog)));
    db_object->customData().set("serverStateCatalog", server_state_catalog);
  }
  db_object->customData().set("originalObjectDDL", grt::StringRef(ddl_script));

  // enable refresh of sql editor contents
  Sql_editor::Ref sql_editor= obj_editor->get_sql_editor();
  if (sql_editor)
  {
    sql_editor->is_refresh_enabled(true);
    sql_editor->is_sql_check_enabled(true);
    // provoke database object to refresh FE control contents
    db_object->signal_changed().emit("", grt::ValueRef());
  }
}


void Db_sql_editor::notify_of_ui_form_creation(bec::UIForm *form)
{
  DBObjectEditorBE *editor= dynamic_cast <DBObjectEditorBE *> (form);
  if (editor)
  {
    editor->on_apply_changes_to_live_object= sigc::mem_fun(this, &Db_sql_editor::apply_changes_to_object);
    editor->on_refresh_live_object= sigc::bind(sigc::mem_fun(this, &Db_sql_editor::refresh_live_object_in_editor), true);
    editor->on_create_live_table_stubs= sigc::mem_fun(this, &Db_sql_editor::create_live_table_stubs);
    editor->on_expand_live_table_stub= sigc::mem_fun(this, &Db_sql_editor::expand_live_table_stub);
  }
}


template <typename T>
static bool remove_model_only_objects(grt::ListRef<T> obj_list)
{
  for (int n= obj_list.count()-1;  n >= 0; --n)
    if (obj_list.get(n)->modelOnly())
      obj_list.remove(n);
  return true;
}

static bool remove_model_only_objects(db_TableRef table)
{
  //remove_model_only_objects(table->columns()); // modelOnly is not a member of db_Column
  remove_model_only_objects(table->triggers());
  //remove_model_only_objects(table->indices()); // modelOnly is not a member of db_Index
  remove_model_only_objects(table->foreignKeys());
  return true;
}

static bool remove_model_only_objects(db_SchemaRef schema)
{
  remove_model_only_objects(schema->tables());
  schema->tables().foreach(sigc::ptr_fun((bool(*)(db_TableRef))&remove_model_only_objects));
  remove_model_only_objects(schema->views());
  remove_model_only_objects(schema->routines());
  return true;
}

static bool remove_model_only_objects(db_CatalogRef catalog)
{
  remove_model_only_objects(catalog->schemata());
  catalog->schemata().foreach(sigc::ptr_fun((bool(*)(db_SchemaRef))&remove_model_only_objects));
  return true;
}

bool Db_sql_editor::apply_changes_to_object(bec::DBObjectEditorBE* obj_editor, bool dry_run)
{
  try
  {
    if (!dry_run && obj_editor->get_sql_editor()->has_sql_errors())
    {
      int res= mforms::Utilities::show_warning(
        _("Apply Changes to Object"),
        _("The object's DDL statement contains syntax errors.\n"
          "Are you sure you want to apply the DDL statement unchanged?"),
        _("Yes")/*1*/,
        _("No")/*0*/,
        _("")/*-1*/);

      if (res != 1)
        return false;
    }

    db_DatabaseObjectRef db_object= obj_editor->get_dbobject();

    DbMySQLImpl *diffsql_module= _grtm->get_grt()->find_native_module<DbMySQLImpl>("DbMySQL"); //! should be db-agnostic

    db_CatalogRef server_cat= db_CatalogRef::cast_from(db_object->customData().get("serverStateCatalog"));
    db_CatalogRef client_cat= db_CatalogRef::cast_from(db_object->customData().get("clientStateCatalog"));
    db_mgmt_RdbmsRef rdbms= db_mgmt_RdbmsRef::cast_from(db_object->customData().get("liveRdbms"));

    db_CatalogRef client_cat_copy= db_CatalogRef::cast_from(grt::copy_object(_grtm->get_grt(), client_cat));
    db_CatalogRef server_cat_copy= db_CatalogRef::cast_from(grt::copy_object(_grtm->get_grt(), server_cat));

    remove_model_only_objects(client_cat_copy);
    remove_model_only_objects(server_cat_copy);

    std::string alter_script= diffsql_module->makeAlterScriptForObject(server_cat_copy, client_cat_copy, db_object);

    if (alter_script.empty())
    {
      if (!dry_run)
        obj_editor->on_live_object_change_error(-1, _("No changes to object were detected."), "");
      return false; // no changes detected
    }

    if (dry_run)
      return true; // some changes were detected

    //DictRef options= DictRef::cast_from(_grtm->get_grt()->get("/wb/options/options"));
    //bool is_live_object_alteration_wizard_enabled= (0 != options.get_int("DbSqlEditor:IsLiveObjectAlterationWizardEnabled", 1));
    if (true/*is_live_object_alteration_wizard_enabled*/)
    {
      run_live_object_alteration_wizard(alter_script, obj_editor);
    }
    else
    {
      apply_object_alter_script(alter_script, obj_editor);
    }
  }
  catch (const std::exception &e)
  {
    obj_editor->on_live_object_change_error(-1, e.what(), "");
  }
  return true; // some changes were detected and applied
}


bool Db_sql_editor::run_live_object_alteration_wizard(std::string &alter_script, bec::DBObjectEditorBE* obj_editor)
{
  on_sql_script_run_error.clear();
  on_sql_script_run_progress.clear();
  on_sql_script_run_statistics.clear();

  SqlScriptRunWizard wizard(_grtm);
  on_sql_script_run_error.connect(sigc::mem_fun(wizard.apply_page, &SqlScriptApplyPage::on_error));
  on_sql_script_run_progress.connect(sigc::mem_fun(wizard.apply_page, &SqlScriptApplyPage::on_exec_progress));
  on_sql_script_run_statistics.connect(sigc::mem_fun(wizard.apply_page, &SqlScriptApplyPage::on_exec_stat));
  wizard.values().gset("sql_script", alter_script);
  wizard.apply_page->apply_sql_script= sigc::bind(sigc::mem_fun(this, &Db_sql_editor::apply_object_alter_script), obj_editor);
  grt::DictRef dict(_grtm->get_grt());
  wizard.run_modal(dict);

  return wizard.applied() && !wizard.has_errors();
}


void Db_sql_editor::apply_object_alter_script(std::string &alter_script, bec::DBObjectEditorBE* obj_editor)
{
  db_DatabaseObjectRef db_object= obj_editor->get_dbobject();

  SqlFacade::Ref sql_splitter= SqlFacade::instance_for_rdbms(rdbms());
  std::list<std::string> statements;
  sql_splitter->splitSqlScript(alter_script, statements);

  // in case of alter script failure:
  // try to restore object since it could had been successfully dropped before the alter script failed
  std::list<std::string> failback_statements;
  {
    std::string original_object_ddl_script= grt::StringRef::cast_from(db_object->customData().get("originalObjectDDL"));
    if (!original_object_ddl_script.empty())
    {
      // reuse the setting schema statement which is the first statement of the alter script
      std::string sql= *statements.begin();
      if ((0 == sql.find("use")) || (0 == sql.find("USE")))
        failback_statements.push_back(sql);
      sql_splitter->splitSqlScript(original_object_ddl_script, failback_statements);
    }
  }

  sql::SqlBatchExec sql_batch_exec;
  sql_batch_exec.stop_on_error(true);
  sql_batch_exec.failback_statements(failback_statements);

  sql_batch_exec.error_cb(on_sql_script_run_error.make_slot());
  sql_batch_exec.batch_exec_progress_cb(on_sql_script_run_progress.make_slot());
  sql_batch_exec.batch_exec_stat_cb(on_sql_script_run_statistics.make_slot());

  on_sql_script_run_error.connect(obj_editor->on_live_object_change_error);
  on_sql_script_run_progress.connect(obj_editor->on_live_object_change_progress);
  on_sql_script_run_statistics.connect(obj_editor->on_live_object_change_statistics);

  {
    GMutexLock aux_dbc_conn_mutex= ensure_valid_aux_connection();
    std::auto_ptr<sql::Statement> stmt(_aux_dbc_conn->ref->createStatement());
    /*long sql_batch_exec_err_count=*/ sql_batch_exec(stmt.get(), statements);
  }

  _history->add_entry(sql_batch_exec.sql_log());

  // refresh state of created/altered object in physical overview
  {
    std::string schema_name= db_SchemaRef::can_wrap(db_object) ? std::string() : *db_object->owner()->name();
    db_SchemaRef schema;
    if (!schema_name.empty())
      schema= db_SchemaRef::cast_from(db_object->owner());

    wb::LiveSchemaTree::ObjectType db_object_type;
    if (db_SchemaRef::can_wrap(db_object))
      db_object_type= wb::LiveSchemaTree::Schema;
    else if (db_TableRef::can_wrap(db_object))
      db_object_type= wb::LiveSchemaTree::Table;
    else if (db_ViewRef::can_wrap(db_object))
      db_object_type= wb::LiveSchemaTree::View;
    else if (db_RoutineRef::can_wrap(db_object))
      db_object_type= wb::LiveSchemaTree::Routine;

    refresh_live_object_in_overview(db_object_type, schema_name, db_object->oldName(), db_object->name());
  }

  refresh_live_object_in_editor(obj_editor, false);
}


void Db_sql_editor::create_live_table_stubs(bec::DBObjectEditorBE *table_editor)
{
  db_DatabaseObjectRef db_object= table_editor->get_dbobject();
  if (db_object->customData().has_key("isLiveTableListLoaded"))
    return;

  GMutexLock aux_dbc_conn_mutex= ensure_valid_aux_connection();

  db_CatalogRef catalog= table_editor->get_catalog();
  grt::ListRef<db_Schema> schemata= catalog->schemata();
  db_SchemaRef schema;
  grt::ListRef<db_Table> tables;
  db_TableRef table;

  std::string database_package= *table_editor->get_rdbms()->databaseObjectPackage();
  std::string schema_typename= database_package + ".Schema";
  std::string table_typename= database_package + ".Table";
  grt::GRT *grt= _grtm->get_grt();

  std::string prev_schema_name;

  std::string query_sql= "select table_schema, table_name from information_schema.tables where table_type='BASE TABLE' order by 1, 2";
  std::auto_ptr<sql::Statement> stmt(_aux_dbc_conn->ref->createStatement());
  boost::shared_ptr<sql::ResultSet> rs(stmt->executeQuery(query_sql));
  while (rs->next())
  {
    std::string schema_name= rs->getString(1);
    std::string table_name= rs->getString(2);
    if (prev_schema_name != schema_name)
    {
      schema= find_named_object_in_list(schemata, schema_name);
      if (!schema.is_valid())
      {
        schema= grt->create_object<db_Schema>(schema_typename);
        schema->owner(catalog);
        schema->name(schema_name);
        schema->oldName(schema_name);
        schema->modelOnly(1);
        schemata.insert(schema);
      }
      tables= schema->tables();
      prev_schema_name= schema_name;
    }
    table= find_named_object_in_list(tables, table_name);
    if (!table.is_valid())
    {
      table= grt->create_object<db_Table>(table_typename);
      table->owner(schema);
      table->name(table_name);
      table->oldName(table_name);
      table->modelOnly(1);
      table->isStub(1);
      tables.insert(table);
    }
  }

  db_object->customData().set("isLiveTableListLoaded", IntegerRef(1));
}


void Db_sql_editor::expand_live_table_stub(bec::DBObjectEditorBE *table_editor, const std::string &schema_name, const std::string &obj_name)
{
  db_CatalogRef catalog= table_editor->get_catalog();
  db_TableRef table;
  db_SchemaRef schema= find_named_object_in_list(catalog->schemata(), schema_name);
  if (schema.is_valid())
  {
    table= find_named_object_in_list(schema->tables(), obj_name);
    if (table.is_valid() && table->customData().has_key("isStubExpanded"))
      return; // stub table has already been expanded
  }

  GMutexLock aux_dbc_conn_mutex= ensure_valid_aux_connection();

  std::string query_sql= strfmt("show create table `%s`.`%s`", schema_name.c_str(), obj_name.c_str());
  std::auto_ptr<sql::Statement> stmt(_aux_dbc_conn->ref->createStatement());
  boost::shared_ptr<sql::ResultSet> rs(stmt->executeQuery(query_sql));
  if (rs->next())
  {
    std::string ddl_script= strfmt("use `%s`;\n", schema_name.c_str());
    ddl_script+= rs->getString(2);
    SqlFacade::Ref sql_facade= SqlFacade::instance_for_rdbms(table_editor->get_rdbms());
    Sql_parser::Ref sql_parser= sql_facade->sqlParser();
    sql_parser->messages_enabled(false);
    grt::DictRef options(_grtm->get_grt());
    sql_parser->parse_sql_script(catalog, ddl_script, options);
  }

  if (table.is_valid())
  {
    table->modelOnly(0);
    table->isStub(1);
    table->customData().set("isStubExpanded", IntegerRef(1));
  }
}


db_mgmt_RdbmsRef Db_sql_editor::rdbms()
{
  if (_connection.is_valid())
    return db_mgmt_RdbmsRef::cast_from(_connection->driver()->owner());
  else
    return db_mgmt_RdbmsRef::cast_from(_grtm->get_grt()->get("/wb/doc/physicalModels/0/rdbms"));
}


std::string Db_sql_editor::caption() const
{
  std::string caption= "SQL Editor";
  if (*_connection->name().c_str())
    caption+= strfmt(" (%s)", _connection->name().c_str());
  return caption;
}


std::string Db_sql_editor::connection_info()
{
  return _connection_info;
}

bool Db_sql_editor::save_snippet(const std::string &text)
{
  if (text.empty()) return false;
  
  std::string name;
  if (!wbsql()->get_wbui()->get_wb()->request_input("Enter a Name for the Snippet (optional):", 0, name))
    return false;

  DbSqlEditorSnippets::get_instance()->add_snippet(name, text);
  do_partial_ui_refresh(RefreshSnippets);
  
  return true;
}


LivePhysicalOverviewBE * Db_sql_editor::live_physical_overview()
{
  return _live_physical_overview;
}


bool Db_sql_editor::activate_live_object(GrtObjectRef object)
{
  std::string obj_name= *object->name();
  std::string owner_name= *object->owner()->name();

  if (db_SchemaRef::can_wrap(object))
    schema_object_activated("activate", LiveSchemaTree::Schema, "", obj_name);
  else if (db_TableRef::can_wrap(object))
    schema_object_activated("activate", LiveSchemaTree::Table, owner_name, obj_name);
  else if (db_ViewRef::can_wrap(object))
    schema_object_activated("activate", LiveSchemaTree::View, owner_name, obj_name);
  else if (db_RoutineRef::can_wrap(object))
    schema_object_activated("activate", LiveSchemaTree::Routine, owner_name, obj_name);
  else
    return false;

  return true;
}


bool Db_sql_editor::create_live_object(GrtObjectRef object_type, std::string owner_name, std::string obj_name)
{
  if (db_SchemaRef::can_wrap(object_type))
    do_create_live_object(LiveSchemaTree::Schema, "", "");
  else if (db_TableRef::can_wrap(object_type))
    do_create_live_object(LiveSchemaTree::Table, owner_name, "");
  else if (db_ViewRef::can_wrap(object_type))
    do_create_live_object(LiveSchemaTree::View, owner_name, "");
  else if (db_RoutineRef::can_wrap(object_type))
    do_create_live_object(LiveSchemaTree::Routine, owner_name, "");
  else
    return false;

  return true;
}


bool Db_sql_editor::drop_live_object(GrtObjectRef object_type, std::string owner_name, std::string obj_name)
{
  if (db_SchemaRef::can_wrap(object_type))
    do_drop_live_object(LiveSchemaTree::Schema, "", obj_name);
  else if (db_TableRef::can_wrap(object_type))
    do_drop_live_object(LiveSchemaTree::Table, owner_name, "");
  else if (db_ViewRef::can_wrap(object_type))
    do_drop_live_object(LiveSchemaTree::View, owner_name, "");
  else if (db_RoutineRef::can_wrap(object_type))
    do_drop_live_object(LiveSchemaTree::Routine, owner_name, "");
  else
    return false;

  return true;
}


void Db_sql_editor::refresh_physical_overview()
{
  do_refresh_physical_overview(true);
}


void Db_sql_editor::do_refresh_physical_overview(Ref ref, bool hard_refresh)
{
  do_refresh_physical_overview(hard_refresh);
}


void Db_sql_editor::do_refresh_physical_overview(bool hard_refresh)
{
  _live_physical_overview->pre_refresh_groups();

  if (hard_refresh)
  {
    // remove all existing schemata
    std::string database_package= rdbms()->databaseObjectPackage();
    workbench_physical_ModelRef model= workbench_physical_ModelRef::cast_from(_live_physical_overview->get_model());
    db_CatalogRef catalog= model->catalog();
    db_SchemaRef schema= _grtm->get_grt()->create_object<db_Schema>(database_package + ".Schema");
    catalog->defaultSchema(schema);
    catalog->schemata().remove_all();
  }

  _schema_tree.refresh_ex(hard_refresh);

  _live_physical_overview->refresh_schemata_node();
  _live_physical_overview->send_refresh_schema_list();

#ifdef _WIN32
  do_ui_refresh();
#else
  do_partial_ui_refresh(RefreshSidebar);
#endif
}


bool Db_sql_editor::can_close()
{
  bool has_pending_changes= false;
  {
    GMutexLock recordsets_mutex(_recordsets_mutex);
    BOOST_FOREACH (Recordset::Ref rs, _recordsets)
    {
      if (rs->has_pending_changes())
      {
        has_pending_changes= true;
        break;
      }
    }
  }

  if (has_pending_changes)
  {
    int res= mforms::Utilities::show_warning(
      _("Closing Query Editor"),
      _("At least one result set has pending changes.\n"
        "If you choose to proceed all pending changes will be discarded."),
      _("OK")/*1*/,
      _("Cancel")/*0*/,
      _("")/*-1*/);

    if (res != 1)
      return false;
  }

  return true;
}
