#ifndef _DB_SQL_EDITOR_BE_H_
#define _DB_SQL_EDITOR_BE_H_

// renamed from db_sql_editor_be.h

#include "workbench/wb_backend_public_interface.h"
#include "sqlide/recordset_be.h"
#include "sqlide/sql_editor_be.h"
#include "sqlide/db_sql_editor_log_be.h"
#include "sqlide/db_sql_editor_history_be.h"
#include "cppdbc.h"
#include "grts/structs.workbench.h"
#include "grts/structs.db.mgmt.h"
#include "ui_form.h"
#include "grt/refresh_ui.h"
#include "sqlide/wb_context_sqlide.h"
#include "sqlide/wb_live_schema_tree.h"
#include "sqlide/wb_overview_live_physical.h"

namespace bec
{
class DBObjectEditorBE;
}

class MYSQLWBBACKEND_PUBLIC_FUNC Db_sql_editor : public bec::UIForm, public bec::RefreshUI
{
public:
  enum
  {
    RefreshSidebar, // update the sidebar
    RefreshSchemaTree, // refresh live schema tree
    RefreshEditor, // refresh the text editor using data from backend
    RefreshEditorBackend, // save text editor contents to backend so that it can be saved to file
    RefreshSnippets, // snippets list
    RefreshSnippetCollections, // snippet collections list
    RefreshOverview, // refresh the live overview
    RunCurrentScript, // running sql needs to be initiated by front-end
    RunCurrentStatement // run only statement under cursor position (if there any statement at all)
  };

public:
  typedef boost::shared_ptr<Db_sql_editor> Ref;
  static Ref create(wb::WBContextSQLIDE *wbsql, const db_mgmt_ConnectionRef &conn);
protected:
  Db_sql_editor(wb::WBContextSQLIDE *wbsql, const db_mgmt_ConnectionRef &conn);
public:
  virtual ~Db_sql_editor();
/*
 self reference which will be passed to every threaded task to retain until it's finished.
 the reference will prevent:
  - destroying object before threaded tasks are finished
  - initiation of new threaded tasks when being empty reference Ref()
 the reference is:
  - set in `create` static method
  - released in `close` virtual method
*/
public:
  Ref self_ref();
protected:
  Ref _self;
protected:
  virtual void dispose();

public:
  virtual bool close();
  virtual bool is_main_form() { return true; }
  virtual std::string get_form_context_name() const;
  
public:
  bec::GRTManager * grt_manager() const { return _grtm; }
  wb::WBContextSQLIDE *wbsql() const { return _wbsql; }
  void context_ui(wb::WBContextUI *val);
private:
  void request_refresh_of_menu_and_toolbar();
private:
  wb::WBContextSQLIDE *_wbsql;
  wb::WBContextUI *_context_ui;
  bec::GRTManager *_grtm;
  std::string _connection_info;

public:
  db_mgmt_RdbmsRef rdbms();

public:
  Sql_editor::Ref sql_editor();
  Sql_editor::Ref sql_editor(int index);
  std::string sql_editor_path(int index) { return _sql_editors[index].filename; }
  bool sql_editor_dirty(int index) { return _sql_editors[index].dirty; }
  void sql_editor_dirty(int index, bool flag) { _sql_editors[index].dirty= flag; }
private:
  struct Sql_editor_info {
    std::string filename;
    Sql_editor::Ref editor;
    bool dirty;
  };
  typedef std::vector<Sql_editor_info> Sql_editors;
  Sql_editors _sql_editors;
  GMutex *_sql_editors_mutex;
public:
  int add_sql_editor(); // returns index of the added sql_editor
  void remove_sql_editor(int index);
  int sql_editor_count();
  int active_sql_editor_index() const { return _active_sql_editor_index; }
  void active_sql_editor_index(int val) { _active_sql_editor_index= val; }
private:
  int _active_sql_editor_index;
private:
  void on_sql_editor_text_change();
  void on_sql_editor_text_selection_change();
  void set_sql_editor_text(const std::string &sql, bool run_sql);
public:
  std::string caption() const;

public:
  std::string connection_info();
private:
  std::map<std::string, std::string> _connection_details;

public:
  void connect();
  bool connected() const;
  void cancel_query();
  void reset();
  void commit();
  void rollback();
  bool auto_commit();
  void auto_commit(bool value);
private:
  void do_commit();

public:  
  db_mgmt_ConnectionRef connection_descriptor() const { return _connection; }

private:
  void create_connection(sql::Dbc_connection_handler::Ref dbc_conn, db_mgmt_ConnectionRef db_mgmt_conn, bool autocommit_mode);
  void init_connection(sql::Connection* dbc_conn_ref, const db_mgmt_ConnectionRef& connectionProperties, sql::Dbc_connection_handler::Ref& dbc_conn);
  bec::GMutexLock ensure_valid_dbc_connection(sql::Dbc_connection_handler::Ref dbc_conn, GMutex *dbc_conn_mutex);
  bec::GMutexLock ensure_valid_aux_connection();
  bec::GMutexLock ensure_valid_usr_connection();

private:
  bec::TimerActionThread *_keep_alive_thread;
  void send_message_keep_alive();
  
  db_mgmt_ConnectionRef _connection;
  // connection for maintenance operations, fetching schema contents & live editors (DDL only)
  sql::Dbc_connection_handler::Ref _aux_dbc_conn;
  GMutex *_aux_dbc_conn_mutex;
  // connection for running sql scripts
  sql::Dbc_connection_handler::Ref _usr_dbc_conn;
  GMutex *_usr_dbc_conn_mutex;

public:
  void exec_sql(std::string &sql, bool sync, bool wrap_with_non_std_delimiter= false);
  bool explain_sql(bool only_check_if_applicable);
  void explain_current_statement();
  bool is_running_query();
private:
  grt::StringRef do_exec_sql(grt::GRT *grt, Ref self, std::string &sql, bool retaining, bool wrap_with_non_std_delimiter);
  void split_sql(const std::string &sql, std::list<std::string> &statements);
  void do_explain_sql(const std::string &sql);
public:
  GrtThreadedTask::Ref exec_sql_task;
private:
  int on_exec_sql_finished();
  bool _is_running_query;

public:
  bool continue_on_error() { return _continue_on_error; }
  void continue_on_error(bool val);
private:
  bool _continue_on_error;

public:
  int recordset_count();
  Recordset::Ref recordset(int index);
  void recordsets_are_pinned_by_default(bool value);
  bool recordsets_are_pinned_by_default() { return _recordsets_are_pinned_by_default; }
private:
  typedef std::vector<Recordset::Ref> Recordsets;
  Recordsets _recordsets;
  GMutex *_recordsets_mutex;
  bool _recordsets_are_pinned_by_default;
public:
  sigc::signal<int, long long> close_recordset_ui;
private:
  void on_close_recordset(Recordset::Ref rs);
  int _rs_sequence;
private:
  void apply_changes_to_recordset(Recordset::Ref rs);
  bool run_data_changes_commit_wizard(Recordset::Ref rs);
  void apply_data_changes_commit(std::string &sql_script_text, Recordset::Ref rs);

  void show_export_recordset(Recordset::Ref rs);
public:
  bool can_close();

public:
  typedef sigc::signal<int, const std::string&> SqlEditorTextInsertSignal;
  SqlEditorTextInsertSignal sql_editor_text_insert_signal;

public:
  void new_sql_script_file();
  void open_sql_script_file(const std::string &file_path, bool run_immediately);
  bool save_sql_script_file(const std::string &file_path, int editor_index);
public:
  sigc::signal<int, int> sql_editor_new_ui;
  sigc::signal<int, std::string> sql_editor_caption_ui;

public:
  void get_schemata(std::list<std::string> &schemata);
  void active_schema(const std::string &value);
  std::string active_schema() const;
  int active_schema_index() const;

  wb::LiveSchemaTree *get_schema_tree();
  void request_refresh_schema_tree();
  
private:
  std::list<std::string> fetch_live_schema_list();
  bool fetch_live_schema_contents(bec::NodeId node, const std::string &schema_name, const wb::LiveSchemaTree::SchemaContentArrivedSlot &arrived_slot);
  grt::StringRef do_fetch_live_schema_contents(grt::GRT *grt, Ref self, bec::NodeId node, std::string schema_name, wb::LiveSchemaTree::SchemaContentArrivedSlot arrived_slot);
  void notify_of_fetched_schema_contents(Ref self, bec::NodeId node, db_SchemaRef schema, bool schemata_tree_needs_refresh, bool live_physical_overview_schema_node_needs_refresh, bool live_physical_overview_needs_refresh);
  bool fetch_object_details(wb::LiveSchemaTree::ObjectType obj_type, const std::string &schema_name, const std::string &obj_name, const wb::LiveSchemaTree::ObjectDetailsArrivedSlot &arrived_slot);
private:
  wb::LiveSchemaTree _schema_tree;
  GrtThreadedTask::Ref live_schema_fetch_task;
  GMutex *_schema_contents_mutex;

private:
  void do_alter_live_object(wb::LiveSchemaTree::ObjectType type, const std::string &schema_name, const std::string &obj_name);
  void do_create_live_object(wb::LiveSchemaTree::ObjectType type, const std::string &schema_name, const std::string &obj_name);
  void do_drop_live_object(wb::LiveSchemaTree::ObjectType type, const std::string &schema_name, const std::string &obj_name);
  std::string get_object_ddl_script(wb::LiveSchemaTree::ObjectType type, const std::string &schema_name, const std::string &obj_name);
  void apply_object_alter_script(std::string &alter_script, bec::DBObjectEditorBE* obj_editor);
  void refresh_live_object_in_editor(bec::DBObjectEditorBE* obj_editor, bool using_old_name);
  void refresh_live_object_in_overview(wb::LiveSchemaTree::ObjectType type, const std::string schema_name, const std::string old_obj_name, const std::string new_obj_name);
  db_SchemaRef create_new_schema(db_CatalogRef owner);
  db_TableRef create_new_table(db_SchemaRef owner);
  db_ViewRef create_new_view(db_SchemaRef owner);
  db_RoutineRef create_new_routine(db_SchemaRef owner);
public:
  void schema_object_activated(const std::string &action, wb::LiveSchemaTree::ObjectType type, const std::string &schema, const std::string &name);
  bool apply_changes_to_object(bec::DBObjectEditorBE* obj_editor, bool dry_run);
  virtual void notify_of_ui_form_creation(bec::UIForm *form);
  bool run_live_object_alteration_wizard(std::string &alter_script, bec::DBObjectEditorBE* obj_editor);
public:
  void create_live_table_stubs(bec::DBObjectEditorBE *table_editor);
  void expand_live_table_stub(bec::DBObjectEditorBE *table_editor, const std::string &schema_name, const std::string &obj_name);

private:
  typedef sigc::signal<int, long long, const std::string&, const std::string&> Error_cb;
  typedef sigc::signal<int, float> Batch_exec_progress_cb;
  typedef sigc::signal<int, long, long> Batch_exec_stat_cb;
  Error_cb on_sql_script_run_error;
  Batch_exec_progress_cb on_sql_script_run_progress;
  Batch_exec_stat_cb on_sql_script_run_statistics;

public:
  wb::LivePhysicalOverviewBE * live_physical_overview();
  bool activate_live_object(GrtObjectRef object);
  bool create_live_object(GrtObjectRef object_type, std::string owner_name, std::string obj_name);
  bool drop_live_object(GrtObjectRef object_type, std::string owner_name, std::string obj_name);
  void refresh_physical_overview();
private:
  void do_refresh_physical_overview(Ref self, bool hard_refresh);
  void do_refresh_physical_overview(bool hard_refresh);
private:
  wb::LivePhysicalOverviewBE * _live_physical_overview;

public:
  DbSqlEditorLog::Ref log() { return _log; }
  DbSqlEditorHistory::Ref history() { return _history; }
  std::string restore_sql_from_history(int entry_index, std::list<int> &detail_indexes);
protected:
  DbSqlEditorLog::Ref _log;
  DbSqlEditorHistory::Ref _history;

public:
  void get_log_event_details(int log_event_no, int &log_event_type_code, std::string &log_event_time, std::string &log_event_action, std::string &log_event_message);
protected:
  RowId add_log_message(int msg_type, const std::string &msg, const std::string &context);
  RowId add_log_message_ex(int msg_type, const std::string &msg, const std::string &context, const std::string &duration);
  void set_log_message_ex(RowId log_message_index, int msg_type, const std::string &msg, const std::string &context, const std::string &duration);
  void advance_exec_sql_task_progress();
private:
  bool _has_pending_progress_messages;

public:
  bool save_snippet(const std::string &text);

  // only used in windows
#ifdef _WIN32
public:
  typedef sigc::slot<int> Call_command_cb;
  typedef sigc::slot<bool> Validate_command_cb;

#define COMMAND_CALL_VALIDATE_CB(cmd) \
public:\
  void call_ ## cmd ## _cb(Call_command_cb cb) { _call_ ## cmd ## _cb= cb; }\
  void call_ ## cmd ## () { _call_ ## cmd ## _cb(); }\
private:\
  Call_command_cb _call_ ## cmd ## _cb;\
public:\
  void validate_ ## cmd ## _cb(Validate_command_cb cb) { _validate_ ## cmd ## _cb= cb; }\
  bool validate_ ## cmd ## () { return _validate_ ## cmd ## _cb(); }\
private:\
  Validate_command_cb _validate_ ## cmd ## _cb;\

COMMAND_CALL_VALIDATE_CB(exec_sql)
COMMAND_CALL_VALIDATE_CB(exec_current_sql_statement)
COMMAND_CALL_VALIDATE_CB(save_edits)
COMMAND_CALL_VALIDATE_CB(discard_edits)

#undef COMMAND_CALL_VALIDATE_CB
#endif

public:
  bool is_sidebar_left_aligned();
};


#endif /* _DB_SQL_EDITOR_BE_H_ */
