/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"

#include <math.h>

#include "home_screen.h"

using namespace wb;
using namespace mforms;
using namespace MySQL::Geometry;

//--------------------------------------------------------------------------------------------------

void delete_surface(cairo_surface_t* surface)
{
  if (surface != NULL)
    cairo_surface_destroy(surface);
}

//--------------------------------------------------------------------------------------------------

void delete_pattern(cairo_pattern_t* pattern)
{
  if (pattern != NULL)
    cairo_pattern_destroy(pattern);
}

//--------------------------------------------------------------------------------------------------

int image_width(cairo_surface_t* image)
{
  if (image != NULL)
    return cairo_image_surface_get_width(image);
  return 0;
}

//--------------------------------------------------------------------------------------------------

int image_height(cairo_surface_t* image)
{
  if (image != NULL)
    return cairo_image_surface_get_height(image);
  return 0;
}

//--------------------------------------------------------------------------------------------------

/**
 * Helper to draw text with a hot decoration.
 */
void text_with_decoration(cairo_t* cr, double x, double y, const char* text, bool hot, int width)
{
  cairo_set_source_rgb(cr, 0x37 / 255.0, 0x3d / 255.0, 0x48 / 255.0);
  cairo_move_to(cr, x, y);
  cairo_show_text(cr, text);
  cairo_stroke(cr);
  
  // TODO: replace this with font decoration once pango is incorporated.
  if (hot)
  {
    cairo_set_line_width(cr, 1);
    cairo_move_to(cr, x, y + 1.5);
    cairo_line_to(cr, x + width, y + 1.5);
    cairo_stroke(cr);
  }
}

//----------------- WorkbenchCentral ---------------------------------------------------------------

WorkbenchCentral::WorkbenchCentral(HomeScreen* owner)
: DrawBox()
{
  _owner= owner;
  _layout_dirty= true;
  _title_image= Utilities::load_icon("wb_central_title_en.png");
  _main_icon= Utilities::load_icon("wb_central_logo.png");
  _pointer_icon= Utilities::load_icon("news_arrow.png");
  _button_parts[0]= Utilities::load_icon("action_button_left.png");
  _button_parts[1]= Utilities::load_icon("action_button_middle.png");
  _button_parts[2]= Utilities::load_icon("action_button_right.png");
  
  if (_button_parts[1] != NULL)
  {
    _button_pattern = cairo_pattern_create_for_surface(_button_parts[1]);
    cairo_pattern_set_extend(_button_pattern, CAIRO_EXTEND_REPEAT);
  }
  else
    _button_pattern= NULL;
  
  _title1_hot= false;
  _title2_hot= false;
}

//--------------------------------------------------------------------------------------------------

WorkbenchCentral::~WorkbenchCentral()
{
  delete_surface(_title_image);
  delete_surface(_main_icon);
  delete_surface(_pointer_icon);
  delete_surface(_button_parts[0]);
  delete_surface(_button_parts[1]);
  delete_surface(_button_parts[2]);
  if (_button_pattern != NULL)
    cairo_pattern_destroy(_button_pattern);
}

//--------------------------------------------------------------------------------------------------

/**
 * Draws an action button with the given text.
 */
void WorkbenchCentral::draw_button(cairo_t* cr, double x, double y)
{
  if (_button_pattern != NULL)
  {
    cairo_set_source_surface(cr, _button_parts[0], x, y);
    cairo_paint(cr);
    
    x += image_width(_button_parts[0]);
    
    // Buttons are all of same height, so use the one from the first button.
    // In order to align the pattern properly to our target bounds we create a
    // translation transformation with the desired amount.
    cairo_matrix_t matrix;
    cairo_matrix_init_translate(&matrix, -x, -y);
    cairo_pattern_set_matrix(_button_pattern, &matrix);
    
    cairo_set_source(cr, _button_pattern);
    cairo_move_to(cr, x, y);
    cairo_rectangle(cr, x, y, _button_inner_width, _button1_bounds.height);
    cairo_fill(cr);
    
    cairo_set_source_surface(cr, _button_parts[2], x + _button_inner_width, y);
    cairo_paint(cr);
  }
}

//--------------------------------------------------------------------------------------------------

#define TITLE "Welcome to MySQL Workbench"
#define CHAPTER1_TITLE "What's New in This Release?"
#define CHAPTER1_DETAILS "Read about all changes in this MySQL Workbench release."
#define CHAPTER2_TITLE "Dev-Central Blogs"
#define CHAPTER2_DETAILS "Keep up to date with the main product and its extensions."

#define BUTTON1_TEXT "Check for Updates"
#define BUTTON2_TEXT "Submit a Bug Report"
#define BUTTON3_TEXT "Discuss a Topic"

#define CENTRAL_RIGHT_SPACING 8          // Horizontal distance between right border and button block.

void WorkbenchCentral::repaint(cairo_t* cr, int areax, int areay, int areaw, int areah)
{
  layout(cr);
  
  cairo_pattern_t* gradient = cairo_pattern_create_linear(0, 0, 0, get_height());
  cairo_pattern_add_color_stop_rgb(gradient, 0, 0xff / 255.0, 0xff / 255.0, 0xff / 255.0);
  cairo_pattern_add_color_stop_rgb(gradient, 0.33, 0xff / 255.0, 0xff / 255.0, 0xff / 255.0);
  cairo_pattern_add_color_stop_rgb(gradient, 1, 0xf4 / 255.0, 0xf7 / 255.0, 0xfc / 255.0);
  
  cairo_set_source(cr, gradient);
  cairo_paint(cr);
  
  cairo_pattern_destroy(gradient);

  double width= get_width();
  double button_offset= width - _button1_bounds.width - CENTRAL_RIGHT_SPACING;
  
  cairo_set_source_surface(cr, _main_icon, 14, 11);
  cairo_paint(cr);

  cairo_set_source_surface(cr, _title_image, _title_bounds.left, _title_bounds.top);
  cairo_paint(cr);

  cairo_select_font_face(cr, HOME_NORMAL_FONT, CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, HOME_NORMAL_FONT_SIZE);
  text_with_decoration(cr, _chapter1_title_bounds.left, _chapter1_title_bounds.top, CHAPTER1_TITLE, 
                       _title1_hot, (int) _chapter1_title_bounds.width);
  text_with_decoration(cr, _chapter2_title_bounds.left, _chapter2_title_bounds.top, CHAPTER2_TITLE,
                       _title2_hot, (int) _chapter2_title_bounds.width);

  cairo_select_font_face(cr, HOME_DETAILS_FONT, CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, HOME_DETAILS_FONT_SIZE);
  
  // The second part might not have enough room, so we need to shorten the text eventually.
  std::string text= Utilities::shorten_string(cr, CHAPTER1_DETAILS, button_offset - _chapter1_details_bounds.left - 15);
  text_with_decoration(cr, _chapter1_details_bounds.left, _chapter1_details_bounds.top, text.c_str(),
                       false, 0);

  text= Utilities::shorten_string(cr, CHAPTER2_DETAILS, button_offset - _chapter2_details_bounds.left - 15);
  text_with_decoration(cr, _chapter2_details_bounds.left, _chapter2_details_bounds.top, text.c_str(), 
                       false, 0);

  cairo_set_source_surface(cr, _pointer_icon, _arrow1_bounds.left, _arrow1_bounds.top);
  cairo_paint(cr);
  
  cairo_set_source_surface(cr, _pointer_icon, _arrow2_bounds.left, _arrow2_bounds.top);
  cairo_paint(cr);
  
  // Action buttons. These are computed such that they have the same size.
  draw_button(cr, button_offset, _button1_bounds.top);
  draw_button(cr, button_offset, _button2_bounds.top);
  draw_button(cr, button_offset, _button3_bounds.top);

  cairo_select_font_face(cr, HOME_NORMAL_FONT, CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, ACTION_BUTTON_FONT_SIZE);
  cairo_set_source_rgb(cr, 0x37 / 255.0, 0x3d / 255.0, 0x48 / 255.0);
  
  cairo_move_to(cr, button_offset + _button1_text_left, _button1_bounds.top + _button1_text_top);
  cairo_show_text(cr, BUTTON1_TEXT);
  cairo_move_to(cr, button_offset + _button2_text_left, _button2_bounds.top + _button2_text_top);
  cairo_show_text(cr, BUTTON2_TEXT);
  cairo_move_to(cr, button_offset + _button3_text_left, _button3_bounds.top + _button3_text_top);
  cairo_show_text(cr, BUTTON3_TEXT);
  cairo_stroke(cr);
}

//--------------------------------------------------------------------------------------------------

#define CENTRAL_LEFT_SPACING 12          // Space between left border and icon.
#define CENTRAL_TOP_SPACING 24           // Vertical space between top border and heading.
#define CENTRAL_ICON_SPACING 18          // Horizontal space between icon and content.
#define CENTRAL_ARROW_OFFSET 2           // Horizontal movement of the left arrow (there is some inset).
#define CENTRAL_HEADLINE_ICON_SPACING 7  // Vertical space between headline and pointer icon.
#define CENTRAL_HEADLINE_SPACING 8       // Vertical distance between headline image and normal text.
#define CENTRAL_CHAPTER_SPACING 32       // Space between the chapter titles.
#define CENTRAL_CHAPTER_ARROW_SPACING 2  // Space between chapter arrow and chapter title.
#define CENTRAL_LINE_SPACING 2           // Extra space between two lines of normal text.
#define CENTRAL_BUTTON_SPACING 2         // Vertical distance between action buttons.
#define CENTRAL_BUTTON_PADDING 7         // Additional space left and right to the button text.
#define CENTRAL_BUTTON_TEXT_SHIFT 1      // Correction factor to align the button text vertically (the button
                                         // images include some outer area too, so centering the text is not correct
                                         // without some manual adjustment).

void WorkbenchCentral::layout(cairo_t* cr)
{
  if (_layout_dirty)
  {
    cairo_text_extents_t extents;

    _layout_dirty= false;
    
    double offset= CENTRAL_LEFT_SPACING + image_width(_main_icon) + CENTRAL_ICON_SPACING;

    _title_bounds.left= offset;
    _title_bounds.top= CENTRAL_TOP_SPACING;
    _title_bounds.width= image_width(_title_image);
    _title_bounds.height= image_height(_title_image);
    
    // Text sizes for chapter text.
    cairo_select_font_face(cr, HOME_NORMAL_FONT, CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
    cairo_set_font_size(cr, HOME_NORMAL_FONT_SIZE);
    cairo_text_extents(cr, CHAPTER1_TITLE, &extents);
    _chapter1_title_bounds.width= ceil(extents.width);
    _chapter1_title_bounds.height= ceil(extents.height);

    cairo_text_extents(cr, CHAPTER2_TITLE, &extents);
    _chapter2_title_bounds.width= ceil(extents.width);
    _chapter2_title_bounds.height= ceil(extents.height);
    
    // Make both titles the same (largest) height. This is necessary to give both
    // detail strings the same vertical offset.
    if (_chapter2_title_bounds.height > _chapter1_title_bounds.height)
      _chapter1_title_bounds.height= _chapter2_title_bounds.height;
    else
      _chapter2_title_bounds.height= _chapter1_title_bounds.height;

    cairo_select_font_face(cr, HOME_DETAILS_FONT, CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
    cairo_set_font_size(cr, HOME_DETAILS_FONT_SIZE);
    cairo_text_extents(cr, CHAPTER1_DETAILS, &extents);
    _chapter1_details_bounds.width= ceil(extents.width);
    _chapter1_details_bounds.height= ceil(extents.height);

    cairo_text_extents(cr, CHAPTER2_DETAILS, &extents);
    _chapter2_details_bounds.width= ceil(extents.width);
    _chapter2_details_bounds.height= ceil(extents.height);

    // Chapter 1 arrow, title and details text positioning.
    _chapter1_title_bounds.top= _title_bounds.top + _title_bounds.height + _chapter1_title_bounds.height + CENTRAL_HEADLINE_SPACING;

    _arrow1_bounds.top= _title_bounds.top + _title_bounds.height + CENTRAL_HEADLINE_ICON_SPACING;
    _arrow1_bounds.left= _title_bounds.left + CENTRAL_ARROW_OFFSET;
    _arrow1_bounds.width= image_width(_pointer_icon);
    _arrow1_bounds.height= image_height(_pointer_icon);
    
    _chapter1_title_bounds.left= _arrow1_bounds.left + _arrow1_bounds.width + CENTRAL_CHAPTER_ARROW_SPACING;
    _chapter1_details_bounds.left= _chapter1_title_bounds.left;

    _chapter1_details_bounds.top= _chapter1_title_bounds.top + _chapter1_title_bounds.height + CENTRAL_LINE_SPACING;
    
    // Chapter 2 arrow, title and details text positioning.
    offset= MAX(_chapter1_title_bounds.width, _chapter1_details_bounds.width);
    _chapter2_title_bounds.top= _chapter1_title_bounds.top;
    
    _arrow2_bounds.top= _arrow1_bounds.top;
    _arrow2_bounds.left= _chapter1_title_bounds.left + offset + CENTRAL_CHAPTER_SPACING;
    _arrow2_bounds.width= image_width(_pointer_icon);
    _arrow2_bounds.height= image_height(_pointer_icon);
    
    _chapter2_title_bounds.left= _arrow2_bounds.left + _arrow1_bounds.width + CENTRAL_CHAPTER_ARROW_SPACING;
    _chapter2_details_bounds.left= _chapter2_title_bounds.left;

    _chapter2_details_bounds.top= _chapter2_title_bounds.top + _chapter2_title_bounds.height + CENTRAL_LINE_SPACING;
    
    // Action buttons.
    double button_width;
    cairo_select_font_face(cr, HOME_NORMAL_FONT, CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
    cairo_set_font_size(cr, ACTION_BUTTON_FONT_SIZE);
    
    cairo_text_extents_t text1_extents;
    cairo_text_extents(cr, BUTTON1_TEXT, &text1_extents);
    button_width= text1_extents.width;
    
    cairo_text_extents_t text2_extents;
    cairo_text_extents(cr, BUTTON2_TEXT, &text2_extents);
    if (text2_extents.width > button_width)
      button_width= text2_extents.width;

    cairo_text_extents_t text3_extents;
    cairo_text_extents(cr, BUTTON3_TEXT, &text3_extents);
    if (text3_extents.width > button_width)
      button_width= text3_extents.width;
    _button_inner_width= (int) ceil(button_width + 2 * CENTRAL_BUTTON_PADDING);
    button_width= _button_inner_width;
    button_width += image_width(_button_parts[0]) + image_width(_button_parts[2]);
    _button1_bounds.width= button_width;
    _button1_bounds.height= image_height(_button_parts[1]);
    _button2_bounds.width= _button1_bounds.width;
    _button2_bounds.height= _button1_bounds.height;
    _button3_bounds.width= _button1_bounds.width;
    _button3_bounds.height= _button1_bounds.height;
    
    // Compute total height of the button block and center it vertically (right aligned in the window).
    int total_button_height= (int) (3 * _button1_bounds.height + 2 * CENTRAL_BUTTON_SPACING);
    _button1_bounds.top= (get_height() - total_button_height) / 2;
    _button1_bounds.left= get_width() - _button1_bounds.width - CENTRAL_RIGHT_SPACING;
    _button2_bounds.left= _button1_bounds.left;
    _button3_bounds.left= _button1_bounds.left;
    _button2_bounds.top= _button1_bounds.top + _button1_bounds.height + CENTRAL_BUTTON_SPACING;
    _button3_bounds.top= _button2_bounds.top + _button2_bounds.height + CENTRAL_BUTTON_SPACING;
    
    // Compute button text positions. The text is vertically and horizontally centered.
    // Keep in mind: origin for text is lower left corner, not upper left one.
    // These coordinates are relative as the button positions can change (depending on window size).
    _button1_text_top= (int) ((_button1_bounds.height + text1_extents.height) / 2 - CENTRAL_BUTTON_TEXT_SHIFT);
    _button1_text_left= (int) ((_button1_bounds.width - text1_extents.width) / 2);

    _button2_text_top= (int) ((_button2_bounds.height + text2_extents.height) / 2 - CENTRAL_BUTTON_TEXT_SHIFT);
    _button2_text_left= (int) ((_button2_bounds.width - text2_extents.width) / 2);

    _button3_text_top= (int) ((_button3_bounds.height + text3_extents.height) / 2 - CENTRAL_BUTTON_TEXT_SHIFT);
    _button3_text_left= (int) ((_button3_bounds.width - text3_extents.width) / 2);
  }
}

//--------------------------------------------------------------------------------------------------

void WorkbenchCentral::mouse_click(int button, int x, int y)
{
  switch (button)
  {
    case 0: // left button
      HomeScreenAction action= action_from_point(x, y);
      if (action != ActionNone)
        _owner->trigger_callback(action, grt::ValueRef());
      break;
  }
}

//--------------------------------------------------------------------------------------------------

HomeScreenAction WorkbenchCentral::action_from_point(int x, int y)
{
  if (_chapter1_title_bounds.contains_flipped(x, y))
    return ActionWhatsNew;
  else
    if (_chapter2_title_bounds.contains_flipped(x, y))
      return ActionBlogs;
    else
      if (_button1_bounds.contains(x, y))
        return ActionCheckForUpdates;
      else
        if (_button2_bounds.contains(x, y))
          return ActionSubmitBugReport;
        else
          if (_button3_bounds.contains(x, y))
            return ActionDiscussTopic;

  return ActionNone;
}

//--------------------------------------------------------------------------------------------------

void WorkbenchCentral::mouse_leave()
{
  bool repaint_needed= _title1_hot || _title2_hot;
  _title1_hot= false;
  _title2_hot= false;
  if (repaint_needed)
    set_needs_repaint();
}

//--------------------------------------------------------------------------------------------------

void WorkbenchCentral::mouse_move(int x, int y)
{
  bool repaint_needed= false;
  if (_chapter1_title_bounds.contains_flipped(x, y))
  {
    repaint_needed= !_title1_hot;
    _title1_hot= true;
  }
  else
  {
    repaint_needed= _title1_hot;
    _title1_hot= false;
  }

  if (_chapter2_title_bounds.contains_flipped(x, y))
  {
    repaint_needed= repaint_needed || !_title2_hot;
    _title2_hot= true;
  }
  else
  {
    repaint_needed= repaint_needed || _title2_hot;
    _title2_hot= false;
  }
  
  
  if (repaint_needed)
    set_needs_repaint();
}

//----------------- ActionLink ---------------------------------------------------------------------

ActionLink::ActionLink(const std::string& icon_name, const std::string& title, const std::string& description,
                       const grt::ValueRef &object, HomeScreenAction action, bool enabled)
{
  _icon= Utilities::load_icon(icon_name);
  _title= title;
  _description= description;
  _last_text_width= 0;
  _action= action;
  _object= object;
  _enabled= enabled;
}

//--------------------------------------------------------------------------------------------------

ActionLink::~ActionLink()
{
  delete_surface(_icon);
}

//--------------------------------------------------------------------------------------------------

#define ACTION_LINK_PADDING 2      // Left and right padding.
#define ACTION_LINK_ICON_SPACING 4 // Horizontal distance between icon and text.
#define ACTION_LINK_TEXT_SPACING 6 // Vertical distance between title and description.

void ActionLink::paint(cairo_t* cr, Rect bounds, bool hot, bool active)
{
  this->_bounds= bounds;
  cairo_save(cr);
  
  double icon_width= image_width(_icon);
  double icon_height= image_height(_icon);

  double new_width= bounds.width - 2 * ACTION_LINK_PADDING - icon_width - ACTION_LINK_ICON_SPACING;
  if (new_width != _last_text_width)
  {
    _last_text_width= new_width;
    layout(cr);
  }
  
  // Fill a blue background if the item is active.
  if (active && _enabled)
  {
    cairo_set_source_rgb(cr, 0x5a / 255.0, 0x85 / 255.0, 0xdc / 255.0);
    cairo_rectangle(cr, bounds.left, bounds.top, bounds.width, bounds.height);
    cairo_fill(cr);
  }

  cairo_set_source_surface(cr, _icon, bounds.left + ACTION_LINK_PADDING, bounds.top + (int) ((bounds.height - icon_height) / 2));
  if (_enabled)
    cairo_paint(cr);
  else
    cairo_paint_with_alpha(cr, 0.25);

  int text_offset= (int) (bounds.height - _text_height) / 2;
  cairo_select_font_face(cr, ACTION_LINK_NORMAL_FONT, CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, ACTION_LINK_NORMAL_FONT_SIZE);

  if (!_enabled)
    cairo_set_source_rgb(cr, 0.75, 0.75, 0.75);
  else
    if (active)
      cairo_set_source_rgb(cr, 1, 1, 1);
    else
      if (hot)
        cairo_set_source_rgb(cr, 90 / 255.0, 147 / 255.0, 220 / 255.0);
      else
        cairo_set_source_rgb(cr, 0x37 / 255.0, 0x3d / 255.0, 0x48 / 255.0);
  double offset= bounds.left + ACTION_LINK_PADDING + icon_width + ACTION_LINK_ICON_SPACING;
  
  cairo_move_to(cr, offset, bounds.top  + _title_offset + text_offset);
  cairo_show_text(cr, _shorted_title.c_str());
  cairo_stroke(cr);

  cairo_select_font_face(cr, HOME_DETAILS_FONT, CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_NORMAL);
  cairo_set_font_size(cr, ACTION_LINK_DETAILS_FONT_SIZE);
  
  if (!_enabled)
    cairo_set_source_rgb(cr, 0.75, 0.75, 0.75);
  else
    if (active)
      cairo_set_source_rgb(cr, 1, 1, 1);
    else
      cairo_set_source_rgb(cr, 0xc6 / 255.0, 0xc6 / 255.0, 0xc6 / 255.0);
  
  cairo_move_to(cr, offset, bounds.top  + _description_offset + text_offset);
  cairo_show_text(cr, _shorted_description.c_str());
  cairo_stroke(cr);

  // Hot style implementation is currently done manually (draw a line).
  // TODO: replace this with font decoration once pango is incorporated.
  if (hot && _enabled)
  {
    cairo_set_line_width(cr, 1);
    cairo_set_source_rgb(cr, 90 / 255.0, 147 / 255.0, 220 / 255.0);
    cairo_move_to(cr, offset, bounds.top + _title_offset + text_offset + 1.5);
    cairo_line_to(cr, offset + _title_width, bounds.top + _title_offset + text_offset + 1.5);
    cairo_stroke(cr);
  }
  
  cairo_restore(cr);
}

//--------------------------------------------------------------------------------------------------

void ActionLink::layout(cairo_t* cr)
{
  // Re-compute shorted title and its position.
  cairo_select_font_face(cr, ACTION_LINK_NORMAL_FONT, CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
  cairo_set_font_size(cr, ACTION_LINK_NORMAL_FONT_SIZE);
  
  _shorted_title= Utilities::shorten_string(cr, _title, _last_text_width);

  cairo_text_extents_t title_extents;
  cairo_text_extents(cr, _shorted_title.c_str(), &title_extents);
  _title_offset= (int) -title_extents.y_bearing;
  _title_width= title_extents.width;

  // Same for the description.
  cairo_select_font_face(cr, HOME_DETAILS_FONT, CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_NORMAL);
  cairo_set_font_size(cr, ACTION_LINK_DETAILS_FONT_SIZE);
  _shorted_description= Utilities::shorten_string(cr, _description, _last_text_width);

  cairo_text_extents_t description_extents;
  cairo_text_extents(cr, _shorted_description.c_str(), &description_extents);
  _description_offset= _title_offset - (int) description_extents.y_bearing + ACTION_LINK_TEXT_SPACING;

  // Determine overall text height. This is used to center the text during paint.
  _text_height= (int) ceil(title_extents.height + description_extents.height + ACTION_LINK_TEXT_SPACING);
}

//--------------------------------------------------------------------------------------------------

bool ActionLink::contains(double x, double y)
{
  return _bounds.contains(x, y);
}

//-------------------------- WorkspaceGradient -----------------------------------------------------

void WorkspaceGradient::add_stop_color(double position, double r, double g, double b, double a)
{
  StopColor entry= {position, r, g, b, a};
  _values.push_back(entry);
}

//--------------------------------------------------------------------------------------------------

void WorkspaceGradient::add_stop_color(double position, int color)
{
  int red = (color >> 16) & 0xff;
  int green = (color >> 8) & 0xff;
  int blue = color & 0xff;
  StopColor entry= {position, red / 255.0, green / 255.0, blue / 255.0, 1};
  _values.push_back(entry);
}

//--------------------------------------------------------------------------------------------------

cairo_pattern_t* WorkspaceGradient::create_pattern(int height)
{
  cairo_pattern_t* result = cairo_pattern_create_linear(0, 0, 0, height);
  for (std::vector<StopColor>::const_iterator iterator= _values.begin(); iterator != _values.end(); iterator++)
    cairo_pattern_add_color_stop_rgba(result, 
                                      (*iterator).position,
                                      (*iterator).red,
                                      (*iterator).green,
                                      (*iterator).blue,
                                      (*iterator).alpha);
  return result;
}

//-------------------------- ActionLinkBox ---------------------------------------------------------

ActionLinkBox::ActionLinkBox(HomeScreen* owner, const std::string& header_image, WorkspaceGradient* gradient,
                             int top_spacing, int bottom_spacing, int left_spacing, int right_spacing)
{
  _owner= owner;
  _layout_width= 10;
  _layout_height= 10;
  _background= NULL;
  _gradient= gradient; // Weak reference, we don't own it.
  _image= Utilities::load_icon(header_image);
  _hot_link= NULL;
  _selected_link= NULL;
  _layout_dirty= true;
  _top_spacing= top_spacing;
  _bottom_spacing= bottom_spacing;
  _back_red= 1;
  _back_green= 1;
  _back_blue= 1;
  _left_spacing= left_spacing;
  _right_spacing= right_spacing;
  _can_select= false;
  _single_click= true;
}

//--------------------------------------------------------------------------------------------------

ActionLinkBox::~ActionLinkBox()
{
  set_destroying();
  clear();
  delete_surface(_image);
  delete_pattern(_background);
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::add_link(const std::string& icon_name, const std::string& title,
                             const std::string& description, const grt::ValueRef &object, HomeScreenAction action, bool enabled)
{
  ActionLink* link= new ActionLink(icon_name, title, description, object, action, enabled);
  _links.push_back(link);
  _layout_dirty= true;
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::clear()
{
  for (std::vector<ActionLink*>::iterator iterator= _links.begin(); iterator != _links.end(); iterator++)
    delete *iterator;
  _links.clear();

  if (!is_destroying())
    _layout_dirty= true;
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::set_back_color(double red, double green, double blue)
{
  _back_red= red;
  _back_green= green;
  _back_blue= blue;
  set_needs_repaint();
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::set_back_color(int color)
{
  _back_red= ((color >> 16) & 0xff) / 255.0;
  _back_green= ((color >> 8) & 0xff) / 255.0;
  _back_blue= (color & 0xff) / 255.0;
  set_needs_repaint();
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::can_select(bool selectable)
{
  _can_select= selectable;
  if (!_can_select && _selected_link != NULL)
  {
    _selected_link= NULL;
    set_needs_repaint();
  }
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::single_click(bool value)
{
  _single_click= value;
}

//--------------------------------------------------------------------------------------------------

#define ACTION_LINK_HEIGHT 39  // The height of a single action link.
#define ACTION_LINK_SPACING 0  // Vertical distance between two action links.
#define BOX_IMAGE_SPACING 5    // The vertical distance between header image and first action link.

void ActionLinkBox::repaint(cairo_t *cr, int areax, int areay, int areaw, int areah)
{
  layout();
  
  double width= get_width();
  double height= get_height();
  
  if (_background != NULL)
  {
    cairo_set_source(cr, _background);
    cairo_rectangle(cr, 0, 0, width, height);
    cairo_fill(cr);
  }
  else
  {
    cairo_set_source_rgb(cr, _back_red, _back_green, _back_blue);
    cairo_paint(cr);
  }

  Rect link_bounds(_left_spacing, _top_spacing, width - _left_spacing - _right_spacing, ACTION_LINK_HEIGHT);
  
  if (_image != NULL)
  {
    cairo_set_source_surface(cr, _image, link_bounds.left, _top_spacing);
    cairo_paint(cr);

    link_bounds.top += BOX_IMAGE_SPACING + image_height(_image);;
  }
  for (std::vector<ActionLink*>::const_iterator iterator= _links.begin(); iterator != _links.end(); iterator++)
  {
    (*iterator)->paint(cr, link_bounds, *iterator == _hot_link, (*iterator == _selected_link) && _can_select);
    link_bounds.top += ACTION_LINK_HEIGHT + ACTION_LINK_SPACING;
  }
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::mouse_enter()
{
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::mouse_leave()
{
  // Hot link markup is only used for single click lists (to indicate the item can activated).
  // Hence when single click actions are disabled we also don't want hot items.
  if (_single_click && _hot_link != NULL)
  {
    _hot_link= NULL;
    set_needs_repaint();
  }
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::mouse_move(int x, int y)
{
  // See comment about single clicks and hot items in mouse_leave.
  if (_single_click)
  {
    ActionLink* link= action_link_from_point(x, y);
    if (link != _hot_link)
    {
      _hot_link= link;
      set_needs_repaint();
    }
  }
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::mouse_down(int button, int x, int y)
{
  if (button == 0) // Left button.
  {
    ActionLink* link= action_link_from_point(x, y);
    set_selected(link);
  }
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::mouse_up(int button, int x, int y)
{
  switch (button)
  {
    case 1: // Right button
      // TODO: context menu.
      break;
  }
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::mouse_click(int button, int x, int y)
{
  switch (button)
  {
    case 0: // Left button
      if (_single_click)
      {
        ActionLink* link= action_link_from_point(x, y);
        if (link == _selected_link)
        {
          for (int index= 0; index < (int) _links.size(); index++)
            if (_links[index] == _selected_link)
            {
              _owner->trigger_callback(link->get_action(), link->get_object());
              return;
            }
        }
      }
      break;
  }
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::select(int index)
{
  if (index >= 0 && index < (int)_links.size())
    set_selected(_links[index]);
  else
    set_selected(0);
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::mouse_double_click(int button, int x, int y)
{
  switch (button)
  {
    case 0: // Left button
      if (!_single_click)
      {
        ActionLink* link= action_link_from_point(x, y);
        if (link == _selected_link)
        {
          for (int index= 0; index < (int) _links.size(); index++)
            if (_links[index] == _selected_link)
            {
              _owner->trigger_callback(link->get_action(), link->get_object());
              return;
            }
        }
      }
      break;
  }
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::get_layout_size(int* w, int* h)
{
  layout();
  
  *w= _layout_width;
  *h= _layout_height;
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::layout()
{
  if (_layout_dirty)
  {
    _layout_dirty= false;
    
    _layout_height= _top_spacing;
    _layout_width= _left_spacing + _right_spacing;
    
    if (_links.size() > 0)
      _layout_height += _links.size() * ACTION_LINK_HEIGHT + (_links.size() - 1) * ACTION_LINK_SPACING;
    if (_image != NULL)
    {
      _layout_height += BOX_IMAGE_SPACING + image_height(_image);
      _layout_width += image_width(_image);
    }
    
    if (_layout_height < ACTION_LINK_HEIGHT)
      _layout_height= ACTION_LINK_HEIGHT;
    _layout_height += _bottom_spacing;
    
    delete_pattern(_background);
    if (_gradient != NULL)
      _background = _gradient->create_pattern(_layout_height);
    else
      _background = NULL;
  }  
}

//--------------------------------------------------------------------------------------------------

ActionLink* ActionLinkBox::action_link_from_point(double x, double y)
{
  if (x >= 0 && x < get_width() && y >= 0 && y <= get_height())
  {
    for (std::vector<ActionLink*>::const_iterator iterator= _links.begin(); iterator != _links.end(); iterator++)
      if ((*iterator)->contains(x, y) && (*iterator)->enabled())
        return *iterator;
  }
  return NULL;
}

//--------------------------------------------------------------------------------------------------

void ActionLinkBox::set_selected(ActionLink* link)
{
  if (link != _selected_link)
  {
    _selected_link= link;
    set_needs_repaint();
  }
}

//----------------- GradientBox --------------------------------------------------------------------

GradientBox::GradientBox(WorkspaceGradient* gradient)
{
  _gradient= gradient;
  _back_red= 1;
  _back_green= 1;
  _back_blue= 1;
}

//--------------------------------------------------------------------------------------------------

GradientBox::~GradientBox()
{
}

//--------------------------------------------------------------------------------------------------

void GradientBox::set_back_color(double red, double green, double blue)
{
  _back_red= red;
  _back_green= green;
  _back_blue= blue;
  set_needs_repaint();
}

//--------------------------------------------------------------------------------------------------

void GradientBox::set_back_color(int color)
{
  _back_red= ((color >> 16) & 0xff) / 255.0;
  _back_green= ((color >> 8) & 0xff) / 255.0;
  _back_blue= (color & 0xff) / 255.0;
  set_needs_repaint();
}

//--------------------------------------------------------------------------------------------------

void GradientBox::repaint(cairo_t *cr, int areax, int areay, int areaw, int areah)
{
  cairo_pattern_t* background= NULL;
  if (_gradient != NULL)
    background= _gradient->create_pattern(get_height());
  if (background != NULL)
  {
    cairo_set_source(cr, background);
    cairo_rectangle(cr, areax, areay, areaw, areah);
    cairo_fill(cr);
  }
  else
  {
    cairo_set_source_rgb(cr, _back_red, _back_green, _back_blue);
    cairo_paint(cr);
  }
}

//----------------- ActionList ---------------------------------------------------------------------

ActionList::ActionList(HomeScreen* owner, int left_spacing, int right_spacing, WorkspaceGradient* spacer_gradient,
                       const std::string& icon_name, HomeScreenAction action)
: Box(true), _panel(false), _panel_border(false), _icon_name(icon_name)
{
  _owner= owner;
  _action= action;

  _left_spacer= new GradientBox(spacer_gradient);
  _left_spacer->set_size(left_spacing, -1);
  _left_spacer->set_name("left spacer"); // Debug
  add(_left_spacer, false, true);

  _link_box= new ActionLinkBox(_owner, "", NULL, 0, 0, 0, 0);
  _link_box->set_back_color(1, 1, 1);
  _link_box->can_select(true);
  _link_box->single_click(false);
  _link_box->set_back_color(0xeef1f5);

  int w, h;
  _link_box->get_layout_size(&w, &h);
  _link_box->set_size(w, h);
  
  _panel.add(_link_box);
  _panel.set_back_color("#eef1f5");
  _panel.set_visible_scrollers(true, false);
  _panel.set_name("scroll panel"); // Debug

  _panel_border.set_back_color("#93accb");
  _panel_border.set_padding(1);
  _panel_border.add(&_panel, true, true);
  _panel_border.set_name("scroll panel border"); // Debug
  add(&_panel_border, true, true);

  _right_spacer= new GradientBox(spacer_gradient);
  _right_spacer->set_size(right_spacing, -1);
  _right_spacer->set_name("right spacer"); // Debug
  add(_right_spacer, false, true);

  set_name("action list"); // Debug
}

//--------------------------------------------------------------------------------------------------

ActionList::~ActionList()
{
  delete _link_box;
  delete _left_spacer;
  delete _right_spacer;
}

//--------------------------------------------------------------------------------------------------

void ActionList::set_spacer_back_color(int color)
{
  _left_spacer->set_back_color(color);
  _right_spacer->set_back_color(color);
}

//--------------------------------------------------------------------------------------------------

void ActionList::add_entry(const std::string& title, const std::string& description, const grt::ValueRef &object)
{
  _link_box->add_link(_icon_name, title, description, object, _action, true);
  
  // Explicitely set the size of the link box, as this doesn't happen automatically.
  int w, h;
  _link_box->get_layout_size(&w, &h);
  _link_box->set_size(w, h);
}

//--------------------------------------------------------------------------------------------------

void ActionList::select(int index)
{
  _link_box->select(index);
}

//--------------------------------------------------------------------------------------------------

void ActionList::clear()
{
  _link_box->clear();

  // Explicitely set the size of the link box, as this doesn't happen automatically.
  int w, h;
  _link_box->get_layout_size(&w, &h);
  _link_box->set_size(w, h);
}

//----------------- Workspace ----------------------------------------------------------------------

#define COLUMN_TOP_SPACE 18            // Vertical distance between top border and content.
#define COLUMN_BOTTOM_SPACE 18         // Vertical distance between bottom border and content.
#define COLUMN_TOP_LIST_SPACING 7      // Vertical distance between top action list and list box.
#define COLUMN_BOTTOM_LIST_SPACING 18  // Vertical distance between bottom action list and list box.
#define COLUMN_LIST_LEFT_SPACING 46    // Space between left column border and action list.
#define COLUMN_LIST_RIGHT_SPACING 42   // Dito for right border.
#define COLUMN_LEFT_LINK_SPACING 14    // Horizontal distance between left border and action links.
#define COLUMN_RIGHT_LINK_SPACING 14   // Dito for right border.

Workspace::Workspace(HomeScreen* owner)
: Box(true), _left_column(false), _middle_column(false), _right_column(false)
{
  _owner= owner;

  suspend_layout();
  set_back_color("#f4f7fc");

  _top_gradient= new WorkspaceGradient();
  _top_gradient->add_stop_color(0, 0xffffff);
  _top_gradient->add_stop_color(0.4, 0xf4f7fc);
  _top_gradient->add_stop_color(0.5, 0xdde6f3);
  _top_gradient->add_stop_color(0.7, 0xf4f7fc);
  _top_gradient->add_stop_color(1, 0xf4f7fc);
  
  _separator_gradient= new WorkspaceGradient();
  _separator_gradient->add_stop_color(0, 0xf4f7fc);
  _separator_gradient->add_stop_color(0.44, 0xf4f7fc);
  _separator_gradient->add_stop_color(1, 0xe5ebf5);
  
  // SQL column data.
  _sql_top = new ActionLinkBox(_owner, "home_header_sql.png", _top_gradient, COLUMN_TOP_SPACE, COLUMN_TOP_LIST_SPACING,
    COLUMN_LEFT_LINK_SPACING, COLUMN_RIGHT_LINK_SPACING);
  _sql_top->add_link("action_query.png", "Open Connection to start Querying",
                     "Or click a DB connection to open the SQL Editor.", grt::ValueRef(),
                     ActionOpenConnection, true);
  
  _sql_bottom= new ActionLinkBox(_owner, "", NULL, COLUMN_BOTTOM_LIST_SPACING, COLUMN_BOTTOM_SPACE,
    COLUMN_LEFT_LINK_SPACING, COLUMN_RIGHT_LINK_SPACING);
  _sql_bottom->add_link("action_new_connection.png", "New Connection", 
                        "Add a new database connection for querying.", grt::ValueRef(),
                        ActionNewConnection, true);
  _sql_bottom->add_link("action_edit_table_data.png", "Edit Table Data",
                        "Select a connection and schema table to edit.", grt::ValueRef(),
                        ActionEditTable, true);
  _sql_bottom->add_link("action_edit_sql_script.png", "Edit SQL Script",
                        "Open an existing SQL Script file for editing.", grt::ValueRef(),
                        ActionEditSQLScript, true);
  _sql_bottom->add_link("action_manage_connection.png", "Manage Connections",
                        "Modify connection settings or add connections.", grt::ValueRef(),
                        ActionManageConnections, true);
  _sql_bottom->set_back_color(0xf4f7fc);
  
  _connections= new ActionList(_owner, COLUMN_LIST_LEFT_SPACING, COLUMN_LIST_RIGHT_SPACING, NULL,
    "db_connection_34x34.png", ActionOpenConnectionFromList);
  _connections->set_spacer_back_color(0xf4f7fc);
  
  _left_column.set_back_color("#f4f7fc");
  _left_column.add(_sql_top, false, true);
  _left_column.add(_connections, true, true);
  _left_column.add_end(_sql_bottom, false, true);

  // EER column data.
  _eer_top= new ActionLinkBox(_owner, "home_header_eer.png", _top_gradient, COLUMN_TOP_SPACE, COLUMN_TOP_LIST_SPACING,
    COLUMN_LEFT_LINK_SPACING, COLUMN_RIGHT_LINK_SPACING);
  _eer_top->add_link("action_eer_open.png", "Open Existing EER Model",
                     "Or select a model to open or click here to browse.", grt::ValueRef(),
                     ActionOpenEERModel, true);
  
  _eer_bottom= new ActionLinkBox(_owner, "", NULL, COLUMN_BOTTOM_LIST_SPACING, COLUMN_BOTTOM_SPACE,
    COLUMN_LEFT_LINK_SPACING, COLUMN_RIGHT_LINK_SPACING);
  _eer_bottom->set_back_color(0xe5ebf5);
  _eer_bottom->add_link("action_new_eer.png", "Create new EER Model",
                        "Create a new EER Model from scratch.", grt::ValueRef(),
                        ActionNewEERModel, true);
  _eer_bottom->add_link("action_eer_rev_eng_db.png", "Create EER Model from Existing Database",
                        "Create by connecting and reverse engineering.", grt::ValueRef(),
                        ActionNewModelFromDB, true);
  _eer_bottom->add_link("action_eer_rev_eng_file.png", "Create EER Model from SQL Script",
                        "Import an existing SQL file.", grt::ValueRef(),
                        ActionNewModelFromScript, true);
  
  _models= new ActionList(_owner, COLUMN_LIST_LEFT_SPACING, COLUMN_LIST_RIGHT_SPACING, _separator_gradient,
    "db_model_34x34.png", ActionOpenEERModelFromList);

  _middle_column.set_back_color("#f4f7fc");
  _middle_column.add(_eer_top, false, true);
  _middle_column.add(_models, true, true);
  _middle_column.add_end(_eer_bottom, false, true);

  // Admin column data.
  _admin_top= new ActionLinkBox(_owner, "home_header_admin.png", _top_gradient, COLUMN_TOP_SPACE, COLUMN_TOP_LIST_SPACING,
    COLUMN_LEFT_LINK_SPACING, COLUMN_RIGHT_LINK_SPACING);
  _admin_top->add_link("action_admin.png", "Server Administration",
                       "Or click to manage a database server instance.", grt::ValueRef(),
                       ActionServerAdministration, true);
  
  _admin_bottom= new ActionLinkBox(_owner, "", NULL, COLUMN_BOTTOM_LIST_SPACING, COLUMN_BOTTOM_SPACE,
    COLUMN_LEFT_LINK_SPACING, COLUMN_RIGHT_LINK_SPACING);
  _admin_bottom->add_link("action_new_server.png", "New Server Instance",
                          "Register a new server instance to manage.", grt::ValueRef(),
                          ActionNewServerInstance, true);
  _admin_bottom->add_link("action_import_export.png", "Manage Import / Export",
                          "Create a dump file or restore data from a file.", grt::ValueRef(),
                          ActionDumpRestore, true);
  _admin_bottom->add_link("action_security.png", "Manage Security",
                          "Manage user accounts and assign privileges.", grt::ValueRef(),
                          ActionManageSecurity, true);
  _admin_bottom->add_link("action_manage_servers.png", "Manage Server Instances",
                          "Add, delete and update server instance settings.", grt::ValueRef(),
                          ActionManageServerInstances, true);
  _admin_bottom->set_back_color(0xf4f7fc);
  
  _server_instances= new ActionList(_owner, COLUMN_LIST_LEFT_SPACING, COLUMN_LIST_RIGHT_SPACING, NULL,
    "db_server_34x34.png", ActionManageInstanceFromList);
  _server_instances->set_spacer_back_color(0xf4f7fc);
  
  _right_column.set_back_color("#f4f7fc");
  _right_column.add(_admin_top, false, true);
  _right_column.add(_server_instances, true, true);
  _right_column.add_end(_admin_bottom, false, true);
  
  add(&_left_column, true, true);
  add(&_middle_column, true, true);
  add(&_right_column, true, true);
  
  resume_layout();
}

//--------------------------------------------------------------------------------------------------

Workspace::~Workspace()
{
  delete _sql_top;
  delete _sql_bottom;
  delete _eer_top;
  delete _eer_bottom;
  delete _admin_top;
  delete _admin_bottom;
  delete _top_gradient;
  delete _separator_gradient;
  
  delete _connections;
  delete _models;
  delete _server_instances;
}

//--------------------------------------------------------------------------------------------------

void Workspace::add_list_entry(HomeScreenListType list, const std::string& title, const std::string& description, const grt::ValueRef &object)
{
  switch (list)
  {
    case HomeScreenSQLDevList:
      _connections->add_entry(title, description, object);
      break;
    case HomeScreenModellingList:
      _models->add_entry(title, description, object);
      break;
    case HomeScreenAdministrationList:
      _server_instances->add_entry(title, description, object);
      break;
  }
}

//--------------------------------------------------------------------------------------------------

void Workspace::select_list_entry(HomeScreenListType list, int index)
{
  switch (list)
  {
    case HomeScreenSQLDevList:
      _connections->select(index);
      break;
    case HomeScreenModellingList:
      _models->select(index);
      break;
    case HomeScreenAdministrationList:
      _server_instances->select(index);
      break;
  }  
}

//--------------------------------------------------------------------------------------------------

void Workspace::clear_list(HomeScreenListType list)
{
  switch (list)
  {
    case HomeScreenSQLDevList:
      _connections->clear();
      break;
    case HomeScreenModellingList:
      _models->clear();
      break;
    case HomeScreenAdministrationList:
      _server_instances->clear();
      break;
  }
}

//----------------- HomeScreen ---------------------------------------------------------------------

HomeScreen::HomeScreen()
: AppView(false, "home", true), _workbench_central_section(true, "Workbench Central", true),
_workspace_section(false, "Workspace"), _callback(NULL)
{
  // TODO: cannot set a minimum size here as this doesn't work on OS X (box layouter is buggy).
  //set_size(1024, 671); // Minimum size for this entire page.
  _central_content= new WorkbenchCentral(this);
  _central_content->set_size(-1, 112);
  _workbench_central_section.set_content(_central_content);
  add(&_workbench_central_section, false, true);

  //_workspace_content.set_size(10, 10);
  _workspace_content= new Workspace(this);
  _workspace_content->set_name("workspace content");
  _workspace_section.set_content(_workspace_content);
  _workspace_section.set_name("workspace section");
  add(&_workspace_section, true, true);
}

//--------------------------------------------------------------------------------------------------

HomeScreen::~HomeScreen()
{
  delete _central_content;
  delete _workspace_content;
}

//--------------------------------------------------------------------------------------------------

void HomeScreen::set_callback(action_callback callback, void* user_data)
{
  _callback= callback;
  _user_data= user_data;
}

//--------------------------------------------------------------------------------------------------

void HomeScreen::trigger_callback(HomeScreenAction action, const grt::ValueRef &object)
{
  if (_callback != NULL)
    (*_callback)(action, object, _user_data);
}

//--------------------------------------------------------------------------------------------------

/**
 * Adds a new entry to the specified list on the home screen.
 */
void HomeScreen::add_list_entry(HomeScreenListType list, const std::string& title, const std::string& description, const grt::ValueRef &object)
{
  _workspace_content->add_list_entry(list, title, description, object);
}

//--------------------------------------------------------------------------------------------------

void HomeScreen::select_list_entry(HomeScreenListType list, int index)
{
  _workspace_content->select_list_entry(list, index);
}

//--------------------------------------------------------------------------------------------------

void HomeScreen::clear_list(HomeScreenListType list)
{
  _workspace_content->clear_list(list);
}

//--------------------------------------------------------------------------------------------------

