/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#ifndef _HOME_SCREEN_H_
#define _HOME_SCREEN_H_

#include "wb_backend_public_interface.h"
#include "mforms/mforms.h"

#include "grtpp.h"

#ifdef __APPLE__
  #define HOME_NORMAL_FONT "Lucida Grande"
  #define HOME_DETAILS_FONT "Lucida Grande"
  #define ACTION_LINK_NORMAL_FONT "Tahoma"
#elif _WIN32
  #define HOME_NORMAL_FONT "Tahoma"
  #define HOME_DETAILS_FONT "Arial"
  #define ACTION_LINK_NORMAL_FONT "Tahoma"
#else
  #define HOME_NORMAL_FONT "Helvetica"
  #define HOME_DETAILS_FONT "Helvetica"
  #define ACTION_LINK_NORMAL_FONT "Tahoma"
#endif

#define HOME_NORMAL_FONT_SIZE 13
#define HOME_DETAILS_FONT_SIZE 11

#define ACTION_BUTTON_FONT_SIZE 12
#define ACTION_LINK_NORMAL_FONT_SIZE 12
#define ACTION_LINK_DETAILS_FONT_SIZE 11

namespace wb
{
  /**
   * Flag to notify observers which action was triggered on the home screen.
   */
  enum HomeScreenAction
  {
    ActionNone,

    ActionWhatsNew,
    ActionBlogs,
    ActionCheckForUpdates,
    ActionSubmitBugReport,
    ActionDiscussTopic,

    ActionOpenConnection,
    ActionOpenConnectionFromList,
    ActionNewConnection,
    ActionAddSchemaObject,
    ActionEditTable,
    ActionEditSQLScript,
    ActionManageConnections,

    ActionOpenEERModel,
    ActionOpenEERModelFromList,
    ActionNewEERModel,
    ActionNewModelFromDB,
    ActionNewModelFromScript,

    ActionServerAdministration,
    ActionManageInstanceFromList,
    ActionNewServerInstance,
    ActionDumpRestore,
    ActionManageSecurity,
    ActionManageServerInstances
  };

  /**
   * Determines to which list an item must be added.
   */
  enum HomeScreenListType
  {
    HomeScreenSQLDevList,
    HomeScreenModellingList,
    HomeScreenAdministrationList
  };

  typedef void (*action_callback)(HomeScreenAction action, const grt::ValueRef &object, void* user_data);

  class HomeScreen;

  /**
   * The upper part of the home screen.
   */
  class WorkbenchCentral : public mforms::DrawBox
  {
  public:
    WorkbenchCentral(HomeScreen* owner);
    ~WorkbenchCentral();
    
    virtual void repaint(cairo_t *cr, int areax, int areay, int areaw, int areah);
  protected:
    void draw_button(cairo_t* cr, double x, double y);
    void layout(cairo_t* cr);
    virtual void mouse_click(int button, int x, int y);
    HomeScreenAction action_from_point(int x, int y);
    virtual void mouse_leave();
    virtual void mouse_move(int x, int y);
  private:
    HomeScreen* _owner;
    bool _layout_dirty;
    cairo_surface_t* _main_icon;
    cairo_surface_t* _pointer_icon;
    cairo_surface_t* _button_parts[3];
    cairo_surface_t* _title_image;
    MySQL::Geometry::Rect _title_bounds;
    MySQL::Geometry::Rect _chapter1_title_bounds;
    MySQL::Geometry::Rect _chapter1_details_bounds;
    MySQL::Geometry::Rect _arrow1_bounds;
    MySQL::Geometry::Rect _chapter2_title_bounds;
    MySQL::Geometry::Rect _chapter2_details_bounds;
    MySQL::Geometry::Rect _arrow2_bounds;
    
    MySQL::Geometry::Rect _button1_bounds;
    MySQL::Geometry::Rect _button2_bounds;
    MySQL::Geometry::Rect _button3_bounds;
    cairo_pattern_t* _button_pattern;
    int _button_inner_width;                 // Precomputed width for the tile pattern.
    int _button1_text_left;
    int _button1_text_top;
    int _button2_text_left;
    int _button2_text_top;
    int _button3_text_left;
    int _button3_text_top;

    bool _title1_hot;
    bool _title2_hot;
  };
  
  /**
   * The ActionLink class is a lean wrapper for an icon/label combination which can be used
   * to trigger an action. It needs a container to be useful (here the class Workspace).
   */
  class ActionLink
  {
  public:
    ActionLink(const std::string& icon_name, const std::string& title, const std::string& description,
               const grt::ValueRef &object, HomeScreenAction action, bool enabled);
    ~ActionLink();
    
    void paint(cairo_t* cr, MySQL::Geometry::Rect bounds, bool hot, bool active);
    bool contains(double x, double y);
    HomeScreenAction get_action() { return _action; };
    grt::ValueRef get_object() { return _object; }
    bool enabled() { return _enabled; };
  protected:
    void layout(cairo_t* cr);
  private:
    cairo_surface_t* _icon;
    std::string _title;
    std::string _description;
    std::string _shorted_title;        // Contains the description shorted and with ellipses if the
                                       // full description doesn't fit into the available space.
    std::string _shorted_description;
    double _last_text_width;           // The last width for which the shorted description has been
                                       // computed. Used to avoid unnecessary re-computation.
    double _title_offset;              // Vertical position of the title.
    double _description_offset;        // Dito for description.
    double _title_width;               // Width of the (possibly shorted) title. For text decoration.
    MySQL::Geometry::Rect _bounds;     // The link's bounds when it was drawn the last time.
    int _text_height;
    HomeScreenAction _action;          // Which action does this link trigger?
    grt::ValueRef _object;
    bool _enabled;                     // Draw the button in enabled or disabled state.
  };
  
  /**
   * Keeps a list of stop colors along with the position for later cairo gradient construction.
   */
  class WorkspaceGradient
  {
  public:
    void add_stop_color(double position, double r, double g, double b, double a);
    void add_stop_color(double position, int color);
    cairo_pattern_t* create_pattern(int height);
  private:
    struct StopColor {
      double position;
      double red;
      double green;
      double blue;
      double alpha;
    };
    
    std::vector<StopColor> _values;
  };
  
  /**
   * Container for any number of ActionLink instances.
   */
  class ActionLinkBox : public mforms::DrawBox
  {
  public:
    ActionLinkBox(HomeScreen* owner, const std::string& header_image, WorkspaceGradient* gradient,
      int top_spacing, int bottom_spacing, int left_spacing, int right_spacing);
    ~ActionLinkBox();
    
    void add_link(const std::string& icon_name, const std::string& title, const std::string& description,
                  const grt::ValueRef &object, HomeScreenAction action, bool enabled);
    void clear();
    void set_back_color(double red, double green, double blue);
    void set_back_color(int color);
    void can_select(bool selectable);
    void single_click(bool value);
    void select(int index);

    virtual void repaint(cairo_t *cr, int areax, int areay, int areaw, int areah);
    virtual void mouse_enter();
    virtual void mouse_leave();
    virtual void mouse_move(int x, int y);
    virtual void mouse_down(int button, int x, int y);
    virtual void mouse_up(int button, int x, int y);
    virtual void mouse_click(int button, int x, int y);
    virtual void mouse_double_click(int button, int x, int y);
    virtual void get_layout_size(int* w, int* h);
  protected:
    void layout();
    ActionLink* action_link_from_point(double x, double y);
    void set_selected(ActionLink* link);
  private:
    HomeScreen* _owner;
    bool _layout_dirty;
    WorkspaceGradient* _gradient;
    cairo_pattern_t* _background;
    cairo_surface_t* _image;
    std::vector<ActionLink*> _links;
    int _layout_width;
    int _layout_height;
    int _top_spacing;
    int _bottom_spacing;
    int _left_spacing;
    int _right_spacing;
    double _back_red;
    double _back_green;
    double _back_blue;
    bool _can_select;
    bool _single_click;

    ActionLink* _hot_link;
    ActionLink* _selected_link;
  };
  
  /**
   * Special design element used separate columns visually.
   */
  class GradientBox : public mforms::DrawBox
  {
  public:
    GradientBox(WorkspaceGradient* gradient);
    ~GradientBox();

    void set_back_color(double red, double green, double blue);
    void set_back_color(int color);
    virtual void repaint(cairo_t *cr, int areax, int areay, int areaw, int areah);
  private:
    WorkspaceGradient* _gradient;
    double _back_red;
    double _back_green;
    double _back_blue;
  };

  /**
   * Implements a collection of action links forming a scrollable list.
   */
  class ActionList : public mforms::Box
  {
  public:
    ActionList(HomeScreen* owner, int left_spacing, int right_spacing, WorkspaceGradient* spacer_gradient,
      const std::string& icon_name, HomeScreenAction action);
    ~ActionList();

    void set_spacer_back_color(int color);
    void add_entry(const std::string& title, const std::string& description, const grt::ValueRef &object);
    void select(int index);
    void clear();
  private:
    HomeScreen* _owner;
    Box _panel_border;
    mforms::ScrollPanel _panel;
    GradientBox* _left_spacer;
    GradientBox* _right_spacer;
    ActionLinkBox* _link_box;
    std::string _icon_name; // The icon to be used for all entries.
    HomeScreenAction _action;      // One action for all.
  };
  
  /**
   * The lower part of the home screen.
   */
  class Workspace : public mforms::Box
  {
  public:
    Workspace(HomeScreen* owner);
    ~Workspace();
    
    void add_list_entry(HomeScreenListType list, const std::string& title, const std::string& description, const grt::ValueRef &object);
    void select_list_entry(HomeScreenListType list, int index);
    void clear_list(HomeScreenListType list);
  protected:
    ActionLink* action_link_from_point(double x, double y);
  private:
    HomeScreen* _owner;
    mforms::Box _left_column;
    mforms::Box _middle_column;
    mforms::Box _right_column;
    
    ActionLinkBox* _sql_top;
    ActionLinkBox* _sql_bottom;
    ActionLinkBox* _eer_top;
    ActionLinkBox* _eer_bottom;
    ActionLinkBox* _admin_top;
    ActionLinkBox* _admin_bottom;
    
    ActionList* _connections;
    ActionList* _models;
    ActionList* _server_instances;

    WorkspaceGradient* _top_gradient;
    WorkspaceGradient* _separator_gradient;
  };
  
  /**
   * This class implements the main (home) screen in MySQL Workbench, containing
   * workbench central and the workspace sections.
   */
  class MYSQLWBBACKEND_PUBLIC_FUNC HomeScreen : public mforms::AppView
  {
  private:
    mforms::SectionBox _workbench_central_section;
    WorkbenchCentral* _central_content;
    mforms::SectionBox _workspace_section;
    Workspace* _workspace_content;
    action_callback _callback;
    void* _user_data;
  public:
    HomeScreen();
    ~HomeScreen();

    void set_callback(action_callback callback, void* user_data);
    void trigger_callback(HomeScreenAction action, const grt::ValueRef &object);
    void add_list_entry(HomeScreenListType list, const std::string& title, const std::string& description, const grt::ValueRef &object);
    void select_list_entry(HomeScreenListType list, int index);
    void clear_list(HomeScreenListType list);
  };
}

#endif // _HOME_SCREEN_H_