/* 
 * Copyright © 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "stdafx.h"

#include <glib/gstdio.h> // g_stat

#include "wb_context_ui.h"
#include "model/wb_component.h"
#include "model/wb_component_physical.h"

#include "model/wb_model_diagram_form.h"

#include "wb_overview.h"
#include "model/wb_overview_physical.h"
#include "workbench/wb_history_tree.h"
#include "wb_find_dialog.h"
#include "model/wb_user_datatypes.h"
#include "model/wb_diagram_options.h"

#include "grt/editor_base.h"
#include "grt/clipboard.h"
#include "grtui/gui_plugin_base.h"

#include "server_instance_editor.h"
#include "grtui/grtdb_connection_editor.h"
#include "new_server_instance_wizard.h"
#include "new_connection_wizard.h"
#include "edit_sql_script_wizard.h"
#include "edit_table_data_wizard.h"
#include "grt_shell_window.h"
#include "output_view.h"

#include "mdc_back_layer.h"

#include "mforms/utilities.h"
#include "mforms/app.h"
#include "string_utilities.h"

using namespace wb;
using namespace bec;
using namespace base;


WBContextUI::WBContextUI(bool verbose)
  : _wb(new WBContext(this, verbose)), _command_ui(_wb), _usertypes_tree(_wb)
{
  NodeId n; // workaround a bug causing crash when output tab is shown with GLib-ERROR **: The thread system is not yet initialized.


  _catalog_tree= 0;

  _history_tree= 0;

  _shell_window= 0;
  
  _find_dialog= new FindDialogBE(_wb);
  _find_dialog->set_activate_slot(sigc::mem_fun(this, &WBContextUI::activate_find_result));

  _output_view= new OutputView(_wb->get_grt_manager()->get_messages_list());
  _output_view->get_be()->signal_show().connect(sigc::mem_fun(this, &WBContextUI::show_output));
    
  _active_form= 0;
  _active_main_form= 0;
  
  _last_unsaved_changes_state= false;
  _quitting= false;
  
  _home_screen= NULL;

  // to notify that the save status of the doc has changed
  _wb->get_grt()->get_undo_manager()->signal_changed().connect(sigc::mem_fun(this, &WBContextUI::history_changed));
  
  // stuff to do when the active form is switched in the UI (through set_active_form)
  _form_change_signal.connect(sigc::hide(sigc::mem_fun(this, &WBContextUI::form_changed)));
}


WBContextUI::~WBContextUI()
{
  _wb->do_close_document(true);

  delete _find_dialog;
  delete _history_tree;
  delete _home_screen;
  delete _shell_window;

  delete _wb;
}

//--------------------------------------------------------------------------------------------------

bool WBContextUI::init(WBFrontendCallbacks *callbacks, WBOptions *options)
{
  _shell_window = new GRTShellWindow(_wb->get_grt_manager());

  bool flag= _wb->init_(callbacks, options);

  // has to be called after WBContext is initialized
  add_backend_builtin_commands();
  
  // Finally show initial home screen.
  show_home_screen();
  
  return flag;
}

//--------------------------------------------------------------------------------------------------

GRTShellWindow* WBContextUI::get_shell_window()
{
  return _shell_window;
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::init_finish(WBOptions *options)
{  
  _wb->init_finish_(options);
}


void WBContextUI::finalize()
{  
  _wb->finalize();
}

bool WBContextUI::request_quit()
{ 
  if (_quitting)
    return true;
  
  if (!_wb->get_grt_manager()->in_main_thread())
    g_warning("request_quit() called in worker thread");

  if (_wb->has_unsaved_changes())
  {
    int answer= _wb->confirm_action(_("Quit"),
                                    _("Do you want to save pending changes to the document?\n\n"
                                      "If you don't save your changes, they will be lost."),
                                    _("Save"), _("Don't Save"), _("Cancel"));
    if (answer == 1)
    {
      if (!_wb->save_changes())
        return false;
    }
    else if (answer == -1)
      return false;
  }

  return true;
}


void WBContextUI::perform_quit()
{
  _quitting= true;
  _wb->quit_application();
}


void WBContextUI::refresh_toolbar(const std::string &toolbar)
{
  _wb->request_refresh(RefreshToolbar, toolbar, 0);
}


void WBContextUI::refresh_menus()
{
  _wb->request_refresh(RefreshMenubar, "", 0);
}


void WBContextUI::reset()
{
  if (!dynamic_cast<OverviewBE*>(_active_form))
    _active_form= 0;
  if (!dynamic_cast<OverviewBE*>(_active_main_form))
    _active_main_form= 0;
  
  get_physical_overview()->signal_selection_changed().connect(sigc::mem_fun(this, &WBContextUI::overview_selection_changed));
  
  get_physical_overview()->set_model(_wb->get_document()->physicalModels()[0]);
  
  _wb->request_refresh(RefreshSelection, "", 0);

  get_physical_overview()->send_refresh_children(bec::NodeId());
  
  _wb->get_component<WBComponentPhysical>()->update_catalog_tree();

  _usertypes_tree.set_catalog(_wb->get_document()->physicalModels()[0]->catalog());
  _usertypes_tree.set_rdbms(_wb->get_document()->physicalModels()[0]->rdbms());
}


void WBContextUI::history_changed()
{  
  if (_wb->has_unsaved_changes() != _last_unsaved_changes_state)
    _wb->request_refresh(RefreshDocument, "", (NativeHandle)0);

  _last_unsaved_changes_state= _wb->has_unsaved_changes();

  // Signal that the main menu and the main toolbar should be refreshed.
  _wb->request_refresh(RefreshMenubar, "", (NativeHandle)0);
  _wb->request_refresh(RefreshToolbar, "main", (NativeHandle)0);
}

void WBContextUI::update_current_diagram(bec::UIForm *form)
{
  ModelDiagramForm* dform= dynamic_cast<ModelDiagramForm*>(form);
  if (dform)
  {
    model_DiagramRef diagram(dform->get_model_diagram());
    if (diagram.is_valid() && diagram->owner().is_valid())
      diagram->owner()->currentDiagram(diagram);
  }
}

void WBContextUI::overview_selection_changed()
{
  if (get_active_main_form() == get_physical_overview())
  {
    _wb->request_refresh(RefreshSelection, "", (NativeHandle)get_physical_overview()->get_frontend_data());

    _wb->request_refresh(RefreshMenubar, "");
  }
}


void WBContextUI::view_selection_changed(bool, mdc::CanvasItem *item, ModelDiagramForm *view)
{
  if (get_active_main_form() == view)
  {
///    if (_layer_tree)
///      _layer_tree->refresh_selection_status();

    _wb->request_refresh(RefreshMenubar, "");
    _wb->request_refresh(RefreshSelection, "", 0);
  }
}



void WBContextUI::load_app_options(bool update)
{
  if (!update)
    _command_ui.load_data();
}


std::string WBContextUI::get_diagram_path(const model_DiagramRef &diagram)
{
  std::string view_id= diagram.id();

  // look in the physicalModels list for this diagram
  for (size_t c= _wb->get_document()->physicalModels().count(), i= 0; i < c; i++)
  {
    workbench_physical_ModelRef model= _wb->get_document()->physicalModels()[i];
    
    for (size_t d= model->diagrams().count(), j= 0; j < d; j++)
    {
      if (model->diagrams().get(j).id() == view_id)
        return strfmt("/wb/doc/physicalModels/%i/diagrams/%i", (int) i, (int) j);
    }
  }
  return "";
}


static void add_script_file(WBContextUI *wbui)
{
  std::string file= wbui->get_wb()->show_file_dialog("open", _("Add SQL Script File"), "sql");
  if (!file.empty())
  {
    workbench_physical_ModelRef model;

    model= workbench_physical_ModelRef::cast_from(wbui->get_wb()->get_model_context()->get_active_model(false));
    if (model.is_valid())
      wbui->get_wb()->get_component<WBComponentPhysical>()->add_new_stored_script(model, file);
  }
}


static void add_note_file(WBContextUI *wbui)
{
  std::string file= wbui->get_wb()->show_file_dialog("open", _("Add Note File"), "Text Files (*.txt)|*.txt");
  if (!file.empty())
  {
    workbench_physical_ModelRef model;

    model= workbench_physical_ModelRef::cast_from(wbui->get_wb()->get_model_context()->get_active_model(false));
    if (model.is_valid())
      wbui->get_wb()->get_component<WBComponentPhysical>()->add_new_stored_note(model, file);
  }
}


void WBContextUI::call_undo()
{
  _wb->execute_in_grt_thread("Undo",
    sigc::bind_return<grt::ValueRef>(sigc::hide(sigc::mem_fun(_wb->get_grt()->get_undo_manager(),
                                  &grt::UndoManager::undo)), grt::ValueRef()));
}


void WBContextUI::call_redo()
{
  _wb->execute_in_grt_thread("Redo",
    sigc::bind_return<grt::ValueRef>(sigc::hide(sigc::mem_fun(_wb->get_grt()->get_undo_manager(),
                                  &grt::UndoManager::redo)), grt::ValueRef()));
}


/** builtin: commands for use in menus and toolbars that are handled by ourselves
  */
void WBContextUI::add_backend_builtin_commands()
{
  _command_ui.add_builtin_command("overview.home", 
                                  sigc::mem_fun(this, &WBContextUI::show_home_screen));
  _command_ui.add_builtin_command("show_output_form", 
                                  sigc::mem_fun(this, &WBContextUI::show_output));

  _command_ui.add_builtin_command("find_next", 
    sigc::hide_return(sigc::mem_fun(_find_dialog, &FindDialogBE::find_next)));
  _command_ui.add_builtin_command("find_previous", 
    sigc::hide_return(sigc::mem_fun(_find_dialog, &FindDialogBE::find_previous)));

  _command_ui.add_builtin_command("add_script_file", 
    sigc::bind<WBContextUI*>(sigc::ptr_fun(add_script_file), this));
  _command_ui.add_builtin_command("add_note_file", 
    sigc::bind<WBContextUI*>(sigc::ptr_fun(add_note_file), this));


  _command_ui.add_builtin_command("undo", 
    sigc::mem_fun(this, &WBContextUI::call_undo),
    sigc::mem_fun(_wb->get_grt()->get_undo_manager(), &grt::UndoManager::can_undo));
  _command_ui.add_builtin_command("redo",
    sigc::mem_fun(this, &WBContextUI::call_redo),
    sigc::mem_fun(_wb->get_grt()->get_undo_manager(), &grt::UndoManager::can_redo));
  
  // handle some url opening stuff ourselves through mforms
  _command_ui.add_builtin_command("community_blog",
    sigc::bind(sigc::ptr_fun(mforms::Utilities::open_url),"http://wb.mysql.com/"));
  _command_ui.add_builtin_command("community_faq",
    sigc::bind(sigc::ptr_fun(mforms::Utilities::open_url),"http://wb.mysql.com/?page_id=7"));
  _command_ui.add_builtin_command("community_learn",
    sigc::bind(sigc::ptr_fun(mforms::Utilities::open_url),"http://wb.mysql.com/?page_id=10"));
  _command_ui.add_builtin_command("community_discuss",
    sigc::bind(sigc::ptr_fun(mforms::Utilities::open_url),"http://wb.mysql.com/?page_id=16"));
  _command_ui.add_builtin_command("community_contribute",
    sigc::bind(sigc::ptr_fun(mforms::Utilities::open_url),"http://wb.mysql.com/?page_id=17"));
  _command_ui.add_builtin_command("web_mysql_home",
    sigc::bind(sigc::ptr_fun(mforms::Utilities::open_url),"http://mysql.com/"));
  _command_ui.add_builtin_command("web_home",
    sigc::bind(sigc::ptr_fun(mforms::Utilities::open_url),"http://mysql.com/products/tools/workbench/"));
  _command_ui.add_builtin_command("report_a_bug",
    sigc::bind(sigc::ptr_fun(mforms::Utilities::open_url),"http://bugs.mysql.com/"));
  _command_ui.add_builtin_command("list_bugs",
    sigc::bind(sigc::ptr_fun(mforms::Utilities::open_url),"http://bugs.mysql.com/saved/WB%20bugs"));
  
/* moved to frontend
  _command_ui.add_builtin_form_command("cut", 
    sigc::mem_fun(&bec::UIForm::cut), 
    sigc::mem_fun(&bec::UIForm::can_cut), false);
  _command_ui.add_builtin_form_command("copy", 
    sigc::mem_fun(&bec::UIForm::copy), 
    sigc::mem_fun(&bec::UIForm::can_copy), 
    false);
  _command_ui.add_builtin_form_command("paste", 
    sigc::mem_fun(&bec::UIForm::paste), 
    sigc::mem_fun(&bec::UIForm::can_paste), 
    false);
  _command_ui.add_builtin_form_command("delete", 
    sigc::mem_fun(&bec::UIForm::delete_selection), 
    sigc::mem_fun(&bec::UIForm::can_delete), 
    false);
  _command_ui.add_builtin_form_command("selectAll", 
                                       sigc::mem_fun(&bec::UIForm::select_all), 
                                       sigc::hide(sigc::ptr_fun(return_true)),
                                       false);  
 */
}


#ifndef ___specialforms

//-----------------------------------------------------------------------------------
// special forms


bec::ValueTreeBE *WBContextUI::get_catalog_tree()
{
  if (!_catalog_tree)
  {
    WBComponentPhysical *compo= _wb->get_component<wb::WBComponentPhysical>();
    _catalog_tree= compo->get_catalog_tree();
  }
  return _catalog_tree;
}


UserDatatypeListBE *WBContextUI::get_usertypes_tree()
{
  return &_usertypes_tree;
}


PhysicalOverviewBE *WBContextUI::get_physical_overview()
{
  return get_wb()->get_model_context() ? get_wb()->get_model_context()->get_overview() : 0;
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::show_output()
{
  mforms::App::get()->dock_view(_output_view, "maintab");
  mforms::App::get()->set_view_title(_output_view, "Output");
}

//--------------------------------------------------------------------------------------------------

/**
 * Helper method to construct a human-readable server description.
 */
std::string get_server_info(db_mgmt_ServerInstanceRef instance)
{
  std::string text;
  std::string host = instance->connection().is_valid() ?
    instance->connection()->parameterValues().get_string("hostName") : "Invalid";

  if (instance->serverInfo().get_int("remoteAdmin"))
    text = strfmt("Host: %s  Type: %s", instance->loginInfo().get_string("ssh.hostName").c_str(),
      instance->serverInfo().get_string("sys.system").c_str());
  else
    if (host == "localhost" || host.empty() || host == "127.0.0.1")
      text = strfmt("Local  Type: %s", instance->serverInfo().get_string("sys.system").c_str());
    else
      text = strfmt("Host: %s  Type: DB Only", host.c_str());

  return text;
}

//--------------------------------------------------------------------------------------------------

static bool on_home_closing(WBContextUI* owner)
{
  owner->home_screen_closing();
  return true;
}

//--------------------------------------------------------------------------------------------------

/**
 * Creates the main home screen (Workbench Central, Workspace) if not yet done and docks it to 
 * the main application window.
 */
void WBContextUI::show_home_screen()
{
  if (_home_screen == NULL)
  {
    _home_screen= mforms::manage(new HomeScreen());
    _home_screen->set_on_close(sigc::bind(sigc::ptr_fun(on_home_closing), this));
    _home_screen->set_callback((action_callback)&WBContextUI::_handle_home_action, this);
  }
  mforms::App::get()->dock_view(_home_screen, "maintab");
  mforms::App::get()->set_view_title(_home_screen, "Home");
  
  refresh_home_models();
  refresh_home_instances(0);
  refresh_home_connections();  
}

//--------------------------------------------------------------------------------------------------

/**
 * Called when the home screen is closed by the UI. We have to clear our reference then.
 */
void WBContextUI::home_screen_closing()
{
  _home_screen= NULL;
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::_handle_home_action(HomeScreenAction action, const grt::ValueRef &object, WBContextUI *self)
{
  self->handle_home_action(action, object);
}

void WBContextUI::handle_home_action(HomeScreenAction action, const grt::ValueRef &object)
{
  switch (action)
  {
      case ActionNone:
        break;
      case ActionWhatsNew:
        {
          GUILock lock(_wb, _("Opening 'What's New' web page"), _("The MySQL Workbench news web page is being opened "
            "and should be available in a moment.\n\nPlease stand by..."));
          mforms::Utilities::open_url("http://wb.mysql.com/?page_id=49");
        }
        break;
      case ActionBlogs:
        {
          GUILock lock(_wb, _("Opening Workbench Blog site"), _("The MySQL Workbench Blog is being opened "
            "and should be available in a moment.\n\nPlease stand by..."));
          mforms::Utilities::open_url("http://wb.mysql.com/");
        }
        break;
      case ActionCheckForUpdates:
        {
          GUILock lock(_wb, _("Opening Workbench Blog site"), _("The MySQL Workbench Blog is being opened "
            "and should be available in a moment.\n\nPlease stand by..."));
          mforms::Utilities::open_url("http://wb.mysql.com/");
        }
        break;
      case ActionSubmitBugReport:
        {
          GUILock lock(_wb, _("Opening MySQL Bug Reporting Site"), _("The MySQL bug tracker is being opened "
            "and should be available in a moment.\n\nPlease stand by..."));
          mforms::Utilities::open_url("http://bugs.mysql.com/");
        }
        break;
      case ActionDiscussTopic:
        {
          GUILock lock(_wb, _("Opening Workbench Forums"), _("The MySQL Workbench forums are being opened "
            "and should be available in a moment.\n\nPlease stand by..."));
          mforms::Utilities::open_url("http://forums.mysql.com/index.php?151");
        }
        break;

      case ActionOpenConnection:
        {
          _wb->show_status_text("Opening SQL Editor...");
          _wb->add_new_query_window(db_mgmt_ConnectionRef());
          break;
        }

      case ActionOpenConnectionFromList:
        {
          if (object.is_valid())
          {
            db_mgmt_ConnectionRef connection(db_mgmt_ConnectionRef::cast_from(object));
            GUILock lock(_wb, _("Opening SQL Editor"), strfmt(_("An SQL editor instance for '%s' is opening and should be available in a "
              "moment.\n\nPlease stand by..."), connection->name().c_str()));

            _wb->show_status_text("Opening SQL Editor...");          
            _wb->add_new_query_window(connection);
          }
          break;
        }

      case ActionNewConnection:
        {
          NewConnectionWizard wizard(_wb->get_root()->rdbmsMgmt());
          wizard.run();
          refresh_home_connections();
          break;
        }
      case ActionAddSchemaObject:
        break;
      case ActionEditTable:
        {
          EditTableDataWizard wizard(_wb);
          _wb->show_status_text("Edit Table Data Wizard Opened.");
          grt::DictRef options(_wb->get_grt());
          wizard.run_modal(options);
          _wb->show_status_text("");
          break;
        }
      case ActionEditSQLScript:
        {
          EditSqlScriptWizard wizard(_wb);
          _wb->show_status_text("Edit SQL Script Wizard Opened.");
          grt::DictRef options(_wb->get_grt());
          wizard.run_modal(options);
          _wb->show_status_text("");
          break;
        }

      case ActionManageConnections:
        {
          grtui::DbConnectionEditor editor(_wb->get_root()->rdbmsMgmt());
          _wb->show_status_text("Connection Manager Opened.");
          editor.run();
          _wb->show_status_text("");
          // save connection list now
          _wb->save_app_options();
          refresh_home_connections();        
          break;
        }

      case ActionOpenEERModel:
        {
          // Note: wb->open_document has an own GUILock, so we must not set another one here.
          std::string filename = _wb->show_file_dialog("open", _("Open Workbench Model"), "mwb");
          if (!filename.empty())
            _wb->open_document(filename);
          else
            _wb->show_status_text("Cancelled");
          break;
        }

      case ActionOpenEERModelFromList:
        {
          // Note: wb->open_document has an own GUILock, so we must not set another one here.
          if (object.is_valid())
          {
            std::string path = *grt::StringRef::cast_from(object);
            _wb->show_status_text(strfmt("Opening %s...", path.c_str()));
            _wb->open_document(path);
          }
          break;
        }
      case ActionNewEERModel:
        {
          _wb->new_document();
        }
        break;
      
      case ActionNewModelFromDB:
        {
          grt::BaseListRef args(_wb->get_grt());
          _wb->new_document();
          if (_wb->get_document().is_valid())
          {
            args.ginsert(_wb->get_document()->physicalModels()[0]->catalog());
            _wb->execute_plugin("db.plugin.database.rev_eng", args);
          }
          else
            _wb->show_status_text("Error creating document");
          break;
        }
      
      case ActionNewModelFromScript:
        {
          grt::BaseListRef args(_wb->get_grt());
          _wb->new_document();
          if (_wb->get_document().is_valid())
          {
            args.ginsert(_wb->get_document()->physicalModels()[0]->catalog());
            _wb->execute_plugin("db.mysql.plugin.import.sql", args);
          }
          else
            _wb->show_status_text("Error creating document");
          break;
        }

      case ActionServerAdministration:
        {
          grt::BaseListRef args(_wb->get_grt_manager()->get_grt(), grt::AnyType);
          
          app_PluginRef plugin(_wb->get_plugin_manager()->get_plugin("wb.admin.placeholder"));
          
          if (plugin.is_valid())
            try
            {
              _wb->get_plugin_manager()->open_plugin(plugin, args);
            }
            catch(grt::module_error& err)
            {
              std::string msg = strfmt("Error in wb.admin module: %s",err.what());
              _wb->show_status_text(msg);
            }
          break;
        }      
        
      case ActionManageInstanceFromList:
        {
          if (object.is_valid())
          {
            db_mgmt_ServerInstanceRef instance(db_mgmt_ServerInstanceRef::cast_from(object));
            if (!instance->connection().is_valid())
              mforms::Utilities::show_error("Cannot Open Administrator",
                                            "The selected server instance does not have a MySQL server connection assigned and cannot be opened.",
                                            "OK", "", "");
            else
            {
              GUILock lock(_wb, _("Opening MySQL Workbench Administrator"), strfmt(_("The MySQL Workbench Administrator for %s "
                "is being opened and should be available in a moment.\n\nPlease stand by..."), instance->name().c_str()));
              _wb->add_new_admin_window(instance);
            }
          }
          break;
        }
        
      case ActionNewServerInstance:
        {
          NewServerInstanceWizard wizard(_wb);
          grt::DictRef dict;
          
          _wb->show_status_text("Creating new server instance.");
          wizard.run_modal(dict);
          _wb->save_app_options();
          refresh_home_connections();
          refresh_home_instances((int)_wb->get_root()->rdbmsMgmt()->storedInstances().count()-1);
          _wb->show_status_text("New server instance created.");
        }
        break;

      case ActionDumpRestore:
        {
          grt::BaseListRef args(_wb->get_grt_manager()->get_grt(), grt::AnyType);
          
          app_PluginRef plugin(_wb->get_plugin_manager()->get_plugin("wb.admin.dumpManager"));
          
          if (plugin.is_valid())
            try
            {
              _wb->get_plugin_manager()->open_plugin(plugin, args);
            }
            catch(grt::module_error& err)
            {
              std::string msg = strfmt("Error in dump module: %s",err.what());
              _wb->show_status_text(msg);
            }
          break;
        }

    case ActionManageSecurity:
      {
          grt::BaseListRef args(_wb->get_grt_manager()->get_grt(), grt::AnyType);
          
          app_PluginRef plugin(_wb->get_plugin_manager()->get_plugin("wb.admin.securityManager"));
          
          if (plugin.is_valid())
            try
            {
              _wb->get_plugin_manager()->open_plugin(plugin, args);
            }
            catch(grt::module_error& err)
            {
              std::string msg = strfmt("Error in sercurityManager module: %s",err.what());
              _wb->show_status_text(msg);
            }
          break;
        }
    case ActionManageServerInstances:
      {
        ServerInstanceEditor editor(_wb->get_grt_manager(), _wb->get_root()->rdbmsMgmt());
        _wb->show_status_text("Server Instance Manager Opened.");
        db_mgmt_ServerInstanceRef instance(editor.run());
        _wb->show_status_text("");
        // save instance list now
        _wb->save_app_options();
        refresh_home_connections();
        refresh_home_instances((int)_wb->get_root()->rdbmsMgmt()->storedInstances().get_index(instance));        
        break;
      }
  }
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::refresh_home_instances(int selected)
{
  if (!_home_screen) return;
  grt::ListRef<db_mgmt_ServerInstance> instances(_wb->get_root()->rdbmsMgmt()->storedInstances());

  _home_screen->clear_list(HomeScreenAdministrationList);
  for (grt::ListRef<db_mgmt_ServerInstance>::const_iterator end = instances.end(),
       inst = instances.begin(); inst != end; ++inst)
  {
    std::string text= get_server_info(*inst);

    _home_screen->add_list_entry(HomeScreenAdministrationList, (*inst)->name(), text, *inst);
  }

  _home_screen->select_list_entry(HomeScreenAdministrationList, selected);
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::refresh_home_connections(int selected)
{
  if (!_home_screen) return;
  grt::ListRef<db_mgmt_Connection> connections(_wb->get_root()->rdbmsMgmt()->storedConns());  
  
  _home_screen->clear_list(HomeScreenSQLDevList);
  for (grt::ListRef<db_mgmt_Connection>::const_iterator end = connections.end(),
       inst = connections.begin(); inst != end; ++inst)
  {
    grt::DictRef dict((*inst)->parameterValues());
    std::string tunnel;

    std::string host_entry= strfmt("%s:%i", dict.get_string("hostName").c_str(), (int) dict.get_int("port"));
    if ((*inst)->driver().is_valid() && (*inst)->driver()->name() == "MysqlNativeSSH")
      tunnel = "(ssh tunnel)";
    else
      if ((*inst)->driver().is_valid() && (*inst)->driver()->name() == "MysqlNativeSocket")
      {
        std::string socket= dict.get_string("socket"); // socket or pipe
        if (socket.length() == 0)
          socket= "MySQL"; // default pipe name
        // TODO: what about the default for sockets? Need to distinct the OS here for decision.
        host_entry= "Localhost via pipe " + socket;
      }

    std::string text = strfmt("User: %s  Host: %s %s", dict.get_string("userName").c_str(), 
      host_entry.c_str(), tunnel.c_str());
    
    _home_screen->add_list_entry(HomeScreenSQLDevList, (*inst)->name(), text, *inst);
  }
  
  _home_screen->select_list_entry(HomeScreenSQLDevList, selected);
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::refresh_home_models(int selected)
{
  if (!_home_screen) return;
  _home_screen->clear_list(HomeScreenModellingList);

  for (grt::StringListRef::const_iterator end = _wb->get_root()->options()->recentFiles().end(),
       f = _wb->get_root()->options()->recentFiles().begin(); f != end; ++f)
  {
    if (!g_str_has_suffix((*f).c_str(), ".mwb"))
      continue;
    
#ifdef _WIN32
    struct _stat stbuf;

    if (_stat((*f).c_str(), &stbuf) < 0)
      _home_screen->add_list_entry(HomeScreenModellingList, *f, "File not accessible", grt::ValueRef());
    else
    {
      struct tm t;
      localtime_s(&t, &stbuf.st_mtime);

      std::string text = strfmt("Last modified %s", asctime(&t));
      if (text[text.size()-1] == '\n')
        text = text.substr(0, text.size()-1);

      char *file = g_path_get_basename((*f).c_str());
      _home_screen->add_list_entry(HomeScreenModellingList, file, text, *f);
      g_free(file);
    }
#else
    struct stat stbuf;

    if (g_stat((*f).c_str(), &stbuf) < 0)
      _home_screen->add_list_entry(HomeScreenModellingList, *f, "File not accessible", *f);
    else
    {
      char *buf = asctime(localtime(&stbuf.st_mtime));
      std::string text = strfmt("Last modified %s", buf?buf:"???");
      
      char *file = g_path_get_basename((*f).c_str());
      _home_screen->add_list_entry(HomeScreenModellingList, file, text, *f);
      g_free(file);
    }
#endif
  }
  _home_screen->select_list_entry(HomeScreenModellingList, selected);
}

//--------------------------------------------------------------------------------------------------

void WBContextUI::activate_figure(const grt::ValueRef &value)
{
  ModelDiagramForm *form= 0;
  if (model_FigureRef::can_wrap(value))
  {
    model_FigureRef figure(model_FigureRef::cast_from(value));
    form= get_wb()->get_model_context()->get_diagram_form_for_diagram_id(figure->owner().id());
    if (form)
      form->focus_and_make_visible(figure, true);
  }
  else if (model_ConnectionRef::can_wrap(value))
  {
    model_ConnectionRef conn(model_ConnectionRef::cast_from(value));
    ModelDiagramForm *form= get_wb()->get_model_context()->get_diagram_form_for_diagram_id(conn->owner().id());
    if (form)
      form->focus_and_make_visible(conn, true);
  }
  else if (model_LayerRef::can_wrap(value))
  {
    model_LayerRef layer(model_LayerRef::cast_from(value));
    ModelDiagramForm *form= get_wb()->get_model_context()->get_diagram_form_for_diagram_id(layer->owner().id());
    if (form)
      form->focus_and_make_visible(layer, true);
  }
}


void WBContextUI::form_changed()
{  
  _wb->request_refresh(RefreshZoom, "", (NativeHandle)0);
  _wb->request_refresh(RefreshMenubar, "", (NativeHandle)0);
  _wb->request_refresh(RefreshToolbar, "", (NativeHandle)0);
  
  if (_active_form != NULL && _active_form == _home_screen)
  {
    refresh_home_models();
    refresh_home_instances(0);
    refresh_home_connections();
  }
}

HistoryTreeBE *WBContextUI::get_history_tree()
{
  if (!_history_tree)
  {
    _history_tree= new HistoryTreeBE(_wb, _wb->get_grt()->get_undo_manager());
    
    _wb->get_grt()->get_undo_manager()->signal_changed().
       connect(sigc::bind(sigc::mem_fun(_history_tree, &HistoryTreeBE::tree_changed), -1, bec::NodeId()));
  }
  return _history_tree;
}


bec::ValueInspectorBE *WBContextUI::create_inspector_for_selection(bec::UIForm *form, std::vector<std::string> &items)
{
  grt::ListRef<model_Object> selection;

  //grt::ListRef<model_Object> selection(form->get_selection());
  
  if (dynamic_cast<ModelDiagramForm*>(form))
    selection= dynamic_cast<ModelDiagramForm*>(form)->get_selection();
  else
    return 0;

  if (selection.is_valid() && selection.count()>0)
  {
    if (selection.count()==1)
    {
      items.push_back(strfmt("%s: %s", selection[0]->name().c_str(), selection[0].get_metaclass()->get_attribute("caption").c_str()));

      return ValueInspectorBE::create(_wb->get_grt_manager()->get_grt(), selection[0], false, true);
    }
    else
    {
      std::vector<grt::ObjectRef> list;

      items.push_back(_("Multiple Items"));
      for (size_t c= selection.count(), i= 0; i < c; i++)
      {
        items.push_back(strfmt("%s: %s", selection[i]->name().c_str(), selection[i].get_metaclass()->get_attribute("caption").c_str()));
        list.push_back(selection.get(i));
      }

      return ValueInspectorBE::create(_wb->get_grt_manager()->get_grt(), list);
    }
  }

  return 0;
}


bec::ValueInspectorBE *WBContextUI::create_inspector_for_selection(std::vector<std::string> &items)
{
  std::string res;

  grt::ListRef<GrtObject> selection(get_physical_overview()->get_selection());
  std::string name_mem_name("name");

  if (selection.is_valid() && selection.count()>0)
  {
    if (selection.count()==1)
    {
			GrtObjectRef obj= selection[0];
      if (obj.is_valid() && obj->has_member(name_mem_name))
      {
        items.push_back(strfmt("%s: %s", obj.get_string_member(name_mem_name).c_str(), obj.get_metaclass()->get_attribute("caption").c_str()));

        return ValueInspectorBE::create(_wb->get_grt_manager()->get_grt(), selection[0], false, true);
      }
    }
    else
    {
      std::vector<grt::ObjectRef> list;

      items.push_back(_("Multiple Items"));
      for (size_t c= selection.count(), i= 0; i < c; i++)
      {
        if (!selection[i].is_valid())//skip "Add item" entry in objects list
          continue;
        items.push_back(strfmt("%s: %s", selection[i].get_string_member(name_mem_name).c_str(), selection[i].get_metaclass()->get_attribute("caption").c_str()));
        list.push_back(selection.get(i));
      }

      return ValueInspectorBE::create(_wb->get_grt_manager()->get_grt(), list);
    }
  }

  return 0;
}


static std::list<GrtObjectRef> get_selected_objects_from_form(WBContext *wb, bec::UIForm *form)
{
  std::list<GrtObjectRef> objects;

  if (dynamic_cast<ModelDiagramForm*>(form))
  {
    grt::ListRef<model_Object> selection(dynamic_cast<ModelDiagramForm*>(form)->get_selection());

    if (selection.is_valid())
    {
      for (size_t c= selection.count(), i= 0; i < c; i++)
      {
        model_ObjectRef object(selection[i]);
        WBComponent *comp= wb->get_component_handling(object);
        GrtObjectRef dbobject;

        if (comp) 
          dbobject= comp->get_object_for_figure(object);

        if (dbobject.is_valid())
          objects.push_back(dbobject);
        else 
          objects.push_back(object);
      }
    }
  }
  else if (dynamic_cast<OverviewBE*>(form))
  {
    grt::ListRef<GrtObject> selection(dynamic_cast<OverviewBE*>(form)->get_selection());

    if (selection.is_valid())
    {
      for (size_t c= selection.count(), i= 0; i < c; i++)
      {
        GrtObjectRef object(selection[i]);

        if (object.is_valid())
          objects.push_back(object);
      }
    }
  }
  return objects;
}


std::string WBContextUI::get_description_for_selection(grt::ListRef<GrtObject> &activeObjList, std::vector<std::string> &items)
{
  std::string res;

  if (get_physical_overview() != NULL)
  {
    grt::ListRef<GrtObject> selection(get_physical_overview()->get_selection());
    activeObjList= selection;
    
    std::string comment_mem_name("comment");
    std::string name_mem_name("name");

    if (selection.is_valid() && selection.count()>0)
    {
      if (selection.count()==1)
      {
        GrtObjectRef obj(selection[0]);
        if (obj.is_valid() && obj.has_member(comment_mem_name) && obj.has_member(name_mem_name))
        {
          items.push_back(strfmt("%s: %s", obj->name().c_str(), obj.get_metaclass()->get_attribute("caption").c_str()));
          res= obj.get_string_member(comment_mem_name);
        }
      }
      else
      {
        items.push_back(_("Multiple Items"));
        for (size_t c= selection.count(), i= 0; i < c; i++)
        {
          GrtObjectRef obj(selection[i]);
          if (obj.is_valid() && obj.has_member(comment_mem_name) && obj.has_member(name_mem_name))
          {
            items.push_back(strfmt("%s: %s", obj->name().c_str(), obj.get_metaclass()->get_attribute("caption").c_str()));
            std::string comment= obj.get_string_member(comment_mem_name);
            if (0 == i)
              res= comment;
            else if (0 != res.compare(comment))
              res= "<Multiple Items>\nThat means not all selected items have same comment.\nBeware applying changes will override comments for all selected objects.";
          }
        }
      }
    }
  }
  return res;
}


std::string WBContextUI::get_description_for_selection(bec::UIForm *form, grt::ListRef<GrtObject> &activeObjList, std::vector<std::string> &items)
{
  grt::ListRef<model_Object> selection;

  if (dynamic_cast<ModelDiagramForm*>(form))
    selection= dynamic_cast<ModelDiagramForm*>(form)->get_selection();
  else
    return get_description_for_selection(activeObjList, items);

  std::string res;

  activeObjList= grt::ListRef<model_Object>(selection.get_grt());

  std::string comment_mem_name("comment");

  if (selection.is_valid() && selection.count()>0)
  {
    bool first= true;
    for (size_t c= selection.count(), i= 0; i < c; i++)
    {
      model_ObjectRef figure(selection[i]);
      WBComponent *comp= _wb->get_component_handling(figure);
      GrtObjectRef dbobject;
      if (comp) dbobject= comp->get_object_for_figure(figure);
      
      if (dbobject.is_valid() && dbobject.has_member(comment_mem_name))
      {
        activeObjList.insert(dbobject);
        
        items.push_back(strfmt("%s: %s", figure->name().c_str(), figure->get_metaclass()->get_attribute("caption").c_str()));
        std::string comment= dbobject.get_string_member(comment_mem_name);
        if (first)
          res= comment;
        else if (0 != res.compare(comment))
          res= _("<Multiple Items>\nThat means not all selected items have same comment.\nBeware applying changes will override comments for all selected objects.");

        first= false;
      }
    }
    if (items.size() > 1)
      items.insert(items.begin(), _("Multiple Items"));
  }

  return res;
}


void WBContextUI::set_description_for_selection(const grt::ListRef<GrtObject> &objList, const std::string &val)
{
  if (objList.is_valid() && objList.count()>0)
  {
    std::string comment_mem_name("comment");

    grt::AutoUndo undo(_wb->get_grt());
    
    if (objList.count()==1)
    {
      GrtObjectRef obj(objList[0]);
      if (obj.is_valid() && obj.has_member(comment_mem_name))
        obj.set_member(comment_mem_name, grt::StringRef(val));
    }
    else
    {
      for (size_t c= objList.count(), i= 0; i < c; i++)
      {
        GrtObjectRef obj(objList[i]);
        if (obj.is_valid() && obj.has_member(comment_mem_name))
          obj.set_member(comment_mem_name, grt::StringRef(val));
      }
    }

    undo.end(_("Set Object Description"));
  }
}


DiagramOptionsBE *WBContextUI::create_diagram_options_be(mdc::CanvasView *view)
{
  model_DiagramRef model_diagram(get_wb()->get_model_context()->get_active_model_diagram(false));
  
  if (model_diagram.is_valid())
    return new DiagramOptionsBE(view, model_diagram, _wb);
  else
    return 0;
}


std::string WBContextUI::get_active_diagram_info()
{
  wb::ModelDiagramForm *form= dynamic_cast<wb::ModelDiagramForm*>(get_active_main_form());
  
  if (form)
    return form->get_diagram_info_text();
  
  return "";
}


void WBContextUI::activate_find_result(const GrtObjectRef &location, const GrtObjectRef &object, const std::string &text)
{
  _wb->show_status_text(strfmt("Match: %s (%s)", 
    text.c_str(), object.get_metaclass()->get_attribute("caption").c_str()));

  if (location.is_valid())
  {
    model_DiagramRef view;

    if (location.is_instance<model_Diagram>())
      view= model_DiagramRef::cast_from(location);
    else
      view= _wb->get_parent_for_object<model_Diagram>(location);

    if (view.is_valid())
    {
      ModelDiagramForm *form= get_wb()->get_model_context()->get_diagram_form_for_diagram_id(view.id());

      if (form)
      {
        _wb->switched_view(form->get_view());
        form->focus_and_make_visible(model_FigureRef::cast_from(location), true);
      }
    }
  }
}

#endif // ___specialforms


#ifndef ___preferences
//-----------------------------------------------------------------------------------
// utility functions for user preferences

void WBContextUI::get_doc_properties(std::string& caption, std::string& version, std::string& author, std::string& project, std::string& date_created, std::string& date_changed, std::string& description)
{
  app_DocumentInfoRef info= _wb->get_document()->info();

  caption= info->caption();
  version= info->version();
  author= info->author();
  project= info->project();
  date_created= info->dateCreated();
  date_changed= info->dateChanged();
  description= info->description();  
}

void WBContextUI::set_doc_properties(const std::string &caption, const std::string &version, const std::string &author, const std::string &project, const std::string &date_created, const std::string &date_changed, const std::string &description)
{
  app_DocumentInfoRef info= _wb->get_document()->info();

  info->caption(caption);
  info->version(version);
  info->author(author);
  info->project(project);
  info->dateCreated(date_created);
  info->dateChanged(date_changed);
  info->description(description);  
}


std::list<WBPaperSize> WBContextUI::get_paper_sizes(bool descr_in_inches)
{
  std::list<WBPaperSize> sizes;

  grt::ListRef<app_PaperType> types(_wb->get_root()->options()->paperTypes());

  for (size_t c= types.count(), i= 0; i < c; i++)
  {
    WBPaperSize size;
    size.name= types[i]->name();
    size.caption= types[i]->caption();
    size.width= types[i]->width();
    size.height= types[i]->height();
    size.margins_set= types[i]->marginsSet() != 0;
    size.margin_top= types[i]->marginTop();
    size.margin_bottom= types[i]->marginBottom();
    size.margin_left= types[i]->marginLeft();
    size.margin_right= types[i]->marginRight();

    if (descr_in_inches)
      size.description= strfmt("%.2f in x %.2f in", size.width*0.03937, size.height*0.03937);
    else
      size.description= strfmt("%.2f cm x %.2f cm", size.width/10, size.height/10);

    sizes.push_back(size);
  }

  return sizes;
}


bool WBContextUI::add_paper_size(const std::string &name, double width, double height,
                                   bool margins, double margin_top, double margin_bottom,
                                   double margin_left, double margin_right)
{
  if (grt::find_named_object_in_list(_wb->get_root()->options()->paperTypes(), name).is_valid())
    return false;

  app_PaperTypeRef type(_wb->get_grt_manager()->get_grt());
  type->owner(_wb->get_root()->options());
  type->name(name);
  type->width(width);
  type->height(height);
  type->marginsSet(margins?1:0);
  type->marginTop(margin_top);
  type->marginBottom(margin_bottom);
  type->marginLeft(margin_left);
  type->marginRight(margin_right);
  _wb->get_root()->options()->paperTypes().insert(type);

  return true;
}


app_PageSettingsRef WBContextUI::get_page_settings()
{
  if (_wb->get_document().is_valid())
    return _wb->get_document()->pageSettings();
  else
  {
    // XXX add proper initialization for non-trivial types in structs.app.h too.
    app_PageSettingsRef settings= app_PageSettingsRef(_wb->get_grt());
    settings->scale(1);
    settings->paperType(app_PaperTypeRef());

    return settings;
  }
}


grt::DictRef WBContextUI::get_model_options(const std::string &model_id)
{
  grt::ListRef<workbench_physical_Model> pmodels(_wb->get_document()->physicalModels());

  for (size_t c= pmodels.count(), i= 0; i < c; i++)
  {
    if (pmodels.get(i).id() == model_id)
    {
      return pmodels.get(i)->options();
    }
  }
  return grt::DictRef();
}


std::vector<std::string> WBContextUI::get_wb_options_keys(const std::string &model)
{
  std::vector<std::string> keylist;
  grt::DictRef options= _wb->get_wb_options();

  for (grt::DictRef::const_iterator iter= options.begin(); iter != options.end(); ++iter)
  {
    keylist.push_back(iter->first);
  }

  return keylist;
}


bool WBContextUI::get_wb_options_value(const std::string &model, const std::string &key,
                                              std::string &value)
{
  grt::DictRef options= _wb->get_wb_options();
  grt::ValueRef val;

  // If a model is given check if it is set to use global values or its own.
  if (!model.empty())
  {
    grt::DictRef model_options(get_model_options(model));
    int use_global= model_options.get_int("useglobal", 1);

    if (key == "useglobal")
    {
      if (use_global)
        value= "1";
      else
        value= "0";
      return true;
    }

    if (!use_global && model_options.has_key(key))
      val= model_options.get(key);
  }

  if (!val.is_valid() && options.has_key(key))
    val= options.get(key);

  switch(val.type())
  {
  case grt::StringType: 
  case grt::DoubleType:
  case grt::IntegerType:
      value= val.repr();
      return true;
    default: 
      return false;
  }
}


void WBContextUI::set_wb_options_value(const std::string &model, const std::string &key, const std::string &value, 
                                       const grt::Type default_type)
{
  grt::DictRef options;
  grt::Type type;
  if (_wb->get_wb_options().has_key(key))
    type= _wb->get_wb_options().get(key).type();
  else
    type= default_type;

  if (!model.empty())
  {
    options= get_model_options(model);

    options.gset("useglobal", 0);

    if (options.has_key(key))
      type= options.get(key).type();
  }
  
  if (!options.is_valid())
    options= _wb->get_wb_options();

  switch (type)
  {
  case grt::DoubleType:
      options.set(key, grt::DoubleRef(atof(value.c_str())));
      break;
  case grt::IntegerType:
      options.set(key, grt::IntegerRef(atoi(value.c_str())));
      break;
  case grt::StringType:
      options.set(key, grt::StringRef(value));
      break;
  default:
    throw std::runtime_error("No valid grt type specified when setting options value.");
  }
}

void WBContextUI::discard_wb_model_options(const std::string &model)
{
  grt::DictRef opts= get_model_options(model);
  if (opts.is_valid())
  {
    for (grt::DictRef::const_iterator item= opts.begin(); item != opts.end(); ++item)
    {
      opts.set(item->first, grt::ValueRef());
    }
    opts.gset("useglobal", 1);
  }
}


#endif // ___preferences


#ifndef ___forms

//--------------------------------------------------------------------------------
// Form Management


void *WBContextUI::form_destroyed(void *data)
{
  UIForm *form= reinterpret_cast<UIForm*>(data);
  WBContextUI *wb= reinterpret_cast<WBContextUI*>(form->get_owner_data());

  if (wb->_active_form == form)
    wb->set_active_form(0);
  if (wb->_active_main_form == form)
    wb->_active_main_form= 0;

  return 0;
}


std::string WBContextUI::get_active_context(bool main_context)
{
  bec::UIForm *form= main_context ? get_active_main_form() : get_active_form();

  if (form)
    return form->get_form_context_name();
  
  return "";
}


void WBContextUI::set_active_form(bec::UIForm *form)
{
  if (_active_form == form)
    return;
    
  // register callbacks to form if needed
  
  if (_active_form && form)
    _active_form->remove_destroy_notify_callback(_active_form);
  _active_form= form;
  if (form)
  {
    form->add_destroy_notify_callback(reinterpret_cast<void*>(form), &WBContextUI::form_destroyed);
    form->set_owner_data(reinterpret_cast<void*>(this));
  }

  if (form && form->is_main_form())
  {
    if (_active_main_form != form)
    {
      _active_main_form= form;
      _main_form_change_signal.emit(form);
    }
  }

  _form_change_signal.emit(form);
}



bec::UIForm *WBContextUI::get_active_main_form()
{
  return _active_main_form;
}


bec::UIForm *WBContextUI::get_active_form()
{
  return _active_form;
}

#endif // ___forms


#ifndef ___others

//-----------------------------------------------------------------------------------
// other functionality for UI

std::string WBContextUI::get_document_name()
{
  if (_wb->get_filename().empty())
    return "Untitled";
  else
  {
    gchar *bn= g_path_get_basename(_wb->get_filename().c_str());
    std::string title(bn);
    g_free(bn);
    return title;
  }
}


std::string WBContextUI::get_title()
{
  if (_wb->get_model_context())
  {
#ifndef __APPLE__
    if (_wb->has_unsaved_changes())
      return get_document_name() + "* - MySQL Workbench";
    else
#endif
      return get_document_name() + " - MySQL Workbench";
  }
  else
    return "MySQL Workbench";
}

#endif // ___others
