/* 
 * (c) 2009-2010 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "stdafx.h"

#include <glib.h>
#include "file_charset_dialog.h"
#include "grt/common.h"
#include "string_utilities.h"

using namespace mforms;
using namespace base;

FileCharsetDialog::FileCharsetDialog(const std::string &title, const std::string &message,
                                     const std::string &default_encoding)
  : Form(0)
{
  Box *vbox = manage(new Box(false));
  set_content(vbox);
  vbox->set_padding(12);
  vbox->set_spacing(12);
  
  Label *l = manage(new Label(title));
  l->set_style(BoldStyle);
  vbox->add(l, false, true);
  
  l = manage(new Label(message));
  vbox->add(l, false, true);

  Box *hbox = manage(new Box(true));
  vbox->add(hbox, false, true);

  hbox->add(manage(new Label(_("Character Set Encoding Name:"))), false, true);
  hbox->add(&_charset, true, true);
  _charset.set_value(default_encoding);
  
  Box *bbox = manage(new Box(true));
  vbox->add(bbox, false, true);
  bbox->set_spacing(12);

  _ok.set_text(_("OK"));
  _cancel.set_text(_("Cancel"));
  Utilities::add_end_ok_cancel_buttons(bbox, &_ok, &_cancel);
}


std::string FileCharsetDialog::run()
{
  if (run_modal(&_ok, &_cancel))
    return _charset.get_string_value();
  return "";
}


bool FileCharsetDialog::ensure_filedata_utf8(const char *data, size_t length,
                                             const std::string &filename,
                                             std::string &output_str)
{
  const gchar *end;
  
retry:
  
  if (!g_utf8_validate(data, length, &end))
  {
    std::string default_encoding = "LATIN1";

    // check if utf16
    if (length >= 2 && (guchar)data[0] == 0xff && (guchar)data[1] == 0xfe)
      default_encoding = "UTF-16";

    FileCharsetDialog dlg(_("Unknown File Encoding"), 
                          strfmt("The file '%s'\n"
                                 "has an unknown character set encoding.\n"
                                 "Please select the encoding of the file and press OK for Workbench to convert and open it.\n"
                                 "Note that as Workbench works with UTF-8 text, if you save back to the original file,\n"
                                 "its contents will be replaced with the converted data.", filename.c_str()),
                          default_encoding);

    std::string charset;
    char *converted;
    gsize bytes_read, bytes_written;
    GError *error = NULL;
    
    charset = dlg.run();
    if (charset.empty())
        return false;

    converted = g_convert(data, length, "UTF-8", charset.c_str(),
                          &bytes_read, &bytes_written,
                          &error);
    if (!converted)
    {
      int res;
      
      res = Utilities::show_error(_("Could not Convert Text Data"),
                            strfmt(_("The file contents could not be converted from '%s' to UTF-8:\n%s\n"),
                                   charset.c_str(), error ? error->message : "Unknown error"),
                                  _("Choose Encoding"), _("Cancel"), "");
      if (error)
        g_error_free(error);
      
      if (res == ResultOk)
        goto retry;

      return false;
    }
    else if (bytes_read < length)
    {
      int res;
      
      // give back the partially converted string anyway
      output_str = std::string(converted, bytes_written);
      
      g_free(converted);

      res = Utilities::show_error(_("Could not Convert Text Data"),
                            strfmt(_("Some of the file contents could not be converted from '%s' to UTF-8:\n%s\n"\
                                     "Click Ignore to open the partial file anyway, or choose another encoding."),
                                   charset.c_str(), error ? error->message : "Unknown error"),
                                   _("Ignore"), _("Cancel"), _("Choose Encoding"));
      if (error)
        g_error_free(error);
      if (res == ResultOk)
        return true;
      else if (res == ResultCancel)
        return false;
      else
        goto retry;
    }
    else
    {
      output_str = converted;
      return true;
    }
  }
  
  output_str = std::string(data, length);
  return true;
}

