/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "stdafx.h"

#include "recordset_data_storage.h"
#include "sqlide_generics_private.h"
#include "string_utilities.h"
#include <boost/foreach.hpp>


using namespace bec;
using namespace grt;


const std::string Recordset_data_storage::DATA_STORAGE_MODULE_NAME_PREFIX= "sqlide.data.storage.";
const std::string Recordset_data_storage::DATA_STORAGE_MODULE_GROUP= "sqlide/DataStorage";
const std::string Recordset_data_storage::DATA_STORAGE_MODULE_GETTER= "getDataStorage";


Recordset_data_storage::Recordset_data_storage(GRTManager *grtm)
:
_grtm(grtm),
_readonly(true),
_valid(false),
_limit_rows(false),
_limit_rows_count(1000),
_limit_rows_offset(0),
_limit_rows_applicable(true)
{
}


Recordset_data_storage::~Recordset_data_storage()
{
}


boost::shared_ptr<sqlite::connection> Recordset_data_storage::data_swap_db(const Recordset::Ref recordset)
{
  return recordset->data_swap_db();
}


void Recordset_data_storage::apply_changes(const Recordset::Ref recordset)
{
  boost::shared_ptr<sqlite::connection> data_swap_db= recordset->data_swap_db();
  do_apply_changes(recordset.get(), data_swap_db.get());
}


void Recordset_data_storage::serialize(const Recordset::Ref recordset)
{
  boost::shared_ptr<sqlite::connection> data_swap_db= recordset->data_swap_db();
  do_serialize(recordset.get(), data_swap_db.get());
}


void Recordset_data_storage::unserialize(Recordset::Ref recordset)
{
  boost::shared_ptr<sqlite::connection> data_swap_db= recordset->data_swap_db();
  do_unserialize(recordset.get(), data_swap_db.get());
  recordset->rebuild_data_index(data_swap_db.get(), false, false);
}


void Recordset_data_storage::fetch_blob_value(Recordset::Ref recordset, RowId rowid, ColumnId column, sqlite::Variant &blob_value)
{
  boost::shared_ptr<sqlite::connection> data_swap_db= recordset->data_swap_db();
  fetch_blob_value(recordset.get(), data_swap_db.get(), rowid, column, blob_value);
}


void Recordset_data_storage::fetch_blob_value(Recordset *recordset, sqlite::connection *data_swap_db, RowId rowid, ColumnId column, sqlite::Variant &blob_value)
{
  blob_value= sqlite::Null();

  do_fetch_blob_value(recordset, data_swap_db, rowid, column, blob_value);

  // cache fetched blob in data swap db, blob shouldn't stay in memory for long
  if (!sqlide::is_var_null(blob_value))
  {
    sqlide::Sqlite_transaction_guarder transaction_guarder(data_swap_db);
    update_data_swap_record(data_swap_db, rowid, column, blob_value);
    transaction_guarder.commit();
  }
}


void Recordset_data_storage::create_data_swap_tables(sqlite::connection *data_swap_db, Recordset::Column_names &column_names, Recordset::Column_types &column_types)
{
  // generate sql
  std::ostringstream cr_table_stmt;
  cr_table_stmt << "create table if not exists `data` (";
  {
    sqlide::TypeOfVar type_of_var;
    Recordset::Column_types::iterator column_type_i= column_types.begin();
    for (ColumnId col= 0, count= column_names.size(); col < count; ++col)
    {
      std::string column_type= boost::apply_visitor(type_of_var, *column_type_i);
      cr_table_stmt << "`_" << col << "` " << column_type << ", ";
      ++column_type_i;
    }
  }
  cr_table_stmt << "id integer primary key autoincrement)";

  // execute sql
  sqlite::execute(*data_swap_db, "drop table if exists `data`", true);
  sqlite::execute(*data_swap_db, "drop table if exists `data_index`", true);
  sqlite::execute(*data_swap_db, "drop table if exists `deleted_rows`", true);
  sqlite::execute(*data_swap_db, "drop table if exists `changes`", true);
  sqlite::execute(*data_swap_db, cr_table_stmt.str(), true);
  sqlite::execute(*data_swap_db, "create table if not exists `data_index` (`id` integer)", true);
  sqlite::execute(*data_swap_db, "create table if not exists `deleted_rows` as select * from `data`", true);
  sqlite::execute(*data_swap_db, "create table if not exists `changes` (`id` integer primary key autoincrement, `record` integer, `action` integer, `column` integer)", true);
  sqlite::execute(*data_swap_db, "create index if not exists `changes_idx_1` on `changes` (`record`, `action`, `column`)", true);
}


boost::shared_ptr<sqlite::command> Recordset_data_storage::prepare_data_swap_record_add_statement(sqlite::connection *data_swap_db, Recordset::Column_names &column_names)
{
  std::ostringstream sql;
  sql << "insert into `data` (";
  std::string col_delim;
  for (ColumnId col= 0, count= column_names.size(); col < count; ++col)
  {
    sql << col_delim << "`_" << col << "`";
    col_delim= ", ";
  }
  sql << ") values (";
  col_delim.clear();
  for (int col= 0, count= column_names.size(); col < count; ++col)
  {
    sql << col_delim << "?";
    col_delim= ", ";
  }
  sql << ")";

  return boost::shared_ptr<sqlite::command>(new sqlite::command(*data_swap_db, sql.str()));
}


void Recordset_data_storage::add_data_swap_record(boost::shared_ptr<sqlite::command> &insert_command, Var_vector &values)
{
  insert_command->clear();
  sqlide::BindSqlCommandVar bind_sql_command_var(insert_command.get());
  BOOST_FOREACH (const sqlite::Variant &value, values)
    boost::apply_visitor(bind_sql_command_var, value);
  insert_command->emit();
}


void Recordset_data_storage::update_data_swap_record(sqlite::connection *data_swap_db, RowId rowid, ColumnId column, const sqlite::Variant &value)
{
  boost::shared_ptr<sqlite::command> update_command(
    new sqlite::command(*data_swap_db, base::strfmt("update `data` set `_%u`=? where rowid=%u", column, rowid)));
  sqlide::BindSqlCommandVar bind_sql_command_var(update_command.get());
  boost::apply_visitor(bind_sql_command_var, value);
  update_command->emit();
}
