/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#ifndef _RECORDSET_DATA_STORAGE_BE_H_
#define _RECORDSET_DATA_STORAGE_BE_H_


#include "wbpublic_public_interface.h"
#include "sqlide/recordset_be.h"


namespace sqlite
{
  struct command;
}


class WBPUBLICBACKEND_PUBLIC_FUNC Recordset_data_storage
{
public:
  typedef boost::shared_ptr<Recordset_data_storage> Ref;
  virtual ~Recordset_data_storage();
protected:
  Recordset_data_storage(bec::GRTManager *grtm);

  friend class Recordset;

public:  
  static const std::string DATA_STORAGE_MODULE_NAME_PREFIX;
  static const std::string DATA_STORAGE_MODULE_GROUP;
  static const std::string DATA_STORAGE_MODULE_GETTER;

protected:
  bec::GRTManager *_grtm;

public:
  typedef std::list<sqlite::Variant> Var_list;
  typedef std::vector<sqlite::Variant> Var_vector;

protected:
  boost::shared_ptr<sqlite::connection> data_swap_db(const Recordset::Ref recordset);

public:
  void apply_changes(const Recordset::Ref recordset);
  void serialize(const Recordset::Ref recordset);
  void unserialize(Recordset::Ref recordset);
  void fetch_blob_value(Recordset::Ref recordset, RowId rowid, ColumnId column, sqlite::Variant &blob_value);
protected:
  virtual void fetch_blob_value(Recordset *recordset, sqlite::connection *data_swap_db, RowId rowid, ColumnId column, sqlite::Variant &blob_value);
protected:
  virtual void do_apply_changes(const Recordset *recordset, sqlite::connection *data_swap_db) = 0;
  virtual void do_serialize(const Recordset *recordset, sqlite::connection *data_swap_db) = 0;
  virtual void do_unserialize(Recordset *recordset, sqlite::connection *data_swap_db) = 0;
  virtual void do_fetch_blob_value(Recordset *recordset, sqlite::connection *data_swap_db, RowId rowid, ColumnId column, sqlite::Variant &blob_value) = 0;

public:
  bool valid() { return _valid; }
  bool readonly() { return _readonly; }
  virtual ColumnId aux_column_count() = 0;
protected:
  bool _readonly;
  bool _valid;

public:
  static void create_data_swap_tables(sqlite::connection *data_swap_db, Recordset::Column_names &column_names, Recordset::Column_types &column_types);
protected:
  boost::shared_ptr<sqlite::command> prepare_data_swap_record_add_statement(sqlite::connection *data_swap_db, Recordset::Column_names &column_names);
  void add_data_swap_record(boost::shared_ptr<sqlite::command> &insert_command, Var_vector &values);
  void update_data_swap_record(sqlite::connection *data_swap_db, RowId rowid, ColumnId column, const sqlite::Variant &value);

protected:
  static Recordset::Column_names & get_column_names(Recordset *recordset) { return recordset->_column_names; }
  static Recordset::Column_types & get_column_types(Recordset *recordset) { return recordset->_column_types; }
  static Recordset::Column_types & get_real_column_types(Recordset *recordset) { return recordset->_real_column_types; }
  static const Recordset::Column_names & get_column_names(const Recordset *recordset) { return recordset->_column_names; }
  static const Recordset::Column_types & get_column_types(const Recordset *recordset) { return recordset->_column_types; }
  static const Recordset::Column_types & get_real_column_types(const Recordset *recordset) { return recordset->_real_column_types; }

public:
  bool limit_rows() { return _limit_rows; }
  void limit_rows(bool value) { _limit_rows= value; }
  int limit_rows_count() { return _limit_rows_count; }
  void limit_rows_count(RowId value) { _limit_rows_count= value; }
  bool limit_rows_applicable() { return _limit_rows_applicable; }
  void limit_rows_applicable(bool val) { _limit_rows_applicable= val; }
protected:
  bool _limit_rows;
  int _limit_rows_count;
  int _limit_rows_offset;
  bool _limit_rows_applicable;
};


#endif /* _RECORDSET_DATA_STORAGE_BE_H_ */
