/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "stdafx.h"

#include "sql_script_run_wizard.h"


//----------------------
// SqlScriptReviewPage
//----------------------


SqlScriptReviewPage::SqlScriptReviewPage(grtui::WizardForm *form)
:
grtui::WizardPage(form, "review"),
_box(false),
_sql_script(mforms::BothScrollBars)
{
  set_title(_("Review the SQL Script to be Applied on the Database"));
  set_short_title(_("Review SQL Script"));
  
  _box.set_spacing(10);
  add(&_box, true, true);

  _page_heading.set_text(_(
    "Please review the following SQL script that will be applied to the database.\n"
    "Note that once applied, these statements may not be revertible without losing some of the data.\n"
    "You can also manually change the SQL statements before execution."));
  _page_heading.set_wrap_text(true);
  _box.add(&_page_heading, false, false);

  _sql_script_heading.set_text(_("SQL Statement(s):"));
  _sql_script_heading.set_wrap_text(true);
  _box.add(&_sql_script_heading, false, false);

  _box.add(&_sql_script, true, true);
}


void SqlScriptReviewPage::enter(bool advancing)
{
  _sql_script.set_value(values().get_string("sql_script"));
  grtui::WizardPage::enter(advancing);
}


bool SqlScriptReviewPage::advance()
{
  values().gset("sql_script", _sql_script.get_string_value());
  return grtui::WizardPage::advance();
}


std::string SqlScriptReviewPage::next_button_caption()
{
  return _("Apply SQL Script");
}


//---------------------
// SqlScriptApplyPage
//---------------------


SqlScriptApplyPage::SqlScriptApplyPage(grtui::WizardForm *form)
:
grtui::WizardProgressPage(form, "apply"),
_err_count(0)
{
  set_title(_("Applying SQL script to the database ..."));
  set_short_title(_("Apply SQL Script"));

  /*TaskRow *task=*/ add_task(
    _("Execute SQL Statements"), 
    sigc::mem_fun(this, &SqlScriptApplyPage::execute_sql_script),
    _("Executing SQL Statements..."));

  end_adding_tasks(true, _("SQL script was successfully applied to the database."));

  set_status_text("");
}


int SqlScriptApplyPage::on_error(long long err_code, const std::string& err_msg, const std::string& err_sql)
{
  std::string sql= base::strip_text(err_sql, true, true);
  _log+= base::strfmt("ERROR %lli: %s\nSQL Statement:\n%s\n\n", err_code, err_msg.c_str(), sql.c_str());
  return 0;
}


int SqlScriptApplyPage::on_exec_progress(float progress)
{
  update_progress(progress, "");
  return 0;
}


int SqlScriptApplyPage::on_exec_stat(long success_count, long err_count)
{
  _err_count= err_count;
  return 0;
}


bool SqlScriptApplyPage::execute_sql_script()
{
  values().gset("applied", 1);
  values().gset("has_errors", 0);
  std::string sql_script= values().get_string("sql_script");
  apply_sql_script(sql_script);
  if (_err_count)
  {
    values().gset("has_errors", 1);
    add_log_text(_log);
    throw std::runtime_error(_("There was an error while applying the SQL script to the database."));
  }
  else
  {
    add_log_text(_("SQL script was successfully applied to the database."));
  }
  return true;
}


std::string SqlScriptApplyPage::next_button_caption()
{
#ifdef __APPLE__
  return _("Close");
#elif defined(_WIN32)
  return _("Finish");
#else
  return _("_Close");
#endif
}


bool SqlScriptApplyPage::allow_back()
{
  return true;
}


void SqlScriptApplyPage::enter(bool advancing)
{
  if (advancing)
  {
    _log_text.set_value("");
  }
  WizardProgressPage::enter(advancing);
}


//-----------------------
// SqlScriptRunWizard
//-----------------------


SqlScriptRunWizard::SqlScriptRunWizard(bec::GRTManager *grtm)
:
grtui::WizardForm(grtm)
{
  set_title(_("Apply SQL Script to Database"));
  review_page= new SqlScriptReviewPage(this);
  add_page(mforms::manage(review_page));
  apply_page= new SqlScriptApplyPage(this);
  add_page(mforms::manage(apply_page));
  values().gset("has_errors", 0);
  values().gset("applied", 0);
}


bool SqlScriptRunWizard::has_errors()
{
  return values().get_int("has_errors") != 0;
}


bool SqlScriptRunWizard::applied()
{
  return values().get_int("applied") != 0;
}
