/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#ifndef _SQLIDE_GENERICS_H_
#define _SQLIDE_GENERICS_H_


#include "wbpublic_public_interface.h"
#include <sqlite/database_exception.hpp>
#include <sqlite/variant.hpp>
#include <sqlite/connection.hpp>
#include <sigc++/sigc++.h>
#include <limits>
#include <ctime>


namespace sqlide
{

using namespace sqlite;

class WBPUBLICBACKEND_PUBLIC_FUNC VarEq : public boost::static_visitor<bool>
{
public:
  template<typename T> result_type operator()(const Null &v1, const T &v2) const { return false; }
  template<typename T> result_type operator()(const T &v1, const Null &v2) const { return false; }
  result_type operator()(const Null &v1, const Null &v2) const { return true; }

  template<class T> result_type cmp(const Unknown &v1, const T &v2) const { return false; }
  template<class T> result_type cmp(const T &v1, const Unknown &v2) const { return false; }
  result_type operator()(const Unknown &v1, const Unknown &v2) const { return false; }

  template<class T1, class T2> result_type operator()(const T1 &v1, const T2 &v2) const { return false; }
  template<typename T> result_type operator()(const T &v1, const T &v2) const { return v1 == v2; }
};


class WBPUBLICBACKEND_PUBLIC_FUNC VarConvBase : public boost::static_visitor<std::string>
{
public:
  VarConvBase()
  {
    _ss.precision(std::numeric_limits<long double>::digits10);
  }

protected:
  class StateKeeper {
  public:
    inline StateKeeper(VarConvBase *obj) : _obj(obj) {}
    inline ~StateKeeper() { _obj->reset(); }
  private:
    VarConvBase *_obj;
  };
  friend class StateKeeper;
  mutable std::stringstream _ss;
  inline void reset() { _ss.str(""); }
};


class WBPUBLICBACKEND_PUBLIC_FUNC VarToStr : public VarConvBase
{
public:
  VarToStr() : VarConvBase(), is_truncation_enabled(false), truncation_threshold(std::string::npos) {}
  bool is_truncation_enabled;
  std::string::size_type truncation_threshold;

  result_type operator()(const Unknown &v) const { return ""; }
  result_type operator()(const Null &v) const { return ""; }
  result_type operator()(const BlobRef &v) const
  {
    //if (is_truncation_enabled)
      return "...";
    //else
    //  throw std::runtime_error("Can't convert BLOB to string value");
  }

  result_type operator()(const std::string &v) const
  {
    return (is_truncation_enabled && (truncation_threshold < v.size())) ? v.substr(0, truncation_threshold) + "..." : v;
  }

  template<typename T>
  result_type operator()(const T &v) const
  {
    StateKeeper sk(const_cast<VarConvBase*>((const VarConvBase*)this));
    _ss << v;
    return _ss.str();
  }
};


class WBPUBLICBACKEND_PUBLIC_FUNC VarCast : public boost::static_visitor<Variant>
{
public:
  result_type operator()(const Null &t, const std::string &v) { return t; }
  result_type operator()(const Unknown &t, const std::string &v) { return v; }
  result_type operator()(const BlobRef &t, const std::string &v) { BlobRef res(new Blob()); res->reserve(v.size()); std::copy(v.begin(), v.end(), std::back_inserter(*res)); return res; }
  result_type operator()(const std::string &t, const std::string &v) { return v; }
  template<typename T> result_type operator()(const T &t, const std::string &v)
  {
    T res;
    std::stringstream ss(v);
    ss.precision(std::numeric_limits<long double>::digits10);
    ss >> res;
    return res;
  }
  template<typename T> result_type operator()(const T &t, const T &v) { return v; }
  result_type operator()(const Null &t, const Null &v) { return v; }
  template<typename T> result_type operator()(const T &t, const Null &v) { return v; }
  template<typename T, typename V> result_type operator()(const T &t, const V &v)
  {
    return t;
    //!throw std::runtime_error("Conversion for types pair is not implemented");
  }
};


class VarToInt : public boost::static_visitor<boost::int64_t>
{
public:
  result_type operator()(const int &v) const { return v; }
  result_type operator()(const boost::int64_t &v) const { return v; }
  result_type operator()(const Null &v) const { return 0; }

  template<typename T>
  result_type operator()(const T &v) const
  {
    return -1;
    //!throw std::runtime_error(std::string("Variant: wrong type: '")+typeid(T).name()+"' instead of '"+typeid(result_type).name()+"'");
  }
};


class WBPUBLICBACKEND_PUBLIC_FUNC VarToLongDouble : public boost::static_visitor<long double>
{
public:
  result_type operator()(const long double &v) const { return v; }
  result_type operator()(const int &v) const { return v; }
  result_type operator()(const boost::int64_t &v) const { return (long double)v; }
  result_type operator()(const Null &v) const { return 0; }

  template<typename T>
  result_type operator()(const T &v) const
  {
    return -1;
    //!throw std::runtime_error(std::string("Variant: wrong type: '")+typeid(T).name()+"' instead of '"+typeid(result_type).name()+"'");
  }
};


class WBPUBLICBACKEND_PUBLIC_FUNC QuoteVar : public VarConvBase
{
public:
  QuoteVar() : store_unknown_as_string(true), allow_func_escaping(false) {}
  typedef sigc::slot<std::string, const std::string&> Escape_sql_string;
  Escape_sql_string escape_string;
  bool store_unknown_as_string;
  bool allow_func_escaping;

  static std::string escape_c_string(const std::string &text) // used by mysql
  {
    std::string escaped;
    std::string::size_type p, s, len= text.length();

    // escapes ' " \n \r backslash

    p= 0;
    s= 0;
    while (p < len)
    {
      switch (text[p])
      {
      case '\'':
      case '"':
        if (p > s)
          escaped.append(text.substr(s, p-s));
        escaped.append("\\");
        escaped.append(text.substr(p, 1));
        s= p+1;
        break;
      case '\n':
        if (p > s)
          escaped.append(text.substr(s, p-s));
        escaped.append("\\n");
        s= p+1;
        break;
      case '\r':
        if (p > s)
          escaped.append(text.substr(s, p-s));
        escaped.append("\\r");
        s= p+1;
        break;
      case '\\':
        if (p > s)
          escaped.append(text.substr(s, p-s));
        escaped.append("\\\\");
        s= p+1;
        break;
      }
      p++;
    }
    if (p > s)
      escaped.append(text.substr(s));

    return escaped;
  }

  static std::string escape_ansi_sql_string(const std::string &text) // used by sqlite
  {
    std::string escaped;
    std::string::size_type p, s, len= text.length();

    // escapes '

    p= 0;
    s= 0;
    while (p < len)
    {
      switch (text[p])
      {
      case '\'':
        if (p > s)
          escaped.append(text.substr(s, p-s));
        escaped.append("\'");
        escaped.append(text.substr(p, 1));
        s= p+1;
        break;
      }
      p++;
    }
    if (p > s)
      escaped.append(text.substr(s));

    return escaped;
  }

  result_type operator()(const Unknown&, const std::string &v) const
  {
    static std::string t;
    return store_unknown_as_string ? operator()(t, v) : v;
  }
  template<typename T>
  result_type operator()(const T&, const Unknown &) const { return ""; }
  template<typename T>
  result_type operator()(const T&, const Null &) const { return "NULL"; }
  template<typename T>
  result_type operator()(const T&, const std::string &v) const
  {
    if (allow_func_escaping)
    {
      //! temporary convention to allow function call as a cell value
      // escape sequnce "\func " at start denotes the rest of the value is actual value and it should go unquoted
      // to enter "\func " value as literal user will have to enter "\\func "
      static const std::string func_call_seq= "\\func ";
      static const std::string func_call_exc= "\\\\func ";
      if (!v.empty() && (v[0] == '\\'))
      {
        if ((v.size() > func_call_seq.size()) && (v.compare(0, func_call_seq.size(), func_call_seq) == 0))
          return v.substr(func_call_seq.size());
        else if ((v.size() > func_call_exc.size()) && (v.compare(0, func_call_exc.size(), func_call_exc) == 0))
          return "'" + escape_string(v.substr(1)) + "'";
      }
    }
    return "'" + escape_string(v) + "'";
  }
  template<typename T>
  result_type operator()(const T&, const BlobRef &) const { return "?"; /*bind variable placeholder*/ }
  template<typename V>
  result_type operator()(const BlobRef&, const BlobRef&) const { return "?"; /*bind variable placeholder*/ }
  result_type operator()(const BlobRef&, const Null&) const { return "?"; /*bind variable placeholder*/ }

  template<typename T, typename V>
  result_type operator()(const T&, const V &v) const
  {
    StateKeeper sk(const_cast<VarConvBase*>((const VarConvBase*)this));
    _ss << v;
    return _ss.str();
  }
};


class WBPUBLICBACKEND_PUBLIC_FUNC TypeOfVar : public boost::static_visitor<std::string>
{
public:
  result_type operator()(const long double &v) const { return "FLOAT"; }
  result_type operator()(const int &v) const { return "INTEGER"; }
  result_type operator()(const std::string &v) const { return "VARCHAR"; }
  result_type operator()(const BlobRef &v) const { return "BLOB"; }

  template<typename T>
  result_type operator()(const T &v) const
  {
    return "VARCHAR";
  }
};


WBPUBLICBACKEND_PUBLIC_FUNC bool is_var_null(const sqlite::Variant &value);
WBPUBLICBACKEND_PUBLIC_FUNC bool is_var_unknown(const sqlite::Variant &value);
WBPUBLICBACKEND_PUBLIC_FUNC bool is_var_blob(const sqlite::Variant &value);


WBPUBLICBACKEND_PUBLIC_FUNC void optimize_sqlite_connection_for_speed(sqlite::connection *conn);

class WBPUBLICBACKEND_PUBLIC_FUNC Sqlite_transaction_guarder
{
public:
  Sqlite_transaction_guarder(sqlite::connection *conn);
  ~Sqlite_transaction_guarder();
  void commit();
  void commit_and_start_new_transaction();
private:
  sqlite::connection *_conn;
  bool _in_trans;
};


}


typedef size_t RowId;
typedef size_t ColumnId;
typedef std::vector<sqlite::Variant> Data;


template<typename C>
inline void reinit(C &c)
{
  C tmp;
  c.swap(tmp);
}


template<typename T>
class AutoSwap
{
public:
  AutoSwap(T &v1, T &v2) : _v1(v1), _v2(v2) { std::swap(_v1, _v2); }
  ~AutoSwap() { std::swap(_v1, _v2); }
private:
  T &_v1;
  T &_v2;
};


time_t WBPUBLICBACKEND_PUBLIC_FUNC timestamp();
std::string WBPUBLICBACKEND_PUBLIC_FUNC current_time();


#endif /* _SQLIDE_GENERICS_H_ */
