/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "stdafx.h"

#include "var_grid_model_be.h"
#include <sqlite/execute.hpp>
#include <sqlite/query.hpp>
#include <boost/foreach.hpp>
#include "glib/gstdio.h"


using namespace bec;
using namespace grt;

#ifdef __APPLE__
#define GMutexLock GMutex*
#endif

VarGridModel::VarGridModel(GRTManager *grtm)
:
_grtm(grtm),
_readonly(true),
_is_field_value_truncation_enabled(false),
_edited_field_row(-1),
_edited_field_col(-1)
{
  _data_mutex= g_mutex_new();
}


VarGridModel::~VarGridModel()
{
  g_mutex_free(_data_mutex);

  // clean temporary file to prevent crowding of files
  if (!_data_swap_db_path.empty())
  {
    _data_swap_db.reset();
    g_remove(_data_swap_db_path.c_str());
  }
}


void VarGridModel::dispose()
{
  GridModel::dispose();

  refresh_ui_cb= sigc::slot<int>();
}


void VarGridModel::reset()
{
  _data_swap_db.reset();
  if (_data_swap_db_path.empty())
  {
    _data_swap_db_path= _grtm->get_unique_tmp_subdir();
    _data_swap_db_path.resize(_data_swap_db_path.size()-1); // remove trailing path separator
    _data_swap_db_path+= ".db";

    boost::shared_ptr<sqlite::connection> data_swap_db= this->data_swap_db();

    sqlite::execute(*data_swap_db, "create table `data` (`id` integer)", true);
    sqlite::execute(*data_swap_db, "create table `data_index` (`id` integer)", true);
    sqlite::execute(*data_swap_db, "create table `deleted_rows` (`id` integer)", true);
    sqlite::execute(*data_swap_db, "create table `changes` (`id` integer primary key autoincrement, `record` integer, `action` integer, `column` integer)", true);
  }

  {
    GMutexLock data_mutex(_data_mutex);
    reinit(_data);
  }
  reinit(_column_names);
  reinit(_column_types);
  reinit(_real_column_types);

  _column_count= 0;
  _row_count= 0;
  _data_frame_begin= 0;
  _data_frame_end= 0;
}


int VarGridModel::floating_point_visible_scale()
{
  grt::DictRef options= grt::DictRef::cast_from(_grtm->get_grt()->get("/wb/options/options"));
  int scale= options.get_int("Recordset:FloatingPointVisibleScale");
  return scale;
}


boost::shared_ptr<sqlite::connection> VarGridModel::data_swap_db() const
{
  if (_grtm->in_main_thread())
    return (_data_swap_db) ? _data_swap_db : _data_swap_db= create_data_swap_db_connection();
  else
    return create_data_swap_db_connection();
}


boost::shared_ptr<sqlite::connection> VarGridModel::create_data_swap_db_connection() const
{
  boost::shared_ptr<sqlite::connection> data_swap_db;
  if (!_data_swap_db_path.empty())
  {
    data_swap_db.reset(new sqlite::connection(_data_swap_db_path));
    sqlide::optimize_sqlite_connection_for_speed(data_swap_db.get());
  }
  return data_swap_db;
}


bool VarGridModel::call_refresh_ui()
{
  refresh_ui_cb();
  return false;
}


int VarGridModel::refresh_ui()
{
  if (_grtm->in_main_thread())
    return refresh_ui_cb();
  else
  {
    //_grtm->run_when_idle(sigc::bind_return(sigc::hide_return(refresh_ui_cb), false));
    // this method is better because it will auto-invalidate the scheduled slot when the object
    // is deleted and also reduce call stack size from sigc wrappers
    _grtm->run_when_idle(sigc::mem_fun(this, &VarGridModel::call_refresh_ui));
    return 0;
  }
}


int VarGridModel::count()
{
  return _row_count + (_readonly ? 0 : 1);
}


class VarType
{
public:
  typedef VarGridModel::ColumnType result_type;
  result_type operator()(const sqlite::BlobRef &) const { return VarGridModel::BlobType; }
  result_type operator()(int) const { return VarGridModel::NumericType; }
  result_type operator()(const boost::int64_t &) const { return VarGridModel::NumericType; }
  result_type operator()(const long double &) const { return VarGridModel::FloatType; }
  template<typename T> result_type operator()(const T &v) const { return VarGridModel::StringType; }
};
VarGridModel::ColumnType VarGridModel::get_column_type(int column)
{
  static VarType vt;
  return boost::apply_visitor(vt, _column_types[column]);
}


VarGridModel::ColumnType VarGridModel::get_real_column_type(int column)
{
  static VarType vt;
  return boost::apply_visitor(vt, _real_column_types[column]);
}


std::string VarGridModel::get_column_caption(int column)
{
  return _column_names.at(column);
}


VarGridModel::Cell VarGridModel::cell(RowId row, ColumnId column)
{
  if (row >= _row_count)
    return _data.end();

  // cache rows if needed
  if ((_data_frame_begin > row) || (_data_frame_end <= row))
    cache_data_frame(row, false);

  // translate to absolute cell address
  RowId cell_index= (row - _data_frame_begin) * _column_count + column;
  return _data.begin() + cell_index;
}


bool VarGridModel::get_cell(VarGridModel::Cell &cell, const NodeId &node, ColumnId column, bool allow_new_row)
{
  if (!node.is_valid())
    return false;

  RowId row= node[0];

  if ( (row > _row_count)
    || (column >= _column_count)
    || (!allow_new_row && (_row_count == row)))
    return false;

  cell= this->cell(row, column);
  return true;
}


bool VarGridModel::is_field_null(const NodeId &node, int column)
{
  GMutexLock data_mutex(_data_mutex);

  // returns true for out of the range addresses
  Cell cell;
  if (get_cell(cell, node, column, false))
  {
    if (sqlide::is_var_blob(_column_types[column]))
      return false;
    else
      return sqlide::is_var_null(*cell);
  }
  else
  {
    return true;
  }
}


bool VarGridModel::set_field_null(const bec::NodeId &node, int column)
{
  return is_field_null(node, column) ? true : set_field(node, column, sqlite::Null());
}


class IconForVal : public boost::static_visitor<IconId>
{
public:
  IconForVal()
  {
    IconManager *icon_man= IconManager::get_instance();
    _null_icon= icon_man->get_icon_id("field_overlay_null.png");
    _blob_icon= icon_man->get_icon_id("field_overlay_blob.png");
  }
private:
  IconId _null_icon;
  IconId _blob_icon;

public:
  template<typename T> result_type operator()(const T &t, const sqlite::Null &v) const { return _null_icon; }
  result_type operator()(const sqlite::Null &v) const { return _null_icon; }
  result_type operator()(const sqlite::BlobRef &t, const sqlite::Null &v) const { return _blob_icon; }
  template<typename V> result_type operator()(const sqlite::BlobRef &t, const V &v) const { return _blob_icon; }

  template<typename T, typename V>
  result_type operator()(const T &t, const V &v) const { return 0; }
};
IconId VarGridModel::get_field_icon(const NodeId &node, int column, IconSize size)
{
  GMutexLock data_mutex(_data_mutex);

  static IconForVal icon_for_val;
  Cell cell;
  static const sqlite::Variant null_value= sqlite::Null();
  const sqlite::Variant &var= get_cell(cell, node, column, false) ? *cell : null_value;
  return boost::apply_visitor(icon_for_val, _column_types[column], var);
}


bool VarGridModel::get_field(const NodeId &node, int column, std::string &value)
{
  GMutexLock data_mutex(_data_mutex);
  return get_field_(node, column, value);
}


bool VarGridModel::get_field_(const NodeId &node, int column, std::string &value)
{
  Cell cell;
  bool res= get_cell(cell, node, column, false);
  if (res)
    value= boost::apply_visitor(_var_to_str, *cell);
  return res;
}


bool VarGridModel::get_field_repr(const NodeId &node, int column, std::string &value)
{
  GMutexLock data_mutex(_data_mutex);
  return get_field_repr_(node, column, value);
}


bool VarGridModel::get_field_repr_(const NodeId &node, int column, std::string &value)
{
  Cell cell;
  bool res= get_cell(cell, node, column, false);
  if (res)
  {
    if (_is_field_value_truncation_enabled)
    {
      int row= (int)node[0];
      _var_to_str_repr.is_truncation_enabled= (row != _edited_field_row) || (column != _edited_field_col);
    }
    value= boost::apply_visitor(_var_to_str_repr, *cell);
  }
  return res;
}


bool VarGridModel::get_field(const NodeId &node, int column, int &value)
{
  GMutexLock data_mutex(_data_mutex);
  return get_field_(node, column, value);
}


bool VarGridModel::get_field_(const NodeId &node, int column, int &value)
{
  Cell cell;
  bool res= get_cell(cell, node, column, false);
  if (res)
    value= (int)boost::apply_visitor(_var_to_int, *cell);
  return res;
}


bool VarGridModel::get_field(const NodeId &node, int column, long long &value)
{
  GMutexLock data_mutex(_data_mutex);
  return get_field_(node, column, value);
}


bool VarGridModel::get_field_(const NodeId &node, int column, long long &value)
{
  Cell cell;
  bool res= get_cell(cell, node, column, false);
  if (res)
    value= boost::apply_visitor(_var_to_int, *cell);
  return res;
}


bool VarGridModel::get_field(const NodeId &node, int column, double &value)
{
  GMutexLock data_mutex(_data_mutex);
  return get_field_(node, column, value);
}


bool VarGridModel::get_field_(const NodeId &node, int column, double &value)
{
  Cell cell;
  bool res= get_cell(cell, node, column, false);
  if (res)
    value= (double)boost::apply_visitor(_var_to_long_double, *cell);
  return res;
}


bool VarGridModel::get_field_grt(const NodeId &node, int column, grt::ValueRef &value)
{
  std::string val;
  bool res= get_field(node, column, val);
  if (res)
    value= grt::StringRef(val);
  return res;
}


bool VarGridModel::set_field(const NodeId &node, int column, const sqlite::Variant &value)
{
  bool res= false;

  {
    GMutexLock data_mutex(_data_mutex);

    Cell cell;
    res= get_cell(cell, node, column, true);
    if (res && !sqlide::is_var_blob(_column_types[column]))
    {
      static const sqlide::VarEq var_eq;
      res= !boost::apply_visitor(var_eq, value, *cell);
      if (res)
        *cell= value;
    }
  }

  if (res)
    after_set_field(node, column, value);

  return res;
}


bool VarGridModel::set_field(const NodeId &node, int column, const std::string &value)
{
  return set_field(node, column, sqlite::Variant(value));
}


bool VarGridModel::set_field(const NodeId &node, int column, double value)
{
  return set_field(node, column, sqlite::Variant((long double)value));
}


bool VarGridModel::set_field(const NodeId &node, int column, int value)
{
  return set_field(node, column, sqlite::Variant(value));
}


bool VarGridModel::set_field(const NodeId &node, int column, long long value)
{
  return set_field(node, column, sqlite::Variant((boost::int64_t)value));
}


void VarGridModel::add_column(const std::string &name, const sqlite::Variant &type)
{
  _column_names.push_back(name);
  _column_types.push_back(type);
  _real_column_types.push_back(type);
  ++_column_count;
}


void VarGridModel::cache_data_frame(RowId center_row, bool force_reload)
{
  static const RowId half_row_count= 100; //! load from options
  RowId row_count= half_row_count * 2;

  // center_row of -1 means only to forcibly reload current data frame
  if (-1 != (int)center_row)
  {
    RowId starting_row= (center_row < half_row_count) ? 0 : (center_row - half_row_count);

    // adjust range borders to comply with row count & cache frame size
    // e.g. shift back and/or resize requested cache frame
    if (starting_row + row_count > _row_count)
    {
      if (row_count < _row_count)
      {
        starting_row= _row_count - row_count;
      }
      else
      {
        starting_row= 0;
        row_count= _row_count;
      }
    }

    if (!force_reload &&
      (_data_frame_begin == starting_row) &&
      (_data_frame_begin != _data_frame_end) &&
      (_data_frame_end - _data_frame_begin == row_count))
    {
      return;
    }

    _data_frame_begin= starting_row;
    _data_frame_end= starting_row + row_count;
  }
  else
  {
    row_count= _data_frame_end - _data_frame_begin;
  }

  _data.clear();

  // load data
  {
    boost::shared_ptr<sqlite::connection> data_swap_db= this->data_swap_db();
    sqlite::query select_data_frame_statement(*data_swap_db, "select d.* from `data` d inner join `data_index` di on (di.`id`=d.`id`) order by di.`rowid` limit ? offset ?");
    select_data_frame_statement % (int)row_count;
    select_data_frame_statement % (int)_data_frame_begin;
    if (select_data_frame_statement.emit())
    {
      boost::shared_ptr<sqlite::result> rs= select_data_frame_statement.get_result();
      int col_count= rs->get_column_count();

      std::vector<bool> blob_columns(col_count);
      for (ColumnId col= 0; col_count > (int) col; ++col)
        blob_columns[col]= sqlide::is_var_blob(_column_types[col]);

      _data.reserve(row_count * col_count);
      do
      {
        for (int n= 0; n < col_count; ++n)
        {
          sqlite::Variant v;
          if (blob_columns[n])
          {
            v= sqlite::Null();
          }
          else
          {
            rs->get_variant(n, v);
            v= boost::apply_visitor(_var_cast, _column_types[n], v);
          }
          _data.push_back(v);
        }
      }
      while (rs->next_row());
    }
  }
}


void VarGridModel::set_edited_field(int row_index, int col_index)
{
  _edited_field_row= row_index;
  _edited_field_col= col_index;
}


bool VarGridModel::is_field_value_truncation_enabled(bool val)
{
  _is_field_value_truncation_enabled= val;
  if (_is_field_value_truncation_enabled)
  {
    grt::DictRef options= grt::DictRef::cast_from(_grtm->get_grt()->get("/wb/options/options"));
    int field_value_truncation_threshold= options.get_int("Recordset:FieldValueTruncationThreshold", 256);
    if (field_value_truncation_threshold < 0)
      _var_to_str_repr.is_truncation_enabled= _is_field_value_truncation_enabled= false;
    else
      _var_to_str_repr.truncation_threshold= field_value_truncation_threshold;
  }
  else
  {
    _var_to_str_repr.is_truncation_enabled= _is_field_value_truncation_enabled;
  }
  return _is_field_value_truncation_enabled;
}
