/* 
 * Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "stdafx.h"

#include "sqlide/wb_sql_editor_form.h"

#include "Overview.h"
#include "FindDialogBE.h"
#include "Wb.h"

namespace MySQL {
namespace Workbench {
  
String^ WbContext::read_state(String^ name, String^ domain, String^ default_value)
{
  std::string value= inner->get_wb()->read_state(NativeToCppStringRaw(name), NativeToCppStringRaw(domain), 
    NativeToCppString(default_value));
  
  return CppStringToNative(value);
}

int WbContext::read_state(String^ name, String^ domain, const int default_value)
{
  int value= inner->get_wb()->read_state(NativeToCppStringRaw(name), NativeToCppStringRaw(domain), 
    default_value);
  
  return value;
}

double WbContext::read_state(String^ name, String^ domain, const double default_value)
{
  double value= inner->get_wb()->read_state(NativeToCppStringRaw(name), NativeToCppStringRaw(domain), 
    default_value);
  
  return value;
}

bool WbContext::read_state(String^ name, String^ domain, const bool default_value)
{
  bool value= inner->get_wb()->read_state(NativeToCppStringRaw(name), NativeToCppStringRaw(domain), 
    default_value);
  
  return value;
}

void WbContext::save_state(String^ name, String^ domain, String^ value)
{
  inner->get_wb()->save_state(NativeToCppStringRaw(name), NativeToCppStringRaw(domain), 
    NativeToCppString(value));
}

void WbContext::save_state(String^ name, String^ domain, const int value)
{
  inner->get_wb()->save_state(NativeToCppStringRaw(name), NativeToCppStringRaw(domain), value);
}

void WbContext::save_state(String^ name, String^ domain, const double value)
{
  inner->get_wb()->save_state(NativeToCppStringRaw(name), NativeToCppStringRaw(domain), value);
}

void WbContext::save_state(String^ name, String^ domain, const bool value)
{
  inner->get_wb()->save_state(NativeToCppStringRaw(name), NativeToCppStringRaw(domain), value);
}

String^ WbContext::read_option_value(String^ model, String^ key, String^ default_value)
{
  std::string raw_result;
  if (inner->get_wb_options_value(NativeToCppStringRaw(model), NativeToCppStringRaw(key), raw_result))
    return CppStringToNativeRaw(raw_result);
  else
    return default_value;
}

//----------------- WbFrontendCallbacks ------------------------------------------------------------

void WbFrontendCallbacks::create_main_form_view_wrapper(const std::string& view_name, bec::UIForm *form_be)
{
  UIForm ^form = nullptr;

  if (0 == view_name.compare(WB_MAIN_VIEW_DB_QUERY))
  {
    form= gcnew MySQL::GUI::Workbench::Db_sql_editor(
      IntPtr(dynamic_cast<::Db_sql_editor*>(form_be)));
  }

  create_main_form_view_delegate(CppStringToNativeRaw(view_name), form);
}

//--------------------------------------------------------------------------------------------------

Object^ WbFrontendCallbacks::GetFromFixedId(IntPtr ip)
{
  GCHandle gcHandle = GCHandle::FromIntPtr(ip);
  return (Object^) gcHandle.Target;
}

//--------------------------------------------------------------------------------------------------

void WbFrontendCallbacks::dock_view(mforms::App *app, mforms::AppView *view, const std::string &position)
{
  WbFrontendCallbacks^ callbacks= (WbFrontendCallbacks^) GetFromFixedId((IntPtr) app->get_data_ptr());
  callbacks->app_command_wrapper(AppCommand::AppDockView, view, position);
}

//--------------------------------------------------------------------------------------------------

void WbFrontendCallbacks::undock_view(mforms::App *app, mforms::AppView *view)
{
  WbFrontendCallbacks^ callbacks= (WbFrontendCallbacks^) GetFromFixedId((IntPtr) app->get_data_ptr());
  callbacks->app_command_wrapper(AppCommand::AppUndockView, view, "");
}

//--------------------------------------------------------------------------------------------------

void WbFrontendCallbacks::set_view_title(mforms::App *app, mforms::AppView *view, const std::string &title)
{
  WbFrontendCallbacks^ callbacks= (WbFrontendCallbacks^) GetFromFixedId((IntPtr) app->get_data_ptr());
  callbacks->app_command_wrapper(AppCommand::AppSetViewTitle, view, title);
}

//--------------------------------------------------------------------------------------------------

std::string WbFrontendCallbacks::get_resource_path(mforms::App *app, const std::string &file)
{
  WbFrontendCallbacks^ callbacks= (WbFrontendCallbacks^) GetFromFixedId((IntPtr) app->get_data_ptr());
  return callbacks->app_command_wrapper(AppCommand::AppGetResourcePath, nullptr, file);
}

//--------------------------------------------------------------------------------------------------

void WbFrontendCallbacks::set_status_text(mforms::App *app, const std::string &text)
{
  WbFrontendCallbacks^ callbacks= (WbFrontendCallbacks^) GetFromFixedId((IntPtr) app->get_data_ptr());
  callbacks->app_command_wrapper(AppCommand::AppSetStatusText, nullptr, text);
}

//--------------------------------------------------------------------------------------------------

std::string WbFrontendCallbacks::get_bounds(mforms::App *app)
{
  WbFrontendCallbacks^ callbacks= (WbFrontendCallbacks^) GetFromFixedId((IntPtr) app->get_data_ptr());
  return callbacks->app_command_wrapper(AppCommand::AppGetBounds, nullptr, "");
}

//--------------------------------------------------------------------------------------------------

void WbFrontendCallbacks::set_app_command(MySQL::Forms::Manager^ mgr, StringAppCommandAppViewStringDelegate^ dt)
{
  app_command_delegate= dt;

  // Store a reference to this class in the backend.
  _cb_handle = GCHandle::Alloc(this);
  IntPtr pointer= GCHandle::ToIntPtr(_cb_handle);
  mforms::App::get()->set_data((void*) (intptr_t) pointer);

  ::mforms::ControlFactory *f= ::mforms::ControlFactory::get_instance();

  DEF_CALLBACK3(void, ::mforms::App*, ::mforms::AppView*, const std::string&, mgr, f->_app_impl, WbFrontendCallbacks, dock_view);
  DEF_CALLBACK2(void, ::mforms::App*, ::mforms::AppView*, mgr, f->_app_impl, WbFrontendCallbacks, undock_view);
  DEF_CALLBACK3(void, ::mforms::App*, ::mforms::AppView*, const std::string&, mgr, f->_app_impl, WbFrontendCallbacks, set_view_title);
  DEF_CALLBACK2(std::string, ::mforms::App*, const std::string&, mgr, f->_app_impl, WbFrontendCallbacks, get_resource_path);
  DEF_CALLBACK2(void, ::mforms::App*, const std::string&, mgr, f->_app_impl, WbFrontendCallbacks, set_status_text);
  DEF_CALLBACK1(std::string, ::mforms::App*, mgr, f->_app_impl, WbFrontendCallbacks, get_bounds);
}

//--------------------------------------------------------------------------------------------------

} // namespace Workbench
} // namespace MySQL