/* 
 * (c) 2010 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include "stdafx.h"
#include "grt_shell_window.h"

//--------------------------------------------------------------------------------------------------

GRTShellWindow::GRTShellWindow(bec::GRTManager *grtm)
: mforms::Form(0, (mforms::FormFlag)(mforms::FormResizable | mforms::FormMinimizable | mforms::FormHideOnClose)), 
  _grtm(grtm),
  _hsplitter(true),
  _global_box1(false), _global_box2(false), _global_splitter(false), _global_combo(mforms::SelectorCombobox), _global_tree(false), _global_list(true), 
  _classes_box(false), _classes_splitter(false), _classes_tree(false), _classes_text(mforms::VerticalScrollBar),
  _modules_splitter(false), _modules_tree(false), _modules_text(mforms::VerticalScrollBar),
  _shell_box(false), _shell_text(mforms::VerticalScrollBar), _shell_hbox(true)
{
  set_title(("Workbench Scripting Shell"));
  set_content(&_hsplitter);
  
  _hsplitter.add(&_side_tab);
  _hsplitter.add(&_main_tab);

  // Side bar consists of 3 pages: globals tree, classes tree and modules tree.
  _first_show= true;

  // 1) Globals tree
  _global_tree_be= 0;
  _side_tab.add_page(&_global_splitter, "Globals");
  _global_splitter.add(&_global_box1, 0);
  _global_splitter.add(&_global_box2, 0);
  _global_box1.set_spacing(4);
  _global_box2.set_spacing(4);
  _global_box1.add(&_global_combo, false, false);
  _global_box1.add(&_global_tree, true, true);

  _global_box2.add(&_global_entry, false, true);
  _global_entry.set_read_only(true);
  _global_entry.set_back_color("#FFFFFF");
  _global_box2.add(&_global_list, true, true);
  _global_list.add_column(mforms::IconStringGRTColumnType, 0, "Name");
  _global_list.add_column(mforms::StringGRTColumnType, 1, "Value");
  _global_list.set_column_width(0, 100);
  _global_list.set_column_width(1, 100);
  _global_tree.add_column(mforms::IconStringGRTColumnType, 0, "GRT Globals");
  _global_tree.set_column_width(0, 220);
  _global_tree.signal_changed().connect(sigc::mem_fun(this, &GRTShellWindow::global_selected));
  _global_combo.signal_changed().connect(sigc::mem_fun(this, &GRTShellWindow::change_global_path));
  _inspector= 0;

  // 2) Classes tree
  _side_tab.add_page(&_classes_splitter, "Classes");
  _classes_splitter.add(&_classes_box, 0);
  _classes_box.set_spacing(4);
  _classes_box.add(&_classes_sorting, false, true);
  _classes_box.add(&_classes_tree, true, true);

  _classes_splitter.add(&_classes_text, 0);
  _classes_text.set_read_only(true);
  _classes_text.set_back_color("#FFFFFF");
  _classes_tree.add_column(mforms::IconStringGRTColumnType, bec::StructsTreeBE::Name, "Name");
  _classes_tree.add_column(mforms::StringGRTColumnType, bec::StructsTreeBE::Type, "Type");
  _classes_tree.set_column_width(0, 150);
  _classes_tree.signal_changed().connect(sigc::mem_fun(this, &GRTShellWindow::class_selected));
  _classes_sorting.add_item("Group by Name");
  _classes_sorting.add_item("Group by Hierarchy");
  _classes_sorting.add_item("Group by Package");
  _classes_sorting.signal_changed().connect(sigc::mem_fun(this, &GRTShellWindow::change_class_sorting));
  
  // 3) Modules tree
  _side_tab.add_page(&_modules_splitter, "Modules");
  _modules_splitter.add(&_modules_tree, 0);
  _modules_splitter.add(&_modules_text, 0);
  _modules_text.set_read_only(true);
  _modules_text.set_back_color("#FFFFFF");
  _modules_tree.add_column(mforms::IconStringGRTColumnType, 0, "GRT Modules");
  _modules_tree.set_column_width(0, 220);
  _modules_tree.signal_changed().connect(sigc::mem_fun(this, &GRTShellWindow::module_selected));
  
  // setup shell
  _shell_box.add(&_shell_text, true, true);
  _shell_text.set_read_only(true);
  _shell_text.set_front_color("#E0E0E0");
  _shell_text.set_back_color("#000000");
  _shell_box.add(&_shell_hbox, false, true);
  _shell_hbox.add(&_shell_prompt, false, true);
  _shell_hbox.add(&_shell_entry, true, true);
  _main_tab.add_page(&_shell_box, "Shell");
  
  _shell_entry.signal_action().connect(sigc::mem_fun(this, &GRTShellWindow::shell_action));

  set_size(800, 600);
  _hsplitter.set_position(250);
  _global_splitter.set_position(400);
  _modules_splitter.set_position(400);
  _classes_splitter.set_position(400);
  
  _grtm->get_shell()->set_ready_handler(sigc::mem_fun(this, &GRTShellWindow::handle_prompt));
  _grtm->get_shell()->set_output_handler(sigc::mem_fun(this, &GRTShellWindow::handle_output));
}

//--------------------------------------------------------------------------------------------------

void GRTShellWindow::shell_action(mforms::TextEntryAction action)
{ 
  switch (action)
  {
    case mforms::EntryActivate:
    {
      std::string command= _shell_entry.get_string_value();
      _shell_entry.set_value("");
      //  _completion->add_completion_text(command);
      command += '\n';
      _grtm->get_shell()->write(_grtm->get_grt()->get_shell()->get_prompt()+" "+command);
      _grtm->get_shell()->process_line_async(command);
      break;
    }
    case mforms::EntryKeyUp:
    {
      std::string line;
      if (_grtm->get_shell()->previous_history_line(_shell_entry.get_string_value(), line))
        _shell_entry.set_value(line);
      break;
    }
    case mforms::EntryCKeyUp:
      break;
    case mforms::EntryKeyDown:
    {
      std::string line;
      if (_grtm->get_shell()->next_history_line(line))
        _shell_entry.set_value(line);
      break;
    }      
    case mforms::EntryCKeyDown:
      break;
  }
}

void GRTShellWindow::show(bool flag)
{
  if (flag)
    refresh_all();
  mforms::Form::show(flag);
}


void GRTShellWindow::refresh_all()
{
  if (_first_show)
  {
    _first_show = false;
    set_globals_tree_path(_global_combo.get_string_value());
    _classes_tree.set_model(_grtm->get_shared_structs_tree());
    _modules_tree.set_model(_grtm->get_shared_modules_tree());
  }

  int idx= 0;
  std::string root = _global_tree_be->get_path_for_node(bec::NodeId(), true);
  std::vector<std::string> l = _grtm->get_shell()->get_grt_tree_bookmarks();
  _global_combo.clear();
  for (std::vector<std::string>::const_iterator i = l.begin(); i != l.end(); ++i, ++idx)
  {
    _global_combo.add_item(*i);
    if (root == *i)
      _global_combo.set_selected(idx);
  }
  
  //refresh values
  _global_tree.refresh(bec::NodeId());
  _grtm->get_shared_structs_tree()->refresh();
  _grtm->get_shared_modules_tree()->refresh();
  
///  _global_tree.expand_row(Gtk::TreePath("0"), false);
  _global_list.refresh(bec::NodeId());
  
  //refresh _struct
  _classes_tree.refresh(bec::NodeId());
  
  // refresh modules
  _modules_tree.refresh(bec::NodeId());  
}


void GRTShellWindow::handle_output(const std::string &text)
{
  _shell_text.append_text(text, true);
}


void GRTShellWindow::handle_error(const std::string &text, const std::string &detail)
{
  _shell_text.append_text(text);
  
  _shell_text.append_text(detail);
  
}


void GRTShellWindow::handle_prompt(const std::string &text)
{
  _shell_prompt.set_text(text);
}



void GRTShellWindow::set_globals_tree_path(const std::string &path)
{
  try
  {
    if (path.empty() || path[0] != '/')
      _global_tree_be = _grtm->get_shared_value_tree("/");
    else
      _global_tree_be = _grtm->get_shared_value_tree(path);
    _global_tree.set_model(_global_tree_be);
    _global_tree.refresh(bec::NodeId());
    _global_list.refresh(bec::NodeId());
  }
  catch (std::exception &exc)
  {
    g_message("error: %s", exc.what());
  }
}

void GRTShellWindow::change_global_path()
{
  set_globals_tree_path(_global_combo.get_string_value());
}

void GRTShellWindow::change_class_sorting()
{
  _grtm->get_shared_structs_tree()->set_display_mode((bec::StructsTreeBE::DisplayMode)_classes_sorting.get_selected_index());
  _classes_tree.refresh(bec::NodeId());
}


void GRTShellWindow::global_selected()
{
  if (_inspector)
  {
    _global_list.set_model(0);
    delete _inspector;
    _inspector= 0;
  }
  
  bec::NodeId selected;
  
  if (_global_tree.get_selected_node(selected))
  {
    grt::ValueRef value(_global_tree_be->get_node_value(selected));
    
    if (value.is_valid())
    {
      _inspector= _grtm->get_new_value_inspector(value, false);
      _global_list.set_model(_inspector);
    }
    
    _global_entry.set_value(_global_tree_be->get_path_for_node(selected, true));
  }
  else
    _global_entry.set_value("");
}

void GRTShellWindow::class_selected()
{
  bec::NodeId selected;
  
  if (_classes_tree.get_selected_node(selected))
  {
    std::string text;
   
    _grtm->get_shared_structs_tree()->get_field(selected, bec::StructsTreeBE::Caption, text);
    text.append("\n");
    text.append(_grtm->get_shared_structs_tree()->get_field_description(selected, 0));
   
    _classes_text.set_value(text);
  }
  else
    _classes_text.set_value("");
}

void GRTShellWindow::module_selected()
{
  bec::NodeId selected;
  
  if (_modules_tree.get_selected_node(selected))
  {
    std::string text(_grtm->get_shared_modules_tree()->get_field_description(selected, 0));
    
    _modules_text.set_value(text);
  }
  else
    _modules_text.set_value("");
}
