/* 
 * (c) 2009-2010 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "stdafx.h"

#include "string_utilities.h"
#include "new_server_instance_wizard.h"
#include "grt/grt_manager.h"
#include "grtui/grtdb_connection_editor.h"

#define INTRO_TEXT \
"This wizard will guide you to create of a Server Profile to manage a MySQL server.\n"\
"To fully support management of a remote MySQL server, a SSH daemon must be running\n"\
"in the target machine. The SSH login is used to start, stop and configure MySQL.\n"\
"You may create a Profile without SSH if you do not need that functionality."

static struct {
  const char *pattern;
  const char *os_name;
} platform_strings[] = {
{"apple-darwin", "MacOS X"},
{"-linux", "Linux"},
{"Win64", "Windows"},
{"Win32", "Windows"},
{NULL, NULL},
};

using namespace base;

//----------------- HostMachinePage ----------------------------------------------------------------

HostMachinePage::HostMachinePage(WizardForm* host, wb::WBContext *wb)
: WizardPage(host, "host machine page"), _context(wb), _top_box(false), 
_top_border(BorderedPanel),
_localhost_button(0), _networkhost_button(0), _dbhost_button(0),
_ip_address_block(true), _dbhost_block(true)
{
  set_short_title(_("Specify Host Machine"));
  set_title(_("Specify the Host Machine the Database Server is running on"));

  set_spacing(12);

  _intro_label.set_text(INTRO_TEXT);
  add(&_intro_label, false, true);
  
  _host_description.set_wrap_text(true);
  _host_description.set_text(_("If your database server is running on the same machine as this application "
    "select localhost. Otherwise please specify the TCP/IP address or the network "
    "name of the remote machine. You may also pick an existing database connection."));
  _top_box.add(&_host_description, false, true);

  _localhost_button.set_text(_("localhost"));
  _localhost_button.signal_toggled().connect(sigc::mem_fun(this, &HostMachinePage::host_selection_changed));
  _top_box.add(&_localhost_button, false, true);
  _networkhost_button.set_text(_("Remote Host"));
  _networkhost_button.set_active(true);
  _networkhost_button.signal_toggled().connect(sigc::mem_fun(this, &HostMachinePage::host_selection_changed));
  _top_box.add(&_networkhost_button, false, true);
  
  Label *indent;

  indent = manage(new Label(""));
  indent->set_size(20, -1);
  _dbhost_block.add(indent, false, true);
  _dbhost_block.add(&_dbhost_selector, true, true);

  indent = manage(new Label(""));
  indent->set_size(20, -1);
  _ip_address_block.add(indent, false, true);
  _ip_address_label.set_text(_("Address:"));
  _ip_address_block.add(&_ip_address_label, false, true);
  _ip_address_block.add(&_ip_address, true, true);
  _ip_address_help.set_text(_("Either IP Address or Hostname"));
  _ip_address_help.set_style(mforms::SmallHelpTextStyle);
  _ip_address_block.add(&_ip_address_help, false, true);
  _ip_address_block.set_spacing(8);
  _top_box.add(&_ip_address_block, false, true);
  
  _dbhost_button.set_text(_("Take Parameters from Existing Database Connection"));
  _dbhost_button.set_active(true);
  _dbhost_button.signal_toggled().connect(sigc::mem_fun(this, &HostMachinePage::host_selection_changed));
  _top_box.add(&_dbhost_button, false, true);
  _top_box.add(&_dbhost_block, false, true);
  
  _top_box.add(manage(new Label("")), false, true);
  
  _localhost_button.set_active(true);
  host_selection_changed();
  
  _top_box.set_spacing(8);
  _top_box.set_padding(8);
  _top_border.set_padding(MF_PANEL_PADDING);
  _top_border.add(&_top_box);

  add(&_top_border, false, true);
  
  refresh_connections();
}

//--------------------------------------------------------------------------------------------------

void HostMachinePage::host_selection_changed()
{
  bool flag= _networkhost_button.get_active();
  _ip_address_block.set_enabled(flag);

  flag = _dbhost_button.get_active();
  _dbhost_block.set_enabled(flag);  
}

//--------------------------------------------------------------------------------------------------

void HostMachinePage::refresh_connections()
{
  grt::ListRef<db_mgmt_Connection> connections(_context->get_root()->rdbmsMgmt()->storedConns());
  
  _dbhost_selector.clear();
  for (grt::ListRef<db_mgmt_Connection>::const_iterator end = connections.end(),
       inst = connections.begin(); inst != end; ++inst)
  {
    grt::DictRef dict((*inst)->parameterValues());
    std::string tunnel;
    
    if ((*inst)->driver().is_valid())
    {
      if ((*inst)->driver()->name() == "MysqlNativeSSH")
        tunnel = "(ssh tunnel)";
      
      std::string text = strfmt("%s (User: %s  Host: %s:%i%s)",
                                (*inst)->name().c_str(),
                                dict.get_string("userName").c_str(),
                                dict.get_string("hostName").c_str(),
                                (int) dict.get_int("port"),
                                tunnel.c_str());
      
      _dbhost_selector.add_item(text);
    }
  } 
  
  _dbhost_button.set_enabled(connections.count() > 0);
}

//--------------------------------------------------------------------------------------------------

bool HostMachinePage::advance()
{
  // Localhost doesn't need further attention, but general network name does.
  std::string machine;
  if (_networkhost_button.get_active())
  {
    machine= base::trim(_ip_address.get_string_value());
    if (machine.size() == 0)
    {
      Utilities::show_error(_("Invalid host name"), _("Target machine name or address is missing."), _("OK"));
      return false;
    }
    
    _form->values().set("connection", grt::ValueRef());
  }
  else if (_dbhost_button.get_active())
  {
    int i = _dbhost_selector.get_selected_index();
    if (i < 0 || i >= (int)_context->get_root()->rdbmsMgmt()->storedConns().count())
    {
      Utilities::show_error(_("Invalid connection"), _("Please select a valid DB connection."), _("OK"));
      return false;
    }

    db_mgmt_ConnectionRef connection(_context->get_root()->rdbmsMgmt()->storedConns()[i]);
     
    if (connection->driver()->name() == "MysqlNativeSSH")
    {
      machine = connection->parameterValues().get_string("sshHost");
      if (machine.find(':') != std::string::npos)
        machine = machine.substr(0, machine.find(':'));
    }
    else if (connection->driver()->name() == "MysqlNativeSocket")
      machine = "localhost";
    else
      machine = connection->parameterValues().get_string("hostName");
    
    _form->values().set("connection", connection);
  }
  else
  {
    _form->values().set("connection", grt::ValueRef());
    machine = "localhost";
  }

  _form->values().gset("host_name", machine);

  return true;
}


//----------------- DatabaseServerPage -------------------------------------------------------------

DatabaseServerPage::DatabaseServerPage(WizardForm* host, wb::WBContext* context)
: WizardPage(host, "database server page"), _panel(false)
{
  _context= context;
  
  set_short_title(_("Database Connection"));
  set_title(_("Set the Database Connection values"));
  
  _conn_name = 0;
  
  add(&_panel, true, true);
}

//--------------------------------------------------------------------------------------------------

bool DatabaseServerPage::skip_page()
{
  return _form->values().get("connection").is_valid();
}

//--------------------------------------------------------------------------------------------------

void DatabaseServerPage::enter(bool advancing)
{
  if (advancing)
  {
    db_mgmt_ConnectionRef conn(_context->get_grt());
    conn->name(_form->values().get_string("host_name", "new_connection"));
    conn->parameterValues().gset("hostName", _form->values().get_string("host_name", "new_connection"));
    
    if (!_conn_name)
    {
      _panel.init(_context->get_root()->rdbmsMgmt(), conn);
    
      _conn_name= _panel.get_name_entry();
    }
    else
    {
      _panel.get_be()->set_connection(conn);
    }
    _conn_name->set_value(conn->name());    
  }
}

//--------------------------------------------------------------------------------------------------

bool DatabaseServerPage::advance()
{
  return true;
}

//--------------------------------------------------------------------------------------------------

void DatabaseServerPage::leave(bool advancing)
{
  if (advancing)
  {
    _panel.get_be()->save_changes();

    db_mgmt_ConnectionRef connection(_panel.get_connection());
    _form->values().set("connection", connection);

    std::string machine;
    if (connection->driver()->name() == "MysqlNativeSSH")
    {
      machine = connection->parameterValues().get_string("sshHost");
      if (machine.find(':') != std::string::npos)
        machine = machine.substr(0, machine.find(':'));
    }
    else if (connection->driver()->name() == "MysqlNativeSocket")
      machine = "localhost";
    else
      machine = connection->parameterValues().get_string("hostName");    
    _form->values().gset("host_name", machine);
    
    // insert the connection to the conn list
  }
}

//----------------- TestDatabaseSettingsPage -------------------------------------------------------

TestDatabaseSettingsPage::TestDatabaseSettingsPage(WizardForm* host)
: WizardProgressPage(host, "test database settings page")
{
  set_short_title(_("Test DB Connection"));
  set_title(_("Testing the Database Connection"));
  
  set_heading(_("The database connection information is being tested. This might take a few "
                "moments depending on your network connection."));
  
  grtui::WizardProgressPage::TaskRow *task;
  
  task = add_task(_("Open Database Connection"),
                  sigc::mem_fun(this, &TestDatabaseSettingsPage::open_connection),
                  _("Connecting to database server..."));
  
  add_task(_("Get Server Version"),
           sigc::mem_fun(this, &TestDatabaseSettingsPage::get_server_version),
           _("Querying server version..."));
  
  add_task(_("Get Server OS"),
           sigc::mem_fun(this, &TestDatabaseSettingsPage::get_server_platform),
           _("Querying server OS type..."));
  
  end_adding_tasks(true, _("Database connection tested successfully."));
  
  set_status_text("");
}

//--------------------------------------------------------------------------------------------------

bool TestDatabaseSettingsPage::open_connection()
{
  try
  {
    db_mgmt_ConnectionRef conn(db_mgmt_ConnectionRef::cast_from(_form->values().get("connection")));
    add_log_text(strfmt("Connecting to MySQL server %s...", conn->name().c_str()));
    _dbc_conn = sql::DriverManager::getDriverManager()->getConnection(conn);
    add_log_text("Connected.");
  }
  catch (std::exception &exc)
  {
    _message = exc.what();
    add_log_text(_message.c_str());
    throw;
  }
  
  return true;
}

//--------------------------------------------------------------------------------------------------

void TestDatabaseSettingsPage::tasks_finished(bool success)
{
  if (!success)
    set_status_text(strfmt("Could not connect to MySQL server: %s\nYou may continue if the server is simply not running.", 
                           _message.c_str()), true);
}

//--------------------------------------------------------------------------------------------------

bool TestDatabaseSettingsPage::get_server_version()
{
  sql::Statement *pstmt = _dbc_conn->createStatement();
  sql::ResultSet *res = pstmt->executeQuery("SELECT VERSION() as VERSION");
  std::string version;
  
  if (res && res->next())
  {
    version = res->getString("VERSION");
  }
  delete res;
  delete pstmt;
  
  if (version.empty())
  {
    current_task()->label.set_text("Get Server Version: Failed");
    throw std::runtime_error("Error querying version of MySQL server");
  }
  
  _form->values().gset("server_version", version);
  
  current_task()->label.set_text("Get Server Version: "+version);
  add_log_text(strfmt("MySQL server version is %s", version.c_str()));
  
  int major, minor;
  if (sscanf(version.c_str(), "%i.%i", &major, &minor) == 2 && major < 5)
  {
    current_task()->label.set_text("Get Server Version: Unsupported Server Version");
    add_log_text(strfmt("MySQL %i.%i is older than 5.0 and is not supported by MySQL Workbench", major, minor));
    throw std::runtime_error(strfmt("MySQL %i.%i is older than 5.0 and is not supported by Workbench", major, minor));
  }
  
  return true;
}

//--------------------------------------------------------------------------------------------------

bool TestDatabaseSettingsPage::get_server_platform()
{
  sql::Statement *pstmt = _dbc_conn->createStatement();
  sql::ResultSet *res = pstmt->executeQuery("SHOW VARIABLES LIKE 'version_compile_%'");
  std::string name, value;
  std::string machine, os;
  
  while (res && res->next())
  {
    name = res->getString("Variable_name");
    value = res->getString("Value");
    
    if (name == "version_compile_machine")
      machine = value;
    if (name == "version_compile_os")
      os = value;
  }
  delete res;
  delete pstmt;
  
  _dbc_conn.reset();
  
  std::string os_type = "unknown";
  
  for (int i= 0; platform_strings[i].pattern; i++)
  {
    if (strstr(os.c_str(), platform_strings[i].pattern))
    {
      os_type = platform_strings[i].os_name;
      _form->values().gset("detected_os_type", os_type);
      break;
    }
  }
  
  current_task()->label.set_text("Get Server OS: "+os_type);
  
  add_log_text(strfmt("MySQL server architecture is %s", machine.empty() ? "unknown" : machine.c_str()));
  add_log_text(strfmt("MySQL server OS is %s", os.empty() ? "unknown" : os.c_str()));
  
  return true;
}

void TestDatabaseSettingsPage::enter(bool advancing)
{
  if (advancing)
  {
    _form->values().remove("server_version");
    _form->values().remove("detected_os_type");
  }
  WizardProgressPage::enter(advancing);
}

//----------------- HostTypePage ---------------------------------------------------------------------

HostTypePage::HostTypePage(WizardForm* host)
: WizardPage(host, "os page"), _top_box(false), _top_border(BorderedPanel)
{
  set_short_title(_("Operating System"));
  set_title(_("Specify the operating system of the machine"));
  
  _os_description.set_wrap_text(true);
  _os_description.set_text(_("Please select the operating system that is running on the target machine and the type "
                             "of database installation. If you are unsure about the type of database "
                             "installation select the (Vendor Package) variant. If your specific operating "
                             "system is not in this list, select a related variant."));
  _top_box.add(&_os_description, false, true);
  
  _params.set_row_count(2);
  _params.set_column_count(2);
  _params.set_row_spacing(MF_TABLE_ROW_SPACING);
  _params.set_column_spacing(MF_TABLE_COLUMN_SPACING);
  
  _os_label.set_text_align(mforms::MiddleRight);
  _os_label.set_text(_("Operating System:"));
  _params.add(&_os_label, 0, 1, 0, 1, mforms::HFillFlag);
  _params.add(&_os_selector, 1, 2, 0, 1, mforms::HFillFlag|mforms::HExpandFlag);
  _os_selector.signal_changed().connect(sigc::mem_fun(this, &HostTypePage::refresh_profile_list));

  _type_label.set_text_align(mforms::MiddleRight);
  _type_label.set_text(_("MySQL Installation Type:"));
  _params.add(&_type_label, 0, 1, 1, 2, mforms::HFillFlag);
  _params.add(&_type_selector, 1, 2, 1, 2, mforms::HFillFlag|mforms::HExpandFlag);
  
  _top_box.add(&_params, true, true);
  _top_box.set_spacing(10);
  _top_border.set_padding(8);
  _top_border.add(&_top_box);
  
  add(&_top_border, false, true);
}

//--------------------------------------------------------------------------------------------------

bool HostTypePage::skip_page()
{
  return !((NewServerInstanceWizard*)_form)->is_admin_enabled();
}

//--------------------------------------------------------------------------------------------------

void HostTypePage::enter(bool advancing)
{
  int last_selected= _os_selector.get_selected_index();
  
  suspend_layout();
  _os_selector.suspend_layout();
  
  // Refresh platform list on each enter. This way a change on disk can be reflected without
  // restarting the wizard.
  std::string path= _form->grtm()->get_data_file_path("mysql.profiles");
  GDir *dir = g_dir_open(path.c_str(), 0, NULL);
  if (dir)
  {
    const gchar *file;
    while ((file = g_dir_read_name(dir)))
    {
      if (g_str_has_suffix(file, ".xml"))
      {
        std::string fname= std::string(file, strlen(file)-4);
        std::string label= bec::replace_string(fname, "_", " ");          
        grt::DictRef dict;
        try
        {
          dict= grt::DictRef::cast_from(_form->grtm()->get_grt()->unserialize(path+"/"+file));
        }
        catch (std::exception &exc)
        {
          g_warning("Profile %s contains invalid data: %s", path.c_str(), exc.what());
          continue;
        }
        _presets[dict.get_string("sys.system")].push_back(std::make_pair(label, path+"/"+file));
      }
    }
    g_dir_close(dir);
  }  
  else
    g_warning("Opening profiles folder failed.");
  
  std::string detected_os_type = _form->values().get_string("detected_os_type");
  std::string host = _form->values().get_string("host_name");
  if (detected_os_type.empty() && (host == "localhost" || host == "127.0.0.1"))
  {
#ifdef _WIN32
    detected_os_type = "Windows";
#elif defined(__APPLE__)
    detected_os_type = "MacOS X";
#else
    detected_os_type = "Linux";
#endif
  }

  _os_selector.clear();
  int i = 0;
  for (std::map<std::string, std::vector<std::pair<std::string,std::string> > >::const_iterator it = _presets.begin();
       it != _presets.end(); ++it, ++i)
  {
    _os_selector.add_item(it->first);
    if (advancing)
    {
      if (detected_os_type == it->first)
        last_selected = i;
    }
  }
  
  if (last_selected < 0)
    last_selected= 0;
  _os_selector.set_selected(last_selected);
  
  _os_selector.resume_layout();
  resume_layout();
  
  refresh_profile_list();
}


void HostTypePage::refresh_profile_list()
{
  std::string system = _os_selector.get_string_value();
  
  _type_selector.clear();
  for (std::vector<std::pair<std::string,std::string> >::const_iterator iter= _presets[system].begin();
       iter != _presets[system].end(); ++iter)
    _type_selector.add_item(iter->first);
}

//--------------------------------------------------------------------------------------------------

bool HostTypePage::advance()
{
  NewServerInstanceWizard* wizard= dynamic_cast<NewServerInstanceWizard*>(_form);
  std::string system = _os_selector.get_string_value();
  
  int selected_index = _type_selector.get_selected_index();
  if (selected_index == -1)
    throw std::runtime_error("MySQL installation type not selected");
  wizard->values().gset("template_path", _presets[system][selected_index].second);
  wizard->values().gset("template", _presets[system][selected_index].first);
  wizard->values().gset("os", system);
  wizard->load_defaults();
  
  return true;
}

//--------------------------------------------------------------------------------------------------

HostMachineConnectionPage::HostMachineConnectionPage(WizardForm* host)
: WizardPage(host, "host machine connection page"), _password(PasswordEntry), _indent(false)
{
  set_short_title(_("Host SSH Connection"));
  set_title(_("Set remote configuration parameters"));
  
  set_spacing(10);
  _main_description1.set_wrap_text(true);
  _main_description1.set_text(_("In order to remotely configure this database instance an SSH account on this "
                                "host with appropriate privileges is required. This account needs write access "
                                "to the my.cnf database config file, read access to the database logs and "
                                "privileges to start/stop the database daemon.\n\n"
                                "If you do not have this information or you do not want to remotely configure "
                                "the database instance please disable the following checkbox."));
  add(&_main_description1, false, true);
  
  _enable_ssh_checkbox.set_active(true);
  _enable_ssh_checkbox.set_text(_("Enable SSH login based administration"));
  _enable_ssh_checkbox.signal_clicked().connect(sigc::mem_fun(this, &HostMachineConnectionPage::ssh_selection_changed));
  add(&_enable_ssh_checkbox, false, true);
  
  _ssh_settings_table.set_row_count(6);
  _ssh_settings_table.set_row_spacing(5);
  _ssh_settings_table.set_column_count(5);
  _ssh_settings_table.set_column_spacing(5);
  
  _indent.set_size(20, -1);
  _ssh_settings_table.add(&_indent, 0, 1, 0, 1, HFillFlag);
  
  _host_name_label.set_text(_("Host Name:"));
  _ssh_settings_table.add(&_host_name_label, 1, 2, 0, 1, HFillFlag);
  _ssh_settings_table.add(&_host_name, 2, 3, 0, 1, HFillFlag | HExpandFlag);
  _port_label.set_text(_("Port:"));
  _ssh_settings_table.add(&_port_label, 3, 4, 0, 1, HFillFlag);
  _port.set_size(50, -1);
  _port.set_value("22");
  _ssh_settings_table.add(&_port, 4, 5, 0, 1, HFillFlag);
  
  _username_label.set_text(_("User Name:"));
  _ssh_settings_table.add(&_username_label, 1, 2, 1, 2, HFillFlag);
  _ssh_settings_table.add(&_username, 2, 3, 1, 2, HFillFlag | HExpandFlag);
  
  _password_label.set_text(_("Password:"));
  _ssh_settings_table.add(&_password_label, 1, 2, 2, 3, HFillFlag);
  _ssh_settings_table.add(&_password, 2, 3, 2, 3, HFillFlag | HExpandFlag);
  _password_info.set_style(SmallStyle);
  _password_info.set_text(_("Leave blank to enter when connecting"));
  _ssh_settings_table.add(&_password_info, 2, 3, 3, 4, HFillFlag);
  
  _use_ssh_key.set_text(_("Authenticate Using SSH Key"));
  _use_ssh_key.signal_clicked().connect(sigc::mem_fun(this, &HostMachineConnectionPage::use_ssh_key_changed));
  _ssh_settings_table.add(&_use_ssh_key, 1, 5, 4, 5, HFillFlag);
  
  _ssh_path_label.set_text(_("SSH Key Path:"));
  _ssh_settings_table.add(&_ssh_path_label, 1, 2, 5, 6, HFillFlag);
  _ssh_settings_table.add(&_ssh_key_path, 2, 3, 5, 6, HFillFlag | HExpandFlag);
  _ssh_settings_table.add(&_ssh_key_browse_button, 3, 4, 5, 6, 0);
  
  _file_selector= mforms::manage(new FsObjectSelector(&_ssh_key_browse_button, &_ssh_key_path));
  std::string homedir=
#ifdef _WIN32
    mforms::Utilities::get_special_folder(mforms::ApplicationData);
#else
    "~";
#endif
  _file_selector->initialize(homedir + "/.ssh/id_rsa", mforms::OpenFile, "", "...", 
    sigc::mem_fun(this, &WizardPage::validate));
  use_ssh_key_changed();
  
  add(&_ssh_settings_table, false, true);
}

//--------------------------------------------------------------------------------------------------

void HostMachineConnectionPage::ssh_selection_changed()
{
  _ssh_settings_table.set_enabled(_enable_ssh_checkbox.get_active());
  use_ssh_key_changed();
}

//--------------------------------------------------------------------------------------------------

void HostMachineConnectionPage::use_ssh_key_changed()
{
  bool value= _use_ssh_key.get_active() && _enable_ssh_checkbox.get_active();
  _ssh_path_label.set_enabled(value);
  _ssh_key_path.set_enabled(value);
  _ssh_key_browse_button.set_enabled(value);
}

//--------------------------------------------------------------------------------------------------

void HostMachineConnectionPage::enter(bool advancing)
{
  if (advancing)
  {
    _host_name.set_value(_form->values().get_string("host_name"));
    const char *username = g_get_user_name();
    _username.set_value(username ? username : "");
  }
}

//--------------------------------------------------------------------------------------------------

bool HostMachineConnectionPage::advance()
{
  NewServerInstanceWizard *wizard = dynamic_cast<NewServerInstanceWizard*>(_form);
  db_mgmt_ServerInstanceRef instance(wizard->assemble_server_instance());
  
  if (_enable_ssh_checkbox.get_active())
  {
    // Check if we have valid SSH credentials.
    std::string value= base::trim(_host_name.get_string_value());
    if (value.empty())
    {
      Utilities::show_error(_("SSH Host Needed"), _("Please specify the host name or address."), _("OK"));
      return false;
    }
    
    value= base::trim(_username.get_string_value());
    if (value.empty())
    {
      Utilities::show_error(_("SSH User Name Needed"), _("Please specify the user name for the SSH account to be used."), _("OK"));
      return false;
    }
    
    // Password might be empty. If there is no keyfile given the password will be queried on connect.
  }
  return true;
}

//--------------------------------------------------------------------------------------------------

void HostMachineConnectionPage::leave(bool advancing)
{
  if (advancing)
  {
    _form->values().gset("enable_admin", _enable_ssh_checkbox.get_active());
    _form->values().gset("host_name", _host_name.get_string_value());
    _form->values().gset("ssh_port", _port.get_string_value());
    _form->values().gset("ssh_user_name", _username.get_string_value());
    _form->values().gset("ssh_password", _password.get_string_value());
    if (_use_ssh_key.get_active())
      _form->values().gset("ssh_key_path", _ssh_key_path.get_string_value());
    else
      _form->values().remove("ssh_key_path");
  }
}

//--------------------------------------------------------------------------------------------------

bool HostMachineConnectionPage::skip_page()
{
  // Don't show this page if we connect to localhost.
  std::string host= _form->values().get_string("host_name", "localhost");
  return (host == "localhost") || (host == "127.0.0.1");
}

//----------------- TestHostMachineSettingsPage ----------------------------------------------------

TestHostMachineSettingsPage::TestHostMachineSettingsPage(WizardForm* host)
: WizardProgressPage(host, "test host machine settings page")
{
  set_short_title(_("Test Settings"));
  set_title(_("Testing Host Machine Settings"));

  set_heading(_("The connection to the host machine is being tested. This might take a few "
    "moments depending on your network connection."));
  _connect_task= add_async_task(_("Connect to host machine"),
    sigc::mem_fun(this, &TestHostMachineSettingsPage::connect_to_host),
    _("Trying to find host machine and connecting to it..."));

  add_async_task(_("Check location of start/stop commands"), 
                 sigc::mem_fun(this, &TestHostMachineSettingsPage::check_admin_commands),
                 _("Checking if commands to start and stop server are in the expected location..."));

  add_async_task(_("Check MySQL configuration file"), 
                 sigc::mem_fun(this, &TestHostMachineSettingsPage::find_config_file),
                 _("Looking for the configuration file of the database server..."));

  end_adding_tasks(true, _("Testing host machine settings is done."));

  set_status_text("");
}

//--------------------------------------------------------------------------------------------------

void TestHostMachineSettingsPage::enter(bool advance)
{
  NewServerInstanceWizard *wizard = dynamic_cast<NewServerInstanceWizard*>(_form);
  reset_tasks();
  
  db_mgmt_ServerInstanceRef instance(wizard->assemble_server_instance());

  std::string host= _form->values().get_string("host_name", "localhost");
  _is_local= (host == "localhost") || (host == "127.0.0.1") || (base::trim(host).empty());
  _connect_task->set_enabled(!_is_local);

  WizardProgressPage::enter(advance);
}

//--------------------------------------------------------------------------------------------------

void TestHostMachineSettingsPage::leave(bool advance)
{
}

//--------------------------------------------------------------------------------------------------

bool TestHostMachineSettingsPage::connect_to_host()
{
  NewServerInstanceWizard* wizard= dynamic_cast<NewServerInstanceWizard*>(_form);

  execute_grt_task(sigc::bind(sigc::mem_fun(wizard, &NewServerInstanceWizard::test_setting_grt),
                              "connect_to_host"), false);

  return true;
}

//--------------------------------------------------------------------------------------------------

bool TestHostMachineSettingsPage::find_config_file()
{
  NewServerInstanceWizard* wizard= dynamic_cast<NewServerInstanceWizard*>(_form);
//  wizard->test_setting_grt(_form->grtm()->get_grt(), _is_local ? "find_config_file/local" : "find_config_file");
  
  execute_grt_task(sigc::bind(sigc::mem_fun(wizard, &NewServerInstanceWizard::test_setting_grt),
                              _is_local ? "find_config_file/local" : "find_config_file"), false);
  return true;
}

//--------------------------------------------------------------------------------------------------

bool TestHostMachineSettingsPage::find_error_files()
{
  NewServerInstanceWizard* wizard= dynamic_cast<NewServerInstanceWizard*>(_form);
  execute_grt_task(sigc::bind(sigc::mem_fun(wizard, &NewServerInstanceWizard::test_setting_grt),
                              _is_local ? "find_error_files/local" : "find_error_files"), false);
  return true;
}

//--------------------------------------------------------------------------------------------------

bool TestHostMachineSettingsPage::check_admin_commands()
{
  NewServerInstanceWizard* wizard= dynamic_cast<NewServerInstanceWizard*>(_form);
//  wizard->test_setting_grt(_form->grtm()->get_grt(), _is_local ? "check_admin_commands/local" : "check_admin_commands");
  execute_grt_task(sigc::bind(sigc::mem_fun(wizard, &NewServerInstanceWizard::test_setting_grt),
                              _is_local ? "check_admin_commands/local" : "check_admin_commands"), false);
  return true;
}

//--------------------------------------------------------------------------------------------------

bool TestHostMachineSettingsPage::skip_page()
{
  return !((NewServerInstanceWizard*)_form)->is_admin_enabled();
}

//--------------------------------------------------------------------------------------------------

void TestHostMachineSettingsPage::tasks_finished(bool success)
{
  // if the template matches reality, we display the settings to the user and ask if they're ok.
  // otherwise, just go straight to customization
  _form->values().gset("template_is_suitable", success);
  _form->values().gset("customize", success ? 0 : 1);
}

//----------------- ReviewPage ----------------------------------------------------------------------

ReviewPage::ReviewPage(WizardForm* host)
: WizardPage(host, "review"), _text(VerticalScrollBar)
{
  set_short_title(_("Review Parameters"));
  set_title(_("Review MySQL Management Parameters"));
  
  _label.set_text(_("Check if the following default settings match your actual settings and toggle 'Change Parameters' "
                  "if you need to make any changes.\n"
                  "Be extra careful if you run more than one instance of MySQL in the same machine."));
  _label.set_wrap_text(true);

  _text.set_read_only(true);
  
  add(&_label, false, true);
  add(&_text, true, true);
  _customize_check.set_text("Change Parameters");
  add(&_customize_check, false, true);
}

//--------------------------------------------------------------------------------------------------

void ReviewPage::leave(bool advancing)
{
  if (advancing)
    values().gset("customize", _customize_check.get_active());
}

//--------------------------------------------------------------------------------------------------

void ReviewPage::enter(bool advancing)
{
  if (advancing)
  {
    std::string summary;

    NewServerInstanceWizard *wizard = (NewServerInstanceWizard*)_form;
    
    grt::DictRef serverInfo(wizard->assemble_server_instance()->serverInfo());
    
    std::string host_name= values().get_string("host_name", "localhost");
    std::string ssh_port= values().get_string("ssh_port", "22");
    std::string ssh_user_name= values().get_string("ssh_user_name");
    std::string ssh_key_path= values().get_string("ssh_key_path");
    bool enable_admin= values().get_int("enable_admin") != 0;
    summary.append(strfmt(_("SSH Based Adminstration:  %s\n"), enable_admin ? _("Yes") : _("No")));
    if (enable_admin)
    {
      summary.append(strfmt(_("    SSH host:  %s:%s\n"), host_name.c_str(), ssh_port.c_str()));
      summary.append(strfmt(_("    SSH user:  %s\n"), ssh_user_name.c_str()));
      summary.append(strfmt(_("    SSH key file:  %s\n"), ssh_key_path.empty() ? "not set" : ssh_key_path.c_str()));
    }
    summary.append("\n");
    std::string os_title= serverInfo.get_string("sys.system", "Unknown");
    std::string ini_path= serverInfo.get_string("sys.config.path");
    std::string ini_section= serverInfo.get_string("sys.config.section");
    std::string mysql_version = serverInfo.get_string("serverVersion");
    summary.append(_("MySQL Configuration\n"));
//    summary.append(strfmt(_("    OS Type:  %s\n"), os_title.c_str()));
    summary.append(strfmt(_("    MySQL Version:  %s\n"), mysql_version.empty() ? "Unknown" : mysql_version.c_str()));
    summary.append(strfmt(_("    Settings Template:  %s\n"), serverInfo.get_string("sys.preset").c_str()));
    summary.append(strfmt(_("    Path to Configuration File:  %s\n"), ini_path.c_str()));
    summary.append(strfmt(_("    Instance Name in Configuration File:  %s\n"), ini_section.c_str()));
    summary.append("\n");
    std::string start= serverInfo.get_string("sys.mysqld.start");
    std::string stop= serverInfo.get_string("sys.mysqld.stop");
    std::string status= serverInfo.get_string("sys.mysqld.status");
    bool use_sudo= serverInfo.get_int("sys.usesudo") != 0;
    summary.append(_("Commands for MySQL Management\n"));
    summary.append(strfmt(_("    Start MySQL:  %s\n"), start.c_str()));
    summary.append(strfmt(_("    Stop MySQL:  %s\n"), stop.c_str()));
    summary.append(strfmt(_("    Check status:  %s\n"), status.c_str()));
    if (os_title != "Windows")
      summary.append(strfmt(_("    Use sudo:  %s\n"), use_sudo ? _("Yes") : _("No")));
    
    _text.set_value(summary);
  }
}

//--------------------------------------------------------------------------------------------------

bool ReviewPage::skip_page()
{
  // if admin is disabled or the defaults failed, go straight to customize
  return !((NewServerInstanceWizard*)_form)->is_admin_enabled() 
          || !_form->values().get_int("template_is_suitable", 0);
}

//----------------- PathsPage ----------------------------------------------------------------------

PathsPage::PathsPage(WizardForm* host, wb::WBContext* context)
: WizardPage(host, "paths page")
{
  _context= context;
  
  set_short_title(_("MySQL Config File"));
  set_title(_("Information about MySQL configuration"));
  
  set_padding(10);
  set_spacing(20);
  _description.set_text(_("In order to manage the settings of the MySQL Server it is necessary to "
                          "know where its configuration file resides.\n\n"
                          "The configuration file may consist of several sections, each of them "
                          "belonging to a different tool or server instance. Hence it is also "
                          "necessary to know which section belongs to the server we are managing.\n\n"
                          "Please specify this information below."));
  _description.set_wrap_text(true);
  add(&_description, false, true);
  
  _content.set_column_count(4);
  _content.set_column_spacing(8);
  _content.set_row_count(5);
  _content.set_row_spacing(8);
  
  _version_label.set_text(_("MySQL Server Version:"));
  _version_label.set_text_align(MiddleRight);
  _content.add(&_version_label, 0, 1, 0, 1, HFillFlag);
  _content.add(&_version, 1, 2, 0, 1, HFillFlag);
  
  _config_path_label.set_text(_("Path to Configuration File:"));
  _config_path_label.set_text_align(MiddleRight);
  _content.add(&_config_path_label, 0, 1, 1, 2, HFillFlag);
  _content.add(&_config_path, 1, 3, 1, 2, HFillFlag);
  _content.add(&_browse_button, 3, 4, 1, 2, HFillFlag);
  
  // Setup for local config file browsing. This will be adjusted if we are at a remote location.
  _file_selector= mforms::manage(new FsObjectSelector(&_browse_button, &_config_path));
  _file_selector->initialize("", mforms::OpenFile, "", "...", sigc::mem_fun(this, &WizardPage::validate));

  _test_config_path_button.set_text(_("Check Path"));
  _test_config_path_button.signal_clicked().connect(sigc::mem_fun(this, &PathsPage::test_path));
  _content.add(&_test_config_path_button, 1, 2, 2, 3, HFillFlag);
  _test_config_path_description.set_text(_("Click to test if your path is correct."));
  _content.add(&_test_config_path_description, 2, 3, 2, 3, HFillFlag | HExpandFlag);
  
  _section_name_label.set_text(_("Section of the Server Instance:"));
  _section_name_label.set_text_align(MiddleRight);
  _content.add(&_section_name_label, 0, 1, 3, 4, HFillFlag);
  _content.add(&_section_name, 1, 3, 3, 4, HFillFlag);
  
  _test_section_button.set_text(_("Check Name"));
  _test_section_button.signal_clicked().connect(sigc::mem_fun(this, &PathsPage::test_section));
  _content.add(&_test_section_button, 1, 2, 4, 5, HFillFlag);
  _test_section_description.set_text(_("Click to test if your instance name is correct."));
  _content.add(&_test_section_description, 2, 3, 4, 5, HFillFlag | HExpandFlag);
  
  add(&_content, true, true);
}

//--------------------------------------------------------------------------------------------------

bool PathsPage::skip_page()
{
  return !((NewServerInstanceWizard*)_form)->is_admin_enabled() || !_form->values().get_int("customize");
}

//--------------------------------------------------------------------------------------------------

void PathsPage::enter(bool advancing)
{
  _test_config_path_description.set_color("#000000");
  _test_config_path_description.set_text(_("Click to test if your path is correct."));
  _test_section_description.set_color("#000000");
  _test_section_description.set_text(_("Click to test if your section is correct."));
 
  if (advancing)
  {
    // Prefill values from defaults.
    NewServerInstanceWizard* wizard= dynamic_cast<NewServerInstanceWizard*>(_form);
    _version.set_value(wizard->get_server_info("serverVersion"));
    _config_path.set_value(wizard->get_server_info("sys.config.path"));
    _section_name.set_value(wizard->get_server_info("sys.config.section"));
  }

  std::string host= _form->values().get_string("host_name", "localhost");
  _is_local= (host == "localhost") || (host == "127.0.0.1") || (base::trim(host).empty());

  bool use_ssh= false; // skip for now _form->values().get_int("enable_admin", 0) != 0;
  if (use_ssh)
  {
    // Setup for remote browsing.
    _file_selector->set_browse_callback(sigc::mem_fun(this, &PathsPage::browse_remote_config_file));
  }
  
}

//--------------------------------------------------------------------------------------------------

bool PathsPage::advance()
{
  std::string version= base::trim(_version.get_string_value());
  int a, b, c;
  if (version.empty() || sscanf(version.c_str(), "%i.%i.%i", &a, &b, &c) < 2
      || a < 4)
  {
    Utilities::show_error(_("Invalid version"), _("The MySQL server version number provided appears to be invalid."), _("OK"));
    return false;
  }
  
  std::string path= base::trim(_config_path.get_string_value());
  if (path.empty())
  {
    Utilities::show_error(_("Empty path"), _("The path to the configuration must not be empty."), _("OK"));
    return false;
  }
  std::string section= base::trim(_section_name.get_string_value());
  if (section.empty())
  {
    Utilities::show_error(_("Empty section"), _("A section must be given which belongs to the given server."), _("OK"));
    return false;
  }
  _form->values().gset("server_version", version);
  _form->values().gset("ini_path", path);
  _form->values().gset("ini_section", section);
  return true;
}

//--------------------------------------------------------------------------------------------------

/**
 * Triggers the remote file open dialog.
 */
void PathsPage::browse_remote_config_file()
{
  app_PluginRef plugin(_context->get_plugin_manager()->get_plugin("wb.admin.remoteFileSelector"));
  
  if (plugin.is_valid())
  {
    grt::BaseListRef args(_context->get_grt_manager()->get_grt(), grt::StringType);
    args.ginsert(grt::StringRef(values().get_string("host_name", "")), 0);
    args.ginsert(grt::StringRef(values().get_string("ssh_port", "22")), 1);
    args.ginsert(grt::StringRef(values().get_string("ssh_user_name")), 2);
    args.ginsert(grt::StringRef(values().get_string("ssh_key_path")), 3);
    
    /*
    grt::BaseListRef result=/ _context->get_plugin_manager()->open_plugin(plugin, args);
    if (result.count() > 0)
    {
      // The result is empty if the user cancelled the selector dialog.
      //grt::StringRef file= result[0];
    }
    */
  }
}

//--------------------------------------------------------------------------------------------------

void PathsPage::test_path()
{
  NewServerInstanceWizard* wizard= dynamic_cast<NewServerInstanceWizard*>(_form);
  std::string detail;

  wizard->values().gset("ini_path", _config_path.get_string_value());
  bool success;
  try
  {
    if (_is_local)
      success= wizard->test_setting("check_config_path/local", detail);
    else
      success= wizard->test_setting("check_config_path", detail);
  }
  catch (std::exception)
  {
    success= false;
  }
  if (success)
  {
    _test_config_path_description.set_color("#00A000");
    _test_config_path_description.set_text(_("The config file path is valid."));
  }
  else
  {
    _test_config_path_description.set_color("#A00000");
    _test_config_path_description.set_text(_("The config file could not be found."));
  }
}

//--------------------------------------------------------------------------------------------------

void PathsPage::test_section()
{
  NewServerInstanceWizard* wizard= dynamic_cast<NewServerInstanceWizard*>(_form);
  std::string detail;
  
  wizard->values().gset("ini_path", _config_path.get_string_value());
  wizard->values().gset("ini_section", _section_name.get_string_value());
  bool success;
  try
  {
    if (_is_local)
      success= wizard->test_setting("check_config_section/local", detail);
    else
      success= wizard->test_setting("check_config_section", detail);
  }
  catch (std::exception)
  {
    success= false;
  }

  if (success)
  {
    _test_section_description.set_color("#00A000");
    _test_section_description.set_text(_("The config file section is valid."));
  }
  else
  {
    _test_section_description.set_color("#A00000");
    _test_section_description.set_text(_("The config file section is invalid."));
  }
}

//----------------- CommandsPage ----------------------------------------------------------------------

CommandsPage::CommandsPage(WizardForm* host)
: WizardPage(host, "commands page")
{
  set_short_title(_("Specify Commands"));
  set_title(_("Specify commands to be used to manage the MySQL server."));
  
  set_spacing(20);
  set_padding(8);
  _description.set_text(_("The values on this page comprise rather low level commands, which are used "
                          "to control the MySQL server instance, check its status and so on.\n\n"
                          "If you are unsure what these values mean leave them untouched. The defaults "
                          "are usually a good choice already (for single server machines)."));
  _description.set_wrap_text(true);
  add(&_description, false, true);
  
  _content.set_column_count(2);
  _content.set_column_spacing(8);
  _content.set_row_count(5);
  _content.set_row_spacing(5);
  
  _start_label.set_text(_("Command to start the MySQL server:"));
  _start_label.set_text_align(MiddleRight);
  _content.add(&_start_label, 0, 1, 0, 1, HFillFlag);
  _content.add(&_start_command, 1, 2, 0, 1, HFillFlag | HExpandFlag);
  
  _stop_label.set_text(_("Command to stop the MySQL server:"));
  _stop_label.set_text_align(MiddleRight);
  _content.add(&_stop_label, 0, 1, 1, 2, HFillFlag);
  _content.add(&_stop_command, 1, 2, 1, 2, HFillFlag | HExpandFlag);
  
  _status_label.set_text(_("Status check command:"));
  _status_label.set_text_align(MiddleRight);
  _content.add(&_status_label, 0, 1, 2, 3, HFillFlag);
  _content.add(&_status_command, 1, 2, 2, 3, HFillFlag | HExpandFlag);
  
  _use_sudo.set_text(_("Check this box if you want or need the above commands \n"
                       "to be executed with elevated Operating System Privileges."));
  _content.add(&_use_sudo, 1, 2, 4, 5, HFillFlag);
  
  add(&_content, false, true);
}

//--------------------------------------------------------------------------------------------------

bool CommandsPage::skip_page()
{
  return !((NewServerInstanceWizard*)_form)->is_admin_enabled() || !_form->values().get_int("customize");
}

//--------------------------------------------------------------------------------------------------

void CommandsPage::enter(bool advancing)
{
  if (advancing)
  {
    // Prefill values from defaults.
    NewServerInstanceWizard* wizard= dynamic_cast<NewServerInstanceWizard*>(_form);
    _start_command.set_value(wizard->get_server_info("sys.mysqld.start"));
    _stop_command.set_value(wizard->get_server_info("sys.mysqld.stop"));
    _status_command.set_value(wizard->get_server_info("sys.mysqld.status"));
    _use_sudo.set_active(wizard->get_server_info("sys.usesudo") != "0");
  }
}

//--------------------------------------------------------------------------------------------------

bool CommandsPage::advance()
{
  _form->values().gset("command_start", base::trim(_start_command.get_string_value()));
  _form->values().gset("command_stop", base::trim(_stop_command.get_string_value()));
  _form->values().gset("command_status", base::trim(_status_command.get_string_value()));
  _form->values().gset("use_sudo", _use_sudo.get_active());

  return true;
}

//----------------- CompleteSetupPage --------------------------------------------------------------

CompleteSetupPage::CompleteSetupPage(WizardForm* host, wb::WBContext* context)
: WizardPage(host, "complete setup page"), _instance_name_box(true), _context(context)
{
  set_short_title(_("Complete Setup"));
  set_title(_("Create the Instance Profile"));

  _description.set_wrap_text(true);
  _description.set_text(_("Please enter a name for this server instance and click Next. "
    "A new Server Instance Profile entry will be created for managing this MySQL server."));
  add(&_description, false, true);

  _instance_name_box.set_spacing(8);
  _instance_name_label.set_text(_("Server Instance Name:"));
  _instance_name_box.add(&_instance_name_label, false, true);
  _instance_name_box.add(&_instance_name, true, true);
  
  add(&_instance_name_box, false, true);
}

//--------------------------------------------------------------------------------------------------

bool CompleteSetupPage::advance()
{
  // Check if the instance name already exists.
  grt::ListRef<db_mgmt_ServerInstance> instances= _context->get_root()->rdbmsMgmt()->storedInstances();
  std::string name= base::trim(_instance_name.get_string_value());

  if (name.empty())
  {
    std::string text= _("Please specify a proper name for the new server instance.");
    Utilities::show_error(_("Empty instance name"), text, _("OK"));
    return false;
  }

  GRTLIST_FOREACH(db_mgmt_ServerInstance, instances, inst)
  {
    if ((*inst)->name() == name)
    {
      std::string text= strfmt(_("An instance with the name %s exists already. Please choose another one."), name.c_str());
      Utilities::show_error(_("Duplicate name"), text, _("OK"));
      return false;
    }
  }
  return true;
}

//--------------------------------------------------------------------------------------------------

void CompleteSetupPage::enter(bool advancing)
{
  if (advancing)
  {
    NewServerInstanceWizard *wizard= (NewServerInstanceWizard*)_form;
    db_mgmt_ServerInstanceRef instance(wizard->assemble_server_instance());
        
    if (wizard->is_admin_enabled())
      _instance_name.set_value(strfmt("%s@%s",
                                      instance->serverInfo().get_string("sys.config.section").c_str(), 
                                      _form->values().get_string("host_name").c_str()));
    else
      _instance_name.set_value(instance->connection()->name());
  }
}

//--------------------------------------------------------------------------------------------------

void CompleteSetupPage::leave(bool advancing)
{
  if (advancing)
  {
    _form->values().gset("instance_name", _instance_name.get_string_value());
    create_instance();

//    NewServerInstanceWizard* wizard= dynamic_cast<NewServerInstanceWizard*>(_form);
//    wizard->update_summary();
  }
}

//--------------------------------------------------------------------------------------------------

void CompleteSetupPage::create_instance()
{
  db_mgmt_ManagementRef rdbms(_context->get_root()->rdbmsMgmt());
  grt::ListRef<db_mgmt_ServerInstance> instances= rdbms->storedInstances();

  NewServerInstanceWizard* wizard= dynamic_cast<NewServerInstanceWizard*>(_form);
  db_mgmt_ServerInstanceRef instance= wizard->assemble_server_instance();
  instances.insert(instance);
  
  if (rdbms->storedConns().get_index(instance->connection()) == grt::BaseListRef::npos)
    rdbms->storedConns().insert(instance->connection());
}

//----------------- NewServerInstanceWizard ---------------------------------------------------------

NewServerInstanceWizard::NewServerInstanceWizard(wb::WBContext* context)
:WizardForm(context->get_grt_manager()), _instance(context->get_grt_manager()->get_grt())
{
  set_title(_("Create New Server Instance Profile"));
    
  _context= context;
  _host_machine_page= new HostMachinePage(this, context);
  add_page(manage(_host_machine_page));
  
  _database_server_page= new DatabaseServerPage(this, _context);
  add_page(manage(_database_server_page));
  
  _test_database_settings_page= new TestDatabaseSettingsPage(this);
  add_page(manage(_test_database_settings_page));
  
  _host_machine_connection_page= new HostMachineConnectionPage(this);
  add_page(manage(_host_machine_connection_page));
    
  _os_page= new HostTypePage(this);
  add_page(manage(_os_page));
  
  _test_host_machine_settings_page= new TestHostMachineSettingsPage(this);
  add_page(manage(_test_host_machine_settings_page));

  _review_page= new ReviewPage(this);
  add_page(manage(_review_page));
  
  _paths_page= new PathsPage(this, _context);
  add_page(manage(_paths_page));
  
  _commands_page= new CommandsPage(this);
  add_page(manage(_commands_page));
  
  _complete_setup_page= new CompleteSetupPage(this, _context);
  add_page(manage(_complete_setup_page));
  
//  _finish_page= new WizardFinishedPage(this, "finish page");
//  add_page(manage(_finish_page));
}

//--------------------------------------------------------------------------------------------------

NewServerInstanceWizard::~NewServerInstanceWizard()
{
  // Pages are freed by the WizardForm ancestor.
  puts("DISCONNECT");
  // disconnect the test SSH session
  std::string s;
  test_setting("disconnect", s);
}

//--------------------------------------------------------------------------------------------------
/*
void NewServerInstanceWizard::update_summary()
{
  _finish_page->set_title(_("A new server instance has been created."));
  
  std::string instance_name= values().get_string("instance_name", "new instance");
  std::string heading= strfmt(_("Your server instance \"%s\" has been successfully created. You can now double "
                    "click the database connection entry that has been created for this server to "
                    "execute SQL queries or double click the server instance entry to administrate "
                    "the database server."), instance_name.c_str());
  _finish_page->set_heading(heading);
  
  std::string summary;
  int db_connection= values().get_int("db_connection", -1);
  
  summary.append(_("Settings in the instance:\n"));
  
  grt::ListRef<db_mgmt_Connection> connections(_context->get_root()->rdbmsMgmt()->storedConns());
  db_mgmt_ConnectionRef connection= connections[db_connection];
  grt::DictRef dict(connection->parameterValues());
  std::string tunnel;
  
  if (connection->driver().is_valid() && connection->driver()->name() == "MysqlNativeSSH")
    tunnel = "(ssh tunnel)";
  
  std::string connection_string= strfmt("%s (User: %s  Host: %s:%i%s)",
                                        connection->name().c_str(),
                                        dict.get_string("userName").c_str(),
                                        dict.get_string("hostName").c_str(),
                                        (int) dict.get_int("port"),
                                        tunnel.c_str());
  summary.append(strfmt(_("  MySQL Connection: %s\n"), connection_string.c_str()));

  std::string host_name= values().get_string("host_name", "localhost");
  std::string ssh_port= values().get_string("ssh_port", "22");
  std::string ssh_user_name= values().get_string("ssh_user_name");
  std::string ssh_key_path= values().get_string("ssh_key_path");
  bool enable_admin= values().get_int("enable_admin") != 0;
  summary.append(strfmt(_("  SSH based adminstration: %s\n"), enable_admin ? _("Yes") : _("No")));
  summary.append(strfmt(_("    SSH host: %s:%s\n"), host_name.c_str(), ssh_port.c_str()));
  summary.append(strfmt(_("    SSH user: %s\n"), ssh_user_name.c_str()));
  summary.append(strfmt(_("    SSH key file: %s\n"), ssh_key_path.c_str()));
  
  std::string os_title= "Unknown";
///  if (!_templates.empty())
///    os_title= _templates[os].first;
  std::string ini_path= values().get_string("ini_path");
  std::string ini_section= values().get_string("ini_section");
  summary.append(strfmt(_("  Instance template: %s\n"), os_title.c_str()));
  summary.append(strfmt(_("    Path to configuration file: %s\n"), ini_path.c_str()));
  summary.append(strfmt(_("    Section in configuration file: %s\n"), ini_section.c_str()));
  
  std::string start= values().get_string("command_start");
  std::string stop= values().get_string("command_stop");
  std::string status= values().get_string("command_status");
  bool use_sudo= values().get_int("use_sudo") != 0;
  summary.append(_("  MySQL management\n"));
  summary.append(strfmt(_("    Start MySQL: %s\n"), start.c_str()));
  summary.append(strfmt(_("    Stop MySQL: %s\n"), stop.c_str()));
  summary.append(strfmt(_("    Check status: %s\n"), status.c_str()));
  summary.append(strfmt(_("    Use sudo: %s\n"), use_sudo ? _("Yes") : _("No")));
  
  _finish_page->set_summary(summary);
}
*/
//--------------------------------------------------------------------------------------------------

/**
 * Creates a server instance object from the current values.
 */
db_mgmt_ServerInstanceRef NewServerInstanceWizard::assemble_server_instance()
{
  extern std::string get_admin_script_for_os(const std::string &os);

  _instance->owner(_context->get_root()->rdbmsMgmt());
  
  std::string instance_name= values().get_string("instance_name", "new instance");
  std::string host_name= values().get_string("host_name", "localhost");
  std::string ssh_host= values().get_string("host_name");
  std::string ssh_port= values().get_string("ssh_port", "22");
  std::string ssh_user_name= values().get_string("ssh_user_name");
  std::string ssh_password= values().get_string("ssh_password");
  std::string ssh_key_path= values().get_string("ssh_key_path");
  
  _instance->name(instance_name);
  _instance->loginInfo().gset("ssh.hostName", ssh_host);
  _instance->loginInfo().gset("ssh.port", ssh_port);
  _instance->loginInfo().gset("ssh.userName", ssh_user_name);
  _instance->loginInfo().gset("ssh.password", ssh_password);
  _instance->loginInfo().gset("ssh.useKey", !ssh_key_path.empty());
  if (!ssh_key_path.empty())
    _instance->loginInfo().gset("ssh.key", ssh_key_path);
  
  std::string version = values().get_string("server_version", "");
  if (!version.empty())
    _instance->serverInfo().gset("serverVersion", version);
  
  if (values().get_int("customize", 0))
    _instance->serverInfo().gset("sys.preset", "Custom");
  else
    _instance->serverInfo().gset("sys.preset", values().get_string("template"));
  
  std::string os = values().get_string("os");
  _instance->serverInfo().gset("sys.system", os);
  
  bool enable_admin= values().get_int("enable_admin", 0) != 0;
  _instance->serverInfo().gset("remoteAdmin", enable_admin);

  _instance->serverInfo().gset("sys.script", get_admin_script_for_os(os));
  
  if (get_active_page_number() >= 7
      && values().get_int("customize")) // don't overwrite defaults unless user has had time to enter them
  {
    std::string ini_path= values().get_string("ini_path");
    std::string ini_section= values().get_string("ini_section");
  
    _instance->serverInfo().gset("sys.config.path", ini_path);
    _instance->serverInfo().gset("sys.config.section", ini_section);
  }
  if (get_active_page_number() >= 8
      && values().get_int("customize")) // don't overwrite defaults unless user has had time to enter them
  {
    std::string start= values().get_string("command_start");
    std::string stop= values().get_string("command_stop");
    std::string status= values().get_string("command_status");
    bool use_sudo= values().get_int("use_sudo") != 0;
  
    _instance->serverInfo().gset("sys.mysqld.start", start);
    _instance->serverInfo().gset("sys.mysqld.stop", stop);
    _instance->serverInfo().gset("sys.mysqld.status", status);
    _instance->serverInfo().gset("sys.usesudo", use_sudo);
  }

  _instance->connection(db_mgmt_ConnectionRef::cast_from(values().get("connection")));
  
  return _instance;
}

//--------------------------------------------------------------------------------------------------

grt::ValueRef NewServerInstanceWizard::test_setting_grt(grt::GRT *grt, const std::string &name)
{
  std::string detail;
  if (!test_setting(name, detail))
    throw std::runtime_error(detail);
  return grt::ValueRef();
}

//--------------------------------------------------------------------------------------------------

bool NewServerInstanceWizard::test_setting(const std::string &name, std::string &detail)
{
  grt::Module *module= grtm()->get_grt()->get_module("WbAdmin");
  if (module)
  {
    grt::BaseListRef args(grtm()->get_grt());
    grt::ValueRef ret;
    args.ginsert(grt::StringRef(name));
    args.ginsert(assemble_server_instance());

    try
    {
      ret = module->call_function("testInstanceSettingByName", args);
      if (ret.is_valid() && grt::StringRef::can_wrap(ret))
      {
        std::string s = *grt::StringRef::cast_from(ret);
        
        if (g_str_has_prefix(s.c_str(), "OK"))
        {
          if (s.size() > 3 && s[2] == ' ')
            detail = s.substr(3);
          return true;
        }
        
        // ERROR
        if (s.size() > 6 && s[5] == ' ')
          detail = s.substr(6);
        
        return false;
      }
    }
    catch (std::exception &exc)
    {
      detail = exc.what();
      return false;      
    }
  }
  return false;
}

//--------------------------------------------------------------------------------------------------

/**
 * Loads all default values for a given instance.
 */
void NewServerInstanceWizard::load_defaults()
{
  std::string template_file = values().get_string("template_path");
  if (!template_file.empty())
  {
    grt::DictRef dict;
    try
    {
      dict= grt::DictRef::cast_from(_grtm->get_grt()->unserialize(template_file));
    }
    catch (std::exception &exc)
    {
      g_warning("Instance %s contains invalid data: %s", template_file.c_str(), exc.what());
      return;
    }
    grt::merge_contents(_instance->serverInfo(), dict, true);
    _instance->serverInfo().gset("sys.preset", values().get_string("template"));
  }
}

//--------------------------------------------------------------------------------------------------

/**
 * Returns the current value for the server info at the specified key. Might be the default
 * value or one set by assemble_server_instance().
 */
std::string  NewServerInstanceWizard::get_server_info(const std::string& key)
{
  grt::ValueRef value= _instance->serverInfo().get(key);
  
  if (!value.is_valid())
    return "";
  if (grt::StringRef::can_wrap(value))
    return grt::StringRef::cast_from(value);
  return value.repr();
}

//--------------------------------------------------------------------------------------------------

bool NewServerInstanceWizard::is_admin_enabled()
{
  bool use_ssh= values().get_int("enable_admin", 0) != 0;
  std::string host= values().get_string("host_name", "localhost");
  bool is_local= (host == "localhost") || (host == "127.0.0.1") || (base::trim(host).empty());
   
  return use_ssh || is_local; 
}
