/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA 02110-1301  USA
 */

#include "stdafx.h"

#include "server_instance_editor.h"
#include "grtui/grtdb_connection_editor.h"

#include "grtpp_util.h"
#include <grt/common.h>
#include "string_utilities.h"

#define SYSTEM_STAT_HEADER "# Customize this script to suit your system if necessary\n"\
"# Shell function names and output format must be maintained\n\n"

static struct SystemStatScript {
  const char *name;
  const char *script;
} system_stat_scripts[] = {
{"Windows", 
"get_cpu_info = wmic cpu get LoadPercentage |  wba_line(1) | wba_arithm(/, 100.0)"
"\n"
"get_mem_info = wmic os get FreePhysicalMemory |  wba_line(1)"
"\n"
"get_mem_total = wmic os get TotalVisibleMemorySize |  wba_line(1)"
"\n"
},
{"Linux", 
"get_cpu_info = /usr/bin/uptime | wba_token(' ', 10)"
"\n"
"get_mem_info = free | wba_filter(Mem:) | wba_token(' ', 3)"
"\n"
"get_mem_total = free | wba_filter(Mem:) | wba_token(' ', 1)"
"\n"
},
{"MacOS X",
"get_cpu_info = /usr/bin/uptime | wba_token(' ', 10)"
"\n"
"get_mem_info = vm_stat | wba_filter(free) | wba_token(':', 1)"
"\n"
"get_mem_total = sysctl hw.memsize | wba_token(' ', 1) | wba_arithm(/, 4096)"
"\n"
},
{"OpenSolaris",
"get_cpu_info = /usr/bin/uptime | wba_token(' ', 10)"
"\n"
"get_mem_info = free | wba_filter(Mem:) | wba_token(' ', 1, 2)"
"\n"
"get_mem_total = "
"\n"
},
{"FreeBSD",
"get_cpu_info = /usr/bin/uptime | wba_token(' ', 10)"
"\n"
"get_mem_info = free | wba_filter(Mem:) | wba_token(' ', 1, 2)"
"\n"
"get_mem_total = sysctl hw.memsize | wba_token(' ', 1) | wba_arithm(/, 4096)"
"\n"
},
{NULL, NULL}
};


std::string get_admin_script_for_os(const std::string &os)
{
  for (int i= 0; system_stat_scripts[i].name; i++)
    if (strcmp(system_stat_scripts[i].name, os.c_str()) == 0)
      return system_stat_scripts[i].script;
  return "";
}


static const char *filter_help = 
"Workbench expects status scripts to return 0 on success (running) and 1 otherwise.\n"
"In case of OS where execution status is not always correctly detected\n"
"(as it is in Windows) wba_ internal filters can be used.\n"
"The wba_filter() filter generates a status code internally,\n"
"0 if the pattern is found in the output from the command and 1 if not.\n"
"Filters chain can be debugged using Plugins/Utilities/Test Filters from main menu.\n"
"You may use the following filters provided by Workbench to manipulate\n"
"text output by system commands.\n"
"Filters can be combined (piped) using the | symbol\n"
"\n"
"wba_line(<line number>)\n"
"  Skips all lines in the output except the given one\n"
"wba_token(<separator>, <token number>)\n"
"  Splits line into pieces using separator and returns given numbered token\n"
"wba_filter(<pattern>)\n"
"  Removes all lines which do not contain <pattern>\n"
"wba_arithm(<op>, <arg>)\n"
"  Applies arithmetic operations to the input value";


using namespace mforms;
using namespace base;

inline Label *RLabel(const std::string &text)
{
  Label *l= new Label(text);
  l->set_text_align(MiddleRight);
  return l;
}

inline Table *NewTable(int rows, int cols)
{
  Table *table = new Table();
  table->set_row_count(rows);
  table->set_column_count(cols);
  table->set_row_spacing(8);
  table->set_column_spacing(8);
  table->set_padding(8);
  return table;
}


static bool is_local_connection(const db_mgmt_ConnectionRef &connection)
{
  if (connection.is_valid())
  {
    std::string driver= connection->driver().is_valid() ? connection->driver()->name() : "";
    std::string hostname= connection->parameterValues().get_string("hostName");
  
    if (driver != "MysqlNativeSSH" && (hostname == "localhost" || hostname.empty() || hostname == "127.0.0.1"))
      return true;
  }
  return false;
}


ServerInstanceEditor::ServerInstanceEditor(bec::GRTManager *grtm, const db_mgmt_ManagementRef &mgmt)
: Form(0, FormResizable)
, _grtm(grtm)
, _top_vbox(false)
, _top_hbox(true)
, _content_box(false)
, _inst_list_buttons_hbox(true)
, _stored_instance_list(TreeShowHeader)
, _tabview(false)
, _conn_box(false)
, _ssh_box(false)
, _ssh_pass_box(true)
, _sys_box(false)
, _details_tabview(false)
, _script_text(VerticalScrollBar)
, _bottom_hbox(true)
{
  _mgmt= mgmt;
  _instances= mgmt->storedInstances();
  
  set_title(_("Manage Server Instances"));
  
  _top_vbox.set_padding(MF_WINDOW_PADDING);
  _top_vbox.set_spacing(12);
  _top_hbox.set_spacing(8);
  _top_vbox.add(&_top_hbox, true, true);
  _top_vbox.add(&_bottom_hbox, false, true);
  
  _bottom_hbox.set_spacing(12);
      
  _top_hbox.add(&_stored_instance_list, false, true);
  
  {
    Box *hbox = manage(new Box(true));
    hbox->add(manage(RLabel(_("Instance Name:"))), false, true);
    hbox->add(&_name_entry, true, true);
    hbox->set_padding(12);
    hbox->set_spacing(MF_TABLE_COLUMN_SPACING);
    _name_entry.signal_changed().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::entry_changed),
                                                    &_name_entry));
    
    _content_box.add(hbox, false, true);
    _content_box.add(&_tabview, true, true);
    _top_hbox.add(&_content_box, true, true);
  }
  
  _stored_instance_list.signal_changed().connect(sigc::mem_fun(this, &ServerInstanceEditor::instance_changed));  
  
  _conn_box.set_padding(MF_PANEL_PADDING);
  _ssh_box.set_padding(MF_PANEL_PADDING);
  _sys_box.set_padding(MF_PANEL_PADDING);
  _sys_box.set_spacing(10);
  
  Box *vbox = manage(new Box(false));
  vbox->set_spacing(15);
  vbox->set_padding(MF_PANEL_PADDING);
  
  _tabview.add_page(vbox, _("Connection"));
  
  // Connection
  {
    Panel *panel = manage(new Panel(TitledGroupPanel));
    panel->set_title(_("MySQL Connection"));
    
    Label *label= manage(new Label(_("Pick a preset connection to the MySQL server instance. "
                                     "The connection will be used for\nbasic administration tasks "
                                     "such as managing users and viewing schema objects.")));
    label->set_style(SmallStyle);
    label->set_wrap_text(true);

    _conn_box.set_padding(MF_PANEL_PADDING);
    _conn_box.set_spacing(MF_TABLE_ROW_SPACING);
    _conn_box.add(label, false, false);
    
    Box *box = manage(new Box(true));
    box->set_spacing(MF_TABLE_COLUMN_SPACING);
    box->add(manage(new Label(_("Connection:"))), false, false);
    box->add(&_connection_selector, true, true);
    _conn_box.add(box, false, false);
    
    Box *bbox = manage(new Box(true));
    bbox->set_spacing(MF_TABLE_COLUMN_SPACING);
    _conn_box.add(bbox, false, true);
    Button *b = manage(new Button());
    b->set_text("Manage Connections...");
    b->signal_clicked().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::open_editor), false));
    bbox->add_end(b, false, true);
    b = manage(new Button());
    b->set_text("Edit Selected...");
    b->signal_clicked().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::open_editor), true));
    bbox->add_end(b, false, true);
    
    Box *verbox = manage(new Box(true));
    verbox->set_spacing(MF_TABLE_COLUMN_SPACING);
    _conn_box.add(verbox, false, true);
    
    verbox->add(manage(new Label(_("MySQL Server Version:"))), false, true);
    verbox->add(&_server_version, true, true);
    _server_version.signal_changed().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::entry_changed),
                                                        &_server_version));
    b = manage(new Button());
    b->set_text("Connect and Check...");
    b->signal_clicked().connect(sigc::mem_fun(this, &ServerInstanceEditor::check_version));
    verbox->add(b, false, true);
    
    _connection_selector.signal_changed().connect(sigc::mem_fun(this, &ServerInstanceEditor::connection_changed));
    panel->add(&_conn_box);
    vbox->add(panel, false, true);
  }
    
  // SSH
  {
    Panel *panel = manage(new Panel(TitledGroupPanel));
    panel->set_title(_("SSH Shell Connection"));
    Label *label;
    Table *table = NewTable(6, 2);
    
    _ssh_param_table = table;
    
    _ssh_box.set_spacing(12);

    _enable_ssh_admin.set_text(_("Enable SSH login based administration"));
    _ssh_box.add(&_enable_ssh_admin, false, true);
    _enable_ssh_admin.signal_clicked().connect(sigc::mem_fun(this, &ServerInstanceEditor::toggle_remote_admin));
    label = manage(new Label(_("SSH login based administration allows changing MySQL configuration and starting/stopping\n"
                               "it in remote machines.")));
    label->set_style(SmallStyle);
    _ssh_box.add(label, false, true);
    _ssh_box.add(manage(table), true, true);

    table->add(RLabel(_("Hostname:")), 0, 1, 0, 1, HFillFlag);
    Box *box= manage(new Box(true));
    box->set_spacing(MF_TABLE_COLUMN_SPACING);

    _ssh_host.set_size(200, -1);
    box->add(&_ssh_host, true, true);
    _ssh_host.signal_changed().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::entry_changed), 
                                                  &_ssh_host));
    
    box->add(manage(RLabel(_("Port:"))), false, true);
    _ssh_port.set_size(50, -1);
    box->add(&_ssh_port, true, true);
    _ssh_port.signal_changed().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::entry_changed), 
                                                  &_ssh_port));
    table->add(box, 1, 2, 0, 1, HFillFlag | HExpandFlag);
    
    table->add(manage(RLabel(_("Username:"))), 0, 1, 1, 2, HFillFlag);
    table->add(&_ssh_user, 1, 2, 1, 2, HExpandFlag|HFillFlag);
    _ssh_user.signal_changed().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::entry_changed), 
                                                  &_ssh_user));
    table->add(manage(RLabel(_("Password:"))), 0, 1, 2, 3, HFillFlag);
    table->add(&_ssh_pass_box, 1, 2, 2, 3, HExpandFlag|HFillFlag);

#ifdef _WIN32
    _ssh_pass_set.set_text(_("Store in Vault ..."));
    _ssh_pass_set.set_tooltip(_("Store the password for this connection in the secured vault."));
    _ssh_pass_clear.set_text(_("Remove from Vault"));
    _ssh_pass_clear.set_tooltip(_("Remove previously stored password from the secured vault."));
#else
    _ssh_pass_set.set_text(_("Store in Keychain ..."));
    _ssh_pass_set.set_tooltip(_("Store the password for this connection in the system's keychain."));
    _ssh_pass_clear.set_text(_("Remove from Keychain"));
    _ssh_pass_clear.set_tooltip(_("Remove previously stored password from the system's keychain"));
#endif

    _ssh_pass_set.signal_clicked().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::set_password), false));
    _ssh_pass_clear.signal_clicked().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::set_password), true));
    
    _ssh_pass_box.set_spacing(8);
    _ssh_pass_box.add(&_ssh_pass_set, true, true);
    _ssh_pass_box.add(&_ssh_pass_clear, true, true);
    
    _ssh_usekey.set_text(_("Authenticate Using SSH Key"));
    _ssh_usekey.signal_clicked().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::check_changed), 
                                                  &_ssh_usekey));
    table->add(&_ssh_usekey, 1, 2, 3, 4, HFillFlag);

    table->add(RLabel(_("SSH Key Path:")), 0, 1, 4, 5, HFillFlag);
    box= manage(new Box(true));
    box->set_spacing(MF_TABLE_COLUMN_SPACING);
    box->add(&_ssh_keypath, true, true);
    _ssh_keypath.signal_changed().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::entry_changed), 
                                                  &_ssh_keypath));
    Button *b = manage(new Button());
    b->set_text(_("Browse"));
    box->add(b, false, true);
    b->signal_clicked().connect(sigc::mem_fun(this, &ServerInstanceEditor::browse_file));
    table->add(box, 1, 2, 4, 5, HFillFlag);

    panel->add(&_ssh_box);
    vbox->add(panel, false, true);
  }
  
  {
    Box *box = manage(new Box(true));
    box->set_spacing(8);
    
    _autodetect_button.set_enabled(_grtm->get_grt()->get_module("WbAdmin")!=0);
    _autodetect_button.set_text(_("Detect Server Configuration..."));
    _autodetect_button.set_tooltip(_("Attempt to automatically detect server configuration parameters for the system,\n"
                                     "such as operating system type, path to configuration file, how to start/stop MySQL etc"));
    
    box->add(&_autodetect_button, true, false);
    _autodetect_button.signal_clicked().connect(sigc::mem_fun(this, &ServerInstanceEditor::autodetect_system));
  }
  
  
  _tabview.add_page(&_sys_box, _("System Profile"));

  // Sys
  {
    Label *label = manage(new Label(_("Information about the server and MySQL configuration, such as path to the configuration file, "
                                      "command to start or stop it etc. You may pick a preset configuration profile or customize one for your needs.")));
    label->set_wrap_text(true);
    label->set_style(SmallStyle);
    
    _sys_box.add(label, false, true);

    Table *table = manage(NewTable(4, 2));
    _sys_box.add(table, false, true);

    {
      table->add(manage(RLabel(_("System Type:"))), 0, 1, 0, 1, HFillFlag);
      table->add(&_os_type, 1, 2, 0, 1, HFillFlag | HExpandFlag);
      _os_type.signal_changed().connect(sigc::mem_fun(this, &ServerInstanceEditor::system_type_changed));
    }
    {
      label= manage(RLabel(_("Installation Type:")));
      table->add(label, 0, 1, 1, 2, HFillFlag);

      _sys_profile_type.signal_changed().connect(sigc::mem_fun(this, &ServerInstanceEditor::profile_changed));
      table->add(&_sys_profile_type, 1, 2, 1, 2, HFillFlag | HExpandFlag);
    }

    { // use a custom file browsing field because the files can be local or remote
      Box *box = manage(new Box(true));
      box->set_spacing(8);
      table->add(manage(RLabel(_("Configuration File:"))), 0, 1, 2, 3, HFillFlag);

      box->add(&_sys_config_path, true, true);
      _sys_config_path_browse.enable_internal_padding(false);
      _sys_config_path_browse.set_text("...");
      box->add(&_sys_config_path_browse, false, true);
      _sys_config_path_browse.signal_clicked().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::run_filechooser_wrapper), &_sys_config_path));
      table->add(box, 1, 2, 2, 3, HFillFlag | HExpandFlag);
    }

    _sys_config_path.signal_changed().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::entry_changed), 
                                                  &_sys_config_path));
    table->add(manage(RLabel(_("Instance Name:"))), 0, 1, 3, 4, HFillFlag);
    table->add(&_sys_myini_section, 1, 2, 3, 4, HFillFlag | HExpandFlag);
    _sys_myini_section.signal_changed().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::entry_changed), 
                                                  &_sys_myini_section));

    label = manage(new Label(_("\nThe following options specify how Workbench should perform certain administration actions. "
                                            "Commands given here are the same as if they would be run in a system shell. Commands which return "
                                            "a value might be combined with some predefined filters. See \"Filters Help\" vor more details.")));
    label->set_wrap_text(true);
    label->set_style(SmallHelpTextStyle);
    _sys_box.add(label, false, true);
    
    _sys_box.add(&_details_tabview, true, true);
    
    table = NewTable(6, 2);
    _details_tabview.add_page(manage(table), _("MySQL Management"));
    
    table->add(manage(RLabel(_("Start MySQL:"))), 0, 1, 0, 1, HFillFlag);
    table->add(&_start_cmd, 1, 2, 0, 1, HFillFlag | HExpandFlag);
    _start_cmd.signal_changed().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::entry_changed), 
                                                  &_start_cmd));
    table->add(manage(RLabel(_("Stop MySQL:"))), 0, 1, 1, 2, HFillFlag);
    table->add(&_stop_cmd, 1, 2, 1, 2, HFillFlag | HExpandFlag);    
    _stop_cmd.signal_changed().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::entry_changed), 
                                                  &_stop_cmd));

    table->add(&_sudo_check, 1, 2, 2, 3, HFillFlag | HExpandFlag);
#ifndef _WIN32
    _sudo_check.set_text(_("Elevate privileges to execute start/stop commands\nand write configuration data"));
#else
    _sudo_check.set_text(_("Acquire administrator rights to execute start/stop commands\nand write configuration data"));
#endif
    _sudo_check.signal_clicked().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::check_changed), 
                                                  &_sudo_check));


    table->add(manage(RLabel(_("Check MySQL Status:"))), 0, 1, 3, 4, HFillFlag);
    table->add(&_status_cmd, 1, 2, 3, 4, HFillFlag | HExpandFlag);
    _status_cmd.signal_changed().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::entry_changed), 
                                                  &_status_cmd));


    table->add(&_sudo_status_check, 1, 2, 4, 5, HFillFlag | HExpandFlag);
#ifndef _WIN32
    _sudo_status_check.set_text(_("Elevate privileges to execute status commands"));
#else
    _sudo_status_check.set_text(_("Acquire administrator rights to execute status commands"));
#endif
    _sudo_status_check.signal_clicked().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::check_changed), 
                                                  &_sudo_status_check));

    {
      Box *stats_box = manage(new Box(false));
      _details_tabview.add_page(stats_box, _("Server Stats"));    
      _script_text.signal_changed().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::text_changed), 
                                                  &_script_text));

      stats_box->add(&_script_text, true, true);

      Box *box = manage(new Box(true));
      box->set_spacing(12);
    
      Button* help_filters_button  = manage(new Button());
      help_filters_button->set_text(_("Filters Help"));
      box->add(help_filters_button, true, true);
      help_filters_button->signal_clicked().connect(sigc::mem_fun(this, &ServerInstanceEditor::show_filter_help));
      
      Button* test_filters_button  = manage(new Button());
      test_filters_button->set_text(_("Filters Debugger"));
      box->add(test_filters_button, true, true);
      test_filters_button->signal_clicked().connect(sigc::mem_fun(this, &ServerInstanceEditor::run_filters_debugger));

      stats_box->add(box, false, false);
    }
  }
  

  _dup_inst_button.set_text(_("Duplicate"));
  _dup_inst_button.signal_clicked().connect(sigc::mem_fun(this, &ServerInstanceEditor::duplicate_instance));

  _del_inst_button.set_text(_("Delete"));
  _del_inst_button.signal_clicked().connect(sigc::mem_fun(this, &ServerInstanceEditor::delete_instance));
  _add_inst_button.set_text(_("New"));
  _add_inst_button.signal_clicked().connect(sigc::mem_fun(this, &ServerInstanceEditor::add_instance));
  
  _move_up_button.set_text(_("Move Up"));
  _move_up_button.signal_clicked().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::reorder_instance), true));
  _move_down_button.set_text(_("Move Down"));
  _move_down_button.signal_clicked().connect(sigc::bind(sigc::mem_fun(this, &ServerInstanceEditor::reorder_instance), false));
  
  _bottom_hbox.add(&_add_inst_button, false, true);
  _bottom_hbox.add(&_del_inst_button, false, true);
  _bottom_hbox.add(&_dup_inst_button, false, true);
  _bottom_hbox.add(&_move_up_button, false, true);
  _bottom_hbox.add(&_move_down_button, false, true);
  
  _bottom_hbox.add_end(&_ok_button, false, true);
// not working yet  _bottom_hbox.add_end(&_test_button, false, true);  
  //  _bottom_hbox.add_end(&_cancel_button, false, true);
  
  _ok_button.set_text(_("Close"));

  _test_button.set_text(_("Test Settings"));
  _test_button.set_enabled(_grtm->get_grt()->get_module("WbAdmin")!=0);
  _test_button.signal_clicked().connect(sigc::mem_fun(this, &ServerInstanceEditor::test_settings));
  
  _add_inst_button.enable_internal_padding(true);
  _del_inst_button.enable_internal_padding(true);
  _ok_button.enable_internal_padding(true);
  _test_button.enable_internal_padding(true);

  _stored_instance_list.set_size(180, -1);
  
  set_content(&_top_vbox);
  
  _stored_instance_list.add_column(::mforms::StringColumnType, _("Server Instances"), 150, false);
  _stored_instance_list.end_columns();

  set_size(820, 700);
  center();
  
  ///
  
  std::string path= _grtm->get_data_file_path("mysql.profiles");
  GDir *dir = g_dir_open(path.c_str(), 0, NULL);
  if (dir)
  {
    const gchar *file;
    while ((file = g_dir_read_name(dir)))
    {
      if (g_str_has_suffix(file, ".xml"))
      {
        std::string fname= std::string(file, strlen(file)-4);
        std::string label= bec::replace_string(fname, "_", " ");
        grt::DictRef dict;
        try
        {
          dict= grt::DictRef::cast_from(_grtm->get_grt()->unserialize(path+"/"+file));
        }
        catch (std::exception &exc)
        {
          g_warning("Profile %s contains invalid data: %s", path.c_str(), exc.what());
          continue;
        }
        _presets[dict.get_string("sys.system")].push_back(std::make_pair(label, dict));
      }
    }
    g_dir_close(dir);
  }  
}


void ServerInstanceEditor::set_password(bool clear)
{
  std::string port = _ssh_port.get_string_value();
  std::string storage_key = strfmt("ssh@%s:%s", _ssh_host.get_string_value().c_str(), port.empty() ? "22" : port.c_str());
  std::string username = _ssh_user.get_string_value();
  
  if (username.empty())
  {
    mforms::Utilities::show_warning("Cannot Set Password", "Please fill the username to be used.", "OK", "", "");
    return;
  }

  if (clear)
  {
    try
    {
      mforms::Utilities::forget_password(storage_key, username);
    }
    catch (std::exception &exc)
    {
      mforms::Utilities::show_error("Clear Password", 
                                    base::strfmt("Could not clear password: %s", exc.what()),
                                    "OK");
    }
  }
  else
  {
    std::string password;
    
    try
    {
      if (mforms::Utilities::ask_for_password(_("Store Password For Connection"), 
                                              storage_key, username, password))
        mforms::Utilities::store_password(storage_key, username, password);
    }
    catch (std::exception &exc)
    {
      mforms::Utilities::show_error("Store Password", 
                                    base::strfmt("Could not store password: %s", exc.what()),
                                    "OK");
    }
  }
  show_instance();
}


void ServerInstanceEditor::check_version()
{
  db_mgmt_ServerInstanceRef instance(selected_instance());
  
  if (instance.is_valid() && instance->connection().is_valid())
  {
    try
    {
      sql::ConnectionWrapper dbc_conn= sql::DriverManager::getDriverManager()->getConnection(instance->connection());
      
      if (dbc_conn.get() != NULL)
      {
        boost::shared_ptr<sql::Statement> stmt(dbc_conn->createStatement());
        boost::shared_ptr<sql::ResultSet> res(stmt->executeQuery("SELECT VERSION()"));
        if (res.get() && res->next())
        {
          std::string version = res->getString(1);
          _server_version.set_value(version);
          entry_changed(&_server_version);
          
          int major, minor;
          if (sscanf(version.c_str(), "%i.%i", &major, &minor) == 2 && major < 5)
            mforms::Utilities::show_warning("Unsupported Server Version", 
                                            strfmt("MySQL Server version is %s, which is not supported by Workbench.", version.c_str()),
                                            "OK", "", "");
          return;
        }
      }
      mforms::Utilities::show_warning("Error Verifying Server Version", "Could not fetch server version.", "OK", "", "");
    }
    catch (std::exception &exc)
    {
      mforms::Utilities::show_warning("Error Verifying Server Version", exc.what(), "OK", "", "");
    }
  }
}


void ServerInstanceEditor::show_filter_help()
{
  mforms::Form window(0);
  mforms::Box vbox(false);
  mforms::Box box(true);
  mforms::Button ok;
  mforms::Label label;
  
  window.set_title("MySQL Management Commands Help");

//  window.set_size(400, 200);
  window.set_content(&vbox);

  vbox.set_padding(12);
  vbox.set_spacing(12);
  
  label.set_text(filter_help);
  
  vbox.add(&label, true, true);
  vbox.add(&box, false, true);
  
  ok.set_text("Close");
  box.add_end(&ok, false, true);

  window.center();
  window.run_modal(&ok, 0);
}


db_mgmt_ServerInstanceRef ServerInstanceEditor::run()
{
  refresh_options();
  refresh_instance_list();
    
  _stored_instance_list.set_selected(0);
  instance_changed();
  run_modal(&_ok_button, NULL);

  return selected_instance();
}

void ServerInstanceEditor::run_filters_debugger()
{
  grt::Module *module= _grtm->get_grt()->get_module("WbAdmin");
  if (module)
  {
    db_mgmt_ServerInstanceRef instance(selected_instance());
    grt::BaseListRef args(_grtm->get_grt());
    args.ginsert(instance->serverInfo());
    args.ginsert(instance->loginInfo());

    grt::IntegerRef selection = grt::IntegerRef::cast_from(module->call_function("openFilterDebugger", args));
  }
}

void ServerInstanceEditor::run_filechooser_wrapper(mforms::TextEntry* entry) // Allows to run local or remote file selector
{
  db_mgmt_ServerInstanceRef instance(selected_instance());
  bool is_local = false;

  if (instance.is_valid())
    is_local = is_local_connection(instance->connection());

  if (is_local)
    run_filechooser(entry);
  else
  {
    grt::Module *module= _grtm->get_grt()->get_module("WbAdmin");
    if (module)
    {
      grt::BaseListRef args(_grtm->get_grt());
      args.ginsert(instance->serverInfo());
      args.ginsert(instance->loginInfo());

      grt::StringRef selection = grt::StringRef::cast_from(module->call_function("openRemoteFileSelector", args));
      if (!selection.empty())
      {
        entry->set_value(selection.c_str());
        entry_changed(entry);
      }
    }
  }
}

void ServerInstanceEditor::run_filechooser(mforms::TextEntry* entry)
{
  mforms::FileChooser fc(mforms::OpenFile);
  //TODO: Add set directory
  if (fc.run_modal())
  {
    const std::string path = fc.get_path();
    if (!path.empty() || path != "")
      entry->set_value(path);
    entry->signal_changed().emit();
  }
}

void ServerInstanceEditor::system_type_changed()
{
  db_mgmt_ServerInstanceRef instance(selected_instance());
  
  if (instance.is_valid())
  {
    const int i= _os_type.get_selected_index();
    if (i>=0)
    {
      const std::string &system = system_stat_scripts[i].name;
      instance->serverInfo().gset("sys.system", system);

      const bool is_not_windows = system.find("indows") == std::string::npos;
      _sudo_status_check.set_enabled(is_not_windows);

      if (!is_not_windows)
        _sudo_status_check.set_active(false);

      if (system_stat_scripts[i].script)
      {
        std::string script= SYSTEM_STAT_HEADER;
        script.append(get_admin_script_for_os(system));
        _script_text.set_value(script);
        instance->serverInfo().gset("sys.script", script);
      }

      refresh_profile_list();
    
      profile_changed();
    }
  }
}


void ServerInstanceEditor::refresh_profile_list()
{
  const int i= _os_type.get_selected_index();
  if (i >= 0)
  {
    std::string system = system_stat_scripts[i].name;

    _sys_profile_type.clear();
    for (std::vector<std::pair<std::string,grt::DictRef> >::const_iterator iter= _presets[system].begin();
      iter != _presets[system].end(); ++iter)
    _sys_profile_type.add_item(iter->first);
    _sys_profile_type.add_item(_("Custom"));  
  }
}


void ServerInstanceEditor::refresh_instance_list()
{
  _stored_instance_list.clear_rows();

  GRTLIST_FOREACH(db_mgmt_ServerInstance, _instances, inst)
  {
    int i = _stored_instance_list.add_row();
    if (i >= 0)
    {
      _stored_instance_list.set(i, 0, (*inst)->name().c_str());
    }
  }
}

db_mgmt_ServerInstanceRef ServerInstanceEditor::selected_instance()
{
  int i= _stored_instance_list.get_selected();
  if (i >= 0)
    return _instances[i];

  return db_mgmt_ServerInstanceRef();
}


void ServerInstanceEditor::autodetect_system()
{
  grt::Module *module= _grtm->get_grt()->get_module("WbAdmin");
  if (module)
  {
    grt::BaseListRef args(_grtm->get_grt());
    args.ginsert(selected_instance());

    module->call_function("detectInstanceSettings", args);
  }
}


void ServerInstanceEditor::test_settings()
{
  grt::Module *module= _grtm->get_grt()->get_module("WbAdmin");
  if (module)
  {
    grt::BaseListRef args(_grtm->get_grt());
    grt::ValueRef ret;
    args.ginsert(selected_instance());

    ret = module->call_function("testInstanceSettings", args);
  }
}


void ServerInstanceEditor::toggle_remote_admin()
{
  db_mgmt_ServerInstanceRef instance(selected_instance());
  bool is_local = false;

  if (instance.is_valid())
  {
    instance->serverInfo().gset("remoteAdmin", _enable_ssh_admin.get_active() ? 1 : 0);
    is_local = is_local_connection(instance->connection());
  }

  _ssh_param_table->set_enabled(!is_local && _enable_ssh_admin.get_active());
  
  //_sys_config_path_browse.set_enabled(is_local);
  
  // if local connection or remote with ssh_admin toggle checked, then enable system related options
  if (_enable_ssh_admin.get_active() || is_local)
    _sys_box.set_enabled(true);
  else
    _sys_box.set_enabled(false);
}


void ServerInstanceEditor::add_instance()
{
  db_mgmt_ServerInstanceRef instance(_instances.get_grt());
  std::string name= "new instance";
  int i= 1;
  bool dupe;
  do
  {
    dupe= false;
    GRTLIST_FOREACH(db_mgmt_ServerInstance, _instances, inst)
    {
      if ((*inst)->name() == name)
      {
        name= strfmt("new instance %i", i++);
        dupe= true;
        break;
      }
    }
  } while (dupe);
  
  i = _stored_instance_list.add_row();
  if (i >= 0)
  {
    _stored_instance_list.set(i, 0, name);

    instance->owner(_mgmt);
    instance->name(name);
    instance->loginInfo().gset("ssh.hostName", "");
    instance->loginInfo().gset("ssh.localPort", "3316");
    instance->loginInfo().gset("ssh.userName", "mysql");
    instance->loginInfo().gset("ssh.useKey", 1);
  std::string homedir=
#ifdef _WIN32
    mforms::Utilities::get_special_folder(mforms::ApplicationData);
#else
    "~";
#endif
  instance->loginInfo().gset("ssh.key", homedir + "/.ssh/ssh_private_key");
    
    if (!_presets.empty())
    {
      instance->serverInfo().gset("sys.system", "Windows");
      instance->serverInfo().gset("sys.preset", _presets["Windows"][0].first);
    }
    
    _instances.insert(instance);
    _stored_instance_list.set_selected(i);
  }
  else
    g_warning("add_row returned -1");
  
  //show_instance();
  instance_changed();

  system_type_changed();
  
  connection_changed();
}


void ServerInstanceEditor::delete_instance()
{
  int i= _stored_instance_list.get_selected();
  if (i >= 0 && i < (int)_instances.count())
  {
    _instances.remove(i);
    refresh_instance_list();
  }
}


void ServerInstanceEditor::duplicate_instance()
{
  db_mgmt_ServerInstanceRef orig(selected_instance());
  db_mgmt_ServerInstanceRef instance(_instances.get_grt());
  
  if (!orig.is_valid())
    return;
  
  std::string name= orig->name();
  int i= 1;
  bool dupe;
  do
  {
    dupe= false;
    GRTLIST_FOREACH(db_mgmt_ServerInstance, _instances, inst)
    {
      if ((*inst)->name() == name)
      {
        name= strfmt("%s %i", orig->name().c_str(), i++);
        dupe= true;
        break;
      }
    }
  } while (dupe);
  
  i = _stored_instance_list.add_row();
  if (i >= 0)
  {
    _stored_instance_list.set(i, 0, name);
    
    instance->owner(_mgmt);
    instance->name(name);
    instance->connection(orig->connection());
    grt::merge_contents(instance->loginInfo(), orig->loginInfo(), true);
    grt::merge_contents(instance->serverInfo(), orig->serverInfo(), true);
    
    _instances.insert(instance);
    _stored_instance_list.set_selected(i);
  }
  else
    g_warning("add_row returned -1");
  
  show_instance();  
}


void ServerInstanceEditor::reorder_instance(bool up)
{
  int i= _stored_instance_list.get_selected();

  if (i < 0)
    return;

  if (up)
  {
    if (i > 0)
    {
      _instances.reorder(i, i-1);
      _stored_instance_list.set_selected(i-1);
    }
  }
  else
  {
    if (i < _stored_instance_list.count()-1)
    {
      _instances.reorder(i, i+1);
      _stored_instance_list.set_selected(i+1);
    }
  }

  i= 0;
  GRTLIST_FOREACH(db_mgmt_ServerInstance, _instances, inst)
  {
    _stored_instance_list.set(i++, 0, (*inst)->name().c_str());
  }  
}


void ServerInstanceEditor::browse_file()
{
  FileChooser fsel(mforms::OpenFile);

  fsel.set_title(_("Pick SSH Private Key"));

  if (fsel.run_modal())
  {
    _ssh_keypath.set_value(fsel.get_path());
    entry_changed(&_ssh_keypath);
  }
}


void ServerInstanceEditor::entry_changed(mforms::TextEntry *sender)
{
  std::string value= sender->get_string_value();
  db_mgmt_ServerInstanceRef instance(selected_instance());
  
  if (instance.is_valid())
  {
    if (&_name_entry == sender)
    {
      instance->name(value);
      _stored_instance_list.set(_stored_instance_list.get_selected(), 0, value);
    }
    else if (&_server_version == sender)
      instance->serverInfo().gset("serverVersion", value);
    else if (&_ssh_host == sender)
      instance->loginInfo().gset("ssh.hostName", value);
    else if (&_ssh_port == sender)
      instance->loginInfo().gset("ssh.tunnelPort", value);
    else if (&_ssh_user == sender)
      instance->loginInfo().gset("ssh.userName", value);
    else if (&_ssh_keypath == sender)
    {
      instance->loginInfo().gset("ssh.key", value);
      instance->loginInfo().gset("ssh.useKey", 1);
      _ssh_usekey.set_active(true);
    }
    else if (&_sys_config_path == sender)
    {
      instance->serverInfo().gset("sys.config.path", value);
      _sys_profile_type.set_selected(_sys_profile_type.get_item_count()-1);
      instance->serverInfo().gset("sys.preset", "");
    }
    else if (&_sys_myini_section == sender)
    {
      instance->serverInfo().gset("sys.config.section", value);
      _sys_profile_type.set_selected(_sys_profile_type.get_item_count()-1);
      instance->serverInfo().gset("sys.preset", "");
    }
    else if (&_start_cmd == sender)
    {
      instance->serverInfo().gset("sys.mysqld.start", value);
      _sys_profile_type.set_selected(_sys_profile_type.get_item_count()-1);
      instance->serverInfo().gset("sys.preset", "");
    }
    else if (&_stop_cmd == sender)
    {
      instance->serverInfo().gset("sys.mysqld.stop", value);
      _sys_profile_type.set_selected(_sys_profile_type.get_item_count()-1);
      instance->serverInfo().gset("sys.preset", "");
    }
    else if (&_status_cmd == sender)
    {
      instance->serverInfo().gset("sys.mysqld.status", value);
      _sys_profile_type.set_selected(_sys_profile_type.get_item_count()-1);
      instance->serverInfo().gset("sys.preset", "");
    }
  }
}


void ServerInstanceEditor::check_changed(mforms::CheckBox *sender)
{
  const bool value= sender->get_active();
  db_mgmt_ServerInstanceRef instance(selected_instance());

  if (instance.is_valid())
  {
    grt::DictRef info(instance->serverInfo());

    if (&_ssh_usekey == sender)
      instance->loginInfo().gset("ssh.useKey", value ? 1 : 0);
    else if (&_sudo_check == sender)
      info.gset("sys.usesudo", value ? 1 : 0);
    else if (&_sudo_status_check == sender)
      info.gset("sys.usesudostatus", value ? 1 : 0);
  }
}


void ServerInstanceEditor::text_changed(mforms::TextBox *text)
{
  const std::string value= text->get_string_value();
  db_mgmt_ServerInstanceRef instance(selected_instance());

  if (instance.is_valid())
  {
    instance->serverInfo().gset("sys.script", value);
    _sys_profile_type.set_selected(_sys_profile_type.get_item_count()-1);
  }
}


void ServerInstanceEditor::open_editor(bool edit_selected)
{
  grtui::DbConnectionEditor editor(_mgmt);
  db_mgmt_ConnectionRef selected;
  if (edit_selected)
  {
    const int conn= _connection_selector.get_selected_index();
    if (conn >= 0 && conn < (int)_mgmt->storedConns().count())
      selected = _mgmt->storedConns()[conn];
  }

  selected= editor.run(selected);
  
  if (!edit_selected && selected.is_valid())
  {
    grt::ListRef<db_mgmt_Connection> conns(_mgmt->storedConns());

    size_t index= conns.get_index(selected);
    if (index != grt::BaseListRef::npos)
    {
      refresh_options();
      _connection_selector.set_selected(index);
    }
    else
      refresh_options();
  }
}


void ServerInstanceEditor::connection_changed()
{
  const int conn= _connection_selector.get_selected_index();
  db_mgmt_ServerInstanceRef instance(selected_instance());

  if (instance.is_valid() && conn >= 0 && conn < (int)_mgmt->storedConns().count())
  {
    db_mgmt_ConnectionRef connection(_mgmt->storedConns()[conn]);
    if (is_local_connection(connection))
    {
      instance->loginInfo().gset("ssh.hostName", "");
      _ssh_host.set_value("");

      _ssh_box.set_enabled(false);
      instance->serverInfo().gset("remoteAdmin", 1);
    }
    else
    {
      const std::string driver= connection->driver().is_valid() ? connection->driver()->name() : "";

      _ssh_box.set_enabled(true);

      if (driver == "MysqlNativeSSH")
      {
        std::string host = connection->parameterValues().get_string("sshHost");
        std::string port;

        if (host.find(':') != std::string::npos)
        {
          port = host.substr(host.find(':')+1);
          host = host.substr(0, host.find(':'));
        }
        else
          port = "22";

        _ssh_host.set_value(host);
        _ssh_port.set_value(port);
        _ssh_user.set_value(connection->parameterValues().get_string("sshUserName"));
        _ssh_keypath.set_value(connection->parameterValues().get_string("sshKeyFile"));

        instance->loginInfo().gset("ssh.hostName", host);
        instance->loginInfo().gset("ssh.port", port);
        instance->loginInfo().gset("ssh.userName", _ssh_user.get_string_value());
        instance->loginInfo().gset("ssh.useKey", _ssh_keypath.get_string_value() != "");
        instance->loginInfo().gset("ssh.key", _ssh_keypath.get_string_value());
      }
      else
      {
        const std::string hostname= connection->parameterValues().get_string("hostName");

        if (!hostname.empty())
          instance->loginInfo().gset("ssh.hostName", hostname);
        _ssh_host.set_value(hostname);        
      }
    }
    instance->connection(connection);
  }

  toggle_remote_admin();
}


void ServerInstanceEditor::profile_changed()
{
  db_mgmt_ServerInstanceRef instance(selected_instance());
  int systype = _sys_profile_type.get_selected_index();
  if (systype >= 0 && instance.is_valid())
  {
    std::string system = instance->serverInfo().get_string("sys.system");
    if (systype < (int)_presets[system].size())
    {
      std::string name = _presets[system][systype].first;    
      grt::DictRef dict = _presets[system][systype].second;

      grt::merge_contents(instance->serverInfo(), dict, true);
      instance->serverInfo().gset("sys.preset", name);
      show_instance();
    }
  }
}

//--------------------------------------------------------------------------------------------------

void ServerInstanceEditor::refresh_options()
{
  _connection_selector.clear();
  GRTLIST_FOREACH(db_mgmt_Connection, _mgmt->storedConns(), conn)
  {
    grt::DictRef params = (*conn)->parameterValues();
    std::string username= params.get_string("userName");
    std::string hostname= params.get_string("hostName");
    int port = params.get_int("port");
    std::string driver = (*conn)->driver().is_valid() ? (*conn)->driver()->caption() : "";

    // If host name is empty we have a pipe/socket connection.
    std::string connection_string;
    if (!hostname.empty())
    {
      connection_string= strfmt("%s - %s@%s:%i <%s>", (*conn)->name().c_str(), username.c_str(),
                                hostname.c_str(), port, driver.c_str());
    }
    else
    {
      std::string socket= params.get_string("socket");
      connection_string= (*conn)->name();
      if (socket.empty())
        connection_string += " - (default socket/pipe) ";
      else {
        connection_string += " - socket/pipe: " + socket + " ";
      }

      connection_string += " <" + driver + ">";
    }
    _connection_selector.add_item(connection_string);
  }

  _os_type.clear();
  for (int i = 0; system_stat_scripts[i].name; i++)
  {
    _os_type.add_item(system_stat_scripts[i].name);
  }
}


void ServerInstanceEditor::instance_changed()
{
  int i= _stored_instance_list.get_selected();
  
  if (i >= 0)
  {
    _content_box.set_enabled(true);
    _move_up_button.set_enabled(true);
    _move_down_button.set_enabled(true);
    _dup_inst_button.set_enabled(true);
  }
  else
  {
    _content_box.set_enabled(false);
    _move_up_button.set_enabled(false);
    _move_down_button.set_enabled(false);
    _dup_inst_button.set_enabled(false);
  }
  
  show_instance();
}


void ServerInstanceEditor::show_instance()
{
  int i= _stored_instance_list.get_selected();
  db_mgmt_ServerInstanceRef instance;
  
//  _content_box.suspend_layout();
  
  if (i >= 0)
  {
    instance= _instances[i];
    _del_inst_button.set_enabled(true);
    _dup_inst_button.set_enabled(true);
  }
  else
  {
    instance= db_mgmt_ServerInstanceRef(_mgmt.get_grt());
    _del_inst_button.set_enabled(false);
    _dup_inst_button.set_enabled(false);
  }

  if (instance->connection().is_valid())
  {
    int index = _mgmt->storedConns().get_index(instance->connection());
    if (index >= 0)
      _connection_selector.set_selected(index);
    else
      g_message("Invalid stored DB connection %s referenced from server instance profile", instance->connection()->name().c_str());
  }
  
  _name_entry.set_value(instance->name().c_str());
  
  _server_version.set_value(instance->serverInfo().get_string("serverVersion"));
  
  {
    grt::DictRef info(instance->loginInfo());

    _ssh_host.set_value(info.get_string("ssh.hostName"));
    _ssh_port.set_value(info.get_string("ssh.tunnelPort"));
    _ssh_user.set_value(info.get_string("ssh.userName"));
    _ssh_usekey.set_active(info.get_int("ssh.useKey") != 0);
    _ssh_keypath.set_value(info.get_string("ssh.key"));

    _ssh_box.set_enabled(!is_local_connection(instance->connection()));

    {
      std::string port = _ssh_port.get_string_value();
      std::string storage_key = strfmt("ssh@%s:%s", _ssh_host.get_string_value().c_str(), port.empty() ? "22" : port.c_str());
      std::string username = _ssh_user.get_string_value();
      std::string dummy;
      
      if (mforms::Utilities::find_password(storage_key, username, dummy))
        _ssh_pass_clear.set_enabled(true);
      else
        _ssh_pass_clear.set_enabled(false);
    }
  }
  
  {
    grt::DictRef info(instance->serverInfo());

    _enable_ssh_admin.set_active(info.get_int("remoteAdmin") != 0);
    toggle_remote_admin();
    
    std::string system= info.get_string("sys.system");
    for (int i= 0; system_stat_scripts[i].name; i++)
    {
      if (system.compare(system_stat_scripts[i].name) == 0)
      {
        _os_type.set_selected(i);
        break;
      }
    }
    refresh_profile_list();
    
    bool found= false;
    std::string preset= info.get_string("sys.preset");
    int i= 0;
    for (std::vector<std::pair<std::string,grt::DictRef> >::const_iterator iter= _presets[system].begin();
         iter != _presets[system].end(); ++iter, ++i)
    {
      if (iter->first == preset)
      {
        found= true;
        _sys_profile_type.set_selected(i);
        break;
      }
    }
    if (!found)
      _sys_profile_type.set_selected(_sys_profile_type.get_item_count()-1);

    _sys_config_path.set_value(info.get_string("sys.config.path"));
    _sys_myini_section.set_value(info.get_string("sys.config.section"));

    _start_cmd.set_value(info.get_string("sys.mysqld.start"));
    _stop_cmd.set_value(info.get_string("sys.mysqld.stop"));
    _status_cmd.set_value(info.get_string("sys.mysqld.status"));

    _sudo_check.set_active(info.get_int("sys.usesudo", 1) != 0);
    _sudo_status_check.set_active(info.get_int("sys.usesudostatus", 0) != 0);

    _script_text.set_value(info.get_string("sys.script"));
  }
  
 // _content_box.resume_layout();
}



