/* 
 * (c) 2009-2010 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#import "MGridView.h"
#include "recordset_be.h"

#import "MCPPUtilities.h"

extern const NSString* NSMenuActionNotification;


@implementation MGridView

- (void)setRecordset:(Recordset*)rset
{
  mRecordset = rset;
}


- (void) mouseDown: (NSEvent*) event;
{
  NSPoint localPoint = [self convertPoint: [event locationInWindow]
                                 fromView: nil];
  NSInteger column= [self columnAtPoint: localPoint];
  
  mOSelectedRowIndex = mSelectedRowIndex;
  mOSelectedColumnIndex = mSelectedColumnIndex;
  
  if (column < 0)
    mSelectedColumnIndex = [self numberOfColumns] - 1;
  else
    mSelectedColumnIndex= column;
  mSelectedRowIndex = [self rowAtPoint: localPoint];
  
  if (selectionChangedActionTarget)
    [selectionChangedActionTarget performSelector:selectionChangedAction];

  [self display];
}


- (void)rightMouseDown: (NSEvent*)event
{
  [self mouseDown: event];
  [super rightMouseDown: event];
}


- (void) mouseUp: (NSEvent*) event;
{
  NSPoint localPoint = [self convertPoint: [event locationInWindow]
                                 fromView: nil];
  NSInteger column= [self columnAtPoint: localPoint];
  if (column < 0)
    mSelectedColumnIndex = [self numberOfColumns] - 1;
  else
    mSelectedColumnIndex= column;
  mSelectedRowIndex = [self rowAtPoint: localPoint];
  
  if (selectionChangedActionTarget)
    [selectionChangedActionTarget performSelector:selectionChangedAction];
  
  [self display];
  
  if (column >= 0 && mSelectedRowIndex == mOSelectedRowIndex && mSelectedColumnIndex == mOSelectedColumnIndex)
  {
    if (mSelectedRowIndex >= 0 && mSelectedColumnIndex >= 0 &&
        [[self delegate] respondsToSelector: @selector(tableView:shouldEditTableColumn:row:)] &&      
        [[self delegate] tableView: self
             shouldEditTableColumn: [[self tableColumns] objectAtIndex: mSelectedColumnIndex]
                               row: mSelectedRowIndex])
      [self editColumn: mSelectedColumnIndex
                   row: mSelectedRowIndex
             withEvent: event
                select: YES];
  }
}


- (void)selectCellAtRow:(int)row column:(int)column
{
  mSelectedRowIndex = row;
  mSelectedColumnIndex = column;
  [self setNeedsDisplay:YES];
}


- (void) keyDown: (NSEvent*) event;
{
  unsigned short key = [event keyCode];
  
  if ( (key == 36) || (key == 76) ) { // Return / Enter
    if (mSelectedRowIndex >= 0 && mSelectedColumnIndex >= 0 &&
        [[self delegate] tableView: self
             shouldEditTableColumn: [[self tableColumns] objectAtIndex: mSelectedColumnIndex]
                               row: mSelectedRowIndex])
    {
      // Return or enter. Start edit.
      [self editColumn: mSelectedColumnIndex
                   row: mSelectedRowIndex
             withEvent: event
                select: YES];
    }
  }
  else if (key == 123) { // Left
    mSelectedColumnIndex -= 1;
    mSelectedColumnIndex = MAX(mSelectedColumnIndex, 0);
    if (selectionChangedActionTarget)
      [selectionChangedActionTarget performSelector:selectionChangedAction];
    [self setNeedsDisplay: YES];
  }
  else if (key == 124) { // Right
    mSelectedColumnIndex += 1;
    mSelectedColumnIndex = MIN(mSelectedColumnIndex, [self numberOfColumns] - 1);
    if (selectionChangedActionTarget)
      [selectionChangedActionTarget performSelector:selectionChangedAction];
    [self setNeedsDisplay: YES];
  }
  else if (key == 125) { // Down
    mSelectedRowIndex += 1;
    mSelectedRowIndex = MIN(mSelectedRowIndex, [self numberOfRows] - 1);
    if (selectionChangedActionTarget)
      [selectionChangedActionTarget performSelector:selectionChangedAction];
    [self setNeedsDisplay: YES];
  }
  else if (key == 126) { // Up
    mSelectedRowIndex -= 1;
    mSelectedRowIndex = MAX(mSelectedRowIndex, 0);
    if (selectionChangedActionTarget)
      [selectionChangedActionTarget performSelector:selectionChangedAction];
    [self setNeedsDisplay: YES];
  }
  else if (key == 51) { // Backspace
    [[self delegate] tableView: self
                setObjectValue: nil
                forTableColumn: [[self tableColumns] objectAtIndex: mSelectedColumnIndex]
                           row: mSelectedRowIndex];
  }
  else {
    [super keyDown: event];
  }
  
  [self scrollRowToVisible: mSelectedRowIndex];
  [self scrollColumnToVisible: mSelectedColumnIndex];
}


- (void)deleteBackward:(id)sender
{
  [[self delegate] tableView: self
              setObjectValue: nil
              forTableColumn: nil
                         row: mSelectedRowIndex];
}


- (int) selectedColumnIndex;
{
  return mSelectedColumnIndex;
}



- (int) selectedRowIndex;
{
  return mSelectedRowIndex;
}


- (void)setSelectionChangedAction:(SEL)aSelector
{
  selectionChangedAction= aSelector;
}

- (void)selectionChangedActionTarget:(id)target;
{
  selectionChangedActionTarget = target;
}


static std::vector<int> get_indexes(NSIndexSet *iset, NSInteger clickedRow)
{
  std::vector<int> indexes;
  NSUInteger index = [iset firstIndex];
  while (index != NSNotFound)
  {
    indexes.push_back(index);
    index = [iset indexGreaterThanIndex: index];
  }
  std::reverse(indexes.begin(), indexes.end());
  
  if (indexes.empty() && clickedRow >= 0)
    indexes.push_back(clickedRow);
  
  return indexes;
}


- (NSMenu *)menuForEvent:(NSEvent *)theEvent
{
  if (mRecordset)
  {      
    std::vector<int> rows = get_indexes([self selectedRowIndexes], [self selectedRowIndex]);
    bec::MenuItemList items= mRecordset->get_popup_menu_items(rows, [self selectedColumnIndex]-1);
    
    if (!items.empty())
    {
      NSMenu *menu= [[[NSMenu alloc] initWithTitle: @""] autorelease];
      [menu setAutoenablesItems: NO];
      
      for (bec::MenuItemList::const_iterator iter= items.begin(); iter != items.end(); ++iter)
      {        
        if (iter->type == bec::MenuSeparator)
          [menu addItem: [NSMenuItem separatorItem]];
        else
        {
          NSMenuItem *item= [menu addItemWithTitle: [NSString stringWithCPPString: iter->caption]
                                            action: @selector(activateMenuItem:)
                                     keyEquivalent: @""];
          [item setTarget: self];
          if (!iter->enabled)
            [item setEnabled: NO];
          [item setRepresentedObject: [NSString stringWithCPPString: iter->name]];
        }
      }
      return menu;
    }
  }
  return [super menuForEvent: theEvent];
}



- (void)activateMenuItem:(id)sender
{
  if (mRecordset)
  {
    std::vector<int> rows = get_indexes([self selectedRowIndexes], [self selectedRowIndex]);
    if (mRecordset->activate_popup_menu_item([[sender representedObject] UTF8String], 
                                             rows, [self selectedColumnIndex]-1))
      [self reloadData];
    else
      [[NSNotificationCenter defaultCenter] postNotificationName: NSMenuActionNotification object: sender];
  }
}


@end


