/* 
 * (c) 2009-2010 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import "sigobjc++.h"
#import "MSQLEditorController.h"

#import "ScintillaView.h"
#import "Scintilla.h"

#import "sql_editor_be.h"
#import "WBPluginEditorBase.h"
#import "search_replace.h"

using namespace Scintilla;

//--------------------------------------------------------------------------------------------------

@implementation MSQLEditorController

- (void) setString: (NSString*) string
{
  [mScintillaView setString: string];
}

- (NSString*) string
{
  NSString* text= [mScintillaView selectedString];
  if ([text length] == 0)
    return [mScintillaView string];
  else
    return text;
}

- (void)insertText: (NSString*) text
{
  [mScintillaView insertText: text];
}  

//--------------------------------------------------------------------------------------------------

/**
 * Initialize scintilla editor (styles, colors, markers etc.) for MySQL.
 */
- (void) setupEditor
{  
  [[NSNotificationCenter defaultCenter] addObserver: self
                                           selector: @selector(textDidChange:)
                                               name: NSTextDidChangeNotification
                                             object: mScintillaView];
  [[NSNotificationCenter defaultCenter] addObserver: self
                                           selector: @selector(uiWasUpdated:)
                                               name: SCIUpdateUINotification
                                             object: mScintillaView];
  [WBPluginEditorBase setupCodeEditor: mScintillaView backend: *mBackEnd withStatus: YES];  
  [mScintillaView setStatusText: @""/*"No errors found"*/];
}

//--------------------------------------------------------------------------------------------------

- (void)setView:(ScintillaView*)view
{
  mScintillaView = view;
}

//--------------------------------------------------------------------------------------------------

- (ScintillaView*)view
{
  return mScintillaView;
}

//--------------------------------------------------------------------------------------------------

static int process_sql_statement_border(int begin_lineno, int begin_line_pos, int end_lineno, int end_line_pos, 
                                        MSQLEditorController *self);
static int process_sql_error(const int err_tok_lineno, const int err_tok_line_pos, const int err_tok_len, 
                             const std::string &err_msg, MSQLEditorController *self);
static void callCheckSql(MSQLEditorController *self);
static bool search_replace(const std::string search_text, const std::string replace_text, 
                           int flags, MSQLEditorController *self);
static int insert_text(const std::string &text, MSQLEditorController *self);

//--------------------------------------------------------------------------------------------------

/**
 * Called when the UI in the SQL editor was updated (e.g. caret or selection change).
 */
- (void) uiWasUpdated: (NSNotification*) aNotification
{
  NSRange range= [[mScintillaView content] selectedRange];
  int start= range.location;
  int end= range.location + range.length;
  (*mBackEnd)->set_selected_range(start, end);
};

//--------------------------------------------------------------------------------------------------

/**
 * Called when text was added or removed in the editor.
 */
- (void) textDidChange: (NSNotification*) aNotification
{  
  // Stop the timer in case it is about to trigger.
  [mSqlCheckTimer invalidate];
  
  // Set up a new timer.
  mSqlCheckTimer= [NSTimer scheduledTimerWithTimeInterval: 0.5
                                                   target: self
                                                 selector: @selector(checkSql:)
                                                 userInfo: nil
                                                  repeats: NO];
};

//--------------------------------------------------------------------------------------------------

- (void) setEditorBackEnd:(const Sql_editor::Ref&) be
{
  mBackEnd= new Sql_editor::Ref(be);
  [self setupEditor];
  
  (*mBackEnd)->report_sql_statement_border= sigc::bind(sigc::ptr_fun(process_sql_statement_border), self);
  (*mBackEnd)->sql_parser_err_cb(sigc::bind(sigc::ptr_fun(process_sql_error), self));
  (*mBackEnd)->do_search_slot= sigc::bind(sigc::ptr_fun(search_replace), self);
  (*mBackEnd)->insert_text_slot= sigc::bind(sigc::ptr_fun(insert_text), self);
}

//--------------------------------------------------------------------------------------------------

- (Sql_editor::Ref)backEnd
{
  return *mBackEnd;
}

//--------------------------------------------------------------------------------------------------

- (void) dealloc
{
  delete mBackEnd;
  [super dealloc];
}

//--------------------------------------------------------------------------------------------------

/**
 * Remove all markers we set for previous sql errors.
 */
- (void) resetSqlCheckState
{
  [mScintillaView setStatusText: @""/*"No errors found"*/];
  
  int length= [mScintillaView getGeneralProperty: SCI_GETLENGTH parameter: 0];
  
  [mScintillaView setGeneralProperty: SCI_SETINDICATORCURRENT parameter: 0 value: 0];
  [mScintillaView setGeneralProperty: SCI_INDICATORCLEARRANGE parameter: 0 value: length];
  
  [mScintillaView setGeneralProperty: SCI_MARKERDELETEALL parameter: -1 value: 0];
}

//--------------------------------------------------------------------------------------------------

/**
 * Triggered once a change in the editor content happend and there was a pause of at least 500 ms.
 */
- (void) checkSql:(NSTimer*) timer
{
  [self resetSqlCheckState];
  
  mSqlCheckTimer = nil;
  NSString *text = [mScintillaView string];
  if (text)
  {
    mErrorCount = 0;
    std::string sql = [text UTF8String];
    if (sql != (*mBackEnd)->sql())
      (*mBackEnd)->sql(sql);
    (*mBackEnd)->check_sql(*mBackEnd, false);
  }
}

//--------------------------------------------------------------------------------------------------

/**
 * Triggered once a change in the editor content happend and there was a pause of at least 500 ms.
 */
- (NSString *) currentSqlStatement
{
  int current_pos= [mScintillaView getGeneralProperty: SCI_GETCURRENTPOS parameter: 0];
  int current_line= [mScintillaView getGeneralProperty: SCI_LINEFROMPOSITION parameter: current_pos];
  int current_line_beginning_pos= [mScintillaView getGeneralProperty: SCI_POSITIONFROMLINE parameter: current_line];
  int current_line_pos= current_pos - current_line_beginning_pos;
  
  Sql_editor::SqlStatementBorder sql_statement_border= (*mBackEnd)->get_sql_statement_border_by_line_pos(current_line+1, current_line_pos);
  if (sql_statement_border.begin_lineno == -1)
    return @"";
  CharacterRange cr;
  cr.cpMin= [mScintillaView getGeneralProperty: SCI_POSITIONFROMLINE parameter: sql_statement_border.begin_lineno - 1];
  cr.cpMin+= sql_statement_border.begin_line_pos;
  cr.cpMax= [mScintillaView getGeneralProperty: SCI_POSITIONFROMLINE parameter: sql_statement_border.end_lineno - 1];
  cr.cpMax+= sql_statement_border.end_line_pos;
  int doc_length= [mScintillaView getGeneralProperty: SCI_GETLENGTH parameter: 0];
  if (cr.cpMax > doc_length)
    cr.cpMax= doc_length;
  
  TextRange tr;
  tr.chrg= cr;
  tr.lpstrText= new char[cr.cpMax - cr.cpMin + 1];
  [mScintillaView getGeneralProperty: SCI_GETTEXTRANGE ref: &tr];
  NSString *sql= [NSString stringWithUTF8String: tr.lpstrText];
  delete [] tr.lpstrText;
  
  return sql;
}

//--------------------------------------------------------------------------------------------------

/**
 * Callback from the backend telling us about sql statement range.
 */
static int process_sql_statement_border(int begin_lineno, int begin_line_pos, int end_lineno, int end_line_pos, 
                                        MSQLEditorController *self)
{
  [self->mScintillaView setGeneralProperty: SCI_MARKERADD parameter: begin_lineno - 1 value: 0];
  return 0;
}

//--------------------------------------------------------------------------------------------------

/**
 * Callback from the backend telling us about sql errors it founds.
 */
static int process_sql_error(const int err_tok_lineno, const int err_tok_line_pos, const int err_tok_len,
                             const std::string &err_msg, MSQLEditorController *self)
{
  int line_start_pos= [self->mScintillaView getGeneralProperty: SCI_POSITIONFROMLINE parameter: err_tok_lineno - 1];
  
  [self->mScintillaView setGeneralProperty: SCI_SETINDICATORCURRENT parameter: 0 value: 0];
  [self->mScintillaView setGeneralProperty: SCI_INDICATORFILLRANGE parameter: line_start_pos + err_tok_line_pos value: err_tok_len];
  
  [self->mScintillaView setGeneralProperty: SCI_MARKERADD parameter: err_tok_lineno - 1 value: 1];
  
  ++self->mErrorCount;
  [self->mScintillaView setStatusText: [NSString stringWithFormat: @"%d error(s) found.", self->mErrorCount]];
  
  return 0;
}

//--------------------------------------------------------------------------------------------------

static int insert_text(const std::string &text, MSQLEditorController *self)
{
  NSString* sql_text = [NSString stringWithUTF8String: text.c_str()];
  if (sql_text == nil)
    sql_text= [NSString stringWithCString: text.c_str() encoding: NSISOLatin1StringEncoding];
  [self insertText: sql_text];
  return 0;
}

//--------------------------------------------------------------------------------------------------

/**
 * Callback to let us know the user searchs text.
 */
static bool search_replace(const std::string search_text, const std::string replace_text, 
                           int flags, MSQLEditorController *self)
{
  if (search_text.size() == 0)
    return false;
  
  /* TODO: port find/replace code from .NET.
   mforms::SearchFlags searchFlags= (mforms::SearchFlags) flags;
   int scintillaFlags = 0;
   
   bool previous = (searchFlags & mforms::SearchPrevious) != 0;
   bool useRegEx = (searchFlags & mforms::SearchUseRegularExpression) != 0;
   bool replace = (searchFlags & mforms::SearchDoReplace) != 0;
   bool all = (searchFlags & mforms::SearchAll) != 0;
   */
  
  /*
   Range searchRange = new Range(0, RawText.Length, this);
   if ((searchFlags & Sql_editor.SearchFlags.SearchMatchCase) != 0)
   scintillaFlags |= SearchFlags.MatchCase;
   if ((searchFlags & Sql_editor.SearchFlags.SearchMatchWholeWord) != 0)
   scintillaFlags |= SearchFlags.WholeWord;
   
   Range result = null;
   */
  bool canClose = false;
  /*
   if (useRegEx)
   {
   Regex expression = new Regex(searchText);
   if (replace)
   {
   if (all)
   {
   FindReplace.ReplaceAll(expression, replaceText);
   canClose = true;
   }
   else
   {
   // There is no overload to replace with RE search, so we have to handle that ourselves.
   if (previous)
   result = FindReplace.FindPrevious(expression, true);
   else
   result = FindReplace.FindNext(expression, true);
   
   if (result != null)
   {
   result.Text = replaceText;
   result.End = result.Start + replaceText.Length;
   }
   }
   }
   else
   if (previous)
   result = FindReplace.FindPrevious(expression, true);
   else
   result = FindReplace.FindNext(expression, true);
   }
   else
   {
   if (replace)
   {
   if (all)
   {
   FindReplace.ReplaceAll(searchText, replaceText, scintillaFlags);
   canClose = true;
   }
   else
   {
   if (previous)
   result = FindReplace.ReplacePrevious(searchText, replaceText, true, scintillaFlags);
   else
   result = FindReplace.ReplaceNext(searchText, replaceText, true, scintillaFlags);
   }
   }
   else
   {
   if (previous)
   result = FindReplace.FindPrevious(searchText, true, scintillaFlags);
   else
   result = FindReplace.FindNext(searchText, true, scintillaFlags);
   }
   }
   if (result != null)
   {
   canClose = true;
   result.Select();
   }
   */
  return canClose;
}

//--------------------------------------------------------------------------------------------------

@end
