﻿/* 
 * Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, 
 * Boston, MA 02110-1301  USA.
 */

using System.Windows.Forms;
using System.Collections.Generic;

namespace MySQL.Utilities
{
  /// <summary>
  /// Collection of helper functions working with .NET controls.
  /// </summary>
  public class ControlUtilities
  {
    /// <summary>
    /// Helper method to find out if this control or one of its children has the input focus.
    /// </summary>
    /// <returns></returns>
    public static bool IsHierarchyFocused(Control parent)
    {
      if (parent.Focused)
        return true;

      foreach (Control control in parent.Controls)
        if (IsHierarchyFocused(control))
          return true;

      return false;
    }

    /// <summary>
    /// Does a depth-first search for a control under root. 
    /// </summary>
    /// <returns>Returns the control with the given name or null if not found.</returns>
    public static Control FindControl(Control root, string target)
    {
      if (root.Name == target)
        return root;

      foreach (Control control in root.Controls)
      {
        Control child = FindControl(control, target);
        if (child != null)
          return child;
      }

      return null;
    }

    /// <summary>
    /// Determines if a control is a child within the hierarchy formed by root.
    /// Does a depth-first search for a control.
    /// </summary>
    /// <returns>Returns true if the control was found, false otherwise.</returns>
    public static bool ContainsControl(Control root, Control target)
    {
      if (root == target)
        return true;

      foreach (Control control in root.Controls)
      {
        if (ContainsControl(control, target))
          return true;
      }

      return false;
    }

    /// <summary>
    /// Returns the the currently active non-container control under parent.
    /// </summary>
    /// <param name="parent">The root of the control hierarchy to search in. This must not be null.</param>
    /// <returns>The currently active non-container control under parent.</returns>
    public static Control GetLeafActiveControl(ContainerControl parent)
    {
      Control control = parent.ActiveControl;
      while (control != null)
      {
        ContainerControl child;
        child = control as ContainerControl;
        if (child == null)
          break;
        control = child.ActiveControl;
      }
      return control;
    }
  }

}
