/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.Design;
using System.Drawing;
using System.Drawing.Design;
using System.Drawing.Drawing2D;
using System.Runtime.InteropServices;
using System.Windows.Forms;

using MySQL.Utilities;
using MySQL.Utilities.SysUtils;

namespace MySQL.Controls
{
	/// <summary>
	/// Summary description for FlatTabControl.
	/// </summary>
	[ToolboxBitmap(typeof(System.Windows.Forms.TabControl))] //,
		//Designer(typeof(Designers.FlatTabControlDesigner))]

  public class FlatTabControl : System.Windows.Forms.TabControl
	{
    // Summary: determines position and look of the tabs.
    public enum TabStyleType
    {
      TopNormal,      // Tabs standing on the content above the top line. Focus color is a constant color.
      TopHanging,     // Tabs hanging down above the content. Focus color is a gradient.
      BottomNormal,   // Tabs hanging down below the content, selected color is Tab.BackColor.
    }

    public enum CloseButtonVisiblity
    {
      ShowButton,
      HideButton,
      InheritVisibilitiy,
    }

    // TabInfo keeps all necessary layout information.
    private class TabInfo
    {
      internal TabPage page;
      internal bool isValid;
      internal CloseButtonVisiblity closeButtonVisibility; // For individual close buttons.
      internal Rectangle tabArea;
      internal Rectangle buttonArea;

      internal TabInfo()
      {
        this.page = null;
        isValid = false;
        closeButtonVisibility = CloseButtonVisiblity.InheritVisibilitiy;
      }
    }

    /// <summary> 
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;

    private ImageList leftRightImages = null;
    private Bitmap darkCloseButton;
    private Bitmap lightCloseButton;
    private Size buttonSize;
    private SubClass scroller = null;

    private Color backColor = Color.FromArgb(0xff, 0x28, 0x37, 0x52);
    private bool showCloseButton = true; // Central flag for close buttons. Can be overridden by each page.
    private bool showFocusState = true;
    private bool canCloseLastTab = false;
    private bool hideWhenEmpty = false;

    // Painting/Layouting
    private TabStyleType tabStyle = TabStyleType.TopNormal;
    private List<TabInfo> layoutInfo = new List<TabInfo>();

    // The padding within the tabs.
    private Padding itemPadding = new Padding(6, 0, 6, 0);

    // The padding around the tab pages in the tab control.
    private Padding contentPadding = new Padding();

    // Mouse handling.
    private int lastTabHit = -1;
    private bool buttonHit = false;
    private bool inWheelHandling = false;

    #region Construction and destruction

    public FlatTabControl()
		{
			// This call is required by the Windows.Forms Form Designer.
			InitializeComponent();

			// Some special styles for the control.
			SetStyle(ControlStyles.UserPaint, true);
			SetStyle(ControlStyles.AllPaintingInWmPaint, true);
			SetStyle(ControlStyles.DoubleBuffer, true);
			SetStyle(ControlStyles.ResizeRedraw, true); // Doesn't really work. Need Invalidate() call in OnResize too.
			SetStyle(ControlStyles.SupportsTransparentBackColor, true);
			SetStyle(ControlStyles.OptimizedDoubleBuffer, true);
      UpdateStyles();

			leftRightImages = new ImageList();

			System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(FlatTabControl));
			Bitmap icons = ((System.Drawing.Bitmap)(resources.GetObject("TabIcons.bmp")));
			if (icons != null)
			{
				icons.MakeTransparent(Color.White);
				leftRightImages.Images.AddStrip(icons);
			}

      // Both buttons really should have the same size.
      darkCloseButton = ((System.Drawing.Bitmap)(resources.GetObject("tab_close_dark")));
      lightCloseButton = ((System.Drawing.Bitmap)(resources.GetObject("tab_close_light")));
      buttonSize = new Size(darkCloseButton.Width, darkCloseButton.Height);
    }

    protected override void OnCreateControl()
    {
      base.OnCreateControl();

      FindScroller();
    }

    /// <summary> 
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose(bool disposing)
		{
			if (disposing)
			{
				if(components != null)
				{
					components.Dispose();
				}

				leftRightImages.Dispose();
        darkCloseButton.Dispose();
        lightCloseButton.Dispose();
      }
			base.Dispose(disposing);
		}

    #endregion

    #region Native Windows code needed for some adjustments

    private const int TCM_ADJUSTRECT = 0x1328;
    protected override void WndProc(ref System.Windows.Forms.Message m)
    {
      switch (m.Msg)
      {
        case TCM_ADJUSTRECT:
        {
          // We need to adjust the display rectangle, which directly determines where the content is shown.
          Win32.RECT rectangle = (Win32.RECT) m.GetLParam(typeof(Win32.RECT));

          // The "larger" value indicates the direction of the computation.
          // A 1 means from display rectangle to window rectangle. A 0 the other way around.
          int topOffset = 0;
          if (tabStyle == TabStyleType.TopNormal || tabStyle == TabStyleType.TopHanging)
            topOffset = ItemSize.Height;
          int bottomOffset = 0;
          if (tabStyle == TabStyleType.BottomNormal)
            bottomOffset = ItemSize.Height;

          int larger = (int) m.WParam;
          if (larger == 0)
          {
            rectangle.Left += Margin.Left + contentPadding.Left;
            rectangle.Right -= Margin.Right + contentPadding.Right;
            rectangle.Top += Margin.Top + ContentPadding.Top + topOffset;
            rectangle.Bottom -= Margin.Bottom + ContentPadding.Bottom + bottomOffset;
          }
          else
          {
            rectangle.Left -= Margin.Left + contentPadding.Left;
            rectangle.Right += Margin.Right + contentPadding.Right;
            rectangle.Top -= Margin.Top + contentPadding.Top + topOffset;
            rectangle.Bottom += Margin.Bottom + contentPadding.Bottom + bottomOffset;
          }

          Marshal.StructureToPtr(rectangle, m.LParam, true); 

          break;
        }

        case Win32.WM_NCHITTEST:
        {
          // Declare everything as being part of the client area, so we get mouse events for that.
          m.Result = new IntPtr(1); // HT_CLIENT
          break;
        }

        case Win32.WM_LBUTTONDOWN:
        {
          // Handle this message directly to keep the base class from doing unwanted things.
          uint lparam = (uint)m.LParam.ToInt32();
          int x = Win32.LoWord(lparam);
          int y = Win32.HiWord(lparam);
          MouseEventArgs args = new MouseEventArgs(MouseButtons.Left, 1, x, y, 0);
          OnMouseDown(args);
          m.Result = IntPtr.Zero;

          break;
        }

        case Win32.WM_MOUSEWHEEL:
          if (!inWheelHandling)
          {
            inWheelHandling = true;

            // Forward mouse wheel messages to the control under the mouse pointer.
            // Only handle messages if the mouse pointer is in the form's client area.
            // Also check we don't have any other window (e.g. dialogs) in front.
            Win32.POINT pos = new Win32.POINT(m.LParam.ToInt32() & 0xffff, m.LParam.ToInt32() >> 16);
            IntPtr targetWindow = Win32.WindowFromPoint(pos);
            if (targetWindow != IntPtr.Zero)
            {
              Control target = Control.FromHandle(targetWindow);
              if ((targetWindow != m.HWnd) && (target != null) && (target != this))
              {
                Win32.SendMessage(targetWindow, m.Msg, m.WParam, m.LParam);
                m.Result = IntPtr.Zero;
              }
            }
            inWheelHandling = false;
          }
          break;

        default:
          base.WndProc(ref m);
          break;
      }
    }

    private int ScrollerWndProc(ref Message m)
    {
      switch (m.Msg)
      {
        case Win32.WM_PAINT:
          {
            IntPtr hDC = Win32.GetWindowDC(scroller.Handle);
            Graphics g = Graphics.FromHdc(hDC);
            DrawScroller(g);
            g.Dispose();
            Win32.ReleaseDC(scroller.Handle, hDC);

            // Validate client rectangle, just in case.
            Win32.RECT rect = new Win32.RECT();

            Win32.GetClientRect(scroller.Handle, ref rect);
            Win32.ValidateRect(scroller.Handle, ref rect);

            // return 0 (processed)
            m.Result = IntPtr.Zero;

          }
          return 1;
      }

      return 0;
    }

    protected void RecalculateFrame()
    {
      // TODO: cannot get the content rectangle calculation to start again, investigate.
      // Needed if the tab style changes after the control was already shown.
    }

    #endregion

    #region Drawing the control and its parts.

    protected override void OnPaint(PaintEventArgs e)
	  {
	    base.OnPaint(e); 
  			
	    DrawControl(e.Graphics);
	  }

		internal void DrawControl(Graphics g)
		{
			if (!Visible)
				return;

      bool drawFocused = showFocusState && ControlUtilities.IsHierarchyFocused(this);

			// Fill client area with the control's background color
      // (everything outside the inner rectangle, e.g. including margin).
      Rectangle clientArea = ClientRectangle;
      Brush brush = new SolidBrush(BackgroundColor);
      g.FillRectangle(brush, clientArea);
      brush.Dispose();

      // Prepare other background effects (like gradient).
      // Depending on the tab style this might include or ignore the margin.
      switch (tabStyle)
      {
        case TabStyleType.TopHanging:
          {
            // In top hanging style we need a background shade simulating a shadow.
            LinearGradientBrush gradientBrush = new LinearGradientBrush(
              new Rectangle(0, Margin.Top, 1, ItemSize.Height),
              Color.FromArgb(255, 0x18, 0x25, 0x3b),
              BackgroundColor, LinearGradientMode.Vertical);

            g.FillRectangle(gradientBrush, 0, Margin.Top, clientArea.Width, ItemSize.Height);
          }
          break;
        case TabStyleType.TopNormal:
          // Next fill the inner rectangle (tab control area within margins but not covered by content)
          // with the inner color but keep the top padding area the same color as that of the tab.
          if (SelectedTab != null)
          {
            Rectangle innerArea = clientArea;
            innerArea.X += Margin.Left;
            innerArea.Width -= Margin.Horizontal;
            innerArea.Y += Margin.Top + ItemSize.Height + ContentPadding.Top;
            innerArea.Height -= Margin.Vertical + ItemSize.Height + ContentPadding.Top;

            brush = new SolidBrush(Color.FromArgb(0xff, 0x4a, 0x61, 0x84));
            g.FillRectangle(brush, innerArea);
            brush.Dispose();

            innerArea.Y -= ContentPadding.Top;
            innerArea.Height = ContentPadding.Top;
            if (drawFocused)
              brush = new SolidBrush(Color.FromArgb(0xff, 0xff, 0xe9, 0xb6));
            else
              brush = new SolidBrush(Color.FromArgb(0xff, 0x4a, 0x61, 0x84));
            g.FillRectangle(brush, innerArea);
            brush.Dispose();
          }

          break;
      }

      // Draw only tabs which entirely lie within the client area.
      for (int i = 0; i < this.TabCount; i++)
      {
        ValidateTab(i);
        if (clientArea.Contains(layoutInfo[i].tabArea))
          DrawTab(g, i, drawFocused);
      }
		}

		internal void DrawTab(Graphics g, int index, bool drawFocused)
		{
			RectangleF bounds = GetTabRect(index);

      // For high quality drawing with anti aliased lines we need to specify line and gradient coordinates
      // which lie between two pixels (not exactly *on* one, so adjust by a half pixel offset for these cases.
      bounds.X -= 0.5f;
      bounds.Y -= 0.5f;
      bounds.Height = ItemSize.Height;

      bool isSelected = (SelectedIndex == index);
      if (isSelected)
      {
        // The active tab gets a special background.
        g.SmoothingMode = SmoothingMode.HighQuality;

        // Define form of the tab.
        GraphicsPath path = GetTabOutline(ref bounds);

        // Define gradients based on the tab style.
        ColorBlend gradientTupels = GetTabGradient(index, drawFocused);

        LinearGradientBrush gradientBrush = new LinearGradientBrush(
          new RectangleF(Margin.Left, Margin.Top - 0.5f, 1, ItemSize.Height),
          Color.Black, Color.White,
          LinearGradientMode.Vertical
        );
        gradientBrush.InterpolationColors = gradientTupels;
        gradientBrush.ScaleTransform(-1, -1);
       
        g.FillPath(gradientBrush, path);
        path.Dispose();
        g.SmoothingMode = SmoothingMode.Default;
      }

			// Tab icon.
      TabPage page = TabPages[index];
			if ((page.ImageIndex >= 0) && (ImageList != null) && (ImageList.Images[page.ImageIndex] != null))
			{
				int nLeftMargin = 8;
				int nRightMargin = 2;

				Image img = ImageList.Images[page.ImageIndex];
				
				RectangleF rimage = new RectangleF(bounds.X + nLeftMargin, bounds.Y + 1, img.Width, img.Height);
				
				// adjust rectangles
				float nAdj = (float)(nLeftMargin + img.Width + nRightMargin);

				rimage.Y += ((float) bounds.Height - img.Height) / 2;
				bounds.X += nAdj;
				bounds.Width -= nAdj;

				g.DrawImage(img, rimage);
			}

			// Tab text.
			StringFormat stringFormat = new StringFormat();
			stringFormat.Alignment = StringAlignment.Near;
      stringFormat.LineAlignment = StringAlignment.Center;

      Brush brush;
      if (isSelected)
      {
        if (tabStyle == TabStyleType.TopHanging || tabStyle == TabStyleType.TopNormal)
        {
          if (drawFocused)
            brush = new SolidBrush(Color.Black);
          else
            brush = new SolidBrush(Color.FromArgb(255, 0xcc, 0xd3, 0xdf));
        }
        else
          brush = new SolidBrush(Color.Black);
      }
      else
        brush = new SolidBrush(Color.White);

      bounds.Inflate(-itemPadding.Horizontal, -itemPadding.Vertical);
      g.DrawString(page.Text, Font, brush, bounds, stringFormat);
      brush.Dispose();

      // Finally the close button if this tab is active.
      if (isSelected && IsCloseButtonVisible(index))
      {
        Rectangle buttonRect = CloseButtonRect(index);
        if (drawFocused)
          g.DrawImage(darkCloseButton, buttonRect);
        else
          g.DrawImage(lightCloseButton, buttonRect);
      }
		}

    /// <summary>
    /// Creates the outline of a tab depending on its style.
    /// </summary>
    /// <param name="bounds">The bounding rectangle for the tab.</param>
    /// <returns>The outline of the tab as path, ready to be filled.</returns>
    private GraphicsPath GetTabOutline(ref RectangleF recBounds)
    {
      GraphicsPath path = new GraphicsPath();
      float cornerSize = 6;

      switch (tabStyle)
      {
        case TabStyleType.TopHanging:
          path.AddLine(recBounds.Left, recBounds.Top, recBounds.Right, recBounds.Top);
          path.AddLine(recBounds.Right, recBounds.Top, recBounds.Right, recBounds.Bottom - cornerSize);
          path.AddArc(recBounds.Right - cornerSize, recBounds.Bottom - cornerSize, cornerSize, cornerSize, 0, 90);
          path.AddLine(recBounds.Right - cornerSize, recBounds.Bottom, recBounds.Left + cornerSize, recBounds.Bottom);
          path.AddArc(recBounds.Left, recBounds.Bottom - cornerSize, cornerSize, cornerSize, 90, 90);
          path.AddLine(recBounds.Left, recBounds.Bottom - cornerSize, recBounds.Left, recBounds.Top);
          break;

        case TabStyleType.TopNormal:
          path.AddLine(recBounds.Right, recBounds.Bottom, recBounds.Left, recBounds.Bottom);
          path.AddLine(recBounds.Left, recBounds.Bottom, recBounds.Left, recBounds.Top - cornerSize);
          path.AddArc(recBounds.Left, recBounds.Top, cornerSize, cornerSize, 180, 90);
          path.AddLine(recBounds.Left + cornerSize, recBounds.Top, recBounds.Right - cornerSize, recBounds.Top);
          path.AddArc(recBounds.Right - cornerSize, recBounds.Top, cornerSize, cornerSize, -90, 90);
          path.AddLine(recBounds.Right, recBounds.Top + cornerSize, recBounds.Right, recBounds.Bottom);
          break;

        case TabStyleType.BottomNormal:
          path.AddLine(recBounds.Left, recBounds.Top, recBounds.Right, recBounds.Top);
          path.AddLine(recBounds.Right, recBounds.Top, recBounds.Right, recBounds.Bottom - cornerSize);
          path.AddArc(recBounds.Right - cornerSize, recBounds.Bottom - cornerSize, cornerSize, cornerSize, 0, 90);
          path.AddLine(recBounds.Right - cornerSize, recBounds.Bottom, recBounds.Left + cornerSize, recBounds.Bottom);
          path.AddArc(recBounds.Left, recBounds.Bottom - cornerSize, cornerSize, cornerSize, 90, 90);
          path.AddLine(recBounds.Left, recBounds.Bottom - cornerSize, recBounds.Left, recBounds.Top);
          break;

      }

      return path;
    }

    /// <summary>
    /// Creates a color gradient for a tab depending on its style.
    /// </summary>
    /// <returns>The color gradient which can be used in a gradient brush.</returns>
    private ColorBlend GetTabGradient(int index, bool drawFocused)
    {
      ColorBlend result = new ColorBlend();
      switch (tabStyle)
      {
        case TabStyleType.TopHanging:
          if (drawFocused)
          {
            result.Colors = new Color[] {
                Color.FromArgb(0xff, 0xff, 0xe9, 0xb6),
                Color.FromArgb(0xff, 0xff, 0xe9, 0xb7),
                Color.FromArgb(0xff, 0xff, 0xf3, 0xd9),
                Color.FromArgb(0xff, 0xff, 0xf3, 0xd9),
                Color.FromArgb(0xff, 0xc9, 0xc3, 0xb8),
                Color.FromArgb(0xff, 0x6b, 0x69, 0x66)
              };

            result.Positions = new float[] {
                0, 0.45f, 0.52f, 0.75f, 0.87f, 1
              };
          }
          else
          {
            result.Colors = new Color[] {
                Color.FromArgb(0xff, 0x4a, 0x61, 0x84),
                Color.FromArgb(0xff, 0x4a, 0x61, 0x84),
                Color.FromArgb(0xff, 0x3a, 0x4c, 0x68),
                Color.FromArgb(0xff, 0x1f, 0x29, 0x37)
              };

            result.Positions = new float[] {
                0, 0.75f, 0.87f, 1
              };
          }
          break;
        case TabStyleType.TopNormal:
          if (drawFocused)
          {
            result.Colors = new Color[] {
                Color.FromArgb(0xff, 0xff, 0xe9, 0xb6),
                Color.FromArgb(0xff, 0xff, 0xe9, 0xb7),
                Color.FromArgb(0xff, 0xff, 0xf3, 0xd9),
                Color.FromArgb(0xff, 0xff, 0xfa, 0xf4)
              };

            result.Positions = new float[] {
                0, 0.34f, 0.43f, 1
              };
          }
          else
          {
            result.Colors = new Color[] {
                Color.FromArgb(0xff, 0x4a, 0x61, 0x84),
                Color.FromArgb(0xff, 0x4a, 0x61, 0x84)
              };

            result.Positions = new float[] {
                0, 1
              };
          }
          break;
        case TabStyleType.BottomNormal:
          result.Colors = new Color[] {
              TabPages[index].BackColor,
              TabPages[index].BackColor
            };

          result.Positions = new float[] {
              0, 1
            };
          break;
      }

      return result;
    }

    /// <summary>
    /// Returns the rectangle of the given tab for drawing.
    /// </summary>
    /// <param name="index"></param>
    /// <returns></returns>
    new public Rectangle GetTabRect(int index)
    {
      ValidateTab(index);

      return layoutInfo[index].tabArea;
    }

    /// <summary>
    /// Returns the rectangle of the close button for drawing.
    /// </summary>
    /// <param name="index"></param>
    /// <returns></returns>
    public Rectangle CloseButtonRect(int index)
    {
      ValidateTab(index);

      return layoutInfo[index].buttonArea;
    }

		internal void DrawScroller(Graphics g)
		{
			if ((leftRightImages == null) || (leftRightImages.Images.Count != 4))
				return;

			//----------------------------
			// calc positions
			Rectangle clientArea = this.ClientRectangle;

			Rectangle r0 = new Rectangle();
			//Win32.GetClientRect(scUpDown.Handle, ref r0);

			Brush br = new SolidBrush(SystemColors.Control);
			g.FillRectangle(br, r0);
			br.Dispose();
			
			Pen border = new Pen(SystemColors.ControlDark);
			Rectangle rborder = r0;
			rborder.Inflate(-1, -1);
			g.DrawRectangle(border, rborder);
			border.Dispose();

			int nMiddle = (r0.Width / 2);
			int nTop = (r0.Height - 16) / 2;
			int nLeft = (nMiddle - 16) / 2;

			Rectangle r1 = new Rectangle(nLeft, nTop, 16, 16);
			Rectangle r2 = new Rectangle(nMiddle+nLeft, nTop, 16, 16);
			//----------------------------

			//----------------------------
			// draw buttons
			Image img = leftRightImages.Images[1];
			if (img != null)
			{
				if (this.TabCount > 0)
				{
					Rectangle r3 = this.GetTabRect(0);
					if (r3.Left < clientArea.Left)
						g.DrawImage(img, r1);
					else
					{
						img = leftRightImages.Images[3];
						if (img != null)
							g.DrawImage(img, r1);
					}
				}
			}

			img = leftRightImages.Images[0];
			if (img != null)
			{
				if (this.TabCount > 0)
				{
					Rectangle r3 = this.GetTabRect(this.TabCount - 1);
					if (r3.Right > (clientArea.Width - r0.Width))
						g.DrawImage(img, r2);
					else
					{
						img = leftRightImages.Images[2];
						if (img != null)
							g.DrawImage(img, r2);
					}
				}
			}
			//----------------------------
		}

    /// <summary>
    /// Searches for a child control of type UpDown32, which the standard tab control
    /// maintains and we want to customize.
    /// </summary>
    private void FindScroller()
    {
      scroller = null;

      // Find the UpDown control.
      IntPtr pWnd = Win32.GetWindow(this.Handle, Win32.GW_CHILD);

      while (pWnd != IntPtr.Zero)
      {
        // Get the window class name
        char[] className = new char[33];

        int length = Win32.GetClassName(pWnd, className, 32);
        string s = new string(className, 0, length);

        if (s == "msctls_updown32")
        {
          scroller = new SubClass(pWnd, true);
          scroller.SubClassedWndProc += new SubClass.SubClassWndProcEventHandler(ScrollerWndProc);
          break;
        }

        pWnd = Win32.GetWindow(pWnd, Win32.GW_HWNDNEXT);
      }
    }

    private void UpdateScroller()
    {
      if (scroller != null)
      {
        if (Win32.IsWindowVisible(scroller.Handle))
        {
          Win32.RECT rect = new Win32.RECT();
          Win32.GetWindowRect(scroller.Handle, ref rect);
          //Win32.InvalidateRect(scroller.Handle, ref rect, true);
          if (tabStyle == TabStyleType.BottomNormal)
            rect.Top = ClientSize.Height - rect.Height - Margin.Bottom;
          Win32.MoveWindow(scroller.Handle, rect.Left, rect.Top, rect.Width, rect.Height, true);
        }
      }
    }

    #endregion

    #region Event handling

    protected override void OnHandleCreated(EventArgs e)
    {
      base.OnHandleCreated(e);

      Form container = Application.OpenForms[0].TopLevelControl as Form;
      if (container != null)
      {
        container.Deactivate += new EventHandler(ActivationChanged);
        container.Activated += new EventHandler(ActivationChanged);
      }
    }

    protected override void OnHandleDestroyed(EventArgs e)
    {
      base.OnHandleDestroyed(e);

      // Unregister our activation listener from the application's main form.
      // This assumes the main form hasn't changed in the meantime (which should be the case in 99.99% of all cases).
      if (Application.OpenForms.Count > 0)
      {
        Form container = Application.OpenForms[0].TopLevelControl as Form;
        if (container != null)
        {
          container.Deactivate -= ActivationChanged;
          container.Activated -= ActivationChanged;
        }
      }
    }

    private void ActivationChanged(object sender, EventArgs e)
    {
      Invalidate();
    }

    override protected void OnControlAdded(ControlEventArgs e)
		{
      base.OnControlAdded(e);

      if (e.Control is TabPage)
      {
        TabPage page = e.Control as TabPage;
        page.TextChanged += new EventHandler(PageTextChanged);
        AdjustLayoutInfo(e.Control as TabPage);
        UpdateScroller();

        Show();
      }
		}

		override protected void OnControlRemoved(ControlEventArgs e)
		{
      base.OnControlRemoved(e);

      if (e.Control is TabPage)
      {
        TabPage page = e.Control as TabPage;
        page.TextChanged -= PageTextChanged;
        AdjustLayoutInfo(e.Control as TabPage);
        UpdateScroller();

        // The tab is still there and removed when execution returns to the caller.
        // Hence the test on 1 instead 0.
        if (!DesignMode && hideWhenEmpty && TabCount == 1)
          Hide();
      }
		}

		override protected void OnSelectedIndexChanged(EventArgs e)
		{
      base.OnSelectedIndexChanged(e);

			UpdateScroller();
			Invalidate();	// We need to update border and background colors.
		}

    override protected void OnResize(EventArgs e)
    {
      base.OnResize(e);
      AdjustLayoutInfo(null);
      Invalidate();
      UpdateScroller();
    }

    override protected void OnMouseDown(MouseEventArgs e)
    {
      switch (e.Button)
      {
        case MouseButtons.Left:
          Focus();
          lastTabHit = TabIndexFromPosition(e.Location);
          if (lastTabHit > -1)
          {
            if (SelectedIndex != lastTabHit)
            {
              SuspendLayout();
              SelectedIndex = lastTabHit;
              ResumeLayout(true);
            }
            else
              if (layoutInfo[lastTabHit].buttonArea.Contains(e.Location))
                buttonHit = true;
          }
          break;

        default:
          base.OnMouseDown(e);
          break;
      }
    }

    protected override void OnMouseUp(MouseEventArgs e)
    {
      switch (e.Button)
      {
        case MouseButtons.Left:
          if (lastTabHit > -1)
          {
            if (layoutInfo[lastTabHit].buttonArea.Contains(e.Location) && buttonHit &&
              (TabCount > 1 || canCloseLastTab))
            {
              // Close tab if the application agrees.
              TabPage page = TabPages[lastTabHit];
              CloseTabPage(page);
            }
          }
          break;

        default:
          base.OnMouseUp(e);
          break;
      }
    }

    protected override void OnEnter(EventArgs e)
    {
      base.OnEnter(e);
      Invalidate();
    }

    protected override void OnLeave(EventArgs e)
    {
      base.OnLeave(e);
      Invalidate();
    }

    void PageTextChanged(object sender, EventArgs e)
    {
      AdjustLayoutInfo(sender as TabPage);
    }

    public event EventHandler<TabClosingEventArgs> TabClosing;
    protected internal void OnTabClosing(TabClosingEventArgs args)
    {
      if (TabClosing != null)
        TabClosing(this, args);
    }

    public event EventHandler<TabClosedEventArgs> TabClosed;
    protected internal void OnTabClosed(TabClosedEventArgs args)
    {
      if (TabClosed != null)
        TabClosed(this, args);
    }

    #endregion

    #region Component Designer generated code

    /// <summary>
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			components = new System.ComponentModel.Container();
		}

		#endregion

    #region Other implementation

    /// <summary>
    /// Determines if the the close button for a given tab should be shown.
    /// </summary>
    private bool IsCloseButtonVisible(int index)
    {
      return (layoutInfo[index].closeButtonVisibility == CloseButtonVisiblity.ShowButton) ||
        (layoutInfo[index].closeButtonVisibility == CloseButtonVisiblity.InheritVisibilitiy && showCloseButton);
    }

    /// <summary>
    /// Adjusts the size of the layout info list and marks all entries that follow the given page
    /// as invalid, so that they re-compute their layout next time it is required.
    /// </summary>
    private void AdjustLayoutInfo(TabPage page)
    {
      if (layoutInfo.Count > TabCount)
        layoutInfo.RemoveRange(TabCount, layoutInfo.Count - TabCount);

      for (int i = 0; i < TabCount; i++)
      {
        if (i >= layoutInfo.Count)
          layoutInfo.Add(new TabInfo());
        layoutInfo[i].page = TabPages[i];
      }

      bool invalidate = (page == null) ? true : false;
      foreach (TabInfo entry in layoutInfo)
      {
        // Invalidate this page and all following.
        if (entry.page == page)
          invalidate = true;
        if (invalidate)
          entry.isValid = false;
      }
    }

    /// <summary>
    /// Recomputes layout information for this tab if not yet done.
    /// </summary>
    private void ValidateTab(int index)
    {
      int buttonSpacing = 5; // Number of pixels between text and close button.

      if (!layoutInfo[index].isValid)
      {
        Graphics g = CreateGraphics();

        // Find first invalid entry which is before the given one.
        // We have to validate all entries from there on.
        int currentIndex = 0;
        while (currentIndex < index && layoutInfo[currentIndex].isValid)
          currentIndex++;

        int offsetX = Margin.Left;
        if (currentIndex > 0)
          offsetX = layoutInfo[currentIndex - 1].tabArea.Right;
        int offsetY = Margin.Top;
        if (tabStyle == TabStyleType.BottomNormal)
          offsetY = ClientSize.Height - ItemSize.Height - Margin.Bottom;
        int buttonPart = 0;
        if (IsCloseButtonVisible(currentIndex))
          buttonPart = buttonSpacing + buttonSize.Width;
        for (; currentIndex <= index; currentIndex++)
        {
          TabInfo info = layoutInfo[currentIndex];
          SizeF textSize = g.MeasureString(info.page.Text, Font);

          // Resize tab height if the text would not fit vertically.
          if (textSize.Height > ItemSize.Height)
            ItemSize = new Size(ItemSize.Width, (int) Math.Ceiling(textSize.Height));
          info.tabArea = new Rectangle(offsetX, offsetY,
            2 * itemPadding.Horizontal + (int)Math.Ceiling(textSize.Width) + buttonPart,
            ItemSize.Height);
          offsetX = info.tabArea.Right;

          if (IsCloseButtonVisible(currentIndex))
          {
            int buttonOffset = offsetY + (info.tabArea.Height - buttonSize.Height) / 2;
            info.buttonArea = new Rectangle(offsetX - itemPadding.Horizontal - buttonSize.Width, buttonOffset,
              buttonSize.Width, buttonSize.Height);
          }
          else
            info.buttonArea = Rectangle.Empty;

          info.isValid = true;
        }

        g.Dispose();
      }
    }

    /// <summary>
    /// Allows to set individual close buttons for a tab.
    /// </summary>
    public void SetCloseButtonVisibility(int index, CloseButtonVisiblity visibility)
    {
      AdjustLayoutInfo(TabPages[index]);
      layoutInfo[index].closeButtonVisibility = visibility;
    }

    /// <summary>
    /// Does the actual work to close a tab page, sending out appropriate events and updating
    /// internal structures. Returns true if the page was actually closed.
    /// </summary>
    public bool CloseTabPage(TabPage page)
    {
      if (TabCount > 1 || canCloseLastTab)
      {
        TabClosingEventArgs args = new TabClosingEventArgs(page, true);
        OnTabClosing(args);
        if (args.canClose)
        {
          // Remove this tab and select its predecessor if possible.
          if (SelectedIndex > 0)
            SelectedIndex = SelectedIndex - 1;
          lastTabHit = -1;
          TabPages.Remove(page);

          AdjustLayoutInfo(null);
          Update();

          OnTabClosed(new TabClosedEventArgs(page));

          return true;
        }
      }
      return false;
    }

    /// <summary>
    /// Determines if the given position lies within the bounds of a tab and, if so, returns the tab's
    /// index to the caller. The tab itself must entirely lie within the client bounds.
    /// </summary>
    /// <param name="point">The position to check for in local coordinates.</param>
    /// <returns>The index of the tab found or -1 if none.</returns>
    public int TabIndexFromPosition(Point point)
    {
      Rectangle bounds = ClientRectangle;
      for (int i = 0; i < TabCount; i++)
      {
        ValidateTab(i);
        if (bounds.Contains(layoutInfo[i].tabArea) && layoutInfo[i].tabArea.Contains(point))
          return i;
      }
      return -1;
    }

    // In order to simplify general handling of page content (which is very often just a container
    // with the actual content) we introduce here the concept of documents.
    // A document is the content of a tab. When using this concept you should only have one control on each
    // tab to make this work. Otherwise use the tab pages directly and handle search etc. yourself.
    // See also ITabDocument and TabDocument and the Documents property below.

    /// <summary>
    /// Returns the document (i.e. the first control on a tab page) whose page has the given title.
    /// </summary>
    /// <param name="text"></param>
    /// <returns></returns>
    public ITabDocument FindDocument(string text)
    {
      foreach (TabPage page in TabPages)
        if (page.Text == text)
          if (page.Controls.Count > 0 && page.Controls[0] is ITabDocument && page.Text == text)
            return page.Controls[0] as ITabDocument;
          else
            return null;

      return null;
    }

    /// <summary>
    /// Returns the document stored on the tab page with the given index or null if there is none.
    /// </summary>
    public ITabDocument DocumentFromIndex(int index)
    {
      return DocumentFromPage(TabPages[index]);
    }

    /// <summary>
    /// Returns the document stored on the given tab page or null if there is none.
    /// </summary>
    public ITabDocument DocumentFromPage(TabPage page)
    {
      if (page.Controls.Count > 0 && page.Controls[0] is ITabDocument)
        return page.Controls[0] as ITabDocument;
      else
        return null;
    }

    /// <summary>
    /// Checks if the document is the first control on any of our pages.
    /// </summary>
    /// <returns>True if the document was found, otherwise false.</returns>
    public bool HasDocument(ITabDocument document)
    {
      foreach (TabPage page in TabPages)
        if (page.Controls.Count > 0 && page.Controls[0] == document)
          return true;

      return false;
    }

    /// <summary>
    /// Adds the given document to a new tab page and returns the index of that new page.
    /// </summary>
    /// <param name="document"></param>
    /// <returns></returns>
    public int AddDocument(ITabDocument document)
    {
      TabPage page = new TabPage();
      TabDocument control = document as TabDocument;
      control.SetHost(page);
      page.Controls.Add(control);
      control.Dock = DockStyle.Fill;
      control.Margin = new Padding(0);
      control.Padding = new Padding(0);
      control.Show();
      TabPages.Add(page);

      return TabPages.Count - 1;
    }

    /// <summary>
    /// Removes a previously added document from this control. This method removes the hosting tab,
    /// regardless of other content on it. The document itself is not freed.
    /// </summary>
    /// <param name="document"></param>
    public void RemoveDocument(ITabDocument document)
    {
      foreach (TabPage page in TabPages)
        if (page.Controls.Count > 0 && page.Controls[0] == document)
        {
          page.Controls.Clear();
          TabPages.Remove(page);

          return;
        }
    }

    /// <summary>
    /// Tries to close the given document by sending out TabClosing. If that returns true
    /// the document is closed and its hosting tab removed.
    /// </summary>
    /// <returns>True if the document was closed, otherwise false.</returns>
    public bool CloseDocument(ITabDocument document)
    {
      foreach (TabPage page in TabPages)
        if (page.Controls.Count > 0 && page.Controls[0] == document)
          return CloseTabPage(page);

      return false;
    }

    /// <summary>
    /// Returns the list of documents in an array, e.g. to allow manipulation of this list in a loop.
    /// </summary>
    /// <returns></returns>
    public ITabDocument[] DocumentsToArray()
    {
      ITabDocument[] result = new ITabDocument[TabCount];
      int i = 0;
      foreach (ITabDocument content in Documents)
        result[i++] = content;

      return result;
    }

    #endregion

    #region Properties

    /// <summary>
    /// Returns the currently active document if there is one.
    /// </summary>
    public ITabDocument ActiveDocument
    {
      get
      {
        if (SelectedTab != null && SelectedTab.Controls.Count > 0 && SelectedTab.Controls[0] is ITabDocument)
          return SelectedTab.Controls[0] as ITabDocument;
        return null;
      }
    }

    /// <summary>
    /// Helper enumeration that returns the first control on each tab page, which is often the only
    /// control forming an own "document" with many other content.
    /// </summary>
    /// <returns></returns>
    public IEnumerable<ITabDocument> Documents
    {
      get
      {
        foreach (TabPage page in TabPages)
          if (page.Controls.Count > 0 && page.Controls[0] is ITabDocument)
            yield return page.Controls[0] as ITabDocument;
          else
            yield return null;
      }
    }

		public TabStyleType TabStyle
		{
			get { return tabStyle; }
			set 
      {
        if (value != tabStyle)
        {
          tabStyle = value;
          AdjustLayoutInfo(null);
          Invalidate();
        }
      }
		}

    public Padding ItemPadding
    {
      get { return itemPadding; }
      set
      {
        itemPadding = value;
        AdjustLayoutInfo(null);
        Invalidate();
      }
    }

    public Padding ContentPadding
    {
      get { return contentPadding; }
      set
      {
        contentPadding = value;
        AdjustLayoutInfo(null);
        Invalidate();
      }
    }

    [Browsable(true)]
		public Color BackgroundColor
		{
			get { return backColor; }
			set
      {
        if (value != backColor)
        {
          backColor = value;
          Invalidate();
        }
      }
		}

    [Browsable(true)]
    public bool ShowCloseButton
    {
      get { return showCloseButton; }
      set
      {
        if (value != showCloseButton)
        {
          showCloseButton = value;
          AdjustLayoutInfo(null);
          Invalidate();
        }
      }
    }

    [Browsable(true)]
    public bool ShowFocusState
    {
      get { return showFocusState; }
      set
      {
        if (value != showFocusState)
        {
          showFocusState = value;
          Invalidate();
        }
      }
    }

    [Browsable(true)]
    public bool CanCloseLastTab
    {
      get { return canCloseLastTab; }
      set { canCloseLastTab = value; }
    }

    [Browsable(true)]
    public bool HideWhenEmpty
    {
      get { return hideWhenEmpty; }
      set
      {
        if (value != hideWhenEmpty)
        {
          hideWhenEmpty = value;
          if (!DesignMode && hideWhenEmpty && TabCount == 0)
            Hide();
          else
            Show();
        }
      }
    }

    [Editor(typeof(TabpageExCollectionEditor), typeof(UITypeEditor))]
    public new TabPageCollection TabPages
    {
      get
      {
        return base.TabPages;
      }
    }

    #endregion

		#region TabpageExCollectionEditor

		internal class TabpageExCollectionEditor : CollectionEditor
		{
			public TabpageExCollectionEditor(System.Type type): base(type)
			{
			}
            
			protected override Type CreateCollectionItemType()
			{
				return typeof(TabPage);
			}
		}
        
		#endregion
	}

  //------------------------------------------------------------------------------------------------

  // In order to aid managing documents on tab pages (instead of plain controls) we define an interface and 
  // a base class which implements it, to be used by the application, if it wants to maintain
  // the "document metaphor".
  public interface ITabDocument
  {
    void Activate();
    void Close();

    String TabText { get; set; }
  }

  public class TabDocument : Form, ITabDocument
  {
    private TabPage host = null;
    private String tabText = "";
    private String toolTipText = "";

    public TabDocument()
    {
      FormBorderStyle = FormBorderStyle.None;
      TopLevel = false;
    }

    new public void Activate()
    {
      base.Activate();
      (host.Parent as TabControl).SelectedTab = host;
    }

    public void SetHost(TabPage host)
    {
      if (this.host != host)
      {
        if (this.host != null)
        {
          FlatTabControl tabControl = this.host.Parent as FlatTabControl;
          if (tabControl != null)
          {
            // Remove the tab page this document was docked to if it gets moved to another one.
            this.host.Controls.Clear();
            tabControl.TabPages.Remove(this.host);
            tabControl.OnTabClosed(new TabClosedEventArgs(this.host));
          }
        }

        this.host = host;
        if (host != null)
        {
          host.Text = tabText;
          host.ToolTipText = toolTipText;
        }
      }
    }

    protected override void Dispose(bool disposing)
    {
      host = null;
      base.Dispose(disposing);
    }

    /// <summary>
    /// Called when the document is closed already. Just remove its tab page.
    /// No need to trigger tab closing and closed events.
    /// </summary>
    override protected void OnFormClosed(FormClosedEventArgs e)
    {
      if (host != null && host.Parent != null)
      {
        host.Controls.Clear();
        (host.Parent as FlatTabControl).TabPages.Remove(host);
      }
      base.OnFormClosed(e);
    }

    public virtual String TabText
    {
      get { return tabText; }
      set
      {
        tabText = value;
        if (host != null)
          host.Text = value;
      }
    }

    public virtual String ToolTipText
    {
      get { return toolTipText; }
      set
      {
        toolTipText = value;
        if (host != null)
          host.ToolTipText = value;
      }
    }

    public int Index
    {
      get
      {
        if (host != null)
          return host.TabIndex;
        return -1;
      }
    }
  }

  public class TabClosingEventArgs : EventArgs
  {
    public TabPage page;
    public bool canClose;

    public TabClosingEventArgs(TabPage page, bool canClose)
    {
      this.page = page;
      this.canClose = canClose;
    }
  }

  public class TabClosedEventArgs : EventArgs
  {
    public TabPage page;

    public TabClosedEventArgs(TabPage page)
    {
      this.page = page;
    }
  }

}
