﻿/* 
 * Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

using System;
using System.Drawing;
using System.Windows.Forms;
using System.Drawing.Imaging;
using System.Drawing.Drawing2D;

using MySQL.Utilities.SysUtils;
using System.Threading;

namespace MySQL.Utilities
{
  public partial class HUDForm : Form
  {
    // Make the HUD form a singleton. If we need more than one instance at a later point in time
    // the we need to split that into a singleton message popup and a base HUD window.
    static readonly HUDForm instance = new HUDForm();

    private Size baseSize;
    private float borderSize = 3;
    private float shadowSize = 20;
    private Color shadowColor = Color.Black;
    private float shadowOffset = 5;
    private float cornerSize = 20;
    private int animationSteps = 4;
    private bool useAnimation = false;

    private Bitmap contentBitmap;

    static HUDForm()
    {
      // Thread safety without locks.
    }

    HUDForm()
    {
      InitializeComponent();
      baseSize = Size;
      SetStyle(ControlStyles.Selectable, false);
    }

    public void Show(String title, String message, bool animated)
    {
      label1.Text = title;
      label2.Text = message;

      Reshow(animated);
    }

    public void Reshow(bool animated)
    {
      Form mainForm = Application.OpenForms["MainForm"];
      if (mainForm == null)
        throw new Exception("HUD display needs main form to be named \"MainForm\" to find it.");

      mainForm.Update();

      // The (transparent) HUD will cover the entire application window and by setting the application window
      // as the parent we make it application local instead topmost for the entire desktop.
      Bounds = mainForm.Bounds;
      //Win32.SetParent(Handle, mainForm.Handle);
      //Location = new Point(0, 0);
      //Size = mainForm.Size;

      PrepareBitmap();
      useAnimation = animated;
      if (useAnimation)
      {
        SetBitmap(contentBitmap, 0);
        Win32.ShowWindow(Handle, Win32.SW_SHOWNOACTIVATE);
        for (float i = 1; i <= animationSteps; i++)
          SetBitmap(contentBitmap, (int)(i / animationSteps * 255));
      }
      else
      {
        SetBitmap(contentBitmap, 255);
        Win32.ShowWindow(Handle, Win32.SW_SHOWNOACTIVATE);
      }
    }

    public new void Hide()
    {
      Form mainForm = Application.OpenForms["MainForm"];
      if (mainForm == null)
        mainForm.Update();

      if (useAnimation)
      {
        for (float i = animationSteps; i >= 0;  i--)
          SetBitmap(contentBitmap, (int)(i / animationSteps * 255));
      }
      base.Hide();
    }

    #region Drawing

    protected GraphicsPath GetPath()
    {
      // Generate the outline of the actual content area. Center it around the origin.
      // It will later get transformed to the final position.
      GraphicsPath result = new GraphicsPath();

      float width = baseSize.Width;// -2 * borderSize; excluding the border here hasn't any effect it seems
      float height = baseSize.Height;// -2 * borderSize;
      RectangleF bounds = new RectangleF(-width / 2, -height / 2, width, height);
      result.AddArc(bounds.Left, bounds.Top, cornerSize, cornerSize, 180, 90);
      result.AddArc(bounds.Right - cornerSize, bounds.Top, cornerSize, cornerSize, -90, 90);
      result.AddArc(bounds.Right - cornerSize, bounds.Bottom - cornerSize, cornerSize, cornerSize, 0, 90);
      result.AddArc(bounds.Left, bounds.Bottom - cornerSize, cornerSize, cornerSize, 90, 90);
      result.CloseAllFigures();
      return result;
    }

    /// <summary>
    /// Prepares the bitmap used to draw the window. Layered windows (like this one) use a bitmap for their
    /// content, including alpha channel.
    /// </summary>
    protected void PrepareBitmap()
    {
      contentBitmap = new Bitmap(Width, Height);

      GraphicsPath path = GetPath();
      GraphicsPath innerPath = (GraphicsPath)path.Clone();

      // Increase size of the outline by the shadow size and move it to the center.
      // The inner path keeps the original bounds for clipping.
      Matrix matrix = new Matrix();

      float offsetX = Width / 2;
      float offsetY = Height / 2;
      matrix.Translate(offsetX + shadowOffset, offsetY + shadowOffset);
      matrix.Scale(1 + (2 * shadowSize + borderSize) / (float)baseSize.Width, 1 + (2 * shadowSize + borderSize) / (float)baseSize.Height);
      path.Transform(matrix);

      // Also move the inner part to its final place.
      matrix.Reset();
      matrix.Translate(offsetX, offsetY);
      innerPath.Transform(matrix);

      Graphics g = Graphics.FromImage(contentBitmap);
      g.SmoothingMode = SmoothingMode.HighQuality;
      using (Brush brush = new SolidBrush(Color.FromArgb(10, 0, 0, 0)))
      {
        g.FillRectangle(brush, ClientRectangle);
      }

      // Fill interior.
      using (Brush brush = new SolidBrush(Color.FromArgb(191, 0, 0, 0)))
      {
        g.FillPath(brush, innerPath);
      }

      // ... and draw border around the interior.
      Pen borderPen = new Pen(Color.FromArgb(200, Color.White));
      borderPen.EndCap = LineCap.Round;
      borderPen.StartCap  = LineCap.Round;
      borderPen.Width = borderSize;
      GraphicsPath borderPath = (GraphicsPath)innerPath.Clone();
      borderPath.Widen(borderPen);
      g.FillPath(new SolidBrush(Color.FromArgb(255, Color.White)), borderPath);

      // Clip out interior. Exclude both, the panel itself as well as its border.
      Region region = new Region(innerPath);
      g.SetClip(region, CombineMode.Exclude);

      innerPath.Widen(borderPen);
      region = new Region(innerPath);
      g.SetClip(region, CombineMode.Exclude);

      PathGradientBrush backStyle = new PathGradientBrush(path);

      backStyle.CenterColor = shadowColor;
      backStyle.SurroundColors = new Color[] { Color.Transparent };

      // Make a smooth fade out of the shadow color using the built-in sigma bell curve generator.
      backStyle.SetSigmaBellShape(0.4f, 1f);

      // Now draw the shadow.
      g.FillPath(backStyle, path);

      // Remove clipping for the remaining interior.
      g.ResetClip();
      RectangleF innerBounds = innerPath.GetBounds();
      Point targetLocation = pictureBox1.Location;
      targetLocation.Offset((int)innerBounds.Left, (int)innerBounds.Top);
      g.DrawImageUnscaled(pictureBox1.Image, targetLocation);

      // Message text output.
      StringFormat format = new StringFormat(StringFormatFlags.FitBlackBox);
      targetLocation = label1.Location;
      targetLocation.Offset((int)innerBounds.Left, (int)innerBounds.Top);
      RectangleF textBounds = new RectangleF(targetLocation,
        new SizeF(
          innerBounds.Right - targetLocation.X - Padding.Right,
          innerBounds.Bottom - targetLocation.Y - Padding.Bottom
        )
      );

      using (Brush textBrush = new SolidBrush(label1.ForeColor))
      {
        g.DrawString(label1.Text, label1.Font, textBrush, textBounds, format);
      }

      targetLocation = label2.Location;
      targetLocation.Offset((int)innerBounds.Left, (int)innerBounds.Top);
      textBounds = new RectangleF(targetLocation,
        new SizeF(
          innerBounds.Right - targetLocation.X - Padding.Right,
          innerBounds.Bottom - targetLocation.Y - Padding.Bottom
        )
      );

      using (Brush textBrush = new SolidBrush(label2.ForeColor))
      {
        g.DrawString(label2.Text, label2.Font, textBrush, textBounds, format);
      }
    }

    protected override void OnPaintBackground(PaintEventArgs e)
    {
      //base.OnPaintBackground(e);
    }

    protected override void OnPaint(PaintEventArgs e)
    {
//      SetBitmap(contentBitmap, 255);
    }

    /// <summary>
    /// Sets the given bitmap as window content with transparent parts via the
    /// layered windows API.
    /// </summary>
    /// <param name="bitmap">The bitmap to set.</param>
    /// <param name="opacity">The overall opacity (255 = opaque).</param>
		public void SetBitmap(Bitmap bitmap, int opacity) 
		{
			if (bitmap.PixelFormat != PixelFormat.Format32bppArgb)
				throw new ApplicationException("The bitmap must be 32ppp with alpha-channel.");

			IntPtr screenDc = Win32.GetDC(IntPtr.Zero);
			IntPtr memDc = Win32.CreateCompatibleDC(screenDc);
			IntPtr hBitmap = IntPtr.Zero;
			IntPtr oldBitmap = IntPtr.Zero;

			try 
			{
				hBitmap = bitmap.GetHbitmap(Color.FromArgb(0));  // grab a GDI handle from this GDI+ bitmap
				oldBitmap = Win32.SelectObject(memDc, hBitmap);

				Win32.SIZE size = new Win32.SIZE(bitmap.Width, bitmap.Height);
				Win32.POINT pointSource = new Win32.POINT(0, 0);
				Win32.POINT topPos = new Win32.POINT(Left, Top);
				Win32.BLENDFUNCTION blend = new Win32.BLENDFUNCTION();
				blend.BlendOp = Win32.AC_SRC_OVER;
				blend.BlendFlags = 0;
				blend.SourceConstantAlpha = (byte) opacity;
				blend.AlphaFormat = Win32.AC_SRC_ALPHA;

				Win32.UpdateLayeredWindow(Handle, screenDc, ref topPos, ref size, memDc, ref pointSource, 0, ref blend, Win32.ULW_ALPHA);
			}
			finally
			{
				Win32.ReleaseDC(IntPtr.Zero, screenDc);
				if (hBitmap != IntPtr.Zero) 
				{
					Win32.SelectObject(memDc, oldBitmap);
					Win32.DeleteObject(hBitmap);
				}
				Win32.DeleteDC(memDc);
			}
		}

    #endregion

    #region Properties

    public static HUDForm Instance
    {
      get { return instance;  }
    }

    protected override bool ShowWithoutActivation
    {
      get { return true; }
    }

    #endregion

    protected override CreateParams CreateParams	
		{
			get 
			{
				CreateParams cp = base.CreateParams;

        cp.ExStyle |= Win32.WS_EX_TOPMOST;
        cp.ExStyle |= Win32.WS_EX_LAYERED;
        cp.ExStyle |= Win32.WS_EX_NOACTIVATE;
				return cp;
			}
		}

    #region Events and message handling

    protected override void WndProc(ref Message m)
    {
      switch (m.Msg)
      {
        case Win32.WM_LBUTTONDOWN:
          Hide();
          m.Result = IntPtr.Zero;
          return;
        case Win32.WM_MOUSEACTIVATE:
          m.Result = new IntPtr(Win32.MA_NOACTIVATE);
          return;
      }
      base.WndProc(ref m);
    }

    #endregion

  }
}
