using System;
using System.Runtime.InteropServices;
using System.Windows.Forms;

namespace MySQL.Utilities.SysUtils
{
  /// <summary>
  /// Helper class for invoking Windows functionality directly.
  /// </summary>
  public class Win32
  {
    // GetWindow() constants
    public const int GW_HWNDFIRST         = 0;
    public const int GW_HWNDLAST          = 1;
    public const int GW_HWNDNEXT          = 2;
    public const int GW_HWNDPREV          = 3;
    public const int GW_OWNER             = 4;
    public const int GW_CHILD             = 5;

    // ShowWindow() constants.
    public const int SW_SHOWNOACTIVATE    = 4;

    public const int WM_WINDOWPOSCHANGING = 0x46;
    public const int WM_PAINT             = 0xF;
    public const int WM_CREATE            = 0x1;
    public const int WM_PRINT             = 0x317;
    public const int WM_DESTROY           = 0x2;
    public const int WM_SHOWWINDOW        = 0x18;
    public const int WM_SHARED_MENU       = 0x1E2;
    public const int HC_ACTION            = 0;
    public const int WH_CALLWNDPROC       = 4;
    public const int GWL_WNDPROC          = -4;

    // Mouse messages
    public const int WM_MOUSEACTIVATE     = 0x0021;

    public const int WM_MOUSEFIRST        = 0x0200;
    public const int WM_MOUSEMOVE         = 0x0200;
    public const int WM_LBUTTONDOWN       = 0x0201;
    public const int WM_LBUTTONUP         = 0x0202;
    public const int WM_LBUTTONDBLCLK     = 0x0203;
    public const int WM_RBUTTONDOWN       = 0x0204;
    public const int WM_RBUTTONUP         = 0x0205;
    public const int WM_RBUTTONDBLCLK     = 0x0206;
    public const int WM_MBUTTONDOWN       = 0x0207;
    public const int WM_MBUTTONUP         = 0x0208;
    public const int WM_MBUTTONDBLCLK     = 0x0209;
    public const int WM_MOUSEWHEEL        = 0x020A;
    public const int WM_XBUTTONDOWN       = 0x020B;
    public const int WM_XBUTTONUP         = 0x020C;
    public const int WM_XBUTTONDBLCLK     = 0x020D;
    public const int WM_MOUSEHWHEEL       = 0x020E;
    public const int WM_MOUSELAST         = 0x020E;

    // Mouse message constants.
    public const int MA_ACTIVATE         = 1;
    public const int MA_ACTIVATEANDEAT   = 2;
    public const int MA_NOACTIVATE       = 3;
    public const int MA_NOACTIVATEANDEAT = 4;

    // Other Windows messages.
    public const int WM_NCCREATE          = 0x0081;
    public const int WM_NCDESTROY         = 0x0082;
    public const int WM_NCCALCSIZE        = 0x0083;
    public const int WM_NCHITTEST         = 0x0084;
    public const int WM_NCPAINT           = 0x0085;
    public const int WM_NCACTIVATE        = 0x0086;
    public const int WM_GETDLGCODE        = 0x0087;
    public const int WM_SYNCPAINT         = 0x0088;
    public const int WM_NCMOUSEMOVE       = 0x00A0;
    public const int WM_NCLBUTTONDOWN     = 0x00A1;
    public const int WM_NCLBUTTONUP       = 0x00A2;
    public const int WM_NCLBUTTONDBLCLK   = 0x00A3;
    public const int WM_NCRBUTTONDOWN     = 0x00A4;
    public const int WM_NCRBUTTONUP       = 0x00A5;
    public const int WM_NCRBUTTONDBLCLK   = 0x00A6;
    public const int WM_NCMBUTTONDOWN     = 0x00A7;
    public const int WM_NCMBUTTONUP       = 0x00A8;
    public const int WM_NCMBUTTONDBLCLK   = 0x00A9;

    public const int WM_EXITSIZEMOVE      = 0x0232;

    // Listview message constants.
    public const int LVM_FIRST = 0x1000;
    public const int LVM_GETITEMRECT = (LVM_FIRST + 14);
    public const int LVM_APPROXIMATEVIEWRECT = (LVM_FIRST + 64);

    public const int LVIR_BOUNDS = 0;
    public const int LVIR_ICON = 1;
    public const int LVIR_LABEL = 2;

    public static readonly IntPtr HWND_TOPMOST   = new IntPtr(-1);
    public static readonly IntPtr HWND_NOTOPMOST = new IntPtr(-2);
    public static readonly IntPtr HWND_TOP = new IntPtr(0);

    public const UInt32 SWP_NOSIZE         = 0x0001;
    public const UInt32 SWP_NOMOVE = 0x0002;
    public const UInt32 SWP_NOZORDER = 0x0004;
    public const UInt32 SWP_NOREDRAW = 0x0008;
    public const UInt32 SWP_NOACTIVATE = 0x0010;
    public const UInt32 SWP_FRAMECHANGED = 0x0020;  /* The frame changed: send WM_NCCALCSIZE */
    public const UInt32 SWP_SHOWWINDOW = 0x0040;
    public const UInt32 SWP_HIDEWINDOW = 0x0080;
    public const UInt32 SWP_NOCOPYBITS = 0x0100;
    public const UInt32 SWP_NOOWNERZORDER = 0x0200;  /* Don't do owner Z ordering */
    public const UInt32 SWP_NOSENDCHANGING = 0x0400;  /* Don't send WM_WINDOWPOSCHANGING */

    public const Int32 ULW_COLORKEY = 0x00000001;
    public const Int32 ULW_ALPHA    = 0x00000002;
    public const Int32 ULW_OPAQUE   = 0x00000004;

    public const byte AC_SRC_OVER  = 0x00;
    public const byte AC_SRC_ALPHA = 0x01;

    // Window styles
    public const int WS_POPUP  = unchecked((int) 0x80000000);
    public const int WS_BORDER = 0x00800000;

    public const int WS_EX_TOPMOST = 0x00000008;
    public const int WS_EX_LAYERED    = 0x00080000;
    public const int WS_EX_NOACTIVATE = 0x08000000;

    [StructLayout(LayoutKind.Sequential)]
    public struct RECT
    {
      public int Left;
      public int Top;
      public int Right;
      public int Bottom;

      public int Width
      {
        get { return Right - Left; }
      }

      public int Height
      {
        get { return Bottom - Top; }
      }
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct WINDOWPOS
    {
      public IntPtr hwnd;
      public IntPtr hwndAfter;
      public int x;
      public int y;
      public int cx;
      public int cy;
      public uint flags;
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct NCCALCSIZE_PARAMS
    {
      public RECT rgc;
      public WINDOWPOS wndpos;
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct POINT
    {
      public Int32 x;
      public Int32 y;

      public POINT(Int32 x, Int32 y) { this.x = x; this.y = y; }
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct SIZE
    {
      public Int32 cx;
      public Int32 cy;

      public SIZE(Int32 cx, Int32 cy) { this.cx = cx; this.cy = cy; }
    }

    [StructLayout(LayoutKind.Sequential, Pack = 1)]
    struct ARGB
    {
      public byte Blue;
      public byte Green;
      public byte Red;
      public byte Alpha;
    }

    [StructLayout(LayoutKind.Sequential, Pack = 1)]
    public struct BLENDFUNCTION
    {
      public byte BlendOp;
      public byte BlendFlags;
      public byte SourceConstantAlpha;
      public byte AlphaFormat;
    }

    [DllImport("user32.dll")]
    [return: MarshalAs(UnmanagedType.Bool)]
    public static extern bool SetWindowPos(IntPtr hWnd, IntPtr hWndInsertAfter, int x, int y, int cx, int cy, uint uFlags);

    [DllImport("user32.dll", CharSet = CharSet.Auto)]
    public static extern uint SendMessage(IntPtr hWnd, int msg, int wParam, uint lParam);

    [DllImport("user32.dll", CharSet = CharSet.Auto)]
    public static extern int SendMessage(IntPtr hWnd, int msg, int wParam, int[] lParam);

    [DllImport("user32.dll", CharSet = CharSet.Auto)]
    public static extern int SendMessage(IntPtr hWnd, int msg, IntPtr wParam, IntPtr lParam);

    [DllImport("User32.dll", CharSet = CharSet.Auto)]
    public static extern IntPtr GetWindowDC(IntPtr handle);

    [DllImport("user32.dll", ExactSpelling = true, SetLastError = true)]
    public static extern IntPtr GetDC(IntPtr hWnd);
    
    [DllImport("User32.dll", CharSet = CharSet.Auto)]
    public static extern IntPtr ReleaseDC(IntPtr handle, IntPtr hDC);

    [DllImport("Gdi32.dll", CharSet = CharSet.Auto)]
    public static extern IntPtr CreateCompatibleDC(IntPtr hdc);

    [DllImport("gdi32.dll", ExactSpelling = true, SetLastError = true)]
    [return: MarshalAs(UnmanagedType.Bool)]
    public static extern bool DeleteDC(IntPtr hdc);

    [DllImport("User32.dll", CharSet = CharSet.Auto)]
    public static extern int GetClassName(IntPtr hwnd, char[] className, int maxCount);

    [DllImport("User32.dll", CharSet = CharSet.Auto)]
    public static extern IntPtr GetWindow(IntPtr hwnd, int uCmd);

    [DllImport("User32.dll", CharSet = CharSet.Auto)]
    [return: MarshalAs(UnmanagedType.Bool)]
    public static extern bool IsWindowVisible(IntPtr hwnd);

    [DllImport("user32", CharSet = CharSet.Auto)]
    public static extern int GetClientRect(IntPtr hwnd, [In, Out] ref RECT rect);

    [DllImport("user32", CharSet = CharSet.Auto)]
    [return: MarshalAs(UnmanagedType.Bool)]
    public static extern bool MoveWindow(IntPtr hwnd, int X, int Y, int nWidth, int nHeight, bool bRepaint);

    [DllImport("user32", CharSet = CharSet.Auto)]
    [return: MarshalAs(UnmanagedType.Bool)]
    public static extern bool UpdateWindow(IntPtr hwnd);

    [DllImport("user32", CharSet = CharSet.Auto)]
    [return: MarshalAs(UnmanagedType.Bool)]
    public static extern bool ShowWindow(IntPtr hwnd, int nCmdShow);

    [DllImport("user32", CharSet = CharSet.Auto)]
    [return: MarshalAs(UnmanagedType.Bool)]
    public static extern bool InvalidateRect(IntPtr hwnd, ref RECT rect, bool bErase);

    [DllImport("user32", CharSet = CharSet.Auto)]
    [return: MarshalAs(UnmanagedType.Bool)]
    public static extern bool ValidateRect(IntPtr hwnd, ref RECT rect);

    [DllImport("user32.dll", CharSet = CharSet.Auto)]
    [return: MarshalAs(UnmanagedType.Bool)]
    public static extern bool GetWindowRect(IntPtr hWnd, [In, Out] ref RECT rect);

    [DllImport("user32.dll")]
    [return: MarshalAs(UnmanagedType.Bool)]
    public static extern bool GetCursorPos(ref POINT lpPoint);

    [DllImport("user32.dll")]
    public static extern IntPtr WindowFromPoint(POINT lpPoint);

    [DllImport("user32.dll", ExactSpelling = true, SetLastError = true)]
    [return: MarshalAs(UnmanagedType.Bool)]
    public static extern bool UpdateLayeredWindow(IntPtr hwnd, IntPtr hdcDst, ref POINT pptDst,
      ref SIZE psize, IntPtr hdcSrc, ref POINT pprSrc, Int32 crKey, ref BLENDFUNCTION pblend, Int32 dwFlags);

    [DllImport("gdi32.dll", ExactSpelling = true)]
    public static extern IntPtr SelectObject(IntPtr hDC, IntPtr hObject);

    [DllImport("gdi32.dll", ExactSpelling = true, SetLastError = true)]
    [return: MarshalAs(UnmanagedType.Bool)]
    public static extern bool DeleteObject(IntPtr hObject);

    [DllImport("user32.dll", ExactSpelling = true, SetLastError = true)]
    public static extern long SetWindowLong(IntPtr hWnd, int nIndex, long dwNewLong);

    [DllImport("user32.dll", SetLastError = true)]
    public static extern IntPtr SetParent(IntPtr hWndChild, IntPtr hWndNewParent);

    #region Helper methods

    public static ushort HiWord(uint Number)
    {
      return (ushort) ((Number >> 16) & 0xFFFF);
    }

    public static ushort LoWord(uint Number)
    {
      return (ushort) (Number & 0xFFFF);
    }

    public static uint MakeLong(ushort LoWord, ushort HiWord)
    {
      return (uint) ((HiWord << 16) | LoWord);
    }

    public static uint MakeLParam(ushort LoWord, ushort HiWord)
    {
      return (((uint) HiWord << 16) | LoWord);
    }

    public static POINT PointFromLParam(uint LParam)
    {
      return new POINT(LoWord(LParam), HiWord(LParam));
    }

    #endregion

  }

	#region SubClass Classing Handler Class

	internal class SubClass : System.Windows.Forms.NativeWindow
	{
		public delegate int SubClassWndProcEventHandler(ref System.Windows.Forms.Message m);
		public event SubClassWndProcEventHandler SubClassedWndProc;
		private bool IsSubClassed = false;

		public SubClass(IntPtr Handle, bool _SubClass)
		{
			base.AssignHandle(Handle);
			this.IsSubClassed = _SubClass;
		}

		public bool SubClassed
		{
			get{ return this.IsSubClassed; }
			set{ this.IsSubClassed = value; }
		}

		protected override void WndProc(ref Message m) 
		{
			if (this.IsSubClassed)
			{
				if (OnSubClassedWndProc(ref m) != 0)
					return;
			}
			base.WndProc(ref m);
		}

		public void CallDefaultWndProc(ref Message m)
		{
			base.WndProc(ref m);
		}

		private int OnSubClassedWndProc(ref Message m)
		{
			if (SubClassedWndProc != null)
			{
				return this.SubClassedWndProc(ref m);
			}

			return 0;
		}
	}
	#endregion
}
