using System;
using System.IO;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using MySQL.Utilities;
using MySQL.Grt;
using MySQL.Grt.Db;
using MySQL.Grt.Db.Sql;
using MySQL.Workbench;
using MySQL.Controls;

namespace MySQL.GUI.Workbench
{
  public partial class DbSqlEditor : Plugins.DockablePlugin
  {
    private Db_sql_editor dbSqlEditorBE;
    private List<SqlEditor> sqlEditors;
    private List<String> sqlEditorsAssocFilepath;
    private int sqlEditorsSequence;
    private List<String> storedConnections;
    private GridView logView;
    private int outdatedLogEventCount = 0;
    private GridView historyEntriesView;
    private GridView historyDetailsView;
    private Dictionary<long, TabPage> recordset2page = new Dictionary<long, TabPage>();
    private Dictionary<long, RecordsetView> recordset2placeholder = new Dictionary<long, RecordsetView>();
    private Dictionary<TabPage, MySQL.Grt.Db.Recordset> page2recordset = new Dictionary<TabPage, MySQL.Grt.Db.Recordset>();
    private bool runningSql = false;
    protected WbContext wbContext;
    ModelOverviewForm livePhysicalOverview;

    private SqlSnippetsForm sqlSnippetsForm;
    private SqlIdeSideBar sideBar;

    delegate void CollectGarbageDelegate();
    static void CollectGarbage()
    {
      GC.Collect();
    }

    override public void Destroy()
    {
      if (null == dbSqlEditorBE)
        return;

      sideBar.Destroy();

      List<long> recordsetKeyList = new List<long>(recordset2placeholder.Count);
      foreach (KeyValuePair<long, RecordsetView> pair in recordset2placeholder)
        recordsetKeyList.Add(pair.Key);
      foreach (long key in recordsetKeyList)
        CloseRecordset(key);

      foreach (SqlEditor sqlEditor in sqlEditors)
        sqlEditor.Destroy();
      sqlEditors.Clear();
      sqlEditorsAssocFilepath.Clear();

      dbSqlEditorBE.destroy();
      dbSqlEditorBE = null;
    }

    public DbSqlEditor(WbContext wbContext, UIForm uiForm)
    {
      this.wbContext = wbContext;
      this.uiForm = uiForm;
      Initialize();
    }

    protected void Initialize()
    {
      InitializeComponent();

      dbSqlEditorBE = uiForm as Db_sql_editor;
      grtManager = dbSqlEditorBE.grt_manager();

      sqlSnippetsForm = new SqlSnippetsForm(wbContext);
      sideBar = new SqlIdeSideBar(wbContext);

      TabText = dbSqlEditorBE.caption();

      sqlEditors = new List<SqlEditor>();
      sqlEditorsAssocFilepath = new List<String>();
      editorTabControl.TabClosing += new EventHandler<MySQL.Controls.TabClosingEventArgs>(editorTabControl_TabClosing);
      editorTabControl.TabClosed += new EventHandler<MySQL.Controls.TabClosedEventArgs>(editorTabControl_TabClosed);
      editorTabControl.SelectedIndexChanged += new EventHandler(editorTabControl_SelectedIndexChanged);
      sqlEditorsSequence = 1;
      SqlEditor sqlEditor = AddSqlEditor("SQL Statements", "", dbSqlEditorBE.active_sql_editor_index());
      ActiveControl = sqlEditor;
      dbSqlEditorBE.sql_editor_new_ui_cb(OnSqlEditorNew);
      dbSqlEditorBE.sql_editor_caption_ui_cb(OnSqlEditorCaption);

      logView = new GridView(dbSqlEditorBE.log());
      logView.AutoScroll = true;
      logView.RowHeadersVisible = false;
      logView.Parent = logPanel;
      logView.AllowAutoResizeColumns = false;
      dbSqlEditorBE.log().refresh_ui_cb(logView.ProcessModelRowsChange);
      logView.ProcessModelChange();
      logView.Columns[0].DefaultCellStyle.Alignment = DataGridViewContentAlignment.TopCenter;
      logView.Columns[0].AutoSizeMode = DataGridViewAutoSizeColumnMode.DisplayedCells;
      logView.Columns[1].AutoSizeMode = DataGridViewAutoSizeColumnMode.DisplayedCells;
      logView.Columns[2].AutoSizeMode = DataGridViewAutoSizeColumnMode.DisplayedCells;
      logView.Columns[3].AutoSizeMode = DataGridViewAutoSizeColumnMode.Fill;
      logView.Columns[4].AutoSizeMode = DataGridViewAutoSizeColumnMode.Fill;
      logView.Columns[5].AutoSizeMode = DataGridViewAutoSizeColumnMode.DisplayedCells;
      logView.SelectionMode = DataGridViewSelectionMode.FullRowSelect;
      wrapLogMessagesButton.Checked = logView.WrapMode;
      logView.SelectionChanged += new EventHandler(logView_SelectionChanged);
      logView_SelectionChanged(null, null); // reset log event details
      logView.CellFormatting += new DataGridViewCellFormattingEventHandler(logView_CellFormatting);

      historyEntriesView = new GridView(dbSqlEditorBE.history().entries_model());
      historyEntriesView.MultiSelect = false;
      historyEntriesView.RowHeadersVisible = false;
      historyEntriesView.Parent = historySplitContainer.Panel1;
      dbSqlEditorBE.history().entries_model().refresh_ui_cb(historyEntriesView.ProcessModelRowsChange);
      historyEntriesView.ProcessModelChange();
      historyEntriesView.AutoSizeColumnsMode = DataGridViewAutoSizeColumnsMode.Fill;
      historyEntriesView.SelectionMode = DataGridViewSelectionMode.FullRowSelect;
      historyEntriesView.RowEnter += new DataGridViewCellEventHandler(historyEntriesView_RowEnter);

      historyDetailsView = new GridView(dbSqlEditorBE.history().details_model());
      historyDetailsView.RowHeadersVisible = false;
      historyDetailsView.Parent = historySplitContainer.Panel2;
      dbSqlEditorBE.history().details_model().refresh_ui_cb(historyDetailsView.ProcessModelRowsChange); 
      historyDetailsView.ProcessModelChange();
      historyDetailsView.AutoSizeColumnsMode = DataGridViewAutoSizeColumnsMode.Fill;
      historyDetailsView.SelectionMode = DataGridViewSelectionMode.FullRowSelect;
      historyDetailsView.CellDoubleClick += new DataGridViewCellEventHandler(historyDetailsView_CellDoubleClick);
      wrapHistoryDetailsButton.Checked = historyDetailsView.WrapMode;

      dbSqlEditorBE.exec_sql_task.finish_cb(AfterExecSql);
      dbSqlEditorBE.exec_sql_task.progress_cb(OnExecSqlProgress);
      dbSqlEditorBE.close_recordset_ui_cb(CloseRecordset);

      dbSqlEditorBE.sql_editor_text_insert_cb(OnSqlEditorTextInsert);

      dbSqlEditorBE.refresh_ui().set_refresh_slot(OnRefresh);
      dbSqlEditorBE.refresh_ui().set_partial_refresh_slot(OnPartialRefresh);

      dbSqlEditorBE.call_exec_sql_cb(ExecuteSqlScript);
      dbSqlEditorBE.validate_exec_sql_cb(ValidateExecuteSqlScript);
      dbSqlEditorBE.call_exec_current_sql_statement_cb(ExecuteCurrentSqlStatement);
      dbSqlEditorBE.validate_exec_current_sql_statement_cb(ValidateExecuteCurrentSqlStatement);
      dbSqlEditorBE.call_save_edits_cb(ExecuteSaveEdits);
      dbSqlEditorBE.validate_save_edits_cb(ValidateSaveDiscardEdits);
      dbSqlEditorBE.call_discard_edits_cb(ExecuteDiscardEdits);
      dbSqlEditorBE.validate_discard_edits_cb(ValidateSaveDiscardEdits);

      // load connection list
      storedConnections = new List<String>();
      {
        //!List<String> labels = new List<String>();
        //dbSqlEditorBE.stored_connections(storedConnections, labels);
        //foreach (String name in labels)
        //  storedConnectionsCB.Items.Add(name);
      }

      // set active connection
      if (0 < storedConnections.Count)
      {
        //!String storedConnection = dbSqlEditorBE.stored_connection();
        //int index = 0;
        //foreach (String id in storedConnections)
        //{
        //  if (storedConnection == id)
        //  {
        //    storedConnectionsCB.SelectedIndex = index;
        //    break;
        //  }
        //  ++index;
        //}
      }

      CreateLivePhysicalOverviewPage();

      sideBar.TopLevel = false;
      sideArea.Controls.Add(sideBar);
      sideBar.Dock = DockStyle.Fill;
      sideBar.Show();

      toolStrip2.Renderer = new FlatSubToolStripRenderer();
      toolStrip3.Renderer = new FlatSubToolStripRenderer();
      toolStrip4.Renderer = new FlatSubToolStripRenderer();

      sqlSnippetsForm.Init();
      snippetsPage.Controls.Add(sqlSnippetsForm.contentPanel);
      sqlSnippetsForm.contentPanel.Dock = DockStyle.Fill;
    }

    void logView_CellFormatting(object sender, DataGridViewCellFormattingEventArgs e)
    {
      if (e.RowIndex < outdatedLogEventCount)
        e.CellStyle.ForeColor = Color.Gray;
    }

    void logView_SelectionChanged(object sender, EventArgs e)
    {
      String logEventAction = "";
      String logEventMessage = "";

      if (logView.SelectedRows.Count == 0)
      {
        logEventDetailHeaderLabel.Text = "";
      }
      else
      {
        int logEventNo = logView.SelectedRows[0].Index;
        int logEventTypeCode = 0;
        String logEventTime = "";
        dbSqlEditorBE.get_log_event_details(logEventNo, out logEventTypeCode, out logEventTime, out logEventAction, out logEventMessage);
        String logEventType;
        switch (logEventTypeCode)
        {
          case (int)Msg_type.MT_error:
            logEventType = "Error";
            break;
          case (int)Msg_type.MT_warning:
            logEventType = "Warning";
            break;
          case (int)Msg_type.MT_info:
            logEventType = "Info";
            break;
          default:
            logEventType = "Unknown";
            break;
        }
        logEventDetailHeaderLabel.Text = String.Format("Event {0} of type {1} at {2}", logEventNo+1, logEventType, logEventTime);
      }
      logEventDetailActionTextBox.Text = logEventAction;
      logEventDetailMessageTextBox.Text = logEventMessage;
    }

    void editorTabControl_SelectedIndexChanged(object sender, EventArgs e)
    {
      dbSqlEditorBE.active_sql_editor_index(editorTabControl.SelectedIndex);
    }

    void editorTabControl_TabClosing(object sender, MySQL.Controls.TabClosingEventArgs e)
    {
      editorTabControl.SelectedIndexChanged -= editorTabControl_SelectedIndexChanged;
    }

    void editorTabControl_TabClosed(object sender, MySQL.Controls.TabClosedEventArgs e)
    {
      RemoveSqlEditor(dbSqlEditorBE.active_sql_editor_index());
      editorTabControl.SelectedIndexChanged += new EventHandler(editorTabControl_SelectedIndexChanged);
      dbSqlEditorBE.active_sql_editor_index(editorTabControl.SelectedIndex);
    }

    public Db_sql_editor Backend
    {
      get { return dbSqlEditorBE; }
    }

    private SqlEditor SqlEditor()
    {
      return sqlEditors[dbSqlEditorBE.active_sql_editor_index()];
    }

    /// <summary>
    /// Updates the editor's tab tooltip to contain the current connection info.
    /// </summary>
    public void UpdateTooltip()
    {
      ToolTipText = dbSqlEditorBE.connection_info();
    }

    public void RefreshGUI(RefreshType refresh, String str, IntPtr ptr)
    {
      switch (refresh)
      {
        case RefreshType.RefreshOverviewNodeChildren:
          LivePhysicalOverview.RefreshNodeInfo(new NodeId(str));
          break;

      }
    }

    protected void OnRefresh()
    {
      UpdateTooltip();
      livePhysicalOverview.RebuildModelContents();
      sideBar.RefreshEx(this, true);
    }

    protected void OnPartialRefresh(int what)
    {
      Db_sql_editor.PartialRefreshType refresh_type = (Db_sql_editor.PartialRefreshType)what;
      switch (refresh_type)
      {
        case Db_sql_editor.PartialRefreshType.RefreshSidebar:
          UpdateTooltip();
          sideBar.Refresh(this);
          break;
        case Db_sql_editor.PartialRefreshType.RefreshSchemaTree:
          sideBar.Refresh(this);
          break;
        case Db_sql_editor.PartialRefreshType.RefreshEditor:
          SqlEditor sqlEditor = SqlEditor();
          sqlEditor.Text = dbSqlEditorBE.sql_editor().sql();
          sqlEditor.Focus();
          break;
        case Db_sql_editor.PartialRefreshType.RunCurrentScript:
          ExecuteSqlScript();
          break;
        case Db_sql_editor.PartialRefreshType.RefreshSnippets:
          sqlSnippetsForm.SqlSnippetsModel.RefreshModel();
          break;
      }
    }

    private int OnExecSqlProgress(float progress, String msg)
    {
      logView.ProcessModelRowsChange();
      logView_SelectionChanged(null, null);
      return 0;
    }

    private int AfterExecSql()
    {
      cancelButton.Enabled = false;
      TabPage firstNewPage = null;
      resultTabControl.SelectedTab = tpMessages;
      resultTabControl.Update();
      for (int n = 0, rsCount = dbSqlEditorBE.recordset_count(); n < rsCount; ++n)
      {
        MySQL.Grt.Db.Recordset rs = dbSqlEditorBE.recordset(n);
        if (recordset2page.ContainsKey(rs.key()))
          continue;
        
        TabPage page = new TabPage(rs.caption());
        page.BackColor = SystemColors.ButtonFace;
        if (null == firstNewPage)
          firstNewPage = page;

        RecordsetView recordsetView = new RecordsetView();
        components.Add(recordsetView);

        recordsetView.Embed(page, rs);
        resultTabControl.TabPages.Add(page);
        resultTabControl.SetCloseButtonVisibility(resultTabControl.TabCount - 1,
          FlatTabControl.CloseButtonVisiblity.ShowButton);
        recordset2page.Add(rs.key(), page);
        recordset2placeholder.Add(rs.key(), recordsetView);
        page2recordset.Add(page, rs);
        recordsetView.ProcessModelChange();
      }
      resultTabControl.SelectedTab = (null == firstNewPage) ? tpMessages : firstNewPage;

      logView.ProcessModelRowsChange();
      logView_SelectionChanged(null, null);
      logView.AutoResizeColumn(1);
      logView.AutoResizeColumn(2);
      logView.AutoResizeColumn(5);

      historyEntriesView.ProcessModelRowsChange();
      historyDetailsView.ProcessModelRowsChange();

      runningSql = false;

      if (logView.Rows.Count > 0)
        logView.SetRowSelected(logView.Rows.Count - 1);

      sqlEditors[dbSqlEditorBE.active_sql_editor_index()].Focus();

      return 0;
    }

    private int OnSqlEditorTextInsert(String text)
    {
      SqlEditor sqlEditor = SqlEditor();
      sqlEditor.InsertText(sqlEditor.Caret.Position, text);
      sqlEditor.Selection.Start = sqlEditor.Caret.Position + text.Length;
      sqlEditor.Selection.End = sqlEditor.Selection.Start;
      if (sqlEditor.CanFocus)
        sqlEditor.Focus();
      return 0;
    }
    
    private int CloseRecordset(long key)
    {
      if (recordset2page.ContainsKey(key))
      {
        TabPage page = recordset2page[key];
        resultTabControl.TabPages.Remove(page);

        recordset2page.Remove(key);

        RecordsetView recordsetView = recordset2placeholder[key];
        recordset2placeholder.Remove(key);
        recordsetView.Destroy();
        components.Remove(recordsetView);
        
        page2recordset.Remove(page);
        page.Dispose();

        if (IsHandleCreated)
          BeginInvoke(new CollectGarbageDelegate(CollectGarbage));
      }
      return 0;
    }

    private bool ValidateExecuteSqlScript()
    {
      return !runningSql;
    }

    private int ExecuteSqlScript()
    {
      return DoExecuteSqlScript(false);
    }

    private bool ValidateExecuteCurrentSqlStatement()
    {
      return !runningSql;
    }

    private int ExecuteCurrentSqlStatement()
    {
      return DoExecuteSqlScript(true);
    }

    private int DoExecuteSqlScript(bool currentStatementOnly)
    {
      SqlEditor sqlEditor = SqlEditor();
      String sql;
      if (currentStatementOnly)
      {
        sqlEditor.RunBackgroundAction(true);
        sql = sqlEditor.CurrentSqlStatement;
      }
      else
      {
        sql = (sqlEditor.Selection.Length > 0) ? sqlEditor.Selection.Text : sqlEditor.Text;
      }

      if (sql.Length == 0)
        return 0;

      // close unpinned resultsets
      if (!dbSqlEditorBE.recordsets_are_pinned_by_default())
      {
        Dictionary<long, long> leftRecordsets = new Dictionary<long, long>();
        for (int n = 0, rsCount = dbSqlEditorBE.recordset_count(); n < rsCount; ++n)
        {
          MySQL.Grt.Db.Recordset rs = dbSqlEditorBE.recordset(n);
          if (rs.pinned())
            leftRecordsets.Add(rs.key(), 1);
          else
            CloseRecordset(rs.key());
        }
        List<long> recordsetsToClose = new List<long>();
        foreach (KeyValuePair<long, RecordsetView> pair in recordset2placeholder)
          if (!leftRecordsets.ContainsKey(pair.Key))
            recordsetsToClose.Add(pair.Key);
        foreach (long key in recordsetsToClose)
          CloseRecordset(key);
      }

      outdatedLogEventCount = logView.RowCount;

      resultTabControl.SelectedTab = tpMessages;
      cancelButton.Enabled = true;
      runningSql = true;
      dbSqlEditorBE.exec_sql(sql, currentStatementOnly);

      return 0;
    }

    public MySQL.Grt.Db.Recordset ActiveRecordset
    {
      get
      {
        if (resultTabControl.SelectedTab == null)
          return null;
        return page2recordset.ContainsKey(resultTabControl.SelectedTab) ? page2recordset[resultTabControl.SelectedTab] : null;
      }
    }

    int ExecuteSaveEdits()
    {
      ActiveRecordset.apply_changes();
      return 0;
    }

    int ExecuteDiscardEdits()
    {
      ActiveRecordset.rollback();
      return 0;
    }

    bool ValidateSaveDiscardEdits()
    {
      MySQL.Grt.Db.Recordset rs = ActiveRecordset;
      return (rs != null) ? rs.has_pending_changes() : false;
    }

    private void sqlEditor_KeyDown(object sender, KeyEventArgs e)
    {
      switch (e.KeyCode)
      {
        case Keys.Enter:
          if (e.Control && !e.Shift && !e.Alt)
            ExecuteSqlScript();
          break;
      }
    }

    private void executeButton_Click(object sender, EventArgs e)
    {
      ExecuteSqlScript();
    }

    private void cancelButton_Click(object sender, EventArgs e)
    {
      dbSqlEditorBE.cancel_query();
    }

    private void commitButton_Click(object sender, EventArgs e)
    {
      dbSqlEditorBE.commit();
    }

    private void rollbackButton_Click(object sender, EventArgs e)
    {
      dbSqlEditorBE.rollback();
    }

    private void storedConnectionsCB_SelectedIndexChanged(object sender, EventArgs e)
    {
      //!tabControl.SelectedTab = tpMessages;
      //dbSqlEditorBE.stored_connection(storedConnections[storedConnectionsCB.SelectedIndex]);
      //TabText = Text = dbSqlEditorBE.caption();
      //ReloadSchemata();
    }

    private void reconnectButton_Click(object sender, EventArgs e)
    {
      //!tabControl.SelectedTab = tpMessages;
      //dbSqlEditorBE.disconnect();
      //dbSqlEditorBE.connect();
      //ReloadSchemata();
    }

    void historyEntriesView_RowEnter(Object sender, DataGridViewCellEventArgs e)
    {
      dbSqlEditorBE.history().current_entry(e.RowIndex);
    }

    void loadSelectedHistoryItems(bool overwrite)
    {
      if (null == historyEntriesView.CurrentRow)
        return;

      List<int> sel_indexes = new List<int>();
      foreach (DataGridViewRow row in historyDetailsView.SelectedRows)
        sel_indexes.Add(row.Index);
      sel_indexes.Sort();
      String sql = dbSqlEditorBE.restore_sql_from_history(historyEntriesView.CurrentRow.Index, sel_indexes);

      SqlEditor sqlEditor = SqlEditor();
      if (overwrite)
        sqlEditor.Text = sql;
      else
        sqlEditor.Text += sql;
    }

    void historyDetailsView_CellDoubleClick(Object sender, DataGridViewCellEventArgs e)
    {
      loadSelectedHistoryItems(false);
    }

    private void appendToCurrentSqlMenuItem_Click(object sender, EventArgs e)
    {
      loadSelectedHistoryItems(false);
    }

    private void replaceCurrentSqlMenuItemMenuItem_Click(object sender, EventArgs e)
    {
      loadSelectedHistoryItems(true);
    }

    private void pinResultsetsButton_Click(object sender, EventArgs e)
    {
      dbSqlEditorBE.recordsets_are_pinned_by_default(!dbSqlEditorBE.recordsets_are_pinned_by_default());
      pinResultsetsButton.Checked = dbSqlEditorBE.recordsets_are_pinned_by_default();
      pinResultsetsButton.Image = pinResultsetsButton.Checked ? MySQL.GUI.Workbench.Properties.Resources.PinDown_18x18 : MySQL.GUI.Workbench.Properties.Resources.Pin_18x18;
    }

    private void renamePage_ToolStripMenuItem_Click(object sender, EventArgs e)
    {
      TabPage page = resultTabControl.SelectedTab;
      if (null == page)
        return;
      if (!page2recordset.ContainsKey(page))
        return;

      MySQL.Grt.Db.Recordset rs = page2recordset[page];
      rs.caption();
      
      String newName = page.Text;
      if (DialogResult.OK == StringInputForm.ShowModal("New name of the page", "", "", ref newName))
      {
        rs.caption(newName);
        page.Text = rs.caption();
      }
    }

    private void tabControl_MouseClick(object sender, MouseEventArgs e)
    {
      switch (e.Button)
      {
        case MouseButtons.Right:
          {
            TabPage page = resultTabControl.SelectedTab;
            if (null == page)
              break;
            renamePage_ToolStripMenuItem.Enabled = page2recordset.ContainsKey(page);
            Point p = resultTabControl.PointToScreen((Point)e.Location);
            tabControlContextMenuStrip.Show(p);
          }
          break;
      }
    }

    private void showDbmsConnEditorButton_Click(object sender, EventArgs e)
    {
      //!dbSqlEditorBE.show_connection_editor();
    }

    private bool SaveIsNeeded()
    {
      bool saveIsNeeded = false;
      foreach (SqlEditor sqlEditor in sqlEditors)
      {
        if (sqlEditor.IsDirty)
        {
          saveIsNeeded = true;
          break;
        }
      }
      if (saveIsNeeded)
      {
        saveIsNeeded = (DialogResult.Yes != MessageBox.Show(
          "One or more of SQL scripts contains unsaved changes. Do you want to proceed?",
          "Confirm",
          MessageBoxButtons.YesNo,
          MessageBoxIcon.Question));
      }
      return saveIsNeeded;
    }

    private void newScriptButton_Click(object sender, EventArgs e)
    {
      AddSqlEditor("", "", sqlEditorsSequence);
      //!
      //sqlEditor.ResetText();
      //sqlEditor.UndoRedo.EmptyUndoBuffer();
      //sqlEditor.IsDirty = false;
      //sqlScriptFile = "";
    }

    private int OnSqlEditorNew(int index)
    {
      AddSqlEditor("", "", index);
      return 0;
    }

    private int OnSqlEditorCaption(String filepath)
    {
      editorTabControl.TabPages[dbSqlEditorBE.active_sql_editor_index()].Text = Path.GetFileName(filepath);
      return 0;
    }

    private SqlEditor AddSqlEditor(String caption, String filepath, int index)
    {
      if (-1 == index)
        index = dbSqlEditorBE.add_sql_editor();

      SqlEditor sqlEditor = new SqlEditor(grtManager);
      sqlEditor.BE = dbSqlEditorBE.sql_editor(index);

      if (0 == caption.Length)
      {
        caption = String.Format("SQL Script ({0})", sqlEditorsSequence);
        ++sqlEditorsSequence;
      }
      TabPage page = new TabPage(caption);
      page.Controls.Add(sqlEditor);
      editorTabControl.TabPages.Add(page);
      editorTabControl.SelectedIndex = index;

      sqlEditor.Selection.Start = sqlEditor.Text.Length;
      sqlEditor.Selection.End = sqlEditor.Text.Length;
      sqlEditor.KeyDown += new KeyEventHandler(sqlEditor_KeyDown);

      sqlEditors.Add(sqlEditor);
      sqlEditorsAssocFilepath.Add(filepath);

      return sqlEditor;
    }

    private void RemoveSqlEditor(int index)
    {
      if (-1 == index)
        index = dbSqlEditorBE.active_sql_editor_index();
      dbSqlEditorBE.remove_sql_editor(index);
      SqlEditor sqlEditor = sqlEditors[index];
      sqlEditors.RemoveAt(index);
      sqlEditorsAssocFilepath.RemoveAt(index);
      sqlEditor.Destroy();
    }

    private void openScriptButton_Click(object sender, EventArgs e)
    {
      try
      {
        FileDialog dialog = new OpenFileDialog();
        dialog.RestoreDirectory = true;
        dialog.Title = "Open SQL script ...";
        dialog.Filter = "SQL scripts (*.sql)|*.sql|All files (*.*)|*.*";
        dialog.DefaultExt = "sql";
        dialog.FileName = sqlEditorsAssocFilepath[dbSqlEditorBE.active_sql_editor_index()];
        //!dialog.InitialDirectory = new FileInfo(sqlScriptFile).Directory.FullName;

        if (DialogResult.OK == dialog.ShowDialog())
        {
          String caption = Path.GetFileName(dialog.FileName);
          SqlEditor sqlEditor = AddSqlEditor(caption, dialog.FileName, -1);
          sqlEditor.Text = File.ReadAllText(dialog.FileName, Encoding.UTF8);
          sqlEditor.UndoRedo.EmptyUndoBuffer();
          sqlEditor.IsDirty = false;
        }
      }
      catch (Exception err)
      {
        MessageBox.Show(err.Message, "Error when opening file", MessageBoxButtons.OK, MessageBoxIcon.Error);
      }
    }

    private void saveScriptButton_Click(object sender, EventArgs e)
    {
      SaveFile(false);
    }

    private void saveSqlScriptToPathButton_Click(object sender, EventArgs e)
    {
      SaveFile(true);
    }

    private void SaveFile(bool askForFileLocation)
    {
      try
      {
        String fileLocation = sqlEditorsAssocFilepath[dbSqlEditorBE.active_sql_editor_index()];
        if (("" == fileLocation) || askForFileLocation)
        {
          FileDialog dialog = new SaveFileDialog();
          dialog.RestoreDirectory = true;
          dialog.Title = "Save SQL script ...";
          dialog.Filter = "SQL scripts (*.sql)|*.sql|All files (*.*)|*.*";
          dialog.DefaultExt = "sql";
          dialog.FileName = fileLocation;
          //!dialog.InitialDirectory = new FileInfo(sqlScriptFile).Directory.FullName;

          if (DialogResult.OK == dialog.ShowDialog())
            fileLocation = dialog.FileName;
        }
        if (String.IsNullOrEmpty(fileLocation))
          return;
        SqlEditor sqlEditor = SqlEditor();
        File.WriteAllText(fileLocation, sqlEditor.Text, Encoding.UTF8);
        sqlEditor.IsDirty = false;
        sqlEditorsAssocFilepath[dbSqlEditorBE.active_sql_editor_index()] = fileLocation;
      }
      catch (Exception err)
      {
        MessageBox.Show(err.Message, "Error when saving file", MessageBoxButtons.OK, MessageBoxIcon.Error);
      }
    }

    private void wrapLogMessagesButton_Click(object sender, EventArgs e)
    {
      logView.WrapMode = !logView.WrapMode;
      wrapLogMessagesButton.Checked = logView.WrapMode;
    }

    private void wrapHistoryDetailsButton_Click(object sender, EventArgs e)
    {
      historyDetailsView.WrapMode = !historyDetailsView.WrapMode;
      wrapHistoryDetailsButton.Checked = historyDetailsView.WrapMode;
    }

    private void CreateLivePhysicalOverviewPage()
    {
      schemaOverviewPlaceholder.SuspendLayout();
      try
      {
        livePhysicalOverview = new ModelOverviewForm(wbContext, dbSqlEditorBE.live_physical_overview());
        livePhysicalOverview.skipHeader = true;
        livePhysicalOverview.contentHeaderPanel.Parent = schemaOverviewPlaceholder;
        livePhysicalOverview.contentHeaderPanel.Dock = DockStyle.Fill;
        livePhysicalOverview.contentHeaderPanel.Padding = new Padding();
        livePhysicalOverview.contentHeaderPanel.Show();
        livePhysicalOverview.contentHeaderPanel.BackColor = Color.White;
        livePhysicalOverview.RebuildModelContents();
        livePhysicalOverview.shadowPicture.Visible = false;
        Panel p = new Panel();
        p.Dock = DockStyle.Fill;
        p.BackColor = Color.White;
        p.Parent = livePhysicalOverview.contentHeaderPanel.Parent;

        WorkbenchToolbarManager.ToolbarCallback cb = delegate (String name)
        {
          return dbSqlEditorBE.live_physical_overview().activate_toolbar_item(new NodeId(0), name);
        };

        new WorkbenchToolbarManager(wbContext, toolStrip4).UpdateToolbar(toolStrip4,
          dbSqlEditorBE.live_physical_overview().get_toolbar_items(new NodeId(0)),
          cb);
      }
      finally
      {
        schemaOverviewPlaceholder.ResumeLayout();
      }
    }

    public Overview LivePhysicalOverviewBE
    {
      get { return dbSqlEditorBE.live_physical_overview(); }
    }

    public ModelOverviewForm LivePhysicalOverview
    {
      get { return livePhysicalOverview; }
    }

    private void LoadFormState()
    {
      int sidebarWidth = wbContext.read_state("sidebar_width", "query_editor", 200);
      bool left_aligned = wbContext.read_option_value("", "Sidebar:RightAligned", "0") == "0";

      if (left_aligned)
      {
        mainSplitContainer.SplitterDistance = sidebarWidth;
      }
      else
      {
        mainSplitContainer.FixedPanel = FixedPanel.Panel2;
        contentSplitContainer.Parent = mainSplitContainer.Panel1;
        sideArea.Parent = mainSplitContainer.Panel2;
        mainSplitContainer.SplitterDistance = mainSplitContainer.Width - mainSplitContainer.SplitterWidth - sidebarWidth;
      }
    }

    private void DbSqlEditor_Shown(object sender, EventArgs e)
    {
      LoadFormState();
    }

    private void mainSplitContainer_SplitterMoved(object sender, SplitterEventArgs e)
    {
      int sidebarWidth;
      bool left_aligned = wbContext.read_option_value("", "Sidebar:RightAligned", "0") == "0";
      if (left_aligned)
        sidebarWidth = mainSplitContainer.SplitterDistance;
      else
        sidebarWidth = mainSplitContainer.Width - mainSplitContainer.SplitterWidth - mainSplitContainer.SplitterDistance;
      wbContext.save_state("sidebar_width", "query_editor", sidebarWidth);
    }

  }
}
