/* 
 * Copyright (c) 2008, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

using System;
using System.Windows.Forms;
using System.Runtime.InteropServices;

using MySQL.Grt;
using MySQL.Controls;

namespace MySQL.GUI.Workbench.Plugins
{
	/// <summary>
	/// Generic GRT Object Editor
	/// </summary>
  public partial class DockablePlugin : TabDocument
  {
    #region Member Variables

    /// <summary>
		/// The GRT Manager that controlls the GRT
		/// </summary>
		protected GrtManager grtManager;

		/// <summary>
		/// The GRT Object this Editor is operating on
		/// </summary>
		protected GrtValue grtList;
    public GrtValue GrtList { get { return grtList; } }

		/// <summary>
		/// GCHandle, needed for fixed pointer
		/// </summary>
		GCHandle gcHandle;

    #endregion

    #region Constructors

    /// <summary>
		/// Standard constructor
		/// </summary>
		protected DockablePlugin()
		{
			InitializeComponent();
		}

		/// <summary>
		/// Overloaded constructor taking the GRT Manager and GRT object to edit
		/// </summary>
		/// <param name="grtManager">The GRT Manager</param>
		/// <param name="grtList">The object to edit</param>
		public DockablePlugin(GrtManager grtManager, GrtValue grtList)
			: this()
		{
			this.grtManager = grtManager;
			this.grtList = grtList;
		}

    ~DockablePlugin()
		{
			// Make sure the GCHandle is released
			ReleaseHandle();
    }

    #endregion

    #region Properties

    public GrtManager GrtManager { get { return grtManager;} }

    protected UIForm uiForm;
    public virtual UIForm UIForm
    {
      get { return uiForm; }
    }

    #endregion Properties

    #region Memory Handling

    /// <summary>
		/// Returns a fixed pointer to this object that will not be modified by the GC
		/// </summary>
		/// <returns>fixed int pointer to this object</returns>
		public virtual IntPtr GetFixedPtr()
		{
			if (!gcHandle.IsAllocated)
				 gcHandle = GCHandle.Alloc(this);

			return GCHandle.ToIntPtr(gcHandle);
		}

		/// <summary>
		/// Needs to be called when destroying the object
		/// </summary>
		public virtual void ReleaseHandle()
		{
      if (gcHandle.IsAllocated) // e.g. after exception during module loading
			  gcHandle.Free();
		}

		/// <summary>
		/// Returns the object based on the fixed pointer retrieved by GetFixedPtr()
		/// </summary>
		/// <param name="ptr">The pointer to look up</param>
		/// <returns>The corresponding instance</returns>
		static public DockablePlugin GetFromFixedPtr(IntPtr ptr)
		{
			GCHandle gcHandle = GCHandle.FromIntPtr(ptr);
      return (DockablePlugin)gcHandle.Target;
    }

    virtual public void Destroy() {}

    public void RunOnFormClosing(FormClosingEventArgs e)
    {
      OnFormClosing(e);
    }

    public void RunOnFormClosed(FormClosedEventArgs e)
    {
      OnFormClosed(e);
    }

    #endregion

  }
}