using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using Microsoft.VisualBasic.Devices;
using MySQL.Utilities;
using MySQL.Workbench;
using MySQL.GUI.Workbench;
using MySQL.Controls;

namespace MySQL.Grt.Db
{
  public partial class RecordsetView : Form
  {
    private Recordset model;
    private GridView gridView;
    //private DataExportForm dataExportForm;
    private ColumnFilterDialog columnFilterDialog = new ColumnFilterDialog();
    private int contextColumnIndex;
    private int contextRowIndex;
    private Timer dataSearchTimer;
    private Image columnHeaderFilterImage;

    public void Destroy()
    {
      gridView.EndEdit();
      gridView.Destroy();
      model.destroy();
      model = null;
      Dispose();
    }

    public RecordsetView()
    {
      InitializeComponent();

      dataSearchTimer = new Timer();
      dataSearchTimer.Interval = 2000;
      dataSearchTimer.Tick += new EventHandler(OnDataSearchApply);

      mainToolStrip.Renderer = new FlatSubToolStripRenderer();
    }

    public void Embed(Control parent, Recordset recordset)
    {
      gridView = new GridView(recordset);
      gridView.BorderStyle = BorderStyle.None;
      gridView.Dock = DockStyle.Fill;
      gridView.Parent = viewPanel;
      gridView.StandardTab = false; // Let Tab move the cursor to the next cell instead next control in parent tab order.

      recordsetPanel.Parent = parent;
      gridView.KeyDown += gridView_KeyDown;
      gridView.ColumnHeaderMouseClick += gridView_ColumnHeaderMouseClick;
      gridView.CellContextMenuStripNeeded += gridView_CellContextMenuStripNeeded;
      gridView.CellStateChanged += gridView_CellStateChanged;
      gridView.CellPainting += gridView_CellPainting;

      gridView.AdditionalColumnWidth += gridView_AdditionalColumnWidth;

      // cache icon used for marking columns with applied filters
      {
        int iconId = recordset.column_filter_icon_id();
        columnHeaderFilterImage = GrtIconManager.get_instance().get_icon(iconId);
      }

      // load toolbar items
      {
        List<ToolStripItem> existingToolStripItems = new List<ToolStripItem>(mainToolStrip.Items.Count);
        foreach (ToolStripItem item in mainToolStrip.Items)
          existingToolStripItems.Add(item);
        WorkbenchToolbarManager.get_instance().UpdateToolbar(mainToolStrip,
          recordset.get_toolbar_items(),
          TriggerAction);
        foreach (ToolStripItem item in existingToolStripItems)
          mainToolStrip.Items.Add(item);
      }

      Model = recordset;
    }

    public GridView GridView
    {
      get { return gridView; }
    }

    public Recordset Model
    {
      set
      {
        gridView.Model = value;
        model = value;
        model.refresh_ui_cb(ProcessModelChange);
        model.refresh_ui_status_bar_cb(UpdateStatusBar);
        model.task.msg_cb(ProcessTaskMsg);

        ActionList action_list = model.action_list;
        action_list.register_action("record_first", OnRecordFirst);
        action_list.register_action("record_back", OnRecordBack);
        action_list.register_action("record_next", OnRecordNext);
        action_list.register_action("record_last", OnRecordLast);
        //action_list.register_action("record_fetch_all", OnRecordFetchAll);
        action_list.register_action("record_wrap_vertical", OnRecordWrapVertical);
        action_list.register_action("record_sort_asc", OnRecordSortAsc);
        action_list.register_action("record_sort_desc", OnRecordSortDesc);
        action_list.register_action("record_edit", OnRecordEdit);
        action_list.register_action("record_add", OnRecordAdd);
        action_list.register_action("record_del", OnRecordDel);
        action_list.register_action("record_save", OnRecordSave);
        action_list.register_action("record_discard", OnRecordDiscard);
        //action_list.register_action("record_refresh", OnRecordRefresh);

        // init popup menu
        {
          action_list.register_rows_col_action("set_to_null", OnSelectedFieldsNullify);
          action_list.register_rows_col_action("delete_row", OnSelectedRowsDelete);

          MenuManager.MenuContext popupMenuContext = new MenuManager.MenuContext();
          popupMenuContext.ActionList = model.action_list;
          popupMenuContext.GetRowsColMenuItems = model.get_popup_menu_items;
          popupMenuContext.GetSelectedRowsCol = GetSelectedRowsCol;
          popupMenuContext.TriggerRowsColAction = model.activate_popup_menu_item;

          MenuManager.InitMenu(contextMenuStrip, popupMenuContext);
        }
      }
      get { return model; }
    }

    private int ProcessTaskMsg(int msgType, String message, String detail)
    {
      messagesTB.Text = message;
      int splitterDistance = (message.Length == 0) ? splitContainer1.Height : splitContainer1.Height-25-splitContainer1.SplitterWidth;
      if (splitterDistance < 0)
        splitterDistance = 0;
      splitContainer1.SplitterDistance = splitterDistance;
      
      switch ((Msg_type)msgType)
      {
        case Msg_type.MT_error:
          messagesTB.ForeColor = Color.Red;
          break;
        case Msg_type.MT_warning:
        case Msg_type.MT_info:
          messagesTB.ForeColor = SystemColors.WindowText;
          break;
      }

      return 0;
    }

    void UpdateToolBar()
    {
      //!startEditButton.Visible =
      //  insertRowButton.Visible =
      //  deleteRowButton.Visible =
      //  commitButton.Visible =
      //  rollbackButton.Visible =
      //  tsTransactionGroupSeparator.Visible = !gridView.ReadOnly;

      dataSearchExprTB.Text = model.data_search_string();
      dataSearchTimer.Stop();
    }

    private int UpdateStatusBar()
    {
      String limitText;
      if (model.limit_rows_applicable() && model.limit_rows())
      {
        //!fetchAllRowsButton.Enabled = true;
        limitText = ", more available";
      }
      else
      {
        //!fetchAllRowsButton.Enabled = false;
        limitText = "";
      }

      String statusText = String.Format("Fetched {0} records{1}", model.real_row_count(), limitText);
      if (!gridView.ReadOnly)
      {
        int upd_count = 0, ins_count = 0, del_count = 0;
        model.pending_changes(ref upd_count, ref ins_count, ref del_count);
        if (upd_count > 0)
          statusText += String.Format(", updated {0}", upd_count);
        if (ins_count > 0)
          statusText += String.Format(", inserted {0}", ins_count);
        if (del_count > 0)
          statusText += String.Format(", deleted {0}", del_count);
      }

      dataStatusLabel.Text = statusText;

      return 0;
    }

    public int ProcessModelChange()
    {
      gridView.ProcessModelChange();

      UpdateStatusBar();
      UpdateToolBar();

      setNullMenuItem.Enabled = !gridView.ReadOnly;
      deleteRowsMenuItem.Enabled = !gridView.ReadOnly;

      return 0;
    }

    private bool TriggerAction(String name)
    {
      return model.action_list.trigger_action(name);
    }

    public bool PinButtonVisible
    {
      set
      {
        pinButton.Visible = value;
      }
    }

    private void pinButton_Click(object sender, EventArgs e)
    {
      model.pinned(!model.pinned());
      pinButton.Checked = model.pinned();
      pinButton.Image = pinButton.Checked ? Properties.Resources.tiny_pinned : Properties.Resources.tiny_pin;
    }

    private void OnRecordRefresh()
    {
      model.refresh();
    }

    private void OnRecordDiscard()
    {
      model.rollback();
    }

    private void OnRecordSave()
    {
      // Apply any pending changes in the grid before writing them to model.
      gridView.EndEdit();
      model.apply_changes();
    }

    private void setNullMenuItem_Click(object sender, EventArgs e)
    {
      int column = -1;
      List<int> rows = GetSelectedRowsCol(ref column);
      OnSelectedFieldsNullify(rows, column);
    }

    private void OnSelectedFieldsNullify(List<int> rows, int column)
    {
      if (gridView.ReadOnly)
        return;
      foreach (DataGridViewCell cell in gridView.SelectedCells)
      {
        model.set_field_null(new NodeId(cell.RowIndex), cell.ColumnIndex);
        gridView.InvalidateCell(cell);
      }
    }

    private void gridView_KeyDown(object sender, KeyEventArgs e)
    {
      switch (e.KeyCode)
      {
        case Keys.Delete:
        case Keys.Back:
          {
            if (1 == gridView.SelectedCells.Count)
              setNullMenuItem_Click(gridView, new EventArgs());
          }
          break;
        default:
          break;
      }
    }

    /*
    private void exportButton_Click(object sender, EventArgs e)
    {
      if (null == dataExportForm)
        dataExportForm = new DataExportForm(model);
      dataExportForm.ShowDialog();
    }*/

    private void OnRecordWrapVertical()
    {
      gridView.WrapMode = !gridView.WrapMode;
      //!toggleWrapModeButton.Checked = gridView.WrapMode;
    }

    private void closeButton_Click(object sender, EventArgs e)
    {
      model.close(); // after this call the object is disposed
    }

    private void OnRecordFirst()
    {
      gridView.SetRowSelected(0);
    }

    private void OnRecordBack()
    {
      if (contextRowIndex == -1)
        return;
      int rowIndex = contextRowIndex - 1;
      if (rowIndex >= 0)
        gridView.SetRowSelected(rowIndex);
    }

    private void OnRecordNext()
    {
      if (contextRowIndex == -1)
        return;
      int rowIndex = contextRowIndex + 1;
      if (rowIndex < gridView.RowCount)
        gridView.SetRowSelected(rowIndex);
    }

    private void OnRecordLast()
    {
      gridView.SetRowSelected(gridView.RowCount - 1);
    }

    private void OnRecordEdit()
    {
      if (gridView.SelectedCells.Count > 0)
        gridView.BeginEdit(false);
    }

    private void OnRecordAdd()
    {
      OnRecordLast();
      if (gridView.Columns.Count > 0)
        gridView.BeginEdit(false);
    }

    private void OnRecordDel()
    {
      deleteRowsMenuItem_Click(null, null);
    }

    private void OnSelectedRowsDelete(List<int> rows, int column)
    {
      deleteRowsMenuItem_Click(null, null);
    }

    private List<DataGridViewRow> SelectedRows()
    {
      SortedDictionary<int, int> rowIndeces = new SortedDictionary<int, int>();
      int indecesEnd = gridView.RowCount;
      if (!gridView.ReadOnly)
        --indecesEnd;
      foreach (DataGridViewCell cell in gridView.SelectedCells)
      {
        int rowIndex = cell.RowIndex;
        if (rowIndex < indecesEnd)
          rowIndeces[rowIndex] = 0;
      }
      List<DataGridViewRow> res = new List<DataGridViewRow>(rowIndeces.Count);
      foreach (KeyValuePair<int, int> pair in rowIndeces)
        res.Add(gridView.Rows[pair.Key]);
      return res;
    }

    private List<int> GetSelectedRowsCol(ref int column)
    {
      List<DataGridViewRow> rows = SelectedRows();
      List<int> indexes = new List<int>(rows.Count);
      foreach (DataGridViewRow row in rows)
        indexes.Add(row.Index);
      column = (0 < gridView.SelectedColumns.Count) ? gridView.SelectedColumns[gridView.SelectedColumns.Count-1].Index : -1;
      if (-1 == column)
        column = (0 < gridView.SelectedCells.Count) ? gridView.SelectedCells[gridView.SelectedCells.Count - 1].ColumnIndex : -1;
      return indexes;
    }

    private void deleteRowsMenuItem_Click(object sender, EventArgs e)
    {
      List<DataGridViewRow> selectedRows = SelectedRows();
      foreach (DataGridViewRow row in selectedRows)
        gridView.Rows.Remove(row);
    }

    private void copyRowValuesMenuItem_Click(object sender, EventArgs e)
    {
      List<DataGridViewRow> selectedRows = SelectedRows();
      List<int> selectedRowIndeces = new List<int>(selectedRows.Count);
      foreach (DataGridViewRow row in selectedRows)
        selectedRowIndeces.Add(row.Index);
      model.copy_rows_to_clipboard(selectedRowIndeces);
    }

    private void CopyFieldContentMenuItem_Click(object sender, EventArgs e)
    {
      if (gridView.SelectedCells.Count > 0)
      {
        DataGridViewCell cell = gridView.SelectedCells[0];
        model.copy_field_to_clipboard(cell.RowIndex, cell.ColumnIndex);
      }
    }

    private void gridView_CellContextMenuStripNeeded(object sender, DataGridViewCellContextMenuStripNeededEventArgs e)
    {
      contextColumnIndex = e.ColumnIndex;
      contextRowIndex = e.RowIndex;
      if (e.RowIndex == -1)
      {
        e.ContextMenuStrip = (e.ColumnIndex == -1) ? null : columnHeaderContextMenuStrip;
        resetAllColumnFiltersToolStripMenuItem.Enabled = model.has_column_filters();
        resetColumnFilterToolStripMenuItem.Enabled = model.has_column_filter(contextColumnIndex);
      }
      else
      {
        e.ContextMenuStrip = contextMenuStrip;
      }
    }

    private void gridView_ColumnHeaderMouseClick(object sender, DataGridViewCellMouseEventArgs e)
    {
      if (MouseButtons.Left == e.Button)
      {
        int direction = (gridView.Columns[e.ColumnIndex].HeaderCell.SortGlyphDirection == SortOrder.Ascending ? -1 : 1);
        Keyboard keyboard = new Keyboard();
        direction = (keyboard.AltKeyDown ? 0 : direction);
        bool retaining = (0 == direction) || keyboard.ShiftKeyDown;
        SortByColumn(e.ColumnIndex, direction, retaining);
      }
    }

    private void setColumnFilterToolStripMenuItem_Click(object sender, EventArgs e)
    {
      columnFilterDialog.ColumnName = model.get_column_caption(contextColumnIndex);
      columnFilterDialog.FilterExpression = model.get_column_filter_expr(contextColumnIndex);
      columnFilterDialog.Location = new Point(columnHeaderContextMenuStrip.Left, columnHeaderContextMenuStrip.Top);
      columnFilterDialog.ShowDialog();
      if (columnFilterDialog.DialogResult == DialogResult.OK)
      {
        model.set_column_filter(contextColumnIndex, columnFilterDialog.FilterExpression);
      }
    }

    private void resetColumnFilterToolStripMenuItem_Click(object sender, EventArgs e)
    {
      model.reset_column_filter(contextColumnIndex);
    }

    private void resetAllColumnFiltersToolStripMenuItem_Click(object sender, EventArgs e)
    {
      model.reset_column_filters();
    }

    private void dataSearchExprTB_TextChanged(object sender, EventArgs e)
    {
      dataSearchTimer.Stop();
      dataSearchTimer.Start();
    }

    private void dataSearchExprTB_KeyDown(object sender, KeyEventArgs e)
    {
      switch (e.KeyCode)
      {
        case Keys.Return:
          OnDataSearchApply(null, null);
          break;
      }
    }

    private void SortByColumn(int columnIndex, int direction, bool retaining)
    {
      SortOrder sortOrder;
      switch (direction)
      {
        case -1: sortOrder = SortOrder.Ascending; break;
        case 1: sortOrder = SortOrder.Descending; break;
        case 0: default: sortOrder = SortOrder.None; break;
      }
      gridView.Columns[columnIndex].HeaderCell.SortGlyphDirection = sortOrder;
      model.sort_by(columnIndex, direction, retaining);
    }

    private void OnRecordSortAsc()
    {
      if (contextColumnIndex == -1)
        return;
      int direction = (gridView.Columns[contextColumnIndex].HeaderCell.SortGlyphDirection == SortOrder.Ascending ? 0 : 1);
      SortByColumn(contextColumnIndex, direction, true);
    }

    private void OnRecordSortDesc()
    {
      if (contextColumnIndex == -1)
        return;
      int direction = (gridView.Columns[contextColumnIndex].HeaderCell.SortGlyphDirection == SortOrder.Descending ? 0 : -1);
      SortByColumn(contextColumnIndex, direction, true);
    }

    private void OnDataSearchApply(object sender, EventArgs e)
    {
      dataSearchTimer.Stop();
      if (dataSearchExprTB.Text.Length == 0)
        model.reset_data_search_string();
      else
        model.set_data_search_string(dataSearchExprTB.Text);
    }

    void gridView_CellStateChanged(object sender, DataGridViewCellStateChangedEventArgs e)
    {
      if (e.StateChanged == DataGridViewElementStates.Selected)
      {
        if (e.Cell.Selected)
        {
          contextColumnIndex = e.Cell.ColumnIndex;
          contextRowIndex = e.Cell.RowIndex;
        }
        else if (gridView.SelectedCells.Count > 0)
        {
          DataGridViewCell cell = gridView.SelectedCells[gridView.SelectedCells.Count-1];
          contextColumnIndex = cell.ColumnIndex;
          contextRowIndex = cell.RowIndex;
        }
        else
        {
          contextColumnIndex = -1;
          contextRowIndex = -1;
        }
      }
    }

    private int gridView_AdditionalColumnWidth(DataGridViewColumn column)
    {
      int additionalColumnWidth = 0;
      if (column.Index >= 0)
        if (model.has_column_filter(column.Index))
          additionalColumnWidth += columnHeaderFilterImage.Size.Width + 2;
      return additionalColumnWidth;
    }

    private void gridView_CellPainting(object sender, DataGridViewCellPaintingEventArgs e)
    {
      if ((0 <= e.RowIndex) || (-1 == e.ColumnIndex))
        return;

      if (!model.has_column_filter(e.ColumnIndex))
        return;

      e.PaintBackground(e.ClipBounds, true);

      Rectangle glyphRect = new Rectangle(new Point(0, 0), columnHeaderFilterImage.Size);
      glyphRect.X = e.CellBounds.Right - glyphRect.Width - 2;
      if (gridView.Columns[e.ColumnIndex].HeaderCell.SortGlyphDirection != SortOrder.None)
        glyphRect.X -= glyphRect.Width;
      if (glyphRect.X < 0)
        glyphRect.X = 0;
      glyphRect.Y = e.ClipBounds.Top + (e.ClipBounds.Height - glyphRect.Height) / 2;
      if (glyphRect.Y < 0)
        glyphRect.Y = 0;

      Rectangle contentRect = e.ClipBounds;
      if (contentRect.Width > (e.CellBounds.Width - glyphRect.Width))
        contentRect.Width = e.CellBounds.Width - glyphRect.Width;
      e.PaintContent(contentRect);
      e.Graphics.DrawImageUnscaledAndClipped(columnHeaderFilterImage, glyphRect);

      e.Handled = true;
    }

    private void searchButton_Click(object sender, EventArgs e)
    {
      OnDataSearchApply(null, null);
    }
  }
}
