﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Windows;
using System.Windows.Forms;
using System.Drawing;
using System.Text.RegularExpressions;
using System.IO;

using ScintillaNet;

namespace MySQL.Grt.Db.Sql
{
  public class SqlEditor : Scintilla
  {
    private Sql_editor sqlEditorBE;
    private GrtManager grtManager;
    private Timer timer;
    private List<ManagedRange> messagesRanges = new List<ManagedRange>();
    private List<String> messages = new List<String>();

    public delegate void BackgroundActionDelegate(bool sync);
    public BackgroundActionDelegate BackgroundAction;

    public SqlEditor(GrtManager grtManager)
    {
      GrtManager = grtManager;
      Initialize();
    }

    public void Destroy()
    {
      BackgroundAction = null;
      sqlEditorBE.destroy();
      sqlEditorBE = null;
    }

    public GrtManager GrtManager
    {
      get { return grtManager; }
      set { grtManager = value; }
    }

    public Sql_editor BE
    {
      get { return sqlEditorBE; }
      set
      {
        sqlEditorBE = value;
        sqlEditorBE.set_report_sql_statement_border_delegate(ProcessSqlStatementBorder);
        sqlEditorBE.sql_parser_err_cb(ProcessSqlError);
        sqlEditorBE.set_replace_selected_text_delegate(DoReplaceSelectedText);
        sqlEditorBE.set_insert_text_delegate(DoInsertText);
        sqlEditorBE.set_do_seach_delegate(DoTextSeach);
        sqlEditorBE.set_current_statement_delegate(GetCurrentSqlStatement);

        SelectionChanged += new EventHandler(TextSelectionChanged);

        SuspendLayout();

        // customized parameters
        ConfigurationManager.Language = sqlEditorBE.string_option("ConfigurationManager.Language");
        Indentation.IndentWidth = sqlEditorBE.int_option("Indentation.IndentWidth");
        Indentation.TabWidth = sqlEditorBE.int_option("Indentation.TabWidth");
        Indentation.UseTabs = sqlEditorBE.int_option("Indentation.UseTabs") != 0;
        //! todo: virtualize the rest params

        ResumeLayout();
      }
    }

    void TextSelectionChanged(object sender, EventArgs e)
    {
      sqlEditorBE.set_selected_range(Selection.Start, Selection.End);
      sqlEditorBE.set_cursor_pos(CurrentPos);
    }

    protected virtual void Initialize()
    {
      BeginInit();

      Caret.IsSticky = true;
      Dock = DockStyle.Fill;
      Printing.PageSettings.Color = false;
      UseFont = true;
      //!DocumentChange += new System.EventHandler<NativeScintillaEventArgs>(sqlEditorControl_Change);
      Font = grtManager.get_font_option("workbench.general.Editor:Font");
      Margins.Margin0.Width = 35; // line numbers
      Margins.Margin1.Width = 16; // markers
      Margins.Margin2.Width = 16; // indicators
      Indentation.BackspaceUnindents = true;
      Indentation.SmartIndentType = SmartIndent.Simple;

      // errors indication
      Indicator errInd = Indicators[0];
      errInd.Color = Color.Red;
      errInd.Style = IndicatorStyle.Squiggle;
      errInd.IsDrawnUnder = true;

      errInd = Indicators[1];
      errInd.Color = Color.Red;
      errInd.Style = IndicatorStyle.RoundBox;
      errInd.IsDrawnUnder = true;

      Marker marker = Markers[1];
      String path = "images/ui/editor_error.xpm";
      if (File.Exists(path))
      {
        StreamReader reader = File.OpenText(path);
        String icon = reader.ReadToEnd();
        marker.SetImage(icon);
      }
      else
        marker.BackColor = Color.Red;

      marker = Markers[0];
      path = "images/ui/editor_statement.xpm";
      if (File.Exists(path))
      {
        StreamReader reader = File.OpenText(path);
        String icon = reader.ReadToEnd();
        marker.SetImage(icon);
      }
      else
        marker.BackColor = Color.Blue;

      // fixes improper ViewportSize from start.
      // once row is stretched enough to make thumb appear this thumb becomes irreversible.
      Scrolling.HorizontalWidth = 1;

      NativeInterface.SetCodePage((int)Constants.SC_CP_UTF8);
      EndOfLine.Mode = EndOfLineMode.Crlf;

      timer = new Timer();
      timer.Interval = 2000;
      timer.Tick += new EventHandler(BackgroundActionTimer);

      BackgroundAction = CheckSql;

      TextDeleted += new System.EventHandler<TextModifiedEventArgs>(TextModified);
      TextInserted += new System.EventHandler<TextModifiedEventArgs>(TextModified);

      DwellStart += new EventHandler<ScintillaMouseEventArgs>(OnDwellStart);
      DwellEnd += new EventHandler(OnDwellEnd);
      NativeInterface.SetMouseDwellTime(200);

      EndInit();
    }

    public String SqlText
    {
      get { return Text; }
      set
      {
        if (!IsRefreshEnabled)
          return;

        IsRefreshEnabled = false;

        BackgroundActionDelegate backgroundAction = BackgroundAction;
        BackgroundAction = null;

        int caretPosition = Caret.Position;

        Text = value;
        IsDirty = false;

        if (caretPosition > Text.Length)
          caretPosition = Text.Length;
        Selection.Start = Selection.End = caretPosition;

        BackgroundAction = backgroundAction;

        CheckSql(false);
      }
    }

    private void TextModified(object sender, TextModifiedEventArgs e)
    {
      if (null != BackgroundAction)
      {
        timer.Stop();
        timer.Start();
      }
      sqlEditorBE.sql(Text);
    }

    public void RunBackgroundAction(bool sync)
    {
      if (!IsDirty)
        return;

      IsDirty = false;
      if (null != BackgroundAction)
        BackgroundAction(sync);
    }

    private void BackgroundActionTimer(Object obj, EventArgs args)
    {
      RunBackgroundAction(false);
    }

    public void CheckSql(bool sync)
    {
      if (!IsSqlCheckEnabled)
        return;
      if (CheckSql != BackgroundAction)
        IsSqlCheckEnabled = false;
      ResetSqlCheckState();
      if (null == sqlEditorBE)
        return;
      sqlEditorBE.sql(Text);
      sqlEditorBE.check_sql(sync);
    }

    public int ProcessSqlStatementBorder(int beginLineNo, int beginLinePos, int endLineNo, int endLinePos)
    {
      beginLineNo -= 1;
      endLineNo -= 1;
      try
      {
        SuspendLayout();
        Lines[beginLineNo].AddMarker(0);
        ResumeLayout();
      }
      catch (Exception) { }
      return 0;
    }

    public int ProcessSqlError(int errTokLine, int errTokLinePos, int errTokLen, String msg)
    {
      errTokLine -= 1;
      try
      {
        errTokLinePos += Lines[errTokLine].Range.Start;
        ManagedRange range = new ManagedRange(errTokLinePos, errTokLinePos + errTokLen, this);
        
        SuspendLayout();
        range.SetIndicator(0);
        range.SetIndicator(1);
        Lines[errTokLine].AddMarker(1);
        ResumeLayout();

        messages.Add(msg);
        messagesRanges.Add(range);
        ManagedRanges.Add(range);
      }
      catch (Exception) { }
      return 0;
    }

    public void ResetSqlCheckState()
    {
      // errors
      {
        timer.Stop();
        foreach (ManagedRange managedRange in messagesRanges)
          if (!managedRange.IsDisposed)
            managedRange.Dispose();
        messagesRanges.Clear();
        messages.Clear();
      }

      SuspendLayout();
      Range range = new Range(0, RawText.Length, this);
      range.ClearIndicator(0);
      range.ClearIndicator(1);
      Markers.DeleteAll(Markers[1]);
      Markers.DeleteAll(Markers[0]);
      ResumeLayout();
    }

    public bool IsRefreshEnabled
    {
      get { return (null == BE) ? true : BE.is_refresh_enabled(); }
      set { BE.is_refresh_enabled(value); }
    }

    public bool IsSqlCheckEnabled
    {
      get { return (null == BE) ? true : BE.is_sql_check_enabled(); }
      set { BE.is_sql_check_enabled(value); }
    }

    private String GetCurrentSqlStatement()
    {
      return CurrentSqlStatement;
    }

    public String CurrentSqlStatement
    {
      get
      {
        int stmtBeginLineNo = -1;
        int stmtBeginLinePos = -1;
        int stmtEndLineNo = -1;
        int stmtEndLinePos = -1;
        sqlEditorBE.get_sql_statement_border_by_line_pos(
          (Selection.Range.StartingLine.Number)+1, (Selection.Range.Start - Selection.Range.StartingLine.StartPosition),
          ref stmtBeginLineNo, ref stmtBeginLinePos, ref stmtEndLineNo, ref stmtEndLinePos);
        if (stmtBeginLineNo == -1)
          return "";
        stmtBeginLineNo -= 1;
        stmtEndLineNo -= 1;
        return GetRange(Lines[stmtBeginLineNo].StartPosition + stmtBeginLinePos, Lines[stmtEndLineNo].StartPosition + stmtEndLinePos).Text;
      }
    }

    void OnDwellStart(object sender, ScintillaMouseEventArgs e)
    {
      int pos = PositionFromPoint(e.X, e.Y);
      if (pos == -1)
        return;
      int messageIndex = -1;
      if (NativeInterface.IndicatorValueAt(0, pos) == 1)
      {
        for (int n = 0; n < messagesRanges.Count; ++n)
        {
          if (messagesRanges[n].PositionInRange(pos))
          {
            messageIndex = n;
            break;
          }
        }
      }
      if ((messageIndex >= 0) && (messageIndex < messages.Count))
        CallTip.Show(messages[messageIndex], pos);
    }

    void OnDwellEnd(object sender, EventArgs e)
    {
      CallTip.Cancel();
    }

    int DoReplaceSelectedText(String newText)
    {
      Selection.Text = newText;
      Focus();
      return 0;
    }

    int DoInsertText(String newText)
    {
      InsertText(newText);
      Focus();
      return 0;
    }

    bool DoTextSeach(String searchText, String replaceText, int flags)
    {
      if (searchText.Length == 0)
        return false;

      Sql_editor.SearchFlags searchFlags = (Sql_editor.SearchFlags) flags;
      SearchFlags scintillaFlags = SearchFlags.Empty;

      bool previous = (searchFlags & Sql_editor.SearchFlags.SearchPrevious) != 0;
      bool useRegEx = (searchFlags & Sql_editor.SearchFlags.SearchUseRegularExpression) != 0;
      bool replace = (searchFlags & Sql_editor.SearchFlags.SearchDoReplace) != 0;
      bool all = (searchFlags & Sql_editor.SearchFlags.SearchAll) != 0;

      Range searchRange = new Range(0, RawText.Length, this);
      if ((searchFlags & Sql_editor.SearchFlags.SearchMatchCase) != 0)
        scintillaFlags |= SearchFlags.MatchCase;
      if ((searchFlags & Sql_editor.SearchFlags.SearchMatchWholeWord) != 0)
        scintillaFlags |= SearchFlags.WholeWord;

      Range result = null;
      bool canClose = false;
      if (useRegEx)
      {
        Regex expression = new Regex(searchText);
        if (replace)
        {
          if (all)
          {
            FindReplace.ReplaceAll(expression, replaceText);
            canClose = true;
          }
          else
          {
            // There is no overload to replace with RE search, so we have to handle that ourselves.
            if (previous)
              result = FindReplace.FindPrevious(expression, true);
            else
              result = FindReplace.FindNext(expression, true);

            if (result != null)
            {
              result.Text = replaceText;
              result.End = result.Start + replaceText.Length;
            }
          }
        }
        else
          if (previous)
            result = FindReplace.FindPrevious(expression, true);
          else
            result = FindReplace.FindNext(expression, true);
      }
      else
      {
        if (replace)
        {
          if (all)
          {
            FindReplace.ReplaceAll(searchText, replaceText, scintillaFlags);
            canClose = true;
          }
          else
          {
            if (previous)
              result = FindReplace.ReplacePrevious(searchText, replaceText, true, scintillaFlags);
            else
              result = FindReplace.ReplaceNext(searchText, replaceText, true, scintillaFlags);
          }
        }
        else
        {
          if (previous)
            result = FindReplace.FindPrevious(searchText, true, scintillaFlags);
          else
            result = FindReplace.FindNext(searchText, true, scintillaFlags);
        }
      }
      if (result != null)
      {
        canClose = true;
        result.Select();
      }

      return canClose;
    }
  }
}
