using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using System.ComponentModel;

using MySQL.Grt;
using MySQL.Workbench;

namespace MySQL.GUI.Workbench
{
  public class WorkbenchToolbarManager
  {
    #region Member Variables
    
    // The Workbench context.
    private WbContext wbContext;

    // Search support.
    public String lastSearchString = "";
    public ToolStripTextBox searchBox = null;

    private ToolStrip mainToolStrip;

    private static WorkbenchToolbarManager instance = null;

    #endregion

    static public WorkbenchToolbarManager get_instance()
    {
      return instance;
    }

    #region Constructors

    public WorkbenchToolbarManager(WbContext WbContext, ToolStrip MainToolStrip)
	{
      instance = this;

	  wbContext = WbContext;
      mainToolStrip = MainToolStrip;
    }

    #endregion

    public delegate bool ToolbarCallback(String command);

    public void UpdateMainToolbar()
    {
      mainToolStrip.Items.Clear();
      mainToolStrip.Items.AddRange(buildToolbar(wbContext.get_items_for_toolbar("main")));
      mainToolStrip.Items.AddRange(buildToolbar(wbContext.get_items_for_toolbar("options")));
    }

    public void UpdateToolsToolbar(ToolStrip WorkToolStrip)
    {
      WorkToolStrip.Items.Clear();
      WorkToolStrip.Items.AddRange(buildToolbar(wbContext.get_items_for_toolbar("tools"), true));
    }

    public void UpdateToolbar(ToolStrip toolstrip, List<ToolbarItem> toolbarItems,
          ToolbarCallback callback)
    {
      toolstrip.Tag = callback;
      toolstrip.Items.Clear();
      toolstrip.Items.AddRange(buildToolbar(toolbarItems));
    }

    /// <summary>
    /// Search for a specific ToolStripItem by the command name
    /// </summary>
    /// <param name="cmd">command name string</param>
    /// <returns>the ToolStripItem if found</returns>
    public ToolStripItem GetToolStripItem(String cmd)
    {
      foreach (ToolStripItem item in mainToolStrip.Items)
        if (item.Tag != null && item.Tag.ToString().Equals(cmd))
          return item;

      return null;
    }

    ToolStripItem[] buildToolbar(List<ToolbarItem> toolbarItems)
    {
      return buildToolbar(toolbarItems, false);
    }

    ToolStripItem[] buildToolbar(List<ToolbarItem> toolbarItems, bool addSpacer)
    {
      ToolStripItem[] itemlist;
      int i = 0;
      GrtIconManager iconManager = GrtIconManager.get_instance();

      if (!addSpacer)
        itemlist = new ToolStripItem[toolbarItems.Count];
      else
        itemlist = new ToolStripItem[toolbarItems.Count + 1];

      // rebuild the menu
      foreach (ToolbarItem subitem in toolbarItems)
      {
        switch (subitem.get_type())
        {
          case ToolbarItemType.ToolbarAction:
            {
              ToolStripButton smitem;

              smitem = new ToolStripButton();
              smitem.Name = subitem.get_name();
              smitem.Tag = subitem.get_command();
              smitem.Image = iconManager.get_icon(subitem.get_icon());
              smitem.ToolTipText = subitem.get_tooltip();
              smitem.ImageScaling = ToolStripItemImageScaling.None;

              smitem.Enabled = subitem.get_enabled();
              smitem.Click += new System.EventHandler(toolbarItem_Click);

              if (smitem.Name.EndsWith("__right"))
                smitem.Alignment = ToolStripItemAlignment.Right;

              itemlist[i++] = smitem;
              break;
            }

          case ToolbarItemType.ToolbarLabel:
            {
              ToolStripLabel smitem;

              smitem = new ToolStripLabel();
              smitem.Name = subitem.get_name();
              smitem.Tag = subitem.get_command();
              smitem.Text = subitem.get_caption();
              smitem.Font = new System.Drawing.Font(smitem.Font.FontFamily, 7);
              smitem.Enabled = subitem.get_enabled();

              if (smitem.Name.EndsWith("__right"))
                smitem.Alignment = ToolStripItemAlignment.Right;

              itemlist[i++] = smitem;
              break;
            }

          case ToolbarItemType.ToolbarSeparator:
            {
              itemlist[i++] = new ToolStripSeparator();
              if (subitem.get_name().EndsWith("__right"))
                itemlist[i - 1].Alignment = ToolStripItemAlignment.Right;
              break;
            }
          case ToolbarItemType.ToolbarRadio:
          case ToolbarItemType.ToolbarToggle:
            {
              ToolStripButton smitem;

              smitem = new ToolStripButton();
              smitem.Name = subitem.get_name();
              smitem.Tag = subitem.get_command();
              if (subitem.get_checked() && subitem.get_alt_icon() != 0)
                  smitem.Image = iconManager.get_icon(subitem.get_alt_icon());
              else
                  smitem.Image = iconManager.get_icon(subitem.get_icon());
              smitem.ImageScaling = ToolStripItemImageScaling.None;
              smitem.DisplayStyle = ToolStripItemDisplayStyle.Image;
              smitem.ToolTipText = subitem.get_tooltip();
              smitem.Enabled = subitem.get_enabled();
              smitem.Checked = subitem.get_checked();
              smitem.Click += new System.EventHandler(toolbarItem_Click);

              if (smitem.Name.EndsWith("__right"))
                smitem.Alignment = ToolStripItemAlignment.Right;

              itemlist[i++] = smitem;
              break;
            }

          case ToolbarItemType.ToolbarCheck:
            {
              CheckBox cb = new CheckBox();
              cb.Name = subitem.get_name();
              cb.Text = subitem.get_caption();
              cb.Tag = subitem.get_command();
              cb.Enabled = subitem.get_enabled();
              cb.Checked = subitem.get_checked();
              cb.Font = new System.Drawing.Font(cb.Font.FontFamily, 7);
              cb.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
              cb.Click += new System.EventHandler(checkbox_Click);
              cb.AutoSize = true;
              ToolStripControlHost chost = new ToolStripControlHost(cb);
              chost.ToolTipText = subitem.get_tooltip();
              itemlist[i++] = chost;

              if (chost.Name.EndsWith("__right"))
                chost.Alignment = ToolStripItemAlignment.Right;
              break;
            }

          case ToolbarItemType.ToolbarDropDown:
            {
              String selected_item;
              List<String> items = wbContext.get_dropdown_items(subitem.get_name(), subitem.get_command(), out selected_item);

              if (items.Count > 0 && items[0][0] == '#')
              {
                ToolStripDropDownButton smitem;
                ToolStripDropDown dropdown= new ToolStripDropDown();
                System.Drawing.Bitmap selected= null;
                System.Drawing.SolidBrush brush= new System.Drawing.SolidBrush(System.Drawing.Color.White);
								System.Drawing.Pen pen = new System.Drawing.Pen(System.Drawing.Color.LightGray);
                List<ToolStripButton> buttons = new List<ToolStripButton>();

                smitem = new ToolStripDropDownButton();
                smitem.Name = subitem.get_name();
                smitem.Tag = subitem.get_command();
                smitem.Enabled = subitem.get_enabled();
                smitem.ToolTipText = subitem.get_tooltip();
                smitem.DropDown= dropdown;
                smitem.ShowDropDownArrow= true;
                smitem.ImageScaling= ToolStripItemImageScaling.None;
                smitem.DropDownDirection= ToolStripDropDownDirection.BelowRight;
                smitem.AutoSize = true;

                foreach (String item in items)
                {
                  try
                  {
                    ToolStripButton button = new ToolStripButton();
                    System.Drawing.Color color = System.Drawing.ColorTranslator.FromHtml(item);
                    System.Drawing.Bitmap bmp = new System.Drawing.Bitmap(20, 15);
                    System.Drawing.Graphics gfx = System.Drawing.Graphics.FromImage(bmp);
                    brush.Color = color;
                    gfx.FillRectangle(brush, new System.Drawing.Rectangle(1, 0, 18, 14));
                    gfx.DrawRectangle(pen, new System.Drawing.Rectangle(1, 0, 18, 14));

                    button.Name = item;
                    button.Image = bmp;
                    button.ImageScaling = ToolStripItemImageScaling.None;
                    button.Tag = smitem;
                    button.Click += new EventHandler(colorDropDownItem_Click);

                    if (item == selected_item || selected == null)
                      selected = bmp;

                    buttons.Add(button);
                  }
                  catch
                  {
                  }
                }
                dropdown.Items.AddRange(buttons.ToArray());

                smitem.Image = selected;

                itemlist[i++] = smitem;
              }
              else
              {
                ToolStripComboBox smitem;
                String[] strings= new String[items.Count];
                String selected= null;
                int j = 0;

                smitem = new ToolStripComboBox();
                smitem.Name = subitem.get_name();
                smitem.Tag = subitem.get_command();
                smitem.Enabled = subitem.get_enabled();
                smitem.ToolTipText = subitem.get_tooltip();
                smitem.DropDownHeight = 250;
                smitem.SelectedIndexChanged += new EventHandler(textDropDownItem_Changed);

                foreach (String item in items)
                {
                  strings[j++]= item;

                  if (item == selected_item || selected == null)
                    selected = item;
                }
                smitem.Items.AddRange(strings);

                smitem.Text = selected;

                itemlist[i++] = smitem;
              }
              break;
            }
          case ToolbarItemType.ToolbarSearch:
            {
              ToolStripTextBox tb= new ToolStripTextBox();
              tb.Name = subitem.get_name();
              tb.Tag = subitem.get_command();
              tb.Enabled = true;// subitem.get_enabled();
              tb.AutoSize = false;
              tb.Size = new Size(180, 20);
              tb.TextBox.MaximumSize = new Size(180, 20);
              tb.ToolTipText = subitem.get_tooltip();
              tb.KeyPress += new KeyPressEventHandler(tb_KeyPress);

              if (tb.Name.EndsWith("__right"))
                tb.Alignment = ToolStripItemAlignment.Right;

              itemlist[i++] = tb;
              searchBox = tb; // Keep to allow focussing it via hotkey.

              break;
            }
        }        
      }

      if (addSpacer)
      {
        ToolStripLabel smitem;

        smitem = new ToolStripLabel();
        smitem.Name = "spacerLabel";
        smitem.Tag = null;
        smitem.Text = "        ";
        smitem.Font = new System.Drawing.Font("Tahoma", 8);
        smitem.Enabled = false;

        itemlist[i++] = smitem;
      }

      return itemlist;
    }

    void tb_KeyPress(object sender, KeyPressEventArgs e)
    {
      ToolStripTextBox item = sender as ToolStripTextBox;

      lastSearchString = item.Text;
      if (e.KeyChar == '\r')
      {
        e.Handled = true;

        if (item.Owner.Tag != null)
        {
          ToolbarCallback activate = item.Owner.Tag as ToolbarCallback;
          activate((String)item.Tag);
        }
        else
          wbContext.activate_command((String)item.Tag);
      }
    }


    void toolbarItem_Click(object sender, EventArgs e)
    {
      ToolStripItem item = sender as ToolStripItem;

      if (item != null)
      {
        if (item.Owner.Tag != null)
        {
          ToolbarCallback activate = item.Owner.Tag as ToolbarCallback;
          activate((String)item.Tag);
        }
        else
          wbContext.activate_command((String)item.Tag);
      }
    }

    void checkbox_Click(object sender, EventArgs e)
    {
      CheckBox item = sender as CheckBox;

      if (item != null)
      {
        wbContext.toggle_checkbox_item(item.Name, (String)item.Tag, item.Checked);
      }
    }

    void colorDropDownItem_Click(object sender, EventArgs e)
    {
      ToolStripButton button = sender as ToolStripButton;
      ToolStripDropDownButton smitem= button.Tag as ToolStripDropDownButton;

      smitem.Image = button.Image;

      wbContext.select_dropdown_item(smitem.Name, (String)smitem.Tag, button.Name);
    }

    void textDropDownItem_Changed(object sender, EventArgs e)
    {
      ToolStripComboBox combo = sender as ToolStripComboBox;

      wbContext.select_dropdown_item(combo.Name, (String)combo.Tag, (String)combo.SelectedItem);
    }
  }


  public class MenuManager
  {
    public class MenuContext
    {
      public MySQL.Grt.ActionList ActionList;

      public delegate List<NodeId> GetSelectedNodes_Delegate();
      public GetSelectedNodes_Delegate GetSelectedNodes;
      public List<NodeId> nodes;
      public delegate List<MySQL.Grt.MenuItem> GetNodesMenuItems_Delegate(List<NodeId> nodes);
      public GetNodesMenuItems_Delegate GetNodesMenuItems;
      public delegate bool TriggerNodesAction_Delegate(String action, List<NodeId> nodes);
      public TriggerNodesAction_Delegate TriggerNodesAction;

      public delegate List<int> GetSelectedRowsCol_Delegate(ref int column);
      public GetSelectedRowsCol_Delegate GetSelectedRowsCol;
      public List<int> rows;
      public int column;
      public delegate List<MySQL.Grt.MenuItem> GetRowsColMenuItems_Delegate(List<int> rows, int column);
      public GetRowsColMenuItems_Delegate GetRowsColMenuItems;
      public delegate bool TriggerRowsColAction_Delegate(String action, List<int> rows, int column);
      public TriggerRowsColAction_Delegate TriggerRowsColAction;
    }

    static public void InitMenu(ContextMenuStrip menu, MenuContext menuContext)
    {
      MenuContext prevMenuContext = menu.Tag as MenuContext;
      if (null == prevMenuContext)
        menu.Opening += OnMenuOpening;
      menu.Tag = menuContext;
    }

    static private void OnMenuOpening(object sender, CancelEventArgs e)
    {
      ContextMenuStrip menu = sender as ContextMenuStrip;
      MenuContext menuContext = menu.Tag as MenuContext;
      List<MySQL.Grt.MenuItem> itemsBE = null;
      menuContext.nodes = null;
      menuContext.rows = null;
      menuContext.column = -1;
      if (null != menuContext.GetSelectedNodes)
      {
        menuContext.nodes = menuContext.GetSelectedNodes();
        if (null == menuContext.nodes)
          menuContext.nodes = new List<NodeId>();
        itemsBE = menuContext.GetNodesMenuItems(menuContext.nodes);
      }
      else if (null != menuContext.GetSelectedRowsCol)
      {
        menuContext.rows = menuContext.GetSelectedRowsCol(ref menuContext.column);
        if (null == menuContext.rows)
          menuContext.rows = new List<int>();
        itemsBE = menuContext.GetRowsColMenuItems(menuContext.rows, menuContext.column);
      }
      menu.Items.Clear();
      if (null == itemsBE)
      {
        e.Cancel = true;
      }
      else
      {
        FillMenuItems(itemsBE, menu.Items);
        e.Cancel = false;
      }
    }

    static private void FillMenuItems(List<MySQL.Grt.MenuItem> itemsBE, ToolStripItemCollection itemsFE)
    {
      foreach (MySQL.Grt.MenuItem itemBE in itemsBE)
      {
        switch (itemBE.get_type())
        {
          case MySQL.Grt.MenuItemType.MenuSeparator:
            {
              itemsFE.Add(new ToolStripSeparator());
            }
            break;
          default:
            {
              ToolStripMenuItem itemFE = new ToolStripMenuItem();
              itemFE.Tag = itemBE.get_name();
              itemFE.Text = itemBE.get_caption();
              itemFE.Enabled = itemBE.get_enabled();
              if (MySQL.Grt.MenuItemType.MenuCascade == itemBE.get_type())
              {
                FillMenuItems(itemBE.get_subitems(), itemFE.DropDownItems);
              }
              else
              {
                itemFE.Click += new EventHandler(OnMenuItemClick);
              }
              itemsFE.Add(itemFE);
            }
            break;
        }
      }
    }

    static private void OnMenuItemClick(object sender, EventArgs e)
    {
      ToolStripMenuItem menuItem = sender as ToolStripMenuItem;
      MenuContext menuContext = GetMenuContext(menuItem.Owner);
      if (null == menuContext)
        return;
      string action = menuItem.Tag as string;
      bool res = false;
      if (null != menuContext.nodes)
      {
        if (!res && (null != menuContext.ActionList))
          res = menuContext.ActionList.trigger_action(action, menuContext.nodes);
        if (!res && (null != menuContext.TriggerNodesAction) && (null != menuContext.nodes))
          res = menuContext.TriggerNodesAction(action, menuContext.nodes);
      }
      else if (null != menuContext.rows)
      {
        if (!res && (null != menuContext.ActionList))
          res = menuContext.ActionList.trigger_action(action, menuContext.rows, menuContext.column);
        if (!res && (null != menuContext.TriggerRowsColAction))
          res = menuContext.TriggerRowsColAction(action, menuContext.rows, menuContext.column);
      }
    }

    static private MenuContext GetMenuContext(ToolStrip toolStrip)
    {
      if (null == toolStrip)
        return null;
      MenuContext menuContext = toolStrip.Tag as MenuContext;
      if (null == menuContext)
      {
        ToolStripDropDownMenu dropDownMenu = toolStrip as ToolStripDropDownMenu;
        if (null == dropDownMenu)
          return null;
        ToolStripItem ownerItem = dropDownMenu.OwnerItem;
        return GetMenuContext(ownerItem.Owner);
      }
      else
        return menuContext;
    }
  }
}
