/* 
 * (c) 2009-2010 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "string_utilities.h"
#include <sigc++/sigc++.h>
#include <functional>
#include <locale>
#include <algorithm>

using namespace std;

namespace base {

string trim_right(const string& s, const string& t)
{
  string d(s);
  string::size_type i (d.find_last_not_of(t));
  if (i == string::npos)
    return "";
  else
    return d.erase(d.find_last_not_of(t) + 1) ;
}  

//--------------------------------------------------------------------------------------------------

string trim_left(const string& s, const string& t)
{
  string d(s);
  return d.erase(0, s.find_first_not_of(t)) ;
}  

//--------------------------------------------------------------------------------------------------

string trim(const string& s, const string& t)
{
  string d(s);
  return trim_left(trim_right(d, t), t) ;
}  

//--------------------------------------------------------------------------------------------------

/**
 * Returns a string containg all characters beginning at "start" from the given string "id", which form
 * a valid, unqualified identifier. The returned identifier does not contain any quoting anymore.
 * Note: this function is UTF-8 safe as it skips over all characters except some which are guaranteed
 *       not to be part of any valid UTF-8 sequence.
 *
 * @param id The string to examine.
 * @param start The start position to search from.
 *
 * @result Returns the first found identifier starting at "start" or an empty string if nothing was 
 *         found. Parameter "start" points to the first character after the found identifier.
 */
string get_identifier(const string& id, string::const_iterator& start)
{
  string::const_iterator token_end= id.end();
  bool is_symbol_quoted= false;
  for (string::const_iterator i= start, i_end= token_end; i != i_end; ++i)
  {
    if (i_end != token_end)
      break;
    switch (*i)
    {
      case '.':
        if (!is_symbol_quoted)
          token_end= i;
        break;
      case ' ':
        if (!is_symbol_quoted)
          token_end= i;
        break;
      case '\'':
      case '"':
      case '`':
        if (*i == *start)
        {
          if (i != start)
            token_end= i + 1;
          else
            is_symbol_quoted= true;
        }
        break;
    }
  }

  if (token_end - start < 2)
    is_symbol_quoted= false;
  string result(start, token_end);
  start= token_end;
  if (is_symbol_quoted)
    return result.substr(1, result.size() - 2);

  return result;
}

//--------------------------------------------------------------------------------------------------

/**
 * Splits the given string into identifier parts assuming a format as allowed by the MySQL syntax for
 * qualified identifiers, e.g. part1.part2.part3 (any of the parts might be quoted).
 * In addition to the traditional syntax also these enhancements are supported:
 * - Unlimited level of nesting.
 * - Quoting might be done using single quotes, double quotes and back ticks.
 *
 * If an identifier is not separated by a dot from the rest of the input then this is considered
 * invalid input and ignored. Only identifiers found until that syntax violation are returned.
 */
vector<string> split_qualified_identifier(const string& id)
{
  vector<string> result;
  string::const_iterator iterator= id.begin();
  string token;
  do
  {
    token= get_identifier(id, iterator);
    if (token == "")
      break;
    result.push_back(token);
  } while (*iterator++ == '.');
  
  return result;
}

//--------------------------------------------------------------------------------------------------

/**
 * Removes the first path part from @path and returns this part as well as the shortend path.
 */
std::string pop_path_front(std::string &path)
{
  std::string::size_type p= path.find('/');
  std::string res;
  if (p == std::string::npos || p == path.length()-1)
  {
    res= path;
    path.clear();
    return res;
  }
  res= path.substr(0, p);
  path= path.substr(p+1);
  return res;
}

//--------------------------------------------------------------------------------------------------

/**
 * Helper routine to format a string into an STL string using the printf parameter syntax.
 */
std::string strfmt(const char *fmt, ...)
{
  va_list args;
  char *tmp;
  std::string ret;
  
  va_start(args, fmt);
  tmp= g_strdup_vprintf(fmt, args);
  va_end(args);
  
  ret= tmp;
  g_free(tmp);
  
  return ret;
}

//--------------------------------------------------------------------------------------------------

/**
 * Helper routine to strip a string into an STL string using the printf parameter syntax.
 */
std::string strip_text(const std::string &text, bool left, bool right)
{
  std::locale loc;
  std::logical_not<bool> nott;
  sigc::slot<bool, bool> not_slot= sigc::mem_fun(nott, &std::logical_not<bool>::operator());
  sigc::slot<bool, std::string::value_type> is_space=
    sigc::bind(sigc::ptr_fun(&std::isspace<std::string::value_type>), sigc::ref(loc));

  std::string::const_iterator l_edge= !left ? text.begin() :
    std::find_if(text.begin(), text.end(), sigc::compose(not_slot, is_space));
  std::string::const_reverse_iterator r_edge= !right ? text.rbegin() :
    std::find_if(text.rbegin(), text.rend(), sigc::compose(not_slot, is_space));

  return std::string(l_edge, r_edge.base());
}

} // namespace base
