/* 
 * (c) 2009-2010 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import "MFTextBox.h"
#include "mforms/mforms.h"

@implementation MFTextBoxImpl

- (id)initWithObject:(::mforms::TextBox*)aTextBox
           scrollers:(mforms::ScrollBars)scrolls
{
  NSRect frame;
  
  frame.origin= NSMakePoint(0, 0);
  frame.size= [NSScrollView frameSizeForContentSize:[self minimumSize]
                              hasHorizontalScroller:NO
                                hasVerticalScroller:YES
                                         borderType:NSLineBorder];
  
  self= [super initWithFrame:frame];
  if (self)
  {
    mOwner= aTextBox;
    mOwner->set_data(self);

    [self setBorderType: NSBezelBorder];
    [self setHasVerticalScroller: scrolls & mforms::VerticalScrollBar];
    [self setHasHorizontalScroller: scrolls & mforms::HorizontalScrollBar];
    
    frame.size= [self minimumSize];
    mContentView= [[[NSTextView alloc] initWithFrame:frame] autorelease];
    [self setDocumentView:mContentView];

    [[mContentView textContainer] setContainerSize: NSMakeSize(1000000000, 1000000000)];
    [mContentView setHorizontallyResizable: YES];
    [mContentView setVerticallyResizable: YES];
    if (scrolls & mforms::VerticalScrollBar)
    {
      [[mContentView textContainer] setHeightTracksTextView: NO];
    }
    else
      [[mContentView textContainer] setHeightTracksTextView: YES];

    if (scrolls & mforms::HorizontalScrollBar)
      [[mContentView textContainer] setWidthTracksTextView: NO];
    else
      [[mContentView textContainer] setWidthTracksTextView: YES];
    
    [mContentView setDelegate: self];
//    [mContentView setAutoresizingMask:NSViewWidthSizable|NSViewHeightSizable|NSViewMinXMargin|NSViewMaxXMargin|NSViewMinYMargin|NSViewMaxYMargin];
    [mContentView setEditable: YES];
  }
  return self;
}

- (mforms::Object*)mformsObject
{
  return mOwner;
}


- (NSSize)minimumSize
{
  NSSize size;
  size.width= [NSScroller scrollerWidth] + 50 + mPadding*2;
  size.height= [NSScroller scrollerWidth] + 50 + mPadding*2;
  return size;
}


- (void)textDidChange:(NSNotification*)notification
{
  mOwner->callback();
}


- (void)setFrame:(NSRect)rect
{
  rect.origin.x+= mPadding;
  rect.origin.y+= mPadding;
  rect.size.width-= mPadding*2;
  rect.size.height-= mPadding*2;
  [super setFrame: rect];
}


- (void)setTag:(NSInteger)tag
{
  mTag= tag;
}


- (NSInteger)tag
{
  return mTag;
}


- (void)setPadding:(float)pad
{
  mPadding= pad;
}


- (void)setBordered:(BOOL)flag
{
  [self setBorderType: flag ? NSBezelBorder : NSNoBorder];
}

- (void)setEnabled:(BOOL)flag
{
  [mContentView setEditable: flag ? YES : NO];
  [mContentView setSelectable: flag ? YES : NO];
}

- (void) setReadOnly: (BOOL) flag
{
  [mContentView setEditable: flag ? NO : YES];
  [mContentView setSelectable: YES];  
}

- (void) setTextColor: (NSColor *) aColor
{
  [mContentView setTextColor: aColor];
}

- (void) setBackgroundColor: (NSColor *) aColor
{
  [mContentView setDrawsBackground: (aColor == nil) ? NO : YES];
  [mContentView setBackgroundColor: aColor];
}

//--------------------------------------------------------------------------------------------------

static bool textbox_create(::mforms::TextBox *self, mforms::ScrollBars scrolls)
{
  MFTextBoxImpl *textbox= [[[MFTextBoxImpl alloc] initWithObject : self 
                                                       scrollers : scrolls] autorelease];
  
  return textbox != nil;
}


static void textbox_set_text(::mforms::TextBox *self, const std::string &text)
{
  if ( self )
  {
    MFTextBoxImpl* textbox = self->get_data();
    
    NSRange range;
    range = NSMakeRange (0, [[textbox->mContentView string] length]);
    [textbox->mContentView replaceCharactersInRange: range withString: wrap_nsstring(text)];
  }
}


static void textbox_append_text(mforms::TextBox *self, const std::string &text, bool scroll_to_end)
{
  if ( self )
  {
    MFTextBoxImpl* textbox = self->get_data();
    
    NSRange range;
    range = NSMakeRange ([[textbox->mContentView string] length], 0);
    [textbox->mContentView replaceCharactersInRange: range withString: wrap_nsstring(text)];
    
    if (scroll_to_end)
    {
      NSRange range;
      range = NSMakeRange ([[textbox->mContentView string] length], 0);
      [textbox->mContentView scrollRangeToVisible: range];
    }
  }  
}


static std::string textbox_get_text(::mforms::TextBox *self)
{
  if ( self )
  {
    MFTextBoxImpl* textbox = self->get_data();
    
    return [[textbox->mContentView string] UTF8String];
  }
  return "";
}


static void textbox_set_padding(::mforms::TextBox *self, int pad)
{
  if ( self )
  {
    MFTextBoxImpl* textbox = self->get_data();
    
    [textbox setPadding:pad];
  }
}


static void textbox_set_bordered(::mforms::TextBox *self, bool flag)
{
  if ( self )
  {
    MFTextBoxImpl* textbox = self->get_data();
    
    [textbox setBordered: flag];
  }
}


static void textbox_set_read_only(mforms::TextBox *self, bool flag)
{
  if (self)
    [self->get_data() setReadOnly: flag];
}

static void textbox_clear(mforms::TextBox *self)
{
  if ( self )
  {
    MFTextBoxImpl* textbox = self->get_data();
    
    [textbox->mContentView setString: @""];
  }  
}

void cf_textbox_init()
{
  ::mforms::ControlFactory *f = ::mforms::ControlFactory::get_instance();
  
  f->_textbox_impl.create= &textbox_create;
  f->_textbox_impl.set_text= &textbox_set_text;
  f->_textbox_impl.get_text= &textbox_get_text;
  f->_textbox_impl.set_padding= &textbox_set_padding;
  f->_textbox_impl.set_bordered= &textbox_set_bordered;
  f->_textbox_impl.set_read_only= &textbox_set_read_only;
  f->_textbox_impl.append_text= &textbox_append_text;
  f->_textbox_impl.clear= &textbox_clear;
}

@end



