/* 
 * (c) 2008-2010 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include "../lf_textbox.h"

namespace mforms {
namespace gtk {


bool TextBoxImpl::create(::mforms::TextBox *self, mforms::ScrollBars scroll_type)
{
  return new TextBoxImpl(self, scroll_type) != 0;
}

void TextBoxImpl::set_text(::mforms::TextBox *self, const std::string &text)
{
  TextBoxImpl* cb = self->get_data<TextBoxImpl>();

  if ( cb )
    cb->_text->get_buffer()->set_text(text);
}

void TextBoxImpl::append_text(::mforms::TextBox *self, const std::string &text, bool scroll_to_end)
{
  TextBoxImpl* cb = self->get_data<TextBoxImpl>();

  if ( cb )
  {
    Gtk::TextView *tv = cb->_text;
    if (tv)
    {
      Glib::RefPtr<Gtk::TextBuffer> buf = tv->get_buffer();
      buf->insert(buf->end(), text);

      if (scroll_to_end)
      {
        Gtk::TextIter it = buf->end();
        tv->scroll_to(it, 0.3);
      }
    }
  }
}

std::string TextBoxImpl::get_text(::mforms::TextBox *self)
{
  TextBoxImpl* cb = self->get_data<TextBoxImpl>();
  std::string ret("");
  if ( cb )
  {
    ret = cb->_text->get_buffer()->get_text().raw();
  }
  return ret;
}

void TextBoxImpl::set_read_only(::mforms::TextBox *self, bool flag)
{
  TextBoxImpl* cb = self->get_data<TextBoxImpl>();
  if (cb && cb->_text)
    cb->_text->set_editable(!flag);
}

void TextBoxImpl::set_padding(::mforms::TextBox *self, int pad)
{
  TextBoxImpl* cb = self->get_data<TextBoxImpl>();
  if (cb && cb->_swin)
    cb->_swin->set_border_width(pad);
}

void TextBoxImpl::set_bordered(::mforms::TextBox *self, bool flag)
{
  TextBoxImpl* cb = self->get_data<TextBoxImpl>();
  if (cb)
    cb->_swin->set_shadow_type(flag ? Gtk::SHADOW_IN : Gtk::SHADOW_NONE);
}

void TextBoxImpl::clear(::mforms::TextBox *self)
{
  TextBoxImpl* cb = self->get_data<TextBoxImpl>();
  if (cb && cb->_text)
    cb->_text->get_buffer()->set_text("");
}

TextBoxImpl::TextBoxImpl(::mforms::TextBox *self, mforms::ScrollBars scroll_type)
  : ViewImpl(self)
{
  _swin= Gtk::manage(new Gtk::ScrolledWindow());
  _text = Gtk::manage(new Gtk::TextView());
  _swin->add(*_text);
  
  Gtk::PolicyType h_scrollbar_policy;
  Gtk::PolicyType v_scrollbar_policy;
  
  switch (scroll_type)
  {
    case ::mforms::NoScrollBar:
    {
      h_scrollbar_policy = Gtk::POLICY_NEVER;
      v_scrollbar_policy = Gtk::POLICY_NEVER;
      break;
    }
    case ::mforms::HorizontalScrollBar:
    {
      h_scrollbar_policy = Gtk::POLICY_AUTOMATIC;
      v_scrollbar_policy = Gtk::POLICY_NEVER;
      break;
    }
    case ::mforms::VerticalScrollBar:
    {
      h_scrollbar_policy = Gtk::POLICY_NEVER;
      v_scrollbar_policy = Gtk::POLICY_AUTOMATIC;
      break;
    }
    case ::mforms::BothScrollBars:
    {
      h_scrollbar_policy = Gtk::POLICY_AUTOMATIC;
      v_scrollbar_policy = Gtk::POLICY_AUTOMATIC;
      break;
    }
  }
  _swin->set_policy(h_scrollbar_policy, v_scrollbar_policy);
  _swin->set_shadow_type(Gtk::SHADOW_IN);
  _text->show();

  _text->get_buffer()->signal_changed().connect(sigc::mem_fun(*self, &::mforms::TextBox::callback));
}

void TextBoxImpl::init()
{
  ::mforms::ControlFactory *f = ::mforms::ControlFactory::get_instance();

  f->_textbox_impl.create   = &TextBoxImpl::create;
  f->_textbox_impl.set_bordered= &TextBoxImpl::set_bordered;
  f->_textbox_impl.set_text = &TextBoxImpl::set_text;
  f->_textbox_impl.get_text = &TextBoxImpl::get_text;
  f->_textbox_impl.append_text = &TextBoxImpl::append_text;
  f->_textbox_impl.set_read_only= &TextBoxImpl::set_read_only;
  f->_textbox_impl.set_padding = &TextBoxImpl::set_padding;
  f->_textbox_impl.clear = &TextBoxImpl::clear;
}


  };
};
