/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#ifndef _MFORMS_Utilities_H_
#define _MFORMS_Utilities_H_

/**
 * Implementation some miscellaneous stuff needed in mforms.
 */

#include <mforms/base.h>
#include <sigc++/sigc++.h>

#include "cairo/cairo.h"

namespace mforms {
  // Constants for special folders on a system. Constants have been taken over from Windows definition
  // and are used directly in the Windows code, hence avoid changing them or fix the platform code if you do.
  enum MFORMS_EXPORT FolderType {
    MyDocuments = 5,          // The "My Documents" folder.
    Documents = 5,            // The directory used to physically store file objects on the desktop.
    DesktopDirectory = 16,
    Desktop = 16,
    ApplicationData = 26,    
    LocalApplicationData = 28 // The roaming user's application data folder. The directory that serves as a common
                              // repository for application-specific data that is used by the current, non-roaming user.
  };

  enum MFORMS_EXPORT PasswordStoreScheme
  {
    SessionStorePasswordScheme = 1,
    PersistentStorePasswordScheme = 2
  };

#ifndef SWIG
  struct MFORMS_EXPORT UtilitiesImplPtrs
  {
    int (__stdcall *show_message)(const std::string &title, const std::string &text,
                                  const std::string &ok, const std::string &cancel,
                                  const std::string &other);
    int (__stdcall *show_error)(const std::string &title, const std::string &text,
                                  const std::string &ok, const std::string &cancel,
                                  const std::string &other);
    int (__stdcall *show_warning)(const std::string &title, const std::string &text,
                                  const std::string &ok, const std::string &cancel,
                                  const std::string &other);
    void (__stdcall *show_wait_message)(const std::string &title, const std::string &text);
    bool (__stdcall *hide_wait_message)();
    
    void (__stdcall *set_clipboard_text)(const std::string &text);
    std::string (__stdcall *get_clipboard_text)();
    std::string (__stdcall *get_special_folder)(mforms::FolderType type);
    
    void (__stdcall *open_url)(const std::string &url);
    
    void (__stdcall *add_timeout)(float delay, const sigc::slot<bool> &callback);
    
    void (__stdcall *store_password)(const std::string &service, const std::string &account, const std::string &password);
    bool (__stdcall *find_password)(const std::string &service, const std::string &account, std::string &password);
    void (__stdcall *forget_password)(const std::string &service, const std::string &account);
  };
#endif

  enum DialogResult {
    ResultOk     =  1,
    ResultCancel =  0,
    ResultOther  = -1
  };
  
  class Box;
  class Button;

  class MFORMS_EXPORT Utilities
  {
#ifdef SWIG
%ignore show_message(const std::string &title, const std::string &text, const std::string &ok, const std::string &cancel);
%ignore show_message(const std::string &title, const std::string &text, const std::string &ok);
%ignore show_warning(const std::string &title, const std::string &text, const std::string &ok, const std::string &cancel);
%ignore show_warning(const std::string &title, const std::string &text, const std::string &ok);
%ignore show_error(const std::string &title, const std::string &text, const std::string &ok, const std::string &cancel);
%ignore show_error(const std::string &title, const std::string &text, const std::string &ok);
#endif
  public:
    static int show_message(const std::string &title, const std::string &text,
                            const std::string &ok, const std::string &cancel="",
                            const std::string &other="");
    static int show_error(const std::string &title, const std::string &text,
                            const std::string &ok, const std::string &cancel="",
                            const std::string &other="");
    static int show_warning(const std::string &title, const std::string &text,
                            const std::string &ok, const std::string &cancel="",
                            const std::string &other="");
    static void show_wait_message(const std::string &title, const std::string &text);
    static bool hide_wait_message();

    static bool ask_for_password_check_store(const std::string &title, const std::string &service, const std::string &username,
      std::string &ret_password /*out*/, bool &ret_store /*out*/);

    static bool ask_for_password(const std::string &title, const std::string &service, const std::string &username,
                                 std::string &ret_password /*out*/);

    static bool find_or_ask_for_password(const std::string &title, const std::string &service, const std::string &username,
                                         bool reset_password, std::string &ret_password /*out*/);

    static void set_clipboard_text(const std::string &text);
    static std::string get_clipboard_text();

    static std::string get_special_folder(FolderType type);

    static void open_url(const std::string &url);

    // callback must return true if it wants to be triggered again
    static void add_timeout(float interval, const sigc::slot<bool> &callback);
    
    static void add_end_ok_cancel_buttons(mforms::Box *box, mforms::Button *ok, mforms::Button *cancel);

    static cairo_surface_t* load_icon(const std::string& name);
    static std::string shorten_string(cairo_t* cr, const std::string& text, double width);

    static void store_password(const std::string &service, const std::string &account, const std::string &password);
    static bool find_password(const std::string &service, const std::string &account, std::string &password);
    static void forget_password(const std::string &service, const std::string &account);
    
#ifdef _WIN32
#ifndef SWIG
  public:
    static void enter_modal_loop();
    static void leave_modal_loop();
    static bool in_modal_loop();
#endif
#endif
  };

};
#endif // _MFORMS_Utilities_H_
