/* 
 * Copyright (c) 2008, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

/**
 * Contains the implementation of the wrapper for a .NET form used by the mforms backend.
 */

#include "stdafx.h"

#include "wf_form.h"

using namespace System;
using namespace System::IO;
using namespace Drawing;
using namespace Windows::Forms;

using namespace MySQL;
using namespace MySQL::Forms;

//--------------------------------------------------------------------------------------------------

bool FormFillLayout::Layout(Object^ container, LayoutEventArgs^ arguments)
{
  // This layout is actually very simple. Simply resize the first (and only) child control so that
  // it fills the entire client area of the container. If enabled resize the container to fit the
  // (preferred) size of the content.

  Control^ parent= (Control^) container;
  if (parent->Controls->Count > 0)
  {
    Size boxSize= parent->Size;
    System::Drawing::Rectangle displayArea= parent->DisplayRectangle;

    Control^ content= parent->Controls[0];
    Size childSize;

    childSize= content->PreferredSize;
    if (ViewImpl::use_min_width_for_layout(content))
      childSize.Width= content->MinimumSize.Width;
    if (ViewImpl::use_min_height_for_layout(content))
      childSize.Height= content->MinimumSize.Height;

    // Adjust width of the container if it is too small or auto resizing is enabled.
    // Don't auto size if this is a standalone window (has no parent).
    bool auto_resize= (parent->Parent != nullptr) && ViewImpl::can_auto_resize_horizontally(parent);
    if (displayArea.Width < childSize.Width || auto_resize)
      displayArea.Width= childSize.Width;

    // Adjust height of the container if it is too small or auto resizing is enabled.
    auto_resize= (parent->Parent != nullptr) && ViewImpl::can_auto_resize_vertically(parent);
    if (displayArea.Height < childSize.Height || auto_resize)
      displayArea.Height= childSize.Height;

    // Now stretch the client control to fill the entire display area.
    ViewImpl::remove_auto_resize(content, mforms::ResizeBoth);
    content->Bounds= displayArea;

    // Make sure the content fits completely in (maybe larger due to min size constraint).
    // This cover-all resize is not controlled by the auto resize flags.
    if (content->Width > displayArea.Width)
      displayArea.Width= content->Width;
    if (content->Height > displayArea.Height)
      displayArea.Height= content->Height;

    // Finally adjust the container.
    bool parentLayoutNeeded= !parent->DisplayRectangle.Size.Equals(displayArea.Size);
    if (parentLayoutNeeded)
    {
      Size newSize;
      newSize.Width= displayArea.Right + parent->Padding.All;
      if (newSize.Width < parent->MinimumSize.Width)
        newSize.Width= parent->MinimumSize.Width;
      newSize.Height= displayArea.Bottom + parent->Padding.All;
      if (newSize.Height < parent->MinimumSize.Height)
        newSize.Height= parent->MinimumSize.Height;

      ViewImpl::resize_with_docking(parent, newSize);
    }

    return parentLayoutNeeded;
  }
  return false;
}

//-------------------------------------------------------------------------------------------------

Drawing::Size FormFillLayout::GetMinimumSize(Control^ container)
{
  if (container->Controls->Count > 0)
  {
    Control^ content= container->Controls[0];
    Drawing::Size size= content->GetPreferredSize(container->Size);

    // Add any space needed to draw decoration (e.g. border + heading label).
    Drawing::Rectangle inner= container->DisplayRectangle;
    Drawing::Size current_size= container->Size;
    size.Width += current_size.Width - inner.Width;
    size.Height += current_size.Height - inner.Height;

    return size;
  }
  return Size::Empty;
}

//----------------- FillForm -----------------------------------------------------------------------

Drawing::Size FillForm::GetPreferredSize(Drawing::Size proposedSize)
{
  Drawing::Size size= layoutEngine->GetMinimumSize(this);
  if (size.Width < MinimumSize.Width)
    size.Width= MinimumSize.Width;
  if (size.Height < MinimumSize.Height)
    size.Height= MinimumSize.Height;
  return size;
}

//-------------------------------------------------------------------------------------------------

bool FormImpl::create(::mforms::Form *self, ::mforms::Form *owner, ::mforms::FormFlag flag)
{
  FormImpl ^form= gcnew FormImpl(self, owner, flag);

  if (form != nullptr)
  {
    Windows::Forms::Form ^f= ViewImpl::create<FillForm>(self, form);
    f->StartPosition= FormStartPosition::Manual;
    f->Disposed += gcnew System::EventHandler(&FormImpl::disposed);
    f->FormClosing += gcnew FormClosingEventHandler(&FormImpl::closing);

    if (File::Exists("images/icons/MySQLWorkbench.ico"))
      f->Icon= gcnew Icon("images/icons/MySQLWorkbench.ico", Size(16, 16));

    if ((flag & mforms::FormSingleFrame) != 0)
      f->FormBorderStyle= FormBorderStyle::FixedToolWindow;
    else
      if ((flag & mforms::FormDialogFrame) != 0)
        f->FormBorderStyle= FormBorderStyle::FixedDialog;
      else
        if ((flag & mforms::FormResizable) != 0)
          f->FormBorderStyle= FormBorderStyle::SizableToolWindow;
        else
          f->FormBorderStyle= FormBorderStyle::None;

    f->MinimizeBox= (flag & mforms::FormMinimizable) != 0;
    f->MaximizeBox= (flag & mforms::FormMinimizable) != 0;
    f->TopMost = (flag & mforms::FormStayOnTop) != 0;

    form->_hide_on_close= (flag & mforms::FormHideOnClose) != 0;

    return true;
  }
  return false;
}

//--------------------------------------------------------------------------------------------------

void FormImpl::set_title(::mforms::Form *self, const std::string &title)
{
  FormImpl^ form= (FormImpl^)ObjectImpl::FromUnmanaged(self);

  if (form != nullptr)
    form->get_control<Form>()->Text= gcnew String(title.c_str());
}

//--------------------------------------------------------------------------------------------------

void FormImpl::show_modal(::mforms::Form *self, ::mforms::Button *accept,::mforms::Button *cancel)
{
  FormImpl^ form= (FormImpl^)ObjectImpl::FromUnmanaged(self);

  if (form != nullptr)
  {
    Form ^f= form->get_control<Form>();

    if (accept)
    {
      f->AcceptButton= ((ViewImpl^)ObjectImpl::FromUnmanaged(accept))->get_control<Button>();
      f->AcceptButton->DialogResult= DialogResult::OK;
    }
    if (cancel)
    {
      f->CancelButton= ((ViewImpl^)ObjectImpl::FromUnmanaged(cancel))->get_control<Button>();
      f->CancelButton->DialogResult= DialogResult::Cancel;
    }
    // Make the window top most (so it stays above all others), but non-blocking.
    f->TopMost= true;
    if (f->InvokeRequired)
      f->BeginInvoke(gcnew MethodInvoker(f, &Control::Show));
    else
      f->Show();
  }
}

//--------------------------------------------------------------------------------------------------

bool FormImpl::run_modal(::mforms::Form *self, ::mforms::Button *accept,::mforms::Button *cancel)
{
  FormImpl^ form= (FormImpl^)ObjectImpl::FromUnmanaged(self);

  if (form != nullptr)
  {
    Form ^f= form->get_control<Form>();

    if (accept)
    {
      f->AcceptButton= ((ViewImpl^)ObjectImpl::FromUnmanaged(accept))->get_control<Button>();
      f->AcceptButton->DialogResult= DialogResult::OK;
    }
    if (cancel)
    {
      f->CancelButton= ((ViewImpl^)ObjectImpl::FromUnmanaged(cancel))->get_control<Button>();
      f->CancelButton->DialogResult= DialogResult::Cancel;
    }

    mforms::Utilities::enter_modal_loop();

    DialogResult dialog_result;
    if (f->InvokeRequired)
    {
      Object^ invocation_result= f->Invoke(gcnew FillForm::ShowModalDelegate(f, &Form::ShowDialog));
      dialog_result= *(DialogResult^) (invocation_result);
    }
    else
      dialog_result= f->ShowDialog();

    bool result = (dialog_result == DialogResult::OK);

    mforms::Utilities::leave_modal_loop();

    f->Hide();

    return result;
  }
  return false;
}

//--------------------------------------------------------------------------------------------------

/**
 * Called by the backend when a form is closed from the application side.
 */
void FormImpl::close(::mforms::Form *self)
{
  FormImpl^ form= (FormImpl^)ObjectImpl::FromUnmanaged(self);

  if (form != nullptr)
    form->get_control<Form>()->Close();
}

//--------------------------------------------------------------------------------------------------

void FormImpl::set_content(::mforms::Form *self, ::mforms::View *view)
{
  FormImpl^ form= (FormImpl^)ObjectImpl::FromUnmanaged(self);

  if (form != nullptr)
  {
    view->set_resize_mode(mforms::ResizeNone);
    ViewImpl^ child= (ViewImpl^)ObjectImpl::FromUnmanaged(view);
    Control^ ctl= child->get_control<Control>();
    ctl->Dock= DockStyle::Fill;

    form->get_control<Form>()->Controls->Add(ctl);
  }
}

//--------------------------------------------------------------------------------------------------


void FormImpl::center(::mforms::Form *self)
{
  FormImpl^ form= (FormImpl^)ObjectImpl::FromUnmanaged(self);
  if (form != nullptr)
    form->center();
}

//--------------------------------------------------------------------------------------------------

void FormImpl::flush_events(::mforms::Form *self)
{
  FormImpl^ form= (FormImpl^)ObjectImpl::FromUnmanaged(self);
  if (form != nullptr)
    form->flush_events();
}

//--------------------------------------------------------------------------------------------------

FormImpl::FormImpl(::mforms::Form *form, ::mforms::Form *owner, ::mforms::FormFlag flag)
  : ViewImpl(form)
{
}

//--------------------------------------------------------------------------------------------------

/**
 * Centers the form over the current main form or, if that cannot be found, on the screen which
 * currently contains the largest portion of it.
 */
void FormImpl::center()
{
  Form^ form= get_control<Form>();
  Drawing::Rectangle workingArea;

  Form^ mainForm= Application::OpenForms["MainForm"];
  if (mainForm == nullptr)
  {
    Screen^ screen= Screen::FromControl(form);
    workingArea= screen->WorkingArea;
  }
  else
    workingArea= mainForm->Bounds;

  // TODO: location is not correct as the layout of the form is not yet done (and hence its size is
  //       not final). Forcing a layout here (form->PerformLayout()) doesn't help at all, so
  //       we need more investigation here.
  form->Location= Point((workingArea.Right - form->Width) / 2, (workingArea.Bottom - form->Height) / 2);
}

//--------------------------------------------------------------------------------------------------

void FormImpl::flush_events()
{
  Application::DoEvents();
}

//--------------------------------------------------------------------------------------------------

/**
 * Called by the OS if the form is destroyed by the user (e.g. via red cross close button and
 * the form is set to destroy itself when closed).
 */
void FormImpl::disposed(System::Object ^sender, System::EventArgs ^e)
{
  Windows::Forms::Form^ native_form= (Windows::Forms::Form^)sender;

  if (native_form->Tag != nullptr)
  {
    ::mforms::Form* form=ViewImpl::get_backend_control<mforms::Form>(native_form);
    form->was_closed();
  }
}

//--------------------------------------------------------------------------------------------------

void FormImpl::closing(System::Object ^sender, FormClosingEventArgs  ^e)
{
  Windows::Forms::Form^ native_form= (Windows::Forms::Form^)sender;
  mforms::Form* be_form= ViewImpl::get_backend_control<mforms::Form>(native_form);
  FormImpl^ wrapper_form= (FormImpl^)ObjectImpl::FromUnmanaged(be_form);
  if (wrapper_form->hide_on_close())
  {
    e->Cancel= true;
    native_form->Hide();
  }

  // Do nothing if hiding is not requested. In this case Windows will dispose of the form implicitely.
}

//--------------------------------------------------------------------------------------------------

bool FormImpl::hide_on_close()
{
  return _hide_on_close;
}

//--------------------------------------------------------------------------------------------------

