/* 
 * (c) 2008-2010 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include "stdafx.h"
#include "wf_panel.h"

using namespace MySQL;
using namespace MySQL::Forms;

//-------------------------------------------------------------------------------------------------

bool FillLayout::Layout(Object^ container, LayoutEventArgs^ arguments)
{
  // This layout is actually very simple. Simply resize the first (and only) child control so that
  // it fills the entire client area of the container. If enabled resize the container to fit the
  // (preferred) size of the content.

  Control^ parent= (Control^) container;
  if (parent->Controls->Count > 0)
  {
    Size boxSize= parent->Size;
    System::Drawing::Rectangle displayArea= parent->DisplayRectangle;

    Control^ content= parent->Controls[0];
    Size childSize;

    childSize= content->PreferredSize;
    if (ViewImpl::use_min_width_for_layout(content))
      childSize.Width= content->MinimumSize.Width;
    if (ViewImpl::use_min_height_for_layout(content))
      childSize.Height= content->MinimumSize.Height;

    // Adjust width of the container if it is too small or auto resizing is enabled.
    if (displayArea.Width < childSize.Width || ViewImpl::can_auto_resize_horizontally(parent))
      displayArea.Width= childSize.Width;

    // Adjust height of the container if it is too small or auto resizing is enabled.
    if (displayArea.Height < childSize.Height || ViewImpl::can_auto_resize_vertically(parent))
      displayArea.Height= childSize.Height;

    // Now stretch the client control to fill the entire display area.
    ViewImpl::remove_auto_resize(content, mforms::ResizeBoth);
    content->Bounds= displayArea;

    // Make sure the content fits completely in (maybe larger due to min size constraint).
    // This cover-all resize is not controlled by the auto resize flags.
    if (content->Width > displayArea.Width)
      displayArea.Width= content->Width;
    if (content->Height > displayArea.Height)
      displayArea.Height= content->Height;

    // Finally adjust the container.
    bool parentLayoutNeeded= !parent->DisplayRectangle.Size.Equals(displayArea.Size);
    if (parentLayoutNeeded)
    {
      Size newSize;
      newSize.Width= displayArea.Right + parent->Padding.All;
      if (newSize.Width < parent->MinimumSize.Width)
        newSize.Width= parent->MinimumSize.Width;
      newSize.Height= displayArea.Bottom + parent->Padding.All;
      if (newSize.Height < parent->MinimumSize.Height)
        newSize.Height= parent->MinimumSize.Height;

      ViewImpl::resize_with_docking(parent, newSize);
      if (parent->GetType() == FillPanel::typeid)
        ((FillPanel^) parent)->SetPreferredSize(newSize);
      else
        if (parent->GetType() == FillGroupBox::typeid)
          ((FillGroupBox^) parent)->SetPreferredSize(newSize);
    }

    return parentLayoutNeeded;
  }
  return false;
}

//-------------------------------------------------------------------------------------------------

Drawing::Size FillLayout::GetMinimumSize(Control^ container)
{
  if (container->Controls->Count > 0)
  {
    Control^ content= container->Controls[0];
    Drawing::Size size= content->GetPreferredSize(container->MinimumSize);
    if (ViewImpl::use_min_width_for_layout(content))
      size.Width= content->MinimumSize.Width;
    if (ViewImpl::use_min_height_for_layout(content))
      size.Height= content->MinimumSize.Height;

    // Add any space needed to draw decoration (e.g. border + heading label).
    Drawing::Rectangle inner= container->DisplayRectangle;
    Drawing::Size current_size= container->Size;
    size.Width += current_size.Width - inner.Width;
    size.Height += current_size.Height - inner.Height;

    return size;
  }
  return Size::Empty;
}

//----------------- FillGroupBox ------------------------------------------------------------------

Drawing::Size FillGroupBox::GetPreferredSize(Drawing::Size proposedSize)
{
  Drawing::Size size= layoutEngine->GetMinimumSize(this);
  if (size.Width < MinimumSize.Width)
    size.Width= MinimumSize.Width;
  if (size.Height < MinimumSize.Height)
    size.Height= MinimumSize.Height;
  return size;
}

//----------------- FillPanel ---------------------------------------------------------------------

Drawing::Size FillPanel::GetPreferredSize(Drawing::Size proposedSize)
{
  Drawing::Size size= layoutEngine->GetMinimumSize(this);
  if (size.Width < MinimumSize.Width)
    size.Width= MinimumSize.Width;
  if (size.Height < MinimumSize.Height)
    size.Height= MinimumSize.Height;
  return size;
}

//----------------- PanelImpl ---------------------------------------------------------------------

bool PanelImpl::create(::mforms::Panel *self, ::mforms::PanelType panelType)
{
  PanelImpl^ panel= gcnew PanelImpl(self);
  if (panel != nullptr)
  {
    panel->type= panelType;
    Control^ control;
    switch (panel->type)
    {
    case ::mforms::TransparentPanel: // just a container with no background
    case ::mforms::FilledPanel: // just a container with color filled background
        // The background color can be specified separately and does not determine the type of panel we create.
        control= ViewImpl::create<FillPanel>(self, panel);
      break;
    case ::mforms::TitledBoxPanel: // native grouping box with a title with border
    case ::mforms::BorderedPanel: // container with native border
    case ::mforms::TitledGroupPanel: // native grouping container with a title (may have no border) 
      // The title can be set separately and does not determine the type of the group box we create.
      // A control with title but no border is not supported on Windows, so we need to create a special composite.
      // TODO: implement box with title, but no border.
      control= ViewImpl::create<FillGroupBox>(self, panel);
      break;
    case ::mforms::LineBorderPanel: // container with a solid line border
      {
        FillPanel^ native_panel= ViewImpl::create<FillPanel>(self, panel);
        control= native_panel;
        native_panel->BorderStyle= BorderStyle::FixedSingle;
        break;
      }
    case ::mforms::CheckBoxTitlePanel: // container with checkbox
      // TODO: implement CheckBoxTitlePanel
      //break;
    default:
      throw std::logic_error("Internal error: unhandled mforms panel type.");
    }
    control->AutoSize= false;
    return true;
  }
  return false;
}

//-------------------------------------------------------------------------------------------------

void PanelImpl::set_title(::mforms::Panel *self, const std::string &title)
{
  PanelImpl^ panel= (PanelImpl^)ObjectImpl::FromUnmanaged(self);

  if (panel != nullptr)
  {
    panel->set_title(title);
    self->set_layout_dirty(true);
  }
}

//-------------------------------------------------------------------------------------------------

void PanelImpl::set_back_color(::mforms::Panel *self, const std::string &color)
{
  PanelImpl^ panel= (PanelImpl^)ObjectImpl::FromUnmanaged(self);

  if (panel != nullptr)
  {
    panel->set_back_color(color);
  }
}

//-------------------------------------------------------------------------------------------------

void PanelImpl::add(::mforms::Panel *self, ::mforms::View *view)
{
  PanelImpl^ panel= (PanelImpl^)ObjectImpl::FromUnmanaged(self);

  if (panel != nullptr)
  {
    panel->add(view);
    self->set_layout_dirty(true);
  }
}

//-------------------------------------------------------------------------------------------------

void PanelImpl::set_active(::mforms::Panel* self, bool value)
{
  PanelImpl^ panel= (PanelImpl^)ObjectImpl::FromUnmanaged(self);

  if (panel != nullptr)
  {
    panel->set_active(value);
  }
}

//-------------------------------------------------------------------------------------------------

bool PanelImpl::get_active(::mforms::Panel* self)
{
  PanelImpl^ panel= (PanelImpl^)ObjectImpl::FromUnmanaged(self);

  if (panel != nullptr)
    return panel->get_active();

  return false;
}

//-------------------------------------------------------------------------------------------------

void PanelImpl::set_padding(::mforms::Panel* self, int value)
{
  PanelImpl^ panel= (PanelImpl^)ObjectImpl::FromUnmanaged(self);

  if (panel != nullptr)
  {
    panel->set_padding(value);
    self->set_layout_dirty(true);
  }
}

//-------------------------------------------------------------------------------------------------

void PanelImpl::remove(::mforms::Panel *self, ::mforms::View* view)
{
  PanelImpl^ panel= (PanelImpl^)ObjectImpl::FromUnmanaged(self);

  if (panel != nullptr)
  {
    panel->remove(view);
    self->set_layout_dirty(true);
  }
}

//----------------- PanelImpl actual implementation -----------------------------------------------

void PanelImpl::set_title(const std::string &title)
{
  switch (type)
  {
  case ::mforms::TitledBoxPanel:
  case ::mforms::TitledGroupPanel:
  case ::mforms::CheckBoxTitlePanel:
    get_control<Control>()->Text= CppStringToNative(title);
    break;
  }
}

//-------------------------------------------------------------------------------------------------

void PanelImpl::set_back_color(const std::string &color)
{
  switch (type)
  {
    case ::mforms::TitledBoxPanel:
    case ::mforms::BorderedPanel: 
    case ::mforms::TitledGroupPanel:
      get_control<GroupBox>()->BackColor= System::Drawing::ColorTranslator::FromHtml(CppStringToNative(color));
      break;
    default:
      get_control<Panel>()->BackColor= System::Drawing::ColorTranslator::FromHtml(CppStringToNative(color));
  }
}

//-------------------------------------------------------------------------------------------------

void PanelImpl::add(::mforms::View *view)
{
  ViewImpl^ child= (ViewImpl^)ObjectImpl::FromUnmanaged(view);
  Control^ ctl= child->get_control<Control>();

  _child= view;
  Control^ panel_ctl= get_control<Control>();
  panel_ctl->Controls->Add(ctl);
}

//-------------------------------------------------------------------------------------------------

void PanelImpl::set_active(bool value)
{
  // TODO: implement as soon as the checkbox is available
}

//-------------------------------------------------------------------------------------------------

bool PanelImpl::get_active()
{
  // TODO: implement as soon as the checkbox is available
  return false;
}

//-------------------------------------------------------------------------------------------------

void PanelImpl::set_padding(int value)
{
  get_control<Control>()->Padding= Padding(value);
}

//-------------------------------------------------------------------------------------------------

void PanelImpl::remove(::mforms::View* view)
{
  ViewImpl^ child= (ViewImpl^)ObjectImpl::FromUnmanaged(view);
  Control^ ctl= child->get_control<Control>();

  Control^ panel_ctl= get_control<Control>();
  if (_child == view)
    _child= NULL;
  panel_ctl->Controls->Remove(ctl);
}

//-------------------------------------------------------------------------------------------------

void PanelImpl::remove()
{
  get_control<Control>()->Controls->Clear();
}

//-------------------------------------------------------------------------------------------------

/*
 * Helper function to set the cached preferred size after a layout operation.
 */
void PanelImpl::set_preferred_size(Control^ control, const Size% size)
{
  mforms::Panel* panel= ViewImpl::get_backend_control<mforms::Panel>(control);
  PanelImpl^ impl= (PanelImpl^)ObjectImpl::FromUnmanaged(panel);
  switch (impl->type)
  {
    case ::mforms::TitledBoxPanel:
    case ::mforms::BorderedPanel: 
    case ::mforms::TitledGroupPanel:
      impl->get_control<FillGroupBox>()->SetPreferredSize(size);
      break;
    default:
      impl->get_control<FillPanel>()->SetPreferredSize(size);
  }
}

//-------------------------------------------------------------------------------------------------

