/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
 * Implementation of the mforms tabview control for Windows.
 */

#include "stdafx.h"
#include "wf_tabview.h"
#include "wf_panel.h"

using namespace MySQL::Forms;

//--------------------------------------------------------------------------------------------------

TabViewImpl::TabViewImpl(::mforms::TabView *self, bool tabless)
  : ViewImpl(self), _tabless(tabless), _active_index(-1)
{
}

//--------------------------------------------------------------------------------------------------

bool TabViewImpl::create(::mforms::TabView *self, bool tabless)
{
  TabViewImpl ^tabview= gcnew TabViewImpl(self, tabless);

  if (tabview != nullptr)
  {
    if (tabless)
      ViewImpl::create<FillPanel>(self, tabview);
    else
    {
      TabControl^ control= ViewImpl::create<TabControl>(self, tabview);

      // Listen to tab changes by the user.
      control->Selected += gcnew TabControlEventHandler(&TabViewImpl::tab_changed);
    }
    return true;
  }
  return false;
}

//--------------------------------------------------------------------------------------------------

void TabViewImpl::set_active_tab(::mforms::TabView *self, int index)
{
  TabViewImpl^ tabview= (TabViewImpl^)ObjectImpl::FromUnmanaged(self);

  if (tabview != nullptr)
  {
    if (tabview->_tabless)
    {
      if (index < 0)
        index= 0;
      if (index > tabview->_pages.Count - 1)
        index= tabview->_pages.Count - 1;

      if (tabview->_active_index != index)
      {
        tabview->_active_index= index;

        FillPanel^ content_container= tabview->get_control<FillPanel>();
        content_container->Controls->Clear();

        Control^ page_container= tabview->_pages[index]->get_control<Control>();
        content_container->Controls->Add(page_container);
        page_container->Dock= DockStyle::Fill;
      }
    }
    else
      tabview->get_control<TabControl>()->SelectedIndex= index;

    self->set_layout_dirty(true);
  }
}

//--------------------------------------------------------------------------------------------------

int TabViewImpl::get_active_tab(::mforms::TabView *self)
{
  TabViewImpl^ tabview= (TabViewImpl^)ObjectImpl::FromUnmanaged(self);

  if (tabview != nullptr)
  {
    if (tabview->_tabless)
      return tabview->_active_index;
    else
      return tabview->get_control<TabControl>()->SelectedIndex;
  }
  return -1;
}

//--------------------------------------------------------------------------------------------------

int TabViewImpl::add_page(::mforms::TabView *self, ::mforms::View *page, const std::string& caption)
{
  TabViewImpl^ tabview= (TabViewImpl^) ObjectImpl::FromUnmanaged(self);

  int new_index= -1;

  if (tabview != nullptr)
  {
    ViewImpl^ view= (ViewImpl^)ObjectImpl::FromUnmanaged(page);
    page->set_resize_mode(::mforms::ResizeNone);

    if (tabview->_tabless)
    {
      ViewImpl^ view= (ViewImpl^) ObjectImpl::FromUnmanaged(page);
      tabview->_pages.Add(view);

      new_index= tabview->_pages.Count - 1;
    }
    else
    {
      TabPage^ tabPage= gcnew TabPage();

      Control^ control= view->get_control<Control>();
      control->Dock = DockStyle::Fill;
      tabPage->Controls->Add(control);
      tabPage->Text= CppStringToNative(caption);

      TabControl^ tabControl= tabview->get_control<TabControl>();
      tabControl->TabPages->Add(tabPage);

      new_index= tabControl->TabPages->Count - 1;
    }

    // Set the page as active page if this is the first page that was added.
    if (new_index == 0)
      set_active_tab(self, new_index);

    self->set_layout_dirty(true);
  }
  return new_index;
}

//--------------------------------------------------------------------------------------------------

void TabViewImpl::tab_changed(System::Object^ sender, TabControlEventArgs^ e)
{
  Windows::Forms::TabControl^ control= (Windows::Forms::TabControl^) sender;

  if (control->Tag != nullptr)
  {
    ::mforms::TabView* tabview= ViewImpl::get_backend_control<::mforms::TabView>(control);
    if (tabview != NULL)
      tabview->signal_tab_changed().emit();
  }
}

//--------------------------------------------------------------------------------------------------

