/* 
 * Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

/**
 * Windows front end implementation of the mforms Utilities APIs.
 */

#include "stdafx.h"
#include "wf_Utilities.h"

using namespace Windows::Forms;
using namespace System::Text;
using namespace System::Drawing;
using namespace System::Threading;

using namespace MySQL::Forms;
using namespace MySQL::Utilities;

//--------------------------------------------------------------------------------------------------

CustomMessageBox::CustomMessageBox()
{
  _picture= gcnew Windows::Forms::PictureBox();
  _button1= gcnew Windows::Forms::Button();
  _button2= gcnew Windows::Forms::Button();
  _button3= gcnew Windows::Forms::Button();
  _messageLabel= gcnew Windows::Forms::Label();

  SuspendLayout();

  Controls->Add(_picture);
  Controls->Add(_button2);
  Controls->Add(_button1);
  Controls->Add(_button3);
  Controls->Add(_messageLabel);

  // picture
  _picture->Name= "picture";
  _picture->TabIndex= 0;
  _picture->TabStop= false;

  // button1
  _button1->Name= "button1";
  _button1->TabIndex= 1;
  _button1->UseVisualStyleBackColor= true;
  _button1->Click += gcnew System::EventHandler(this, &CustomMessageBox::ButtonClick);

  // button2
  _button2->Name= "button2";
  _button2->TabIndex= 2;
  _button2->UseVisualStyleBackColor= true;
  _button2->Click += gcnew System::EventHandler(this, &CustomMessageBox::ButtonClick);

  // button3
  _button3->Name= "button3";
  _button3->TabIndex= 3;
  _button3->UseVisualStyleBackColor= true;
  _button3->Click += gcnew System::EventHandler(this, &CustomMessageBox::ButtonClick);

  // messageText
  _messageLabel->Name= "messageLabel";
  _messageLabel->Padding= Windows::Forms::Padding(2);
  _messageLabel->TabIndex= 4;
  _messageLabel->Text= "label1";

  // Form
  this->Padding= Windows::Forms::Padding(8, 16, 8, 8);
  this->FormBorderStyle= ::FormBorderStyle::FixedDialog;
  this->AutoScaleDimensions= System::Drawing::SizeF(6, 13);
  this->AutoScaleMode= Windows::Forms::AutoScaleMode::Font;
  this->StartPosition = FormStartPosition::CenterScreen;
  this->Name= "MessageBoxEx";
  this->Text= "";
  this->ShowInTaskbar= false;
  this->HelpButton= false;
  this->MinimizeBox= false;
  this->MaximizeBox= false;
  this->TopMost= true;

  ResumeLayout(false);
}

//--------------------------------------------------------------------------------------------------

/// <summary>
/// Displays the CustomMessageBox and returns a dialog result, depending on the button clicked.
/// </summary>
mforms::DialogResult CustomMessageBox::Show(String^ title, String^ text, Image^ image, String^ button1, String^ button2, String^ button3)
{
  CustomMessageBox^ box= gcnew CustomMessageBox();
  
  box->_picture->Image= image; 
  box->_picture->Size= image->Size;
  box->Text= title;
  box->_messageLabel->Text= text;
  box->_button1->Text= button1;
  box->_button1->DialogResult = Windows::Forms::DialogResult::OK;
  box->_button2->Text= button2;
  box->_button2->DialogResult = Windows::Forms::DialogResult::Cancel;
  box->_button3->Text= button3;
  box->_button3->DialogResult = Windows::Forms::DialogResult::Ignore;
  box->ComputeLayout();

  box->AcceptButton = box->_button1;
  box->CancelButton = box->_button2;

  Windows::Forms::DialogResult rc = box->ShowDialog();

  switch (rc)
  {
  case Windows::Forms::DialogResult::OK:
    return mforms::ResultOk;

  default:
  case Windows::Forms::DialogResult::Cancel:
    return mforms::ResultCancel;

  case Windows::Forms::DialogResult::Ignore:
    return mforms::ResultOther;
  }
}

//--------------------------------------------------------------------------------------------------

// A little helper for repeating a button setup.
void AdjustButton(Windows::Forms::Button^ button)
{
  // Note: we need to set the Enabled property too as indicator if this button should be considered
  // when layouting the form. The Visible property cannot be used as long as the button's parent
  // is still hidden (which is the case when we do the layout process).
  if (button->Text == "")
  {
    button->Visible= false;
    button->Enabled= false;
  }
  else
  {
    Size size= button->GetPreferredSize(Drawing::Size::Empty);
    if (size.Width < 75)
      size.Width= 75;
    button->Size= size;
    button->Visible= true;
    button->Enabled= true;
  }
}

//--------------------------------------------------------------------------------------------------

#define MESSAGE_BOX_BUTTON_SPACING 8

/**
 * As the name already says this function computes the layout of the message box depending on the 
 * content (image size, button text etc.).
 */
void CustomMessageBox::ComputeLayout()
{
  SuspendLayout();

  // Buttons have a minimum width of 75 px. Make them all same size and hide those which don't have
  // any text.
  AdjustButton(_button1);
  AdjustButton(_button2);
  AdjustButton(_button3);

  int visibleButtonCount= 0;
  Drawing::Size size= Drawing::Size::Empty;
  if (_button1->Enabled)
  {
    size= _button1->Size;
    visibleButtonCount++;
  }
  if (_button2->Enabled)
  {
    size.Height= _button2->Height;
    if (_button2->Width > size.Width)
      size.Width= _button2->Width;
    visibleButtonCount++;
  }
  if (_button3->Enabled)
  {
    size.Height= _button3->Height;
    if (_button3->Width > size.Width)
      size.Width= _button3->Width;
    visibleButtonCount++;
  }

  // For the common height we use the one from the last visible button we found.
  // Since the font is assumed to be the same for all buttons they also should compute
  // the same height when asked for their preferred size.
  // Compute the total size of the button area on the way (including padding of the container).
  Drawing::Size buttonSize= Drawing::Size::Empty;
  if (size.Width > 0)
  {
    // Compute only if there is at least one visible button.
    buttonSize.Height= size.Height;
    if (_button1->Enabled)
    {
      _button1->Size= size;
      buttonSize.Width= size.Width;
    }
    if (_button2->Enabled)
    {
      _button2->Size= size;
      if (buttonSize.Width > 0)
        buttonSize.Width += MESSAGE_BOX_BUTTON_SPACING;
      buttonSize.Width += size.Width;
    }
    if (_button3->Enabled)
    {
      _button3->Size= size;
      if (buttonSize.Width > 0)
        buttonSize.Width += MESSAGE_BOX_BUTTON_SPACING;
      buttonSize.Width += size.Width;
    }
  }

  // Compute message text layout. Start with a minimum width of 400 pixels.
  int minWidth= (buttonSize.Width > 400) ? buttonSize.Width : 400;
  Drawing::Size proposedSize= Drawing::Size(minWidth, 1);
  
  // We use the golden ratio to create an appealing layout.
  // Increase width in 10% steps until we found the proper ratio.
  int lastWidth= 0;
  while (true)
  {
    size= _messageLabel->GetPreferredSize(proposedSize);

    // Terminate loop if we get a height of 0, the same width as in the last run, beyond a maximum width
    // or reached the golden ratio. Getting the same width means we never can get to the golden
    // ratio as the text is too small.
    if ((size.Height == 0) || (size.Width == lastWidth) || 
      (size.Width >= 1000) || (size.Width / size.Height >= 1.618))
      break;

    lastWidth= size.Width;
    proposedSize.Width += proposedSize.Width / 10;
  };

  // GetPreferredSize might return a value smaller than our minimum width. Account for that.
  if (size.Width < minWidth)
    size.Width= minWidth;
  _messageLabel->Size= size;

  // Now that we have the text size compute the overall size of the message box.
  // The image is vertically centered at the left side (with some padding)
  // and the buttons are at the bottom centered.
  int textHeight= _messageLabel->Padding.Vertical + _messageLabel->Height;
  if (textHeight < _picture->Height + _picture->Padding.Vertical)
    textHeight= _picture->Height + _picture->Padding.Vertical;
  size.Width= Padding.Horizontal + _picture->Padding.Horizontal + _picture->Width +
    _messageLabel->Padding.Horizontal + _messageLabel->Width;
  size.Height= Padding.Vertical + textHeight + buttonSize.Height;
  ClientSize= size;

  // Move picture to its final location (center vertically over the message text's height).
  Point location;
  location.X= Padding.Left;
  location.Y= Padding.Top + (textHeight - _picture->Height - _picture->Padding.Vertical) / 2;
  _picture->Location= location;

  // Text location too.
  location.X= _picture->Right + _messageLabel->Padding.Left;
  location.Y= Padding.Top;
  _messageLabel->Location= location;

  // Move the buttons to their final locations. Center them horizontally.
  if (buttonSize.Width > 0)
  {
    location= Point((ClientSize.Width - buttonSize.Width) / 2, ClientSize.Height - buttonSize.Height - Padding.Bottom);
    if (_button1->Enabled)
    {
      _button1->Location= location;
      location.X += _button1->Width + MESSAGE_BOX_BUTTON_SPACING;
    }
    if (_button2->Enabled)
    {
      _button2->Location= location;
      location.X += _button2->Width + MESSAGE_BOX_BUTTON_SPACING;
    }
    if (_button3->Enabled)
      _button3->Location= location;
  }

  ResumeLayout(true);
}

//--------------------------------------------------------------------------------------------------

void CustomMessageBox::ButtonClick(Object^ sender, EventArgs^ arguments)
{
  DialogResult = ((Windows::Forms::Button^)sender)->DialogResult;

  //this->Close();
}

//----------------- UtilitiesImpl ------------------------------------------------------------------

UtilitiesImpl::UtilitiesImpl()
{
}

//--------------------------------------------------------------------------------------------------

int UtilitiesImpl::show_message(const string &title, const string &text, const string &ok, 
  const string &cancel, const string &other)
{
  String^ boxTitle= CppStringToNative(title);
  String^ boxText= CppStringToNative(text);
  String^ button1= CppStringToNative(ok);
  String^ button2= CppStringToNative(cancel);
  String^ button3= CppStringToNative(other);
  Image^ image= Image::FromFile("images/ui/message_confirm.png");
  mforms::Utilities::enter_modal_loop();
  int i = CustomMessageBox::Show(boxTitle, boxText, image, button1, button2, button3);
  mforms::Utilities::leave_modal_loop();
  return i;
}

//--------------------------------------------------------------------------------------------------

int UtilitiesImpl::show_error(const string &title, const string &text, const string &ok, 
  const string &cancel, const string &other)
{
  String^ boxTitle= CppStringToNative(title);
  String^ boxText= CppStringToNative(text);
  String^ button1= CppStringToNative(ok);
  String^ button2= CppStringToNative(cancel);
  String^ button3= CppStringToNative(other);
  Image^ image= Image::FromFile("images/ui/message_error.png");
  mforms::Utilities::enter_modal_loop();
  int i = CustomMessageBox::Show(boxTitle, boxText, image, button1, button2, button3);
  mforms::Utilities::leave_modal_loop();

  return i;
}

//--------------------------------------------------------------------------------------------------

int UtilitiesImpl::show_warning(const string &title, const string &text, const string &ok, 
  const string &cancel, const string &other)
{
  String^ boxTitle= CppStringToNative(title);
  String^ boxText= CppStringToNative(text);
  String^ button1= CppStringToNative(ok);
  String^ button2= CppStringToNative(cancel);
  String^ button3= CppStringToNative(other);
  Image^ image= Image::FromFile("images/ui/message_warning.png");

  mforms::Utilities::enter_modal_loop();
  int i = CustomMessageBox::Show(boxTitle, boxText, image, button1, button2, button3);
  mforms::Utilities::leave_modal_loop();

  return i;
}

//--------------------------------------------------------------------------------------------------

/**
 * Shows the warning heads-up-display with the given title and text.
 */
void UtilitiesImpl::show_wait_message(const string &title, const string &text)
{
  if (Thread::CurrentThread->Name != "mainthread") // This name was set in Program.cs.
    throw std::runtime_error("Internal error: wait box creation must be done on the main thread.");

  HUDForm::Instance->Show(CppStringToNative(title), CppStringToNative(text), true);
}

//--------------------------------------------------------------------------------------------------

/**
 * Hides a previously shown wait message.
 */
bool UtilitiesImpl::hide_wait_message()
{
  bool result= HUDForm::Instance->Visible;
  HUDForm::Instance->Hide();
  return result;
}

//--------------------------------------------------------------------------------------------------

/**
 * Places the given string on the clipboard.
 * 
 * @param content The text to be placed on the clipboard. It is assume its encoding is UTF-8.
 */
void UtilitiesImpl::set_clipboard_text(const string &content)
{
  if (!content.empty())
    Clipboard::SetText(CppStringToNative(content));
}

//--------------------------------------------------------------------------------------------------

/**
 * Returns the current text on the clipboard, if there is any.
 * 
 * @result If there is text on the clipboard (ANSI or Unicode) it is returned as UTF-8 string.
 * @note The returned text gets all CRLF Windows line breaks converted to pure LF.
 */
string UtilitiesImpl::get_clipboard_text()
{
  String^ unicode= (String^) Clipboard::GetData(DataFormats::UnicodeText);
  if (unicode == nullptr)
    return "";

  return NativeToCppString(unicode);
}

//--------------------------------------------------------------------------------------------------

/**
 * Returns platform specific user folders, e.g. for the desktop, the user's documents etc.
 */
string UtilitiesImpl::get_special_folder(FolderType type)
{
  return NativeToCppString(Environment::GetFolderPath(Environment::SpecialFolder(type)));
}

//--------------------------------------------------------------------------------------------------

void UtilitiesImpl::open_url(const string &url)
{
  try
  {
    System::Diagnostics::Process::Start(CppStringToNative(url));
  }
  catch (Exception ^e)
  {
    MessageBox::Show(e->Message->ToString(), "Error Opening Browser", 
      MessageBoxButtons::OK, MessageBoxIcon::Error, MessageBoxDefaultButton::Button1);
  }
}

//--------------------------------------------------------------------------------------------------

// The password cache is a temporary storage and only used for a short time frame when looking up a password.
static GStaticMutex password_mutex= G_STATIC_MUTEX_INIT;
static std::map<std::string, std::string> password_cache;
typedef std::map<std::string, std::string>::iterator PasswordIterator;

#define DOMAIN_SEPARATOR (char) 2
#define PASSWORD_SEPARATOR (char) 3

/**
 * Loads encrypted passwords from disk. These are typically hold only for a short moment.
 */
void UtilitiesImpl::load_passwords()
{
  g_static_mutex_lock(&password_mutex);

  try
  {
    // Load password cache from disk. Don't throw an error if the cache file doesn't exist yet, though.
    std::string file= get_special_folder(ApplicationData) + "/MySQL/Workbench/workbench_user_data.dat";
    if (g_file_test(file.c_str(), G_FILE_TEST_EXISTS))
    {
      gchar* content;
      gsize length;
      GError* error= NULL;
      bool result= g_file_get_contents(file.c_str(), &content, &length, &error);
      if (!result)
        throw std::runtime_error("Password management error: " + std::string(error->message));

      DATA_BLOB data_in;
      DATA_BLOB data_out;

      data_in.pbData= (BYTE*) content;
      data_in.cbData= length;
      result= CryptUnprotectData(&data_in, NULL, NULL, NULL, NULL, CRYPTPROTECT_UI_FORBIDDEN, &data_out);
      
      g_free(content);

      // Split the string into individual items and fill the password cache with them.
      std::stringstream ss((char*) data_out.pbData);
      std::string item;
      while (std::getline(ss, item, '\n'))
      {
        string::size_type pos= item.find_first_of(PASSWORD_SEPARATOR, 0);

        if (string::npos != pos)
          password_cache[item.substr(0, pos)]= item.substr(pos + 1, item.length());
      }
      LocalFree(data_out.pbData);
    }
  }
  finally
  {
    g_static_mutex_unlock(&password_mutex);
  }
}

//--------------------------------------------------------------------------------------------------

/**
 * Saves the password cache to disk (if store is true) and clears it so passwords aren't kept in
 * memory any longer than necessary.
 */
void UtilitiesImpl::unload_passwords(bool store)
{
  g_static_mutex_lock(&password_mutex);

  try
  {
    // Store all passwords in a string for encryption.
    if (store)
    {
      std::string plain_data;
      for (PasswordIterator iterator= password_cache.begin(); iterator != password_cache.end(); iterator++)
        plain_data += iterator->first + PASSWORD_SEPARATOR + iterator->second + "\n";

      DATA_BLOB data_in;
      DATA_BLOB data_out;

      data_in.pbData= (BYTE*) plain_data.c_str();
      data_in.cbData= (DWORD) plain_data.length() + 1;
      
      if (!CryptProtectData(&data_in, NULL, NULL, NULL, NULL, CRYPTPROTECT_UI_FORBIDDEN, &data_out))
        throw std::runtime_error("Password management error: could not encrypt password cache.");

      // Now write the encrypted data to file.
      std::string file= get_special_folder(ApplicationData) + "/MySQL/Workbench/workbench_user_data.dat";
      GError* error= NULL;
      bool result= g_file_set_contents(file.c_str(), (gchar*) data_out.pbData, data_out.cbData, &error);

      LocalFree(data_out.pbData);

      if (!result)
      {
        std::string message= error->message != NULL ? error->message : "could not store passwort file.";
        throw std::runtime_error("Password management error: " + message);
      }
    }
    password_cache.clear();
  }
  finally
  {
    g_static_mutex_unlock(&password_mutex);

  }
}

//--------------------------------------------------------------------------------------------------

/**
 * This function stores the given password in our password file, parameterized by the given service
 * and user name. The file is encrypted by the system for safety and can only be decrypted by the
 * same user who encrypted it.
 */
void UtilitiesImpl::store_password(const std::string &service, const std::string &account, const std::string &password)
{
  load_passwords();

  // Keep the password in our password cache and write the entire cache to file after that.
  password_cache[service + DOMAIN_SEPARATOR + account]= password;

  unload_passwords(true);
}

//--------------------------------------------------------------------------------------------------

/**
 * Return the plain text password for the given service and account.
 */
bool UtilitiesImpl::find_password(const std::string &service, const std::string &account, std::string &password)
{
  load_passwords();

  bool result= true;
  PasswordIterator iterator= password_cache.find(service + DOMAIN_SEPARATOR + account);
  if (iterator == password_cache.end())
    result= false;
  else
    password= iterator->second;

  unload_passwords(false);
  return result;
}

//--------------------------------------------------------------------------------------------------

/**
 * Remove the password for the given service and account if there is one.
 */
void UtilitiesImpl::forget_password(const std::string &service, const std::string &account)
{
  load_passwords();

  PasswordIterator iterator= password_cache.find(service + DOMAIN_SEPARATOR + account);
  if (iterator != password_cache.end())
  {
    password_cache.erase(iterator);
    unload_passwords(true);
  }
  else
    unload_passwords(false);
}

//--------------------------------------------------------------------------------------------------

public ref class TimerHandler
{
public:
  TimerHandler(float interval, const sigc::slot<bool> &slot)
  {
    _timer = gcnew System::Windows::Forms::Timer();

    _slot = new sigc::slot<bool>(slot);

    _timer->Interval = (int) (interval * 1000);
    _timer->Tick += gcnew EventHandler(this, &TimerHandler::timer_tick);
    _timer->Start();
  }

  ~TimerHandler()
  {
    _timer->Stop();
    delete _timer;
    delete _slot;
  }

private:
  sigc::slot<bool> *_slot;
  System::Windows::Forms::Timer ^_timer;

  void timer_tick(Object^ sender, System::EventArgs ^e)
  {
    // emulate behaviour in MacOS X (timers arent fired when in modal loops)
    // also works around a deadlock of python when a timer is fired while inside a modal loop
    if (!mforms::Utilities::in_modal_loop())
    {
      _timer->Stop();

      // if callback returns true, then restart the timer
      if ((*_slot)())
        _timer->Enabled = true;
      else
        delete this;
    }
  }
};


void UtilitiesImpl::add_timeout(float interval, const sigc::slot<bool> &slot)
{
  TimerHandler ^handler = gcnew TimerHandler(interval, slot);
}
