/* 
 * Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "stdafx.h"

#include "mysql_sql_specifics.h"
#include "mysql_sql_parser_fe.h"
#include "grtdb/charset_utils.h"
#include "string_utilities.h"

#include <sstream>

using namespace grt;
using namespace base;

#define NULL_STATE_KEEPER Null_state_keeper _nsk(this);


class MYSQL_SQL_PARSER_PUBLIC_FUNC Mysql_sql_statement_info : protected Mysql_sql_parser_base
{
public:
  Mysql_sql_statement_info(grt::GRT *grt) : Sql_parser_base(grt), Mysql_sql_parser_base(grt) { NULL_STATE_KEEPER }
  virtual ~Mysql_sql_statement_info() {}

private:
  int *_row_count;
  int *_row_offset;
  bool *_contains_limit_clause;
  bool _statement_valid;
public:
  bool get_limit_clause_params(const std::string &sql, int &row_count, int &row_row_offset, bool &contains_limit_clause)
  {
    NULL_STATE_KEEPER

    _row_count= &row_count;
    _row_offset= &row_row_offset;
    _contains_limit_clause= &contains_limit_clause;
    _statement_valid= false;

    _process_sql_statement= sigc::mem_fun(this, &Mysql_sql_statement_info::process_sql_statement);

    Mysql_sql_parser_fe sql_parser_fe(_grtm->get_grt());
    sql_parser_fe.ignore_dml= false;
    Mysql_sql_parser_base::parse_sql_script(sql_parser_fe, sql.c_str());

    return _statement_valid;
  }

protected:
  int process_sql_statement(const SqlAstNode *tree)
  {
    if (tree)
    {
      _statement_valid= true;

      if (const SqlAstNode *item= tree->subitem(sql::_statement, sql::_select))
        process_select_statement(item);
    }

    return 0; // error count
  }

  Parse_result process_select_statement(const SqlAstNode *tree)
  {
    static sql::symbol path1[]= { sql::_select_init, sql::_select_init2, sql::_select_part2, sql::_select_into, sql::_select_from, sql::_opt_limit_clause, sql::_limit_clause, sql::_ };
    static sql::symbol path2[]= { sql::_select_init, sql::_select_init2, sql::_select_part2, sql::_select_into, sql::_opt_limit_clause, sql::_limit_clause, sql::_ };
    static sql::symbol path3[]= { sql::_select_init, sql::_union_opt, sql::_union_order_or_limit, sql::_order_or_limit, sql::_opt_limit_clause_init, sql::_limit_clause, sql::_ };
    static sql::symbol path4[]= { sql::_select_init, sql::_union_opt, sql::_union_order_or_limit, sql::_order_or_limit, sql::_limit_clause, sql::_ };
    static sql::symbol * paths[]= { path1, path2, path3, path4 };

    const SqlAstNode *limit_clause= tree->search_by_paths(paths, ARR_CAPACITY(paths));
    if (limit_clause)
    {
      const SqlAstNode *limit_options= limit_clause->subitem(sql::_limit_options);
      
      const SqlAstNode *limit_row_offset= *limit_options->subitems()->begin();
      const SqlAstNode *limit_rowcount= *limit_options->subitems()->rbegin();
      if (limit_rowcount == limit_row_offset)
        limit_row_offset= NULL;
      else if (limit_clause->subitem(sql::_OFFSET_SYM))
        std::swap(limit_rowcount, limit_row_offset);

      if (limit_row_offset)
      {
        std::stringstream ss;
        ss << limit_row_offset->restore_sql_text(_sql_statement);
        ss >> *_row_offset;
      }
      else
        *_row_offset= 0;

      {
        std::stringstream ss;
        ss << limit_rowcount->restore_sql_text(_sql_statement);
        ss >> *_row_count;
      }
    }

    *_contains_limit_clause= (limit_clause != NULL);

    return pr_processed;
  }
};



Mysql_sql_specifics::Mysql_sql_specifics(grt::GRT *grt)
:
Sql_specifics(grt)
{
}


std::string Mysql_sql_specifics::limit_select_query(const std::string &sql, int &row_count, int &row_row_offset)
{
  Mysql_sql_statement_info statement_info(_grt);
  bool contains_limit_clause;
  if (statement_info.get_limit_clause_params(sql, row_count, row_row_offset, contains_limit_clause))
  {
    if (!contains_limit_clause)
    {
      std::ostringstream oss;
      oss << sql << std::endl << "LIMIT " << row_row_offset << ", " << row_count;
      return oss.str();
    }
  }
  return sql;
}


void Mysql_sql_specifics::get_connection_startup_script(std::list<std::string> &sql_script)
{
  sql_script.push_back("SET CHARACTER SET utf8");
  sql_script.push_back("SET NAMES utf8");
}


std::string Mysql_sql_specifics::query_connection_id()
{
  return "SELECT CONNECTION_ID()";
}


std::string Mysql_sql_specifics::query_kill_connection(int connection_id)
{
  return strfmt("KILL CONNECTION %i", connection_id);
}


sqlide::QuoteVar::Escape_sql_string Mysql_sql_specifics::escape_sql_string()
{
  bool ansi_sql_strings= false;

  grt::ValueRef sql_mode_value= bec::GRTManager::get_instance_for(_grt)->get_app_option("SqlMode");
  if (sql_mode_value.is_valid() && grt::StringRef::can_wrap(sql_mode_value))
  {
    std::string sql_mode_string= toupper(grt::StringRef::cast_from(sql_mode_value));
    std::istringstream iss(sql_mode_string);
    std::string mode;
    while (std::getline(iss, mode, ','))
    {
      if (mode == "NO_BACKSLASH_ESCAPES")
      {
        ansi_sql_strings= true;
        break;
      }
    }
  }

  return (ansi_sql_strings) ? &sqlide::QuoteVar::escape_ansi_sql_string : &sqlide::QuoteVar::escape_c_string;
}


std::string Mysql_sql_specifics::setting_non_std_sql_delimiter()
{
  return "delimiter " + non_std_sql_delimiter() + "\n";
}


std::string Mysql_sql_specifics::non_std_sql_delimiter()
{
  DictRef options= DictRef::cast_from(_grt->get("/wb/options/options"));
  if (!options.is_valid())
    return "$$";
  return options.get_string("SqlDelimiter", "$$");
}
