#include <qapplication.h>
#include <qdatetime.h>

#include <unistd.h>
#include <iostream>
#include <sys/types.h>
#include <sys/stat.h>
using namespace std;

#include "filetransfer.h"

#include "RingBuffer.h"
#include "libmyth/util.h"

FileTransfer::FileTransfer(QString &filename, QSocket *remote)
{
    rbuffer = new RingBuffer(filename, false);
    pthread_mutex_init(&readthreadLock, NULL);
    sock = remote;
    readthreadlive = false;
    readrequest = 0;
    ateof = false;

    pthread_create(&readthread, NULL, FTReadThread, this);

    while (!readthreadlive)
        usleep(50);
}

FileTransfer::~FileTransfer()
{
    Stop();

    if (rbuffer)
        delete rbuffer;
}

bool FileTransfer::isOpen(void)
{
    if (rbuffer && rbuffer->IsOpen())
        return true;
    return false;
}

void FileTransfer::Stop(void)
{
    if (readthreadlive)
    {
        readthreadlive = false;
        rbuffer->StopReads();
        pthread_join(readthread, NULL);
    }
}

void FileTransfer::Pause(void)
{
    rbuffer->StopReads();
    pthread_mutex_lock(&readthreadLock);
}

void FileTransfer::Unpause(void)
{
    rbuffer->StartReads();
    pthread_mutex_unlock(&readthreadLock);
}

bool FileTransfer::RequestBlock(int size)
{
    if (!readthreadlive)
        return true;

    if (size > 256000)
        size = 256000;

    bool locked = false;
    QDateTime curtime = QDateTime::currentDateTime();
    curtime = curtime.addSecs(15);

    while (QDateTime::currentDateTime() < curtime)
    {
        locked = pthread_mutex_trylock(&readthreadLock);
        if (locked)
            break;
        usleep(50);
    }

    if (!locked)
    {
        cerr << "Backend stopped in RequestBlock\n";
        rbuffer->StopReads();
        return true;
    }

    readrequest = size;
    pthread_mutex_unlock(&readthreadLock);

    curtime = QDateTime::currentDateTime();
    curtime = curtime.addSecs(15);

    while (readrequest > 0 && readthreadlive && !ateof)
    {
        usleep(100);

        if (QDateTime::currentDateTime() > curtime)
        {
            cerr << "Backend stuffed up in RequestBlock\n";
            rbuffer->StopReads();
            break;
        }
    }

    return ateof;
}

long long FileTransfer::Seek(long long curpos, long long pos, int whence)
{
    if (!rbuffer)
        return -1;
    if (!readthreadlive)
        return -1;

    ateof = false;

    Pause();

    if (whence == SEEK_CUR)
    {
        long long desired = curpos + pos;
        long long realpos = rbuffer->GetReadPosition();

        pos = desired - realpos;
    }

    long long ret = rbuffer->Seek(pos, whence);

    Unpause();
    return ret;
}

void FileTransfer::DoFTReadThread(void)
{
    char *buffer = new char[256001];

    readthreadlive = true;
    while (readthreadlive && rbuffer)
    {
        while (readrequest == 0 && readthreadlive)
            usleep(5000);

        if (!readthreadlive)
            break;

        pthread_mutex_lock(&readthreadLock);

        int ret = rbuffer->Read(buffer, readrequest);
        if (!rbuffer->GetStopReads())
        {
            if (ret)
                WriteBlock(sock, buffer, ret);
            else
            {
                ateof = true;
                readrequest = 0;
            }
        }        

        readrequest -= ret;

        pthread_mutex_unlock(&readthreadLock);
    }

    delete [] buffer;
}

void *FileTransfer::FTReadThread(void *param)
{
    FileTransfer *ft = (FileTransfer *)param;
    ft->DoFTReadThread();

    return NULL;
}

long long FileTransfer::GetFileSize(void)
{
    QString filename = rbuffer->GetFilename();

    struct stat st;
    long long size = 0;

    if (stat(filename.ascii(), &st) == 0)
        size = st.st_size;

    return size;
}
