/*
 * dct-channel
 * Copyright (c) 2003 Jim Paris <jim@jtan.com>
 *
 * This is free software; you can redistribute it and/or modify it and
 * it is provided under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation; see COPYING.
 */

#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "crc.h"
#include "serial.h"
#include "packet.h"
#include "debug.h"
#include "remote.h"
#include "opt.h"
#include "version.h"

struct options opt[] = {
	{ 'p', "port", "port", "serial port (default: /dev/ttyS0)" },
	{ 'q', "quiet", NULL, "suppress printing of channel on exit" },
	{ 'v', "verbose", NULL, "be verbose (use twice for extra debugging)" },
	{ 't', "timeout", "scale", "scale all timeouts by this much (default 1.0)" },
	{ 'h', "help", NULL, "this help" },
	{ 'V', "version", NULL, "show program version and exit" },
	{ 0, NULL, NULL, NULL }
};	

int verb_count = 0;

/* Timeouts, in msec. */
double timeout_scale = 1.0;

#define TIMEOUT_TINY (int)(timeout_scale*250)	/* Waiting for extra channel status */
#define TIMEOUT_SHORT (int)(timeout_scale*1500)	/* For most messages */
#define TIMEOUT_LONG (int)(timeout_scale*5000) 	/* Waiting for channel to change */
#define KEY_SLEEP (int)(timeout_scale*200)	/* How long to wait after sending keypress */
#define POWER_SLEEP (int)(timeout_scale*6000)	/* How long to wait for box to power on */

int send_and_get_ack(packet *p)
{
	packet s;
	int ret=0;
	serial_sendpacket(p);
	if(serial_getpacket(&s,TIMEOUT_SHORT)<0) {
		verb("No response to packet\n");
		ret=-1;
	} else if(!packet_is_ack(p,&s)) {
		verb("Got response, but it's not an ACK\n");
		ret=-1;
	}
	return ret;
}

int initialize(void)
{
	packet p;
	verb("Attempting to initialize DCT\n");

	serial_flush();

	/* Send initialize_1 and get ack */
	packet_build_initialize_1(&p);
	if(send_and_get_ack(&p)<0)
		return -1;

	packet_build_initialize_2(&p);
	if(send_and_get_ack(&p)<0)
		return -1;

	/* Now the DCT should send us a status message, so wait
	   for it (and automatically acknowledge it) */
	if(serial_getpacket(&p,TIMEOUT_SHORT)<0) {
		verb("Didn't get expected status message\n");
		return -1;
	}
	return 0;
}

int get_channel(void)
{
	int chan;
	packet p;

	verb("Reading current channel\n");

	/* Send channelquery and get ack */
	packet_build_channelquery(&p);
	if(send_and_get_ack(&p)<0)
		return -1;
	
	/* Now we expect one more packet with the 
	   channel status */
	if(serial_getpacket(&p,TIMEOUT_SHORT)<0) {
		verb("Didn't get channel status message\n");
		return -1;
	}

	if((chan=packet_extract_channel(&p))<0) {
		verb("Returned packet isn't channel status!\n");
		return -1;
	}

	return chan;
}

int send_keypress(int key)
{
	packet p;

	packet_build_keypress(&p,key);
	if(send_and_get_ack(&p)<0)
		return -1;

	/* After sending a keypress, we have to wait. 
	   0.1 sec is not quite enough; 0.2 sec seems to be */
	usleep(KEY_SLEEP * 1000);
	return 0;
}

int set_channel(int chan)
{
	packet p;
	int cur;

	if((send_keypress(KEY_0+((chan/100)%10)))<0 ||
	   (send_keypress(KEY_0+((chan/10)%10)))<0 ||
	   (send_keypress(KEY_0+((chan/1)%10)))<0) {
		verb("Error sending channel keypresses\n");
		return -1;
	}
	
	/* Now the DCT should send us channel status */
	if(serial_getpacket(&p,TIMEOUT_LONG)<0) {
		verb("Didn't get channel status message\n");
		return -1;
	}
	if((cur=packet_extract_channel(&p))<0) {
		verb("Returned packet isn't channel status!\n");
		return -1;
	}

	/* The DCT may send more channel status messages, if it
	   figures out something new about the signal.  Wait
	   for these, but don't wait long. */
	while(serial_getpacket(&p,TIMEOUT_TINY)>=0) {
		int newcur;
		if((newcur=packet_extract_channel(&p))>=0) {
			debug("Got extra status back\n");
			cur = newcur;
		}
	}

	/* Send EXIT to clear the OSD, but don't
	   care if this doesn't work */
	send_keypress(KEY_EXIT);

	return cur;
}

int main(int argc, char *argv[])
{
	int optind;
	char *optarg;
	char c;
	FILE *help = stderr;
	int tries=0;
	int chan=0;
	int cur=0;
	int newcur;
	int quiet=0;
	char *t;
	char *port = "/dev/ttyS0";

	opt_init(&optind);
	while((c=opt_parse(argc,argv,&optind,&optarg,opt))!=0) {
		switch(c) {
		case 'v':
		        verb_count++;
			break;
		case 'q':
			quiet++;
			break;
		case 't':
			timeout_scale = strtod(optarg, &t);
			if(timeout_scale<0 || *t!=0 ||
			   optarg[0]<'0' || optarg[0]>'9') {
				fprintf(stderr,"Invalid time scale: %s\n",
					optarg);
				goto printhelp;
			}
			verb("Scaling timeouts by %f\n",timeout_scale);
			break;				
		case 'p':
			port = optarg;
			break;
		case 'V':
			printf("dct-channel " VERSION "\n");
			printf("Written by Jim Paris <jim@jtan.com>\n");
			printf("This program comes with no warranty and is "
			       "provided under the GPLv2.\n");
			return 0;
			break;
		case 'h':
			help=stdout;
		default:
		printhelp:
			fprintf(help,"Usage: %s [options] [channel]\n",*argv);
			opt_help(opt,help);
			fprintf(help,"Changes channel, if one is supplied, and prints it.\n");
			fprintf(help,"With no arguments, prints current channel.\n");
			return (help==stdout)?0:1;
		}
	}

	if((optind+1)<argc) {
		fprintf(stderr,"Error: too many arguments (%s)\n\n",
			argv[optind+1]);
		goto printhelp;
	}

	if(optind<argc) {
		char *end;
		chan = strtol(argv[optind],&end,10);
		if(*end!='\0' || chan<1 || chan>999) {
			fprintf(stderr,"Invalid channel: %s\n",argv[optind]);
			fprintf(stderr,"valid channels are 1 - 999\n");
			goto printhelp;
		}
	}

	if((serial_init(port))<0) 
		err(1,port);

	atexit(serial_close);

	while(tries < 4) {
		tries++;
		if(initialize()<0) {
			verb("Initialization failed\n");
			continue;
		}

		if((newcur=get_channel())<0) {
			verb("Failed to read current channel\n");
			continue;
		}
		cur = newcur;

		if(chan!=0 && chan!=cur) {
			verb("Changing from channel %d to channel %d\n",cur,chan);
			/* Switch channel. */
			if((newcur=set_channel(chan))<0 &&
			   (newcur=get_channel())!=chan) {
				int i;

				verb("Failed to set new channel\n");
				if(tries==1 || tries==3) {
					/* Try sending exit key a few times to make
					   sure we're not in a menu or something */
					verb("Attempting to exit menus\n");
					for(i=0;i<4;i++)
						if(send_keypress(KEY_EXIT)<0) break;
				} else if(tries==2) {
					/* Try sending power button, in case the box 
					   is turned off.  Need to wait for it to 
					   actually power on, and flush buffers
					   since it will probably send garbage. */
					verb("Attempting to turn the box on\n");
					if(send_keypress(KEY_POWER)>=0) {
						usleep(POWER_SLEEP*1000);
					}
 					serial_flush();
				}
				continue;
			}
			cur = newcur;

			if(cur != chan) {
				debug("channel is wrong; didn't switch?\n");
				continue;
			}

			verb("Success!\n");
		}
		if(!quiet) printf("%d\n",cur);
		return 0;
	}

	if(!quiet) {
		fprintf(stderr,"Communication failed after %d tries\n", tries);
		printf("%d\n",cur);
	}
	return 1;
}
