#include <qsqldatabase.h>
#include <qstring.h>

#include <iostream>
using namespace std;

#include "dbcheck.h"

#include "mythcontext.h"
#include "mythdbcon.h"

/// This is the DB schema version expected by the running MythTV instance.
const QString currentDatabaseVersion = "1123";

static bool UpdateDBVersionNumber(const QString &newnumber);
static bool performActualUpdate(const QString updates[], QString version,
                                QString &dbver);
static bool InitializeDatabase(void);
static bool doUpgradeTVDatabaseSchema(void);

/** \defgroup db_schema MythTV Database Schema

\section db_intro Introduction

\verbatim
The schema contains the following tables:
\endverbatim

\htmlonly
<table>
<tr><td>callsignnetworkmap         <td>pk(id)
<tr><td>capturecard                <td>pk(cardid)
<tr><td>cardinput                  <td>pk(cardinputid)
<tr><td>channel                    <td>pk(chanid) k(channum,sourceid)
<tr><td>codecparams                <td>pk(profile,name)
<tr><td>conflictresolutionany      <td>k(prefertitle) k(disliketitle)
<tr><td>conflictresolutionoverride <td>k(chanid,starttime) k(endtime)
<tr><td>conflictresolutionsingle   <td>k(preferchanid,preferstarttime) k(preferendtime)
<tr><td>credits                    <td>uk(chanid,starttime,person,role) k(person,role)
<tr><td>dtv_multiplex              <td>pk(mplexid)
<tr><td>dtv_privatetypes
<tr><td>dvb_signal_quality         <td>pk(id) k(sampletime,cardid)
<tr><td>favorites                  <td>pk(favid)
<tr><td>housekeeping               <td>pk(tag)
<tr><td>jobqueue                   <td>pk(id) uk(chanid,starttime,type,inserttime)
<tr><td>jumppoints                 <td>pk(destination,hostname)
<tr><td>keybindings                <td>pk(context,action,hostname)
<tr><td>keyword                    <td>uk(phrase,searchtype)
<tr><td>mythlog                    <td>pk(logid)
<tr><td>networkiconmap             <td>pk(id) uk(network)
<tr><td>oldprogram                 <td>pk(oldtitle)
<tr><td>oldrecorded                <td>k(endtime) k(title) k(seriesid) k(programid)
<tr><td>people                     <td>pk(person) uk(name)
<tr><td>pidcache                   <td>
<tr><td>profilegroups              <td>pk(id) uk(name,hostname)
<tr><td>program                    <td>k(endtime) k(title) k(title_pronounce) k(seriesid)
                                       k(programid) k(chanid,starttime,endtime)
<tr><td>programgenres              <td>pk(chanid,starttime,relevance)
<tr><td>programrating              <td>uk(chanid,starttime,system,rating)
                                       k(starttime,system)
<tr><td>recgrouppassword           <td>pk(recgroup) uk(recgroup)
<tr><td>record                     <td>pk(recordid) k(chanid,starttime) k(title)
                                       k(seriesid) k(programid)
<tr><td>recorded                   <td>k(endtime) k(seriesid) k(programid)
<tr><td>recordedmarkup             <td>pk(chanid,starttime,mark,type)
<tr><td>recordingprofiles          <td>pk(id)
<tr><td>recordoverride             <td>
<tr><td>settings                   <td>k(value,hostname)
<tr><td>videosource                <td>pk(sourceid) uk(name)
</table>
\endhtmlonly

Where pk means PRIMARY KEY, uk means "UNIQUE KEY", and k means "KEY".

\section capturecard_table Capure Card Table (capturecard)
This table describes the attributes of a capture card used by MythTV.

The primary key 'cardid' is automatically generated by the database.

The 'videodevice' field normally points to a device used to capture
the video, but for DVB where multiple devices are used it is simply
a number indicating which set of devices do use. The 'audiodevice'
is only used by framegrabber cards, encoding cards like the Haupauge
PVR line encode the audio as part of the video stream, while DTV
cards like the various ATSC and DVB cards recieve the audio as part
of the video stream. The same applies to the 'vbidevice', PVR and
DTV cards have captions encoded in the digital stream, while 
framegrabber cards use this device to provide caption and other
data to MythTV.

The 'audioratelimit' field is used to set the audio capture rate
when using the 'audiodevice' field. This is mostly useful for
V4L cards that have troublesome on-board audio, such as some
cards supported by the bttv driver.


The 'cardtype' is an important field for all cards as it guides the
interpretation of the rest of the fields, it has several possible
values: "V4L", "MJPEG", "HDTV", "DVB", "MPEG", "FIREWIRE", "GO7007"
and  "DBOX2".
"V4L" indicates a V4L compatible device, this could be a
true V4L device or something like a Firewire camera with the 
"firewire->v4l" driver. 
"MJPEG" describes a Matrox G200 or DC10 card which captures
individual frames a JPEG images.
"HDTV" describes a pcHDTV HD-2000 or HD-3000 when using the V4L
drivers.
"DVB" can describe any card supported by the Linux DVB
drivers, including the pcHDTV and other ATSC cards. 
The "MPEG" type describes a device using the ivtv drivers such as 
the Haupauge PVR[150,250,350,500]s +Freestyle cards, the Yuan MPEG 
series, the Avermedia M179, and the Adaptec VideOh! AVC-2410.
The "FIREWIRE" card type does not describe Firewire cameras of
capture cards but instead describes the Firewire port required by
law in the USA on cable boxes for subscribers who recieve HDTV 
channels and request the Firewire port feature. This port lets 
you communicate with your cable box without needing to build an
IR transmitter; in some cases, it also allows you to digitally
capture video. 
The "GO7008" card type describes a USB MPEG-4 encoder such
as the Plextor ConvertX.
Finally, the "DBOX2" card type describes connection to a 
networkable cable box manufactured by Nokia and Sagem for use
in Germany.

The 'hostname' field is another important field for all cards
as it specifies which backend the capture card is connected to.

The 'defaultinput' field is a another important field for all
cards except "FIREWIRE" and "DBOX2" cards. It specifies which
input of the card to use. This does not have to mean a specific
physical input, but may also indicate a different use for the
same phyicial input.

The 'signal_timeout' and 'channel_timeout' indicate in 
milliseconds how long it should take to get a signal and 
channel lock respectively. Signal lock detection is 
currently only supported on "DVB" and "HDTV" card types.

The 'dvb_swfilter', 'dvb_recordts', 'dvb_sat_type', 
'dvb_wait_for_seqstart',
'skipbtaudio', 'dvb_on_demand', 'dvb_diseqc_type' and 
'dvb_hw_decoder' are all "DVB" specific configuration parameters.

Both 'dvb_dmx_buf_size' and 'dvb_pkt_buf_size' are unused, and
will be dropped in future versions of MythTV.

The 'firewire_port', 'firewire_node', 'firewire_speed',
'firewire_model', 'firewire_connection' are all "FIREWIRE" specific
configuration parameters. The first three define the connection,
and function much like 'videodevice' does for other capture "cards".
The 'firewire_model' describes the cable box model, for example
"DCT-6000" decribes a box that communicates using the same protocol
as the Motorola DCT-6000. The 'firewire_connection' field descibes
the overall communication protocol, i.e. are we using "Broadcast"
or "Point-to-Point" communication.

\section dtv_multiplex_table Digital Television Multiplex Table (dtv_multiplex)
This table contains the information needed to tune to a particular
frequency on a particular input.

The primary key, 'mplexid', is automatically generated by the database. 
The most important field is the 'frequency' field which contains the
transmission frequency in Hertz. The 'sourceid' which tells us which
card and on what input of that card this multiplex applies to according
to the \ref videosource_table. Along with a channel number 'sourceid'
also uniquely identifies a channel in the \ref channel_table. The 
'modulation' field tells us whether the modulation mode is "qpsk",
"qam_16", "qam_32", "qam_64", "qam_128", "qam_256", "8vsb", "16vsb",
or "auto". Finally, the 'sistandard' field is the last field which
applies to both DVB and ATSC, it contains on of two strings, "dvb" or
"atsc".

When describing a DVB Multiplex, the networkid, and transportid are
used along with the serviceid in the \ref channel_table to tune to
a particular MPEG stream in DVBChannel.

When descrining an ATSC Multiplex, the \ref channel_table contains
the major and minor channels used to tune to a particular stream.

Other fields are used by different DVBx implementations to tune to a
a particular type of service and Myth just passes them along to the 
drivers to take care of.



\section dtv_privatetypes_table Digital Television Private Types Table (dtv_privatetypes)
This table contains free form data pertaining to DVB services, it is
like a settings table for DVB.



\section videosource_table Video Source Table (videosource)
This table tells us about a particular input on a video capture card.

The primary key, 'sourceid', is automatically generated by the database. 
The 'name' field gives the input name as passed to the video capture card
driver. The 'xmltvgrabber' field tells us where we get our listings from.
The 'userid' and 'password' fields are optionally used to log in to a
listings service, with the 'lineupid' being used to select a particular
lineup available from the listings service. Finally, the 'freqtable'
tells us which channel number to frequency mapping to use with
analog channels, us, us-cable, pal, etc.



\section channel_table Channel Table (channel)
The primary key, 'chanid' is generated by combining the source id with
a unique number from 0 to 99.

The 'channum' field contains the channel number to be input to jump
to a channel. With HDTV using video4linux drivers this field has
a special meaning. If the field contains only numeric characters
then the digits up to the last digit are used as the major channel
number and the last digit is used as the minor channel number.
If it contains an underscore (_), the default, then the digits before
the underscore are used as the major channel, and the ones after
it as the minor channel. If it contains a minus sign (-), then
the digits after the minus sign are used as the program number.

The 'sourceid' field contains the \ref videosource_table key, which
specifies which listings service to use and which input on which video
capture card to find this channel.

The 'callsign' field contains the government mandated licence number
of the channel's broadcaster if this is an over-the-air channel.

The 'name' field provides the alphanumberic name of this channel for
use in the %MythTV user interface.

The 'icon' field provides the path to an optional channel icon for this
channel.

The 'freqid' field provides the channel number to tune to, as
differentiated from the 'channum' used in the %MythTV user interface.
When the 'mplexid' is set it is used instead. And the 'finetune' field
is used to adjust the frequency provided by the combination of 'freqid'
and 'tvformat', this can be used when there is carrier drift in a cable
system.

The 'xmltvid' field is used to identify this channel to the listings
provider.

The 'recpriority' field is used tell the scheduler from which of two 
otherwise equivalent programs on two different channels should be 
prefered, a higher number means this channel is more preferred.

The 'tvformat' is one of "ntsc", "ntsc-jp", "pal", "atsc", etc and is used
for various purposes, such as translating a channel number to a frequency,
setting the recording frame size, etc.

'commfree' tells us whether this is a commercial free channel, such as
those on the BBC and CBC networks. On commercial free channels we
do not need to run the commercial detector.

'visible' tells us whether we should show this channel in the channel
guide.

'outputfilters' contains a list of filters that should be applied
when playing recordings from this channel.

The 'contrast' 'brightness', 'colour', and 'hue' fields are used by frame
grabbing recorders to adjust the picture before encoding each frame.
And, 'videofilters' contains a list of filters that should be applied
when recording from channel using a frame grabbing recorder.

For digital television, the 'mplexid' field contains the 
\ref dtv_multiplex_table key for determining additional digital tuning
parameters. The 'serviceid' field contains the MPEG program number when
describing a DVB service. The 'useonairguide' field is used by DVB to
determine whether to use the on-air-guide data from this channel to
update the listings. Using the on-air-guide is currently experimental
and must be selected when you compile %MythTV. Finally, the 'atscsrcid'
field currently contains both the major and minor atsc channels, encoded
in the form (majorChannel * 256 | minorChannel) when using DVB drivers.

\section program_table Program Entry Table (program)
'category_type' holds one of these exact four strings:
"movie", "series", "sports" or "tvshow".

'airdate' is a string representing the year of release for movies and 
may have no meaning for other types of shows.

'stars' is a floating point number from 0.0 to 1.0. On a four star scale,
1.0 would be four stars, 0.75 would be three stars and so on.

'originalairdate' if provided is the date when a show was, or will be,
first televised. 

'previouslyshown' is a field created by MythTV to try to determine if
a showing is more than 14 days after its original air date or if the
show was marked as a repeat and did not have a date for the first
airing. If this is "0" it usually means that this is a brand new show
or a rebroadcast within the first two weeks.

'programid' is the Tribune Media Service database record identifier
for each program description. In general, these start with a two
letter prefix, MV, EP, SP or SH that equate to the 'category_type'.
For most, the last four digits are "0000" except EP where the last
four digits are the episode number in the series. Note that these
are generated by TMS and not the show's producers but they are
usually in the same order as the original air dates for the episodes.
Detailed information can be found in the Data Direct documentation 
at http://labs.zap2it.com/ .

 */

/** \fn UpdateDBVersionNumber(const QString&)
 *  \brief Updates the schema version stored in the database.
 *
 *   Updates "DBSchemaVer" property in the settings table.
 *  \param newnumber New schema version.
 */
static bool UpdateDBVersionNumber(const QString &newnumber)
{
    // delete old schema version
    MSqlQuery query(MSqlQuery::InitCon());

    QString thequery = "DELETE FROM settings WHERE value='DBSchemaVer';";
    query.prepare(thequery);
    query.exec();

    if (query.lastError().type() != QSqlError::None)
    {
        QString msg =
            QString("DB Error (Deleting old DB version number): \n"
                    "Query was: %1 \nError was: %2 \nnew version: %3")
            .arg(thequery)
            .arg(MythContext::DBErrorMessage(query.lastError()))
            .arg(newnumber);
        VERBOSE(VB_IMPORTANT, msg);
        return false;
    }

    // set new schema version
    thequery = QString("INSERT INTO settings (value, data, hostname) "
                       "VALUES ('DBSchemaVer', %1, NULL);").arg(newnumber);
    query.prepare(thequery);
    query.exec();

    if (query.lastError().type() != QSqlError::None)
    {
        QString msg =
            QString("DB Error (Setting new DB version number): \n"
                    "Query was: %1 \nError was: %2 \nnew version: %3")
            .arg(thequery)
            .arg(MythContext::DBErrorMessage(query.lastError()))
            .arg(newnumber);
        VERBOSE(VB_IMPORTANT, msg);
        return false;
    }

    return true;
}

/** \fn performActualUpdate(const QString updates[], QString, QString&)
 *  \brief Runs a number of SQL commands, and updates the schema version.
 *
 *  \param updates  array of SQL commands to issue, terminated by a "" string.
 *  \param version  version we are updating db to.
 *  \param dbver    the database version at the end of the function is returned
 *                  in this parameter, if things go well this will be 'version'.
 *  \return true on success, false on failure
 */
static bool performActualUpdate(const QString updates[], QString version,
                                QString &dbver)
{
    MSqlQuery query(MSqlQuery::InitCon());

    VERBOSE(VB_IMPORTANT, QString("Upgrading to schema version ") + version);

    int counter = 0;
    QString thequery = updates[counter];

    while (thequery != "")
    {
        query.prepare(thequery);
        query.exec();

        if (query.lastError().type() != QSqlError::None)
        {
            QString msg =
                QString("DB Error (Performing database upgrade): \n"
                        "Query was: %1 \nError was: %2 \nnew version: %3")
                .arg(thequery)
                .arg(MythContext::DBErrorMessage(query.lastError()))
                .arg(version);
            VERBOSE(VB_IMPORTANT, msg);
            return false;
        }

        counter++;
        thequery = updates[counter];
    }

    if (!UpdateDBVersionNumber(version))
        return false;

    dbver = version;
    return true;
}

/** \fn UpgradeTVDatabaseSchema(void)
 *  \brief This is the function called from outside dbcheck.cpp to update the schema.
 *
 *   If the "DBSchemaVer" property equals the currentDatabase version this returns
 *   true immediately. If not we lock the schemalock table. If this fails we return
 *   false. If it succeeds we call doUpgradeTVDatabaseSchema() to do the actual
 *   update, and then we unlock the schemalock table.
 *
 *   If the program running this function is killed while this is running the schema
 *   may be corrupted.
 *
 *  \return true on success, false on failure.
 */
bool UpgradeTVDatabaseSchema(void)
{
    QString dbver = gContext->GetSetting("DBSchemaVer");

    VERBOSE(VB_IMPORTANT, QString("Current Schema Version: %1").arg(dbver));

    if (dbver == currentDatabaseVersion)
        return true;

    MSqlQuery chartype(MSqlQuery::InitCon());
    chartype.prepare("ALTER DATABASE mythconverg DEFAULT CHARACTER SET latin1;");
    chartype.exec();

    VERBOSE(VB_IMPORTANT, QString("Newest Schema Version : %1")
                                  .arg(currentDatabaseVersion));

    MSqlQuery lockquery(MSqlQuery::InitCon());

    lockquery.prepare("CREATE TABLE IF NOT EXISTS "
                      "schemalock ( schemalock int(1));");
    if (!lockquery.exec())
    {
        VERBOSE(VB_IMPORTANT, QString("ERROR: Unable to create database "
                                      "upgrade lock table: %1")
                                      .arg(MythContext::DBErrorMessage(
                                           lockquery.lastError())));
        return false;
    }

    VERBOSE(VB_IMPORTANT, "Setting Lock for Database Schema upgrade. If you "
                          "see a long pause here it means the Schema is "
                          "already locked and is being upgraded by another "
                          "Myth process.");

    lockquery.prepare("LOCK TABLE schemalock WRITE;");
    if (!lockquery.exec())
    {
        VERBOSE(VB_IMPORTANT, QString("ERROR: Unable to acquire database "
                                      "upgrade lock")
                                      .arg(MythContext::DBErrorMessage(
                                           lockquery.lastError())));
        return false;
    }

    bool ret = doUpgradeTVDatabaseSchema();

    if (ret)
        VERBOSE(VB_IMPORTANT, "Database Schema upgrade complete, unlocking.");
    else
        VERBOSE(VB_IMPORTANT, "Database Schema upgrade FAILED, unlocking.");

    lockquery.prepare("UNLOCK TABLES;");
    lockquery.exec();

    return ret;
}

/** \fn doUpgradeTVDatabaseSchema(void)
 *  \brief This is called by UpgradeTVDatabaseSchema() to actually upgrade
 *         the schema to what MythTV expects.
 *
 *   This function Initializes the database if the "DBSchemaVer" property
 *   does not exist in the database. Otherwise it upgrades the schema
 *   from the one in that property to the currentDatabaseVersion, by 
 *   running the SQL commands needed for each upgrade between the two
 *   versions. This will only upgrade the schema, so bad things may
 *   happen if you upgrade and then downgrade MythTV across DB Schema
 *   versions.
 *
 */
static bool doUpgradeTVDatabaseSchema(void)
{
    QString dbver = gContext->GetSetting("DBSchemaVer");

    if (dbver == currentDatabaseVersion)
        return true;
    
    if (dbver == "")
    {
        if (!InitializeDatabase())
            return false;
        dbver = "1112";
    }

    if (dbver == "1027")
    {
        const QString updates[] = {
"CREATE TABLE IF NOT EXISTS dvb_signal_quality ("
"    id INT UNSIGNED AUTO_INCREMENT NOT NULL PRIMARY KEY,"
"    sampletime TIMESTAMP NOT NULL,"
"    cardid INT UNSIGNED NOT NULL,"
"    fe_snr INT UNSIGNED NOT NULL,"
"    fe_ss  INT UNSIGNED NOT NULL,"
"    fe_ber INT UNSIGNED NOT NULL,"
"    fe_unc INT UNSIGNED NOT NULL,"
"    myth_cont INT UNSIGNED NOT NULL,"
"    myth_over INT UNSIGNED NOT NULL,"
"    myth_pkts INT UNSIGNED NOT NULL,"
"    FOREIGN KEY(cardid) REFERENCES capturecard(id),"
"    INDEX (sampletime,cardid)"
");",
"ALTER TABLE capturecard ADD skipbtaudio TINYINT(1) DEFAULT 0;",
""
};
        if (!performActualUpdate(updates, "1028", dbver))
            return false;
    }

    if (dbver == "1028") {
        const QString updates[] = {
"ALTER TABLE channel ADD COLUMN commfree TINYINT NOT NULL default '0';",
"ALTER TABLE record ADD COLUMN recgroup VARCHAR(32) default 'Default';",
"ALTER TABLE record ADD COLUMN dupmethod INT NOT NULL DEFAULT 6;",
"ALTER TABLE record ADD COLUMN dupin INT NOT NULL DEFAULT 15;",
"UPDATE record SET dupmethod = 1 WHERE recorddups = 2;",
"UPDATE record SET dupin = 2 WHERE recorddups = 1;",
"ALTER TABLE record DROP COLUMN recorddups;",
"ALTER TABLE recorded ADD COLUMN recgroup VARCHAR(32) default 'Default';",
"ALTER TABLE recorded ADD COLUMN recordid INT DEFAULT NULL;",
"CREATE TABLE recgrouppassword ("
"  recgroup VARCHAR(32) NOT NULL PRIMARY KEY, "
"  password VARCHAR(10) NOT NULL, "
"  UNIQUE(recgroup)"
");",
""
};
        if (!performActualUpdate(updates, "1029", dbver))
            return false;
    }

    if (dbver == "1029")
    {
        const QString updates[] = {
"ALTER TABLE record CHANGE preroll startoffset INT DEFAULT 0 NOT NULL;",
"ALTER TABLE record CHANGE postroll endoffset INT DEFAULT 0 NOT NULL;",
""
};
        if (!performActualUpdate(updates, "1030", dbver))
            return false;
    }

    if (dbver == "1030") 
    {
        const QString updates[] = {
"ALTER TABLE channel ADD COLUMN visible TINYINT(1) NOT NULL default '1';",
"UPDATE channel SET visible = 1;",
""
};
        if (!performActualUpdate(updates, "1031", dbver))
            return false;
    }

    if (dbver == "1031") {
        const QString updates[] = {
"ALTER TABLE capturecard ADD dvb_on_demand TINYINT NOT NULL DEFAULT 0;",
""
};
        if (!performActualUpdate(updates, "1032", dbver))
            return false;
    }

    if (dbver == "1032")
    {
        const QString updates[] = {
"UPDATE record SET dupmethod = 6 WHERE dupmethod = 22;",
""
};
        if (!performActualUpdate(updates, "1033", dbver))
            return false;
    }

    if (dbver == "1033")
    {
        const QString updates[] = {
"ALTER TABLE program ADD title_pronounce VARCHAR(128) NULL;",
"ALTER TABLE program ADD INDEX (title_pronounce);",
""
};
        if (!performActualUpdate(updates, "1034", dbver))
            return false;
    }

    if (dbver == "1034")
    {
        const QString updates[] = {
"CREATE TABLE mythlog ("
"  logid int(10) unsigned PRIMARY KEY NOT NULL auto_increment,"
"  module char(32) NOT NULL,"
"  priority int(11) NOT NULL,"
"  acknowledged bool default 0,"
"  logdate datetime,"
"  host varchar(128),"
"  message varchar(255) NOT NULL,"
"  details text"
");",
"CREATE TABLE housekeeping ("
"  tag varchar(64) PRIMARY KEY NOT NULL,"
"  lastrun datetime"
");",
""
};
        if (!performActualUpdate(updates, "1035", dbver))
            return false;
    }
    
    if (dbver == "1035")
    {
        const QString updates[] = {
"ALTER TABLE dvb_sat CHANGE pos pos FLOAT;",
"ALTER TABLE dvb_sat ADD diseqc_pos SMALLINT DEFAULT 0 AFTER diseqc_port;",
""
};
        if (!performActualUpdate(updates,"1036", dbver))
            return false;
    }

    if (dbver == "1036")
    {
        const QString updates[] = {
"UPDATE channel SET callsign=chanid WHERE callsign IS NULL OR callsign='';",
"ALTER TABLE record ADD COLUMN station VARCHAR(20) NOT NULL DEFAULT '';",
"ALTER TABLE recordoverride ADD COLUMN station VARCHAR(20) NOT NULL DEFAULT '';",
""
};
        if (!performActualUpdate(updates, "1037", dbver))
            return false;
    }

    if (dbver == "1037")
    {
        const QString updates[] = {
"ALTER TABLE videosource ADD lineupid VARCHAR(64) NULL;",
"ALTER TABLE videosource ADD password VARCHAR(64) NULL;",
"ALTER TABLE program ADD ( "
"    stereo bool, "
"    subtitled bool, "
"    hdtv bool, "
"    closecaptioned bool, "
"    partnumber int, "
"    parttotal int, "
"    seriesid char(12), "
"    originalairdate date, "
"    showtype varchar(30), "
"    colorcode varchar(20), "
"    syndicatedepisodenumber varchar(20), "
"    programid char(12) "
");",
"DELETE FROM credits;",
"ALTER TABLE credits DROP INDEX chanid;",
"ALTER TABLE credits ADD UNIQUE chanid (chanid, starttime, person, role);",
"DELETE FROM people;",
"ALTER TABLE people DROP INDEX name;",
"ALTER TABLE people ADD UNIQUE name (name(41));",
"CREATE TABLE programgenres ( "
"    chanid int unsigned NOT NULL, "
"    starttime timestamp NOT NULL, "
"    relevance char(1) NOT NULL, "
"    genre char(30), "
"    PRIMARY KEY (chanid, starttime, relevance) "
");",
""
};
        if (!performActualUpdate(updates, "1038", dbver))
            return false;
    }

    if (dbver == "1038")
    {
        const QString updates[] = {
"CREATE TABLE IF NOT EXISTS programgenres ( "
"    chanid int unsigned NOT NULL, "
"    starttime timestamp NOT NULL, "
"    relevance char(1) NOT NULL, "
"    genre char(30), "
"    PRIMARY KEY (chanid, starttime, relevance) "
");",
""
};
        if (!performActualUpdate(updates, "1039", dbver))
            return false;
    }

    if (dbver == "1039")
    {
        const QString updates[] = {
"ALTER TABLE channel CHANGE name name VARCHAR(64);",
""
};
        if (!performActualUpdate(updates, "1040", dbver))
            return false;
    }

    if (dbver == "1040")
    {
        const QString updates[] = {
"ALTER TABLE channel ADD outputfilters VARCHAR(255) NULL",
""
};
        if (!performActualUpdate(updates, "1041", dbver))
            return false;
    }

    if (dbver == "1041")
    {
        const QString updates[] = {
"ALTER TABLE record ADD seriesid varchar(12) NULL;",
"ALTER TABLE record ADD programid varchar(12) NULL;",
"ALTER TABLE recorded ADD seriesid varchar(12) NULL;",
"ALTER TABLE recorded ADD programid varchar(12) NULL;",
"ALTER TABLE oldrecorded ADD seriesid varchar(12) NULL;",
"ALTER TABLE oldrecorded ADD programid varchar(12) NULL;",
"ALTER TABLE program ADD INDEX (seriesid);",
"ALTER TABLE program ADD INDEX (programid);",
"ALTER TABLE record ADD INDEX (seriesid);",
"ALTER TABLE record ADD INDEX (programid);",
"ALTER TABLE recorded ADD INDEX (seriesid);",
"ALTER TABLE recorded ADD INDEX (programid);",
"ALTER TABLE oldrecorded ADD INDEX (seriesid);",
"ALTER TABLE oldrecorded ADD INDEX (programid);",
""
};
        if (!performActualUpdate(updates, "1042", dbver))
            return false;
    }

    if (dbver == "1042")
    {
        const QString updates[] = {
"INSERT INTO settings SET value=\"DataDirectMessage\";",
""
};
        if (!performActualUpdate(updates, "1043", dbver))
            return false;
    }

    if (dbver == "1043")
    {
        const QString updates[] = {
"ALTER TABLE program CHANGE title title VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE subtitle subtitle VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE description description TEXT NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE category category VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE category_type category_type VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE title_pronounce title_pronounce VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE seriesid seriesid VARCHAR(12) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE showtype showtype VARCHAR(30) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE colorcode colorcode VARCHAR(20) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE syndicatedepisodenumber syndicatedepisodenumber VARCHAR(20) NOT NULL DEFAULT '';",
"ALTER TABLE program CHANGE programid programid VARCHAR(12) NOT NULL DEFAULT '';",

"ALTER TABLE channel CHANGE channum channum VARCHAR(5) NOT NULL DEFAULT '';",
"ALTER TABLE channel CHANGE callsign callsign VARCHAR(20) NOT NULL DEFAULT '';",
"ALTER TABLE channel CHANGE name name VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE channel CHANGE icon icon VARCHAR(255) NOT NULL DEFAULT 'none';",
"ALTER TABLE channel CHANGE videofilters videofilters VARCHAR(255) NOT NULL DEFAULT '';",
"ALTER TABLE channel CHANGE xmltvid xmltvid VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE channel CHANGE outputfilters outputfilters VARCHAR(255) NOT NULL DEFAULT '';",

"ALTER TABLE record CHANGE title title VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE record CHANGE subtitle subtitle VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE record CHANGE description description TEXT NOT NULL DEFAULT '';",
"ALTER TABLE record CHANGE profile profile VARCHAR(128) NOT NULL DEFAULT 'Default';",
"ALTER TABLE record CHANGE category category VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE record CHANGE recgroup recgroup VARCHAR(32) NOT NULL DEFAULT 'Default';",
"ALTER TABLE record CHANGE station station VARCHAR(20) NOT NULL DEFAULT '';",
"ALTER TABLE record CHANGE seriesid seriesid VARCHAR(12) NOT NULL DEFAULT '';",
"ALTER TABLE record CHANGE programid programid VARCHAR(12) NOT NULL DEFAULT '';",

"ALTER TABLE recorded CHANGE title title VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE recorded CHANGE subtitle subtitle VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE recorded CHANGE description description TEXT NOT NULL DEFAULT '';",
"ALTER TABLE recorded CHANGE hostname hostname VARCHAR(255) NOT NULL DEFAULT '';",
"ALTER TABLE recorded CHANGE category category VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE recorded CHANGE recgroup recgroup VARCHAR(32) NOT NULL DEFAULT 'Default';",
"ALTER TABLE recorded CHANGE seriesid seriesid VARCHAR(12) NOT NULL DEFAULT '';",
"ALTER TABLE recorded CHANGE programid programid VARCHAR(12) NOT NULL DEFAULT '';",

"ALTER TABLE oldrecorded CHANGE title title VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE oldrecorded CHANGE subtitle subtitle VARCHAR(128) NOT NULL DEFAULT '';",
"ALTER TABLE oldrecorded CHANGE description description TEXT NOT NULL DEFAULT '';",
"ALTER TABLE oldrecorded CHANGE category category VARCHAR(64) NOT NULL DEFAULT '';",
"ALTER TABLE oldrecorded CHANGE seriesid seriesid VARCHAR(12) NOT NULL DEFAULT '';",
"ALTER TABLE oldrecorded CHANGE programid programid VARCHAR(12) NOT NULL DEFAULT '';",
""
};
        if (!performActualUpdate(updates, "1044", dbver))
            return false;
    }

    if (dbver == "1044")
    {
        const QString updates[] = {
"UPDATE channel SET icon = 'none' WHERE icon = '';",
"UPDATE record SET profile = 'Default' WHERE profile = '';",
"UPDATE record SET recgroup = 'Default' WHERE recgroup = '';",
"UPDATE recorded SET recgroup = 'Default', starttime = starttime WHERE recgroup = '';",
""
};
        if (!performActualUpdate(updates, "1045", dbver))
            return false;
    }

    if (dbver == "1045")
    {
        const QString updates[] = {
"UPDATE recorded SET recgroup = 'Default', starttime = starttime WHERE recgroup = '';",
""
};
        if (!performActualUpdate(updates, "1046", dbver))
            return false;
    }

    if (dbver == "1046")
    {
        const QString updates[] = {
"ALTER TABLE record ADD COLUMN search INT UNSIGNED NOT NULL DEFAULT 0;",
"UPDATE record SET search = 0 WHERE search IS NULL;",
""
};
        if (!performActualUpdate(updates, "1047", dbver))
            return false;
    }

    if (dbver == "1047")
    {
        const QString updates[] = {
"CREATE TABLE networkiconmap ("
"    id INTEGER NOT NULL AUTO_INCREMENT,"
"    network VARCHAR(20) NOT NULL UNIQUE,"
"    url VARCHAR(255) NOT NULL,"
"    PRIMARY KEY(id)"
");",
"CREATE TABLE callsignnetworkmap ("
"    id INTEGER NOT NULL AUTO_INCREMENT,"
"    callsign VARCHAR(20) NOT NULL UNIQUE,"
"    network VARCHAR(20) NOT NULL,"
"    PRIMARY KEY(id)"
");",
""
};
        if (!performActualUpdate(updates, "1048", dbver))
            return false;
    }

    if (dbver == "1048")
    {
        const QString updates[] = {
"ALTER TABLE cardinput CHANGE preference preference INT NOT NULL DEFAULT 0;",
"UPDATE cardinput SET preference = 0 WHERE preference IS NULL;",
""
};
        if (!performActualUpdate(updates, "1049", dbver))
            return false;
    }

    if (dbver == "1049")
    {
        const QString updates[] = {
"ALTER TABLE keyword ADD COLUMN searchtype INT UNSIGNED NOT NULL DEFAULT 3;",
"ALTER TABLE keyword DROP INDEX phrase;",
"ALTER TABLE keyword DROP PRIMARY KEY;",
"ALTER TABLE keyword ADD UNIQUE(phrase,searchtype);",
""
};
        if (!performActualUpdate(updates, "1050", dbver))
            return false;
    }
    

    if (dbver == "1050")
    {
        const QString updates[] = {
"ALTER TABLE recorded CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE recorded CHANGE endtime endtime DATETIME NOT NULL;",
"ALTER TABLE recorded ADD COLUMN lastmodified TIMESTAMP NOT NULL;",
"ALTER TABLE recorded ADD COLUMN filesize BIGINT(20) DEFAULT 0 NOT NULL;",
"ALTER TABLE credits CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE oldprogram CHANGE airdate airdate DATETIME NOT NULL;",
"ALTER TABLE oldrecorded CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE oldrecorded CHANGE endtime endtime DATETIME NOT NULL;",
"ALTER TABLE program CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE program CHANGE endtime endtime DATETIME NOT NULL;",
"ALTER TABLE programgenres CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE programrating CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE recordedmarkup CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE recordoverride CHANGE starttime starttime DATETIME NOT NULL;",
"ALTER TABLE recordoverride CHANGE endtime endtime DATETIME NOT NULL;",
"ALTER TABLE transcoding CHANGE starttime starttime DATETIME NOT NULL;",
""
};
        if (!performActualUpdate(updates, "1051", dbver))
            return false;
    }

    if (dbver == "1051")
    {
        const QString updates[] = {
"update record set dupmethod = 6 where dupmethod = 0;",
"update record set dupin = 15 where dupin = 0;",
""
};
        if (!performActualUpdate(updates, "1052", dbver))
            return false;
    }

    if (dbver == "1052")
    {
        const QString updates[] = {
"ALTER TABLE recorded ADD COLUMN stars FLOAT NOT NULL DEFAULT 0;",
"ALTER TABLE recorded ADD COLUMN previouslyshown TINYINT(1) DEFAULT 0;",
"ALTER TABLE recorded ADD COLUMN originalairdate DATE;",
"INSERT INTO settings VALUES ('HaveRepeats', '0', NULL);",
""
};
        if (!performActualUpdate(updates, "1053", dbver))
            return false;
    }

    if (dbver == "1053")
    {
        const QString updates[] = {
"ALTER TABLE channel CHANGE freqid freqid VARCHAR(10);",
""
};
        if (!performActualUpdate(updates, "1054", dbver))
            return false;
    }

    if (dbver == "1054")
    {
        const QString updates[] = {
"ALTER TABLE program ADD INDEX id_start_end (chanid,starttime,endtime);",
"ALTER TABLE channel ADD INDEX channel_src (channum,sourceid);",
""
};
        if (!performActualUpdate(updates, "1055", dbver))
            return false;
    }

    if (dbver == "1055")
    {
        const QString updates[] = {
"UPDATE record SET dupmethod=6, dupin=4 WHERE dupmethod=8;",
""
};
        if (!performActualUpdate(updates, "1056", dbver))
            return false;
    }

    if (dbver == "1056")
    {
        const QString updates[] = {
"CREATE TABLE jobqueue ("
"    id INTEGER NOT NULL AUTO_INCREMENT,"
"    chanid INTEGER(10) NOT NULL,"
"    starttime DATETIME NOT NULL,"
"    inserttime DATETIME NOT NULL,"
"    type INTEGER NOT NULL,"
"    cmds INTEGER NOT NULL DEFAULT 0,"
"    flags INTEGER NOT NULL DEFAULT 0,"
"    status INTEGER NOT NULL DEFAULT 0,"
"    statustime TIMESTAMP NOT NULL,"
"    hostname VARCHAR(255) NOT NULL DEFAULT '',"
"    args BLOB NOT NULL DEFAULT '',"
"    comment VARCHAR(128) NOT NULL DEFAULT '',"
"    PRIMARY KEY(id),"
"    UNIQUE(chanid, starttime, type, inserttime)"
");",
"ALTER TABLE record ADD COLUMN autotranscode TINYINT(1) NOT NULL DEFAULT 0;",
"ALTER TABLE record ADD COLUMN autocommflag TINYINT(1) NOT NULL DEFAULT 0;",
"ALTER TABLE record ADD COLUMN autouserjob1 TINYINT(1) NOT NULL DEFAULT 0;",
"ALTER TABLE record ADD COLUMN autouserjob2 TINYINT(1) NOT NULL DEFAULT 0;",
"ALTER TABLE record ADD COLUMN autouserjob3 TINYINT(1) NOT NULL DEFAULT 0;",
"ALTER TABLE record ADD COLUMN autouserjob4 TINYINT(1) NOT NULL DEFAULT 0;",
""
};
        if (!performActualUpdate(updates, "1057", dbver))
            return false;

        if (gContext->GetNumSetting("AutoCommercialFlag", 1))
        {
            MSqlQuery query(MSqlQuery::InitCon());
            query.prepare("UPDATE record SET autocommflag = 1;");
            query.exec();
        }
    }
    if (dbver == "1057")
    {
        const QString updates[] = {
"DROP TABLE IF EXISTS transcoding;",
""
};
        if (!performActualUpdate(updates, "1058", dbver))
            return false;
    }

    if (dbver == "1058")
    {
        const QString updates[] = {
"UPDATE program SET category_type='series' WHERE showtype LIKE '%series%';",
""
};
        if (!performActualUpdate(updates, "1059", dbver))
            return false;
    }

    if (dbver == "1059")
    {
        const QString updates[] = {
"ALTER TABLE recorded ADD COLUMN preserve TINYINT(1) NOT NULL DEFAULT 0;",
""
};
        if (!performActualUpdate(updates, "1060", dbver))
            return false;
    }

    if (dbver == "1060")
    {
        const QString updates[] = {

"ALTER TABLE record ADD COLUMN record.findday TINYINT NOT NULL DEFAULT 0;",
"ALTER TABLE record ADD COLUMN record.findtime TIME NOT NULL DEFAULT '00:00:00';",
"ALTER TABLE record ADD COLUMN record.findid INT NOT NULL DEFAULT 0;",
"ALTER TABLE recorded ADD COLUMN recorded.findid INT NOT NULL DEFAULT 0;",
"ALTER TABLE oldrecorded ADD COLUMN oldrecorded.findid INT NOT NULL DEFAULT 0;",
""
};
        if (!performActualUpdate(updates, "1061", dbver))
            return false;
    }

    if (dbver == "1061")
    {
        const QString updates[] = {
"ALTER TABLE record ADD COLUMN inactive TINYINT(1) NOT NULL DEFAULT 0;",
""
};
        if (!performActualUpdate(updates, "1062", dbver))
            return false;
    }

    if (dbver == "1062")
    {
        const QString updates[] = {
"ALTER TABLE cardinput ADD COLUMN freetoaironly TINYINT(1) DEFAULT 1;",
"ALTER TABLE channel ADD COLUMN useonairguide TINYINT(1) DEFAULT 0;",
"ALTER TABLE capturecard ADD COLUMN dvb_diseqc_type SMALLINT(6);",
"ALTER TABLE cardinput ADD COLUMN diseqc_port SMALLINT(6);",
"ALTER TABLE cardinput ADD COLUMN diseqc_pos FLOAT;",
"ALTER TABLE cardinput ADD COLUMN lnb_lof_switch INT(11) DEFAULT 11700000;",
"ALTER TABLE cardinput ADD COLUMN lnb_lof_hi INT(11) DEFAULT 10600000;",
"ALTER TABLE cardinput ADD COLUMN lnb_lof_lo INT(11) DEFAULT 9750000;",
"ALTER TABLE channel ADD COLUMN mplexid SMALLINT(6);",
"ALTER TABLE channel ADD COLUMN serviceid SMALLINT(6);",
"ALTER TABLE channel ADD COLUMN atscsrcid INT(11) DEFAULT NULL;",
"CREATE TABLE dtv_multiplex ("
"  mplexid smallint(6) NOT NULL auto_increment, "
"  sourceid smallint(6) default NULL,"
"  transportid int(11) default NULL,"
"  networkid int(11) default NULL,"
"  frequency int(11) default NULL,"
"  inversion char(1) default 'a',"
"  symbolrate int(11) default NULL,"
"  fec varchar(10) default 'auto',"
"  polarity char(1) default NULL,"
"  modulation varchar(10) default 'auto',"
"  bandwidth char(1) default 'a',"
"  lp_code_rate varchar(10) default 'auto',"
"  transmission_mode char(1) default 'a',"
"  guard_interval varchar(10) default 'auto',"
"  visible smallint(1) NOT NULL default '0',"
"  constellation varchar(10) default 'auto',"
"  hierarchy varchar(10) default 'auto',"
"  hp_code_rate varchar(10) default 'auto',"
"  sistandard varchar(10) default 'dvb',"
"  serviceversion smallint(6) default 33,"
"  updatetimestamp timestamp(14) NOT NULL,"
"  PRIMARY KEY  (mplexid)"
") TYPE=MyISAM;",
// These should be included in an update after the 0.17 release.
// "DROP TABLE IF EXISTS dvb_channel;",
// "DROP TABLE IF EXISTS dvb_pids;",
// "DROP TABLE IF EXISTS dvb_sat;",
"CREATE TABLE dtv_privatetypes ("
"  sitype varchar(4) NOT NULL, "
"  networkid int(11) NOT NULL, "
"  private_type varchar(20) NOT NULL, "
"  private_value varchar(100) NOT NULL "
");",
//# UK DVB-T
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',9018,'channel_numbers','131');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',9018,'guide_fixup','2');",
//# Bell ExpressVu Canada
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',256,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',257,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',256,'tv_types','1,150,134,133');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',257,'tv_types','1,150,134,133');",

"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4100,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4101,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4102,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4103,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4104,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4105,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4106,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4107,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4097,'sdt_mapping','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4098,'sdt_mapping','1');",

"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4100,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4101,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4102,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4103,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4104,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4105,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4106,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4107,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4097,'tv_types','1,145,154');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4098,'tv_types','1,145,154');",

"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4100,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4101,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4102,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4103,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4104,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4105,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4106,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4107,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4097,'guide_fixup','1');",
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4098,'guide_fixup','1');",

//# NSAB / Sirius
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',94,'tv_types','1,128');",
//# WUNC Guide
"INSERT into dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('atsc',1793,'guide_fixup','3');",
""
};
        if (!performActualUpdate(updates, "1063", dbver))
            return false;
    }

    if (dbver == "1063")
    {
        const QString updates[] = {
"CREATE TABLE IF NOT EXISTS recordmatch (recordid int unsigned, "
"chanid int unsigned, starttime datetime, INDEX (recordid));",
""
};
        if (!performActualUpdate(updates, "1064", dbver))
            return false;
    }

    if (dbver == "1064")
    {
        const QString updates[] = {
"ALTER TABLE `program` CHANGE `stereo` `stereo` TINYINT( 1 ) DEFAULT '0' NOT NULL;",
"ALTER TABLE `program` CHANGE `subtitled` `subtitled` TINYINT( 1 ) DEFAULT '0' NOT NULL;",
"ALTER TABLE `program` CHANGE `hdtv` `hdtv` TINYINT( 1 ) DEFAULT '0' NOT NULL;",
"ALTER TABLE `program` CHANGE `closecaptioned` `closecaptioned` TINYINT( 1 ) DEFAULT '0' NOT NULL;",
"ALTER TABLE `program` CHANGE `partnumber` `partnumber` INT( 11 ) DEFAULT '0' NOT NULL;",
"ALTER TABLE `program` CHANGE `parttotal` `parttotal` INT( 11 ) DEFAULT '0' NOT NULL;",
"ALTER TABLE `program` CHANGE `programid` `programid` VARCHAR( 20 ) NOT NULL;",
"ALTER TABLE `oldrecorded` CHANGE `programid` `programid` VARCHAR( 20 ) NOT NULL;",
"ALTER TABLE `recorded` CHANGE `programid` `programid` VARCHAR( 20 ) NOT NULL;",
"ALTER TABLE `record` CHANGE `programid` `programid` VARCHAR( 20 ) NOT NULL;",
""
};
        if (!performActualUpdate(updates, "1065", dbver))
            return false;
    }
  
    if (dbver == "1065") 
    {
        const QString updates[] = {
"INSERT INTO profilegroups SET name = 'FireWire Input', cardtype = 'FIREWIRE', is_default = 1;",
"ALTER TABLE capturecard ADD COLUMN firewire_port INT UNSIGNED NOT NULL DEFAULT 0;",
"ALTER TABLE capturecard ADD COLUMN firewire_node INT UNSIGNED NOT NULL DEFAULT 2;",
"ALTER TABLE capturecard ADD COLUMN firewire_speed INT UNSIGNED NOT NULL DEFAULT 0;",
"ALTER TABLE capturecard ADD COLUMN firewire_model varchar(32) NULL;",
""
};
        if (!performActualUpdate(updates, "1066", dbver))
            return false;
    } 
 
    if (dbver == "1066")
    {
        const QString updates[] = {
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES('dvb', '40999', 'guide_fixup', '4');",
""
};
        if (!performActualUpdate(updates, "1067", dbver))
            return false;
    }

    if (dbver == "1067")
    {
        const QString updates[] = {
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',70,'force_guide_present','yes');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',70,'guide_ranges','80,80,96,96');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4112,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4115,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',4116,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',12802,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',12803,'channel_numbers','131');",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',12829,'channel_numbers','131');",
""
};
        if (!performActualUpdate(updates, "1068", dbver))
            return false;
    }

    if (dbver == "1068")
    {
        const QString updates[] = {
"ALTER TABLE recorded ADD COLUMN deletepending TINYINT(1) NOT NULL DEFAULT 0;",
""
};
        if (!performActualUpdate(updates, "1069", dbver))
            return false;
    }

    if (dbver == "1069")
    {
        const QString updates[] = {
"UPDATE jumppoints SET description = 'Weather forecasts' "
    "WHERE description = 'Weather forcasts';",
""
};
        if (!performActualUpdate(updates, "1070", dbver))
            return false;
    }

    if (dbver == "1070")
    {
        const QString updates[] = {
"UPDATE recorded SET bookmark = NULL WHERE bookmark = 'NULL';",
""
};
        if (!performActualUpdate(updates, "1071", dbver))
            return false;
    }

    if (dbver == "1071")
    {
        const QString updates[] = {
"ALTER TABLE program ADD COLUMN manualid INT UNSIGNED NOT NULL DEFAULT 0;",
"ALTER TABLE program DROP PRIMARY KEY;",
"ALTER TABLE program ADD PRIMARY KEY (chanid, starttime, manualid);",
"ALTER TABLE recordmatch ADD COLUMN manualid INT UNSIGNED;",
""
};
        if (!performActualUpdate(updates, "1072", dbver))
            return false;
    }

    if (dbver == "1072")
    {
        const QString updates[] = {
"ALTER TABLE recorded ADD INDEX (title);",
""
};
        if (!performActualUpdate(updates, "1073", dbver))
            return false;
    }

    if (dbver == "1073")
    {
        const QString updates[] = {
"ALTER TABLE capturecard ADD COLUMN firewire_connection INT UNSIGNED NOT NULL DEFAULT 0;",
""
};
        if (!performActualUpdate(updates, "1074", dbver))
            return false;
    }

    if (dbver == "1074")
    {
        const QString updates[] = {
"INSERT INTO profilegroups SET name = \"USB Mpeg-4 Encoder (Plextor ConvertX, etc)\", cardtype = 'GO7007', is_default = 1;",
"INSERT INTO recordingprofiles SET name = \"Default\", profilegroup = 8;",
"INSERT INTO recordingprofiles SET name = \"Live TV\", profilegroup = 8;",
"INSERT INTO recordingprofiles SET name = \"High Quality\", profilegroup = 8;",
"INSERT INTO recordingprofiles SET name = \"Low Quality\", profilegroup = 8;",
""
};
        if (!performActualUpdate(updates, "1075", dbver))
            return false;
    }

    if (dbver == "1075")
    {
        const QString updates[] = {
"CREATE TABLE IF NOT EXISTS recordedprogram ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  endtime datetime NOT NULL default '0000-00-00 00:00:00',"
"  title varchar(128) NOT NULL default '',"
"  subtitle varchar(128) NOT NULL default '',"
"  description text NOT NULL,"
"  category varchar(64) NOT NULL default '',"
"  category_type varchar(64) NOT NULL default '',"
"  airdate year(4) NOT NULL default '0000',"
"  stars float unsigned NOT NULL default '0',"
"  previouslyshown tinyint(4) NOT NULL default '0',"
"  title_pronounce varchar(128) NOT NULL default '',"
"  stereo tinyint(1) NOT NULL default '0',"
"  subtitled tinyint(1) NOT NULL default '0',"
"  hdtv tinyint(1) NOT NULL default '0',"
"  closecaptioned tinyint(1) NOT NULL default '0',"
"  partnumber int(11) NOT NULL default '0',"
"  parttotal int(11) NOT NULL default '0',"
"  seriesid varchar(12) NOT NULL default '',"
"  originalairdate date default NULL,"
"  showtype varchar(30) NOT NULL default '',"
"  colorcode varchar(20) NOT NULL default '',"
"  syndicatedepisodenumber varchar(20) NOT NULL default '',"
"  programid varchar(20) NOT NULL default '',"
"  manualid int(10) unsigned NOT NULL default '0',"
"  PRIMARY KEY  (chanid,starttime,manualid),"
"  KEY endtime (endtime),"
"  KEY title (title),"
"  KEY title_pronounce (title_pronounce),"
"  KEY seriesid (seriesid),"
"  KEY programid (programid),"
"  KEY id_start_end (chanid,starttime,endtime)"
");",
"CREATE TABLE IF NOT EXISTS recordedcredits ("
"  person mediumint(8) unsigned NOT NULL default '0',"
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  role set('actor','director','producer','executive_producer','writer','guest_star','host','adapter','presenter','commentator','guest') NOT NULL default '',"
"  UNIQUE KEY chanid (chanid,starttime,person,role),"
"  KEY person (person,role)"
");",
"CREATE TABLE IF NOT EXISTS recordedrating ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  system char(8) NOT NULL default '',"
"  rating char(8) NOT NULL default '',"
"  UNIQUE KEY chanid (chanid,starttime,system,rating),"
"  KEY starttime (starttime,system)"
");",
""
       };
       
        if (!performActualUpdate(updates, "1076", dbver))
            return false;
    }

    if (dbver == "1076")
    {
        const QString updates[] = {
"ALTER TABLE channel MODIFY COLUMN serviceid mediumint unsigned;",
""
        }; 
        if (!performActualUpdate(updates, "1077", dbver))
            return false;
    }

    if (dbver == "1077")
    {
        const QString updates[] = {
"INSERT INTO `dtv_privatetypes` "
"(`sitype`,`networkid`,`private_type`,`private_value`) VALUES "
"('dvb',40999,'parse_subtitle_list',"
"'1070,1049,1041,1039,1038,1030,1016,1131,1068,1069');",
""
        }; 
        if (!performActualUpdate(updates, "1078", dbver))
            return false;
    }

    if (dbver == "1078")
    {
        const QString updates[] = {
"ALTER TABLE capturecard ADD COLUMN dvb_hw_decoder INT DEFAULT '0';",
""
};
        if (!performActualUpdate(updates, "1079", dbver))
            return false;
    }

    if (dbver == "1079")
    {
        const QString updates[] = {
"ALTER TABLE oldrecorded ADD COLUMN recordid INT NOT NULL DEFAULT 0;",
""
};
        if (!performActualUpdate(updates, "1080", dbver))
            return false;
    }

    if (dbver == "1080")
    {
        const QString updates[] = {
"ALTER table recorded ADD INDEX (recordid);",
"ALTER table oldrecorded ADD INDEX (recordid);",
""
};
        if (!performActualUpdate(updates, "1081", dbver))
            return false;
    }

    if (dbver == "1081")
    {
        const QString updates[] = {
"CREATE TABLE IF NOT EXISTS oldfind ("
"recordid INT NOT NULL DEFAULT 0,"
"findid INT NOT NULL DEFAULT 0,"
"PRIMARY KEY (recordid, findid)"
");",
"ALTER TABLE record ADD COLUMN parentid INT NOT NULL DEFAULT 0;",
""
};
        if (!performActualUpdate(updates, "1082", dbver))
            return false;
    }

    if (dbver == "1082")
    {
        const QString updates[] = {
"CREATE TABLE IF NOT EXISTS videobookmarks ("
"  filename varchar(255) NOT NULL,"
"  bookmark varchar(128) default NULL,"
"  PRIMARY KEY (filename)"
");",
""
};
        if (!performActualUpdate(updates, "1083", dbver))
            return false;
    }

    if (dbver == "1083")
    {
        const QString updates[] = {
"DELETE FROM settings where value = 'UseCategoriesAsRecGroups';",
""
};
        if (!performActualUpdate(updates, "1084", dbver))
            return false;
    }

    if (dbver == "1084")
    {
        const QString updates[] = {
"INSERT INTO recordingprofiles SET name = 'High Quality', profilegroup = 6;",
"INSERT INTO recordingprofiles SET name = 'Medium Quality', profilegroup = 6;",
"INSERT INTO recordingprofiles SET name = 'Low Quality', profilegroup = 6;",
"REPLACE INTO settings SET value = 'DefaultTranscoder', data = '0';",
"ALTER TABLE record ADD COLUMN transcoder INT NOT NULL DEFAULT 0;",
"ALTER TABLE recorded ADD COLUMN transcoder INT NOT NULL DEFAULT 0;",
""
};
        if (!performActualUpdate(updates, "1085", dbver))
            return false;

        MSqlQuery recordids(MSqlQuery::InitCon());
        recordids.prepare(
            "SELECT r.recordid"
            "  FROM record r, recordingprofiles p, codecparams c"
            "  WHERE c.name = 'autotranscode'"
            "    AND c.value = 1"
            "    AND c.profile = p.id"
            "    AND r.profile = p.name"
            ";");
        if (recordids.exec() && recordids.isActive() && recordids.size() > 0)
        {
            MSqlQuery update(MSqlQuery::InitCon());
            while (recordids.next())
            {
                update.prepare(QString(
                    "UPDATE record SET autotranscode = 1 WHERE recordid = %1;")
                    .arg(recordids.value(0).toInt()));
                update.exec();
            }
        }
    }

    if (dbver == "1085")
    {
        const QString updates[] = {
"DROP TABLE IF EXISTS pidcache;",
"CREATE TABLE IF NOT EXISTS pidcache ("
"  chanid smallint(6) NOT NULL default '0',"
"  pid int(11) NOT NULL default '-1',"
"  tableid int(11) NOT NULL default '-1',"
"  INDEX(chanid)"
");",
""
};
        if (!performActualUpdate(updates, "1086", dbver))
            return false;
    }

    if (dbver == "1086")
    {
        const QString updates[] = {
"CREATE TABLE IF NOT EXISTS xvmc_buffer_settings ( "
"  id int(11) NOT NULL auto_increment, "
"  description varchar(255) NOT NULL default '',"
"  osd_num int(11) NOT NULL default '0',"
"  osd_res_num int(11) NOT NULL default '0',"
"  min_surf int(11) NOT NULL default '0',"
"  max_surf int(11) NOT NULL default '0',"
"  decode_num int(11) NOT NULL default '0',"
"  agressive int(11) NOT NULL default '1',"
"  PRIMARY KEY  (id) );",
"INSERT INTO xvmc_buffer_settings VALUES (1,'Default / nVidia',2,2,8,16,8,1);",
"INSERT INTO xvmc_buffer_settings VALUES (2,'VLD (More decode buffers)',2,2,8,16,16,1);",
""
}; 
        
        if (!performActualUpdate(updates, "1087", dbver))
            return false;
    }
 
    if (dbver == "1087")
    {
        const QString updates[] = {
"ALTER TABLE record ADD COLUMN tsdefault FLOAT NOT NULL DEFAULT 1.0;",
"ALTER TABLE recorded ADD COLUMN timestretch FLOAT NOT NULL DEFAULT 1.0;",
"UPDATE record SET tsdefault = 1.0;", // we've had the default not take before
"UPDATE recorded SET timestretch = 1.0;", 
""
};
        performActualUpdate(updates, "1088", dbver);
    }

    if (dbver == "1088")
    {
        const QString updates[] = {
"ALTER TABLE oldrecorded ADD COLUMN station VARCHAR(20) NOT NULL DEFAULT '';",
"UPDATE oldrecorded SET station=chanid;",
"ALTER TABLE oldrecorded ADD rectype INT(10) UNSIGNED NOT NULL DEFAULT 0;",
"UPDATE oldrecorded SET rectype=1;",
"ALTER TABLE oldrecorded ADD duplicate TINYINT(1) NOT NULL DEFAULT 0;",
"UPDATE oldrecorded SET duplicate=1;",
"ALTER TABLE oldrecorded ADD recstatus INT NOT NULL DEFAULT 0;",
"UPDATE oldrecorded SET recstatus=-3;",
"ALTER TABLE oldrecorded DROP PRIMARY KEY;",
"ALTER TABLE oldrecorded ADD PRIMARY KEY (station,starttime,title);",
""
}; 
        
        if (!performActualUpdate(updates, "1089", dbver))
            return false;
    }

    if (dbver == "1089")
    {
        const QString updates[] = {
"INSERT INTO profilegroups SET name = 'DBOX2 Input', cardtype = 'DBOX2', is_default = 1;",
"ALTER TABLE capturecard ADD COLUMN dbox2_port INT UNSIGNED NOT NULL DEFAULT 31338;",
"ALTER TABLE capturecard ADD COLUMN dbox2_httpport INT UNSIGNED NOT NULL DEFAULT 80;",
"ALTER TABLE capturecard ADD COLUMN dbox2_host varchar(32) NULL;",
""
};

        if (!performActualUpdate(updates, "1090", dbver))
            return false;
    }

    if (dbver == "1090")
    {
        const QString updates[] = {
"DELETE FROM dtv_privatetypes WHERE sitype='dvb' AND networkid=40999 AND private_type='parse_subtitle_list';",
"INSERT INTO dtv_privatetypes (sitype,networkid,private_type,private_value) VALUES ('dvb',40999,'parse_subtitle_list','1070,1308,1041,1306,1307,1030,1016,1131,1068,1069');",
""
};

        if (!performActualUpdate(updates, "1091", dbver))
            return false;
    }

    if (dbver == "1091")
    {
        const QString updates[] = {
"ALTER TABLE capturecard CHANGE dvb_recordts dvb_recordts INT DEFAULT '1';",
"UPDATE capturecard SET dvb_recordts=1;",
""
};

        if (!performActualUpdate(updates, "1092", dbver))
            return false;
    }

    if (dbver == "1092")
    {
        const QString updates[] = {
"ALTER TABLE recorded ADD COLUMN recpriority INT NOT NULL DEFAULT 0;",
""
};
        
        if (!performActualUpdate(updates, "1093", dbver))
            return false;
    }

    if (dbver == "1093")
    {
        MSqlQuery recordids(MSqlQuery::InitCon());
        recordids.prepare("SELECT recordid,recpriority FROM record;");
        if (!recordids.exec())
            return false;

        if (recordids.isActive() && recordids.size() > 0)
        {
            MSqlQuery update(MSqlQuery::InitCon());
            while (recordids.next())
            {
                update.prepare(QString(
                    "UPDATE recorded SET recpriority = %1 WHERE recordid = %2;")
                    .arg(recordids.value(1).toInt())
                    .arg(recordids.value(0).toInt()));
                if (!update.exec())
                    return false;
            }
        }

        if (!UpdateDBVersionNumber("1094"))
            return false;

        dbver = "1094";
    }

    if (dbver == "1094")
    {
        const QString updates[] = {
"ALTER TABLE recorded ADD COLUMN basename varchar(128) NOT NULL DEFAULT '';",
"UPDATE recorded SET basename = CONCAT(chanid, '_', DATE_FORMAT(starttime, '%Y%m%d%H%i00'), '_', DATE_FORMAT(endtime, '%Y%m%d%H%i00'), '.nuv');",
""
};

        if (!performActualUpdate(updates, "1095", dbver))
            return false;
    }

    if (dbver == "1095")
    {
        const QString updates[] = {
"ALTER TABLE recorded ADD COLUMN progstart DATETIME NOT NULL;",
"UPDATE recorded SET progstart = starttime;",
"ALTER TABLE recorded ADD COLUMN progend DATETIME NOT NULL;",
"UPDATE recorded SET progend = endtime;",
""
};

        if (!performActualUpdate(updates, "1096", dbver))
            return false;
    }

    if (dbver == "1096")
    {
        const QString updates[] = {
"ALTER TABLE capturecard ADD COLUMN signal_timeout int NOT NULL default '1000';",
"ALTER TABLE capturecard ADD COLUMN channel_timeout int NOT NULL default '3000';",
"DROP TABLE IF EXISTS dvb_signal_quality;",
"DROP TABLE IF EXISTS dvb_transport;",
"DROP TABLE IF EXISTS dvb_channel;",
"DROP TABLE IF EXISTS dvb_pids;",
"DROP TABLE IF EXISTS dvb_sat;",
""
};

        if (!performActualUpdate(updates, "1097", dbver))
            return false;
    }

    if (dbver == "1097")
    {
        const QString updates[] = {
"ALTER TABLE oldrecorded ADD COLUMN reactivate SMALLINT NOT NULL DEFAULT 0;",
""
};

        if (!performActualUpdate(updates, "1098", dbver))
            return false;
    }

    if (dbver == "1098")
    {
        const QString updates[] = {
"UPDATE record SET findid=TO_DAYS(startdate) WHERE type=6;",
""
};

        if (!performActualUpdate(updates, "1099", dbver))
            return false;
    }

    if (dbver == "1099")
    {
        const QString updates[] = {
"ALTER TABLE videosource ADD COLUMN useeit SMALLINT NOT NULL DEFAULT 0;",
""
};

        if (!performActualUpdate(updates, "1100", dbver))
            return false;
    }

    if (dbver == "1100")
    {
        const QString updates[] = {
"CREATE TABLE IF NOT EXISTS tvchain ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  chainid varchar(128) NOT NULL default '',"
"  chainpos int(10) NOT NULL default '0',"
"  discontinuity tinyint(1) NOT NULL default '0',"
"  watching int(10) NOT NULL default '0',"
"  PRIMARY KEY  (chanid,starttime)"
");",
""
};

        if (!performActualUpdate(updates, "1101", dbver))
            return false;
    }

    if (dbver == "1101")
    {
        const QString updates[] = {
"ALTER TABLE tvchain ADD COLUMN hostprefix VARCHAR(128) NOT NULL DEFAULT '';",
""
};

        if (!performActualUpdate(updates, "1102", dbver))
            return false;
    }

    if (dbver == "1102")
    {
        const QString updates[] = {
"ALTER TABLE tvchain ADD COLUMN cardtype VARCHAR(32) NOT NULL DEFAULT 'V4L';",
""
};
        if (!performActualUpdate(updates, "1103", dbver))
            return false;
    }

    if (dbver == "1103")
    {
        const QString updates[] = {
"CREATE TABLE IF NOT EXISTS inuseprograms ("
"  chanid int(10) unsigned NOT NULL default '0',"
"  starttime datetime NOT NULL default '0000-00-00 00:00:00',"
"  playid varchar(128) NOT NULL default '',"
"  lastupdatetime datetime NOT NULL default '0000-00-00 00:00:00',"
"  INDEX (chanid,starttime)"
");",
""
};

        if (!performActualUpdate(updates, "1104", dbver))
            return false;
    }

    if (dbver == "1104")
    {
        const QString updates[] = {
"ALTER TABLE tvchain ADD COLUMN input VARCHAR(32) NOT NULL DEFAULT '';",
"ALTER TABLE tvchain ADD COLUMN channame VARCHAR(32) NOT NULL DEFAULT '';",
""
};
        if (!performActualUpdate(updates, "1105", dbver))
            return false;
    }

    if (dbver == "1105")
    {
        const QString updates[] = {
"UPDATE settings SET data = '1' WHERE value = 'AutoExpireMethod' "
      " AND data = '0';",
""
};
        if (!performActualUpdate(updates, "1106", dbver))
            return false;
    }

    if (dbver == "1106")
    {
        const QString updates[] = {
"ALTER TABLE inuseprograms ADD COLUMN hostname VARCHAR(255) NOT NULL DEFAULT '';",
"ALTER TABLE inuseprograms CHANGE playid recusage VARCHAR(128) NOT NULL DEFAULT '';",
""
};
        if (!performActualUpdate(updates, "1107", dbver))
            return false;
    }

    if (dbver == "1107")
    {
        const QString updates[] = {
"ALTER TABLE record ADD COLUMN playgroup VARCHAR(32) NOT NULL DEFAULT 'Default'; ", 
"UPDATE record SET playgroup = 'Default'; ", 
"ALTER TABLE recorded ADD COLUMN playgroup VARCHAR(32) NOT NULL DEFAULT 'Default'; ", 
"UPDATE recorded SET playgroup = 'Default'; ", 
"CREATE TABLE playgroup ("
"  name VARCHAR(32) NOT NULL PRIMARY KEY, "
"  titlematch VARCHAR(255) NOT NULL DEFAULT '', "
"  skipahead INT NOT NULL DEFAULT 0, "
"  skipback INT NOT NULL DEFAULT 0, "
"  timestretch INT NOT NULL DEFAULT 0 "
");",
"INSERT INTO playgroup (name,skipahead,skipback,timestretch) "
"            VALUES ('Default',30,5,100);",
""
};

        if (!performActualUpdate(updates, "1108", dbver))
            return false;
    }

    if (dbver == "1108")
    {
        const QString updates[] = {
"ALTER TABLE oldrecorded ADD COLUMN generic TINYINT(1) DEFAULT 0;",
"ALTER TABLE program ADD COLUMN generic TINYINT(1) DEFAULT 0;",
"UPDATE program SET generic = 1 WHERE "
"  ((programid = '' AND subtitle = '' AND description = '') OR "
"   (programid <> '' AND category_type = 'series' AND "
"    program.programid LIKE '%0000'));",
""
};

        if (!performActualUpdate(updates, "1109", dbver))
            return false;
    }

    if (dbver == "1109")
    {
        const QString updates[] = {
"ALTER TABLE recordedprogram ADD COLUMN generic TINYINT(1) DEFAULT 0;",
""
};

        if (!performActualUpdate(updates, "1110", dbver))
            return false;
    }

    if (dbver == "1110")
    {
        const QString updates[] = {
"ALTER TABLE recorded ADD COLUMN profile VARCHAR(32) NOT NULL DEFAULT '';",
""
};
        if (!performActualUpdate(updates, "1111", dbver))
            return false;
    }

    if (dbver == "1111")
    {
        const QString updates[] = {
"DROP TABLE IF EXISTS conflictresolutionany;",
"DROP TABLE IF EXISTS conflictresolutionoverride;",
"DROP TABLE IF EXISTS conflictresolutionsingle;",
"DROP TABLE IF EXISTS recordoverride;",
""
};
        if (!performActualUpdate(updates, "1112", dbver))
            return false;
    }

    if (dbver == "1112")
    {
        const QString updates[] = {
"DELETE from settings WHERE value LIKE 'DailyWakeup%';",
""
};

        if (!performActualUpdate(updates, "1113", dbver))
            return false;
    }

    if (dbver == "1113")
    {
        const QString updates[] = {
"INSERT INTO settings VALUES ('NextSuggestedMythfilldatabaseRun',NULL,NULL);",
"INSERT INTO settings VALUES ('HonorGrabberNextSuggestedMythfilldatabaseRunTime','1',NULL);",
""
};

        if (!performActualUpdate(updates, "1114", dbver))
            return false;
    }

    if (dbver == "1114")
    {
        const QString updates[] = {
"ALTER TABLE tvchain ADD COLUMN endtime DATETIME NOT NULL default '0000-00-00 00:00:00';",
""
};
        if (!performActualUpdate(updates, "1115", dbver))
            return false;
    }
    
    if (dbver == "1115")
    {
        const QString updates[] = {
"DELETE FROM keybindings WHERE context = 'Music' AND action LIKE 'JUMP%';",
"DELETE FROM keybindings WHERE context = 'Game' AND action LIKE 'JUMP%';",
""
};

        if (!performActualUpdate(updates, "1116", dbver))
            return false;
    }

    if (dbver == "1116")
    {
        const QString updates[] = {
"ALTER TABLE capturecard ADD COLUMN displayname VARCHAR(32) NOT NULL DEFAULT '';",
""
};
        if (!performActualUpdate(updates, "1117", dbver))
            return false;
    }

    if (dbver == "1117")
    {
        const QString updates[] = {
"ALTER TABLE capturecard DROP COLUMN displayname;",
"ALTER TABLE cardinput ADD COLUMN displayname VARCHAR(64) NOT NULL DEFAULT '';",
""
};
        if (!performActualUpdate(updates, "1118", dbver))
            return false;
    }

    if (dbver == "1118")
    {
        const QString updates[] = {
"UPDATE settings SET value = 'MythFillGrabberSuggestsTime' "
    "WHERE value = 'HonorGrabberNextSuggestedMythfilldatabaseRunTime';",
"UPDATE settings SET value = 'MythFillSuggestedRunTime', "
    "    data = '1970-01-01T00:00:00' "
    "WHERE value = 'NextSuggestedMythfilldatabaseRun';",
""
};
        if (!performActualUpdate(updates, "1119", dbver))
            return false;
    }

    if (dbver == "1119")
    {
        const QString updates[] = {
// "UPDATE playgroup, settings set playgroup.skipahead = settings.data"
// " WHERE settings.value = 'FastForwardAmount' AND playgroup.name = 'Default';",
// "UPDATE playgroup, settings set playgroup.skipback = settings.data"
// " WHERE settings.value = 'RewindAmount' AND playgroup.name = 'Default';",
"ALTER TABLE playgroup ADD COLUMN jump INT NOT NULL DEFAULT 0;",
// "UPDATE playgroup, settings set playgroup.jump = settings.data"
// " WHERE settings.value = 'JumpAmount' AND playgroup.name = 'Default';",
""
};
        if (!performActualUpdate(updates, "1120", dbver))
            return false;
    }

    if (dbver == "1120")
    {
        const QString updates[] = {
"UPDATE keybindings SET action = 'ADJUSTSTRETCH' "
"       WHERE action = 'TOGGLESTRETCH';",
""
};
        if (!performActualUpdate(updates, "1121", dbver))
            return false;
    }

    if (dbver == "1121")
    {
        const QString updates[] = {
"ALTER TABLE channel CHANGE channum channum VARCHAR(10) NOT NULL DEFAULT '';",
""
};
        if (!performActualUpdate(updates, "1122", dbver))
            return false;
    }

    if (dbver == "1122")
    {
        const QString updates[] = {
"ALTER TABLE recorded ADD duplicate TINYINT(1) NOT NULL DEFAULT 0;",
"UPDATE recorded SET duplicate=1;",
""
}; 
        
        if (!performActualUpdate(updates, "1123", dbver))
            return false;
    }

// Drop xvmc_buffer_settings table in 0.20
// Drop dvb_dmx_buf_size and dvb_pkt_buf_size columns of channel in 0.20

    return true;
}



bool InitializeDatabase(void)
{
    MSqlQuery query(MSqlQuery::InitCon());
    query.prepare("SHOW TABLES;");

    // check for > 1 table here since the schemalock table should exist
    if (query.exec() && query.isActive() && query.size() > 1)
    {
        QString msg = QString(
            "Told to create a NEW database schema, but the database\n"
            "already has %1 tables.\n"
            "If you are sure this is a good mythtv database, verify\n"
            "that the settings table has the DBSchemaVer variable.\n")
            .arg(query.size() - 1);
        VERBOSE(VB_IMPORTANT, msg);
        return false;   
    }

    VERBOSE(VB_IMPORTANT, "Inserting MythTV initial database information.");

    const QString updates[] = {
"CREATE TABLE IF NOT EXISTS `callsignnetworkmap` ("
"  `id` int(11) NOT NULL auto_increment,"
"  `callsign` varchar(20) NOT NULL default '',"
"  `network` varchar(20) NOT NULL default '',"
"  PRIMARY KEY  (`id`),"
"  UNIQUE KEY `callsign` (`callsign`)"
");",
"CREATE TABLE IF NOT EXISTS `capturecard` ("
"  `cardid` int(10) unsigned NOT NULL auto_increment,"
"  `videodevice` varchar(128) default NULL,"
"  `audiodevice` varchar(128) default NULL,"
"  `vbidevice` varchar(128) default NULL,"
"  `cardtype` varchar(32) default 'V4L',"
"  `defaultinput` varchar(32) default 'Television',"
"  `audioratelimit` int(11) default NULL,"
"  `hostname` varchar(255) default NULL,"
"  `dvb_swfilter` int(11) default '0',"
"  `dvb_recordts` int(11) default '1',"
"  `dvb_sat_type` int(11) NOT NULL default '0',"
"  `dvb_wait_for_seqstart` int(11) NOT NULL default '1',"
"  `dvb_dmx_buf_size` int(11) NOT NULL default '8192',"
"  `dvb_pkt_buf_size` int(11) NOT NULL default '8192',"
"  `skipbtaudio` tinyint(1) default '0',"
"  `dvb_on_demand` tinyint(4) NOT NULL default '0',"
"  `dvb_diseqc_type` smallint(6) default NULL,"
"  `firewire_port` int(10) unsigned NOT NULL default '0',"
"  `firewire_node` int(10) unsigned NOT NULL default '2',"
"  `firewire_speed` int(10) unsigned NOT NULL default '0',"
"  `firewire_model` varchar(32) default NULL,"
"  `firewire_connection` int(10) unsigned NOT NULL default '0',"
"  `dvb_hw_decoder` int(11) default '0',"
"  `dbox2_port` int(10) unsigned NOT NULL default '31338',"
"  `dbox2_httpport` int(10) unsigned NOT NULL default '80',"
"  `dbox2_host` varchar(32) default NULL,"
"  `signal_timeout` int(11) NOT NULL default '1000',"
"  `channel_timeout` int(11) NOT NULL default '3000',"
"  PRIMARY KEY  (`cardid`)"
");",
"CREATE TABLE IF NOT EXISTS `cardinput` ("
"  `cardinputid` int(10) unsigned NOT NULL auto_increment,"
"  `cardid` int(10) unsigned NOT NULL default '0',"
"  `sourceid` int(10) unsigned NOT NULL default '0',"
"  `inputname` varchar(32) NOT NULL default '',"
"  `externalcommand` varchar(128) default NULL,"
"  `preference` int(11) NOT NULL default '0',"
"  `shareable` char(1) default 'N',"
"  `tunechan` varchar(5) NOT NULL default '',"
"  `startchan` varchar(5) NOT NULL default '',"
"  `freetoaironly` tinyint(1) default '1',"
"  `diseqc_port` smallint(6) default NULL,"
"  `diseqc_pos` float default NULL,"
"  `lnb_lof_switch` int(11) default '11700000',"
"  `lnb_lof_hi` int(11) default '10600000',"
"  `lnb_lof_lo` int(11) default '9750000',"
"  PRIMARY KEY  (`cardinputid`)"
");",
"CREATE TABLE IF NOT EXISTS `channel` ("
"  `chanid` int(10) unsigned NOT NULL default '0',"
"  `channum` varchar(5) NOT NULL default '',"
"  `freqid` varchar(10) default NULL,"
"  `sourceid` int(10) unsigned default NULL,"
"  `callsign` varchar(20) NOT NULL default '',"
"  `name` varchar(64) NOT NULL default '',"
"  `icon` varchar(255) NOT NULL default 'none',"
"  `finetune` int(11) default NULL,"
"  `videofilters` varchar(255) NOT NULL default '',"
"  `xmltvid` varchar(64) NOT NULL default '',"
"  `recpriority` int(10) NOT NULL default '0',"
"  `contrast` int(11) default '32768',"
"  `brightness` int(11) default '32768',"
"  `colour` int(11) default '32768',"
"  `hue` int(11) default '32768',"
"  `tvformat` varchar(10) NOT NULL default 'Default',"
"  `commfree` tinyint(4) NOT NULL default '0',"
"  `visible` tinyint(1) NOT NULL default '1',"
"  `outputfilters` varchar(255) NOT NULL default '',"
"  `useonairguide` tinyint(1) default '0',"
"  `mplexid` smallint(6) default NULL,"
"  `serviceid` mediumint(8) unsigned default NULL,"
"  `atscsrcid` int(11) default NULL,"
"  PRIMARY KEY  (`chanid`),"
"  KEY `channel_src` (`channum`,`sourceid`)"
");",
"CREATE TABLE IF NOT EXISTS `codecparams` ("
"  `profile` int(10) unsigned NOT NULL default '0',"
"  `name` varchar(128) NOT NULL default '',"
"  `value` varchar(128) default NULL,"
"  PRIMARY KEY  (`profile`,`name`)"
");",
"CREATE TABLE IF NOT EXISTS `credits` ("
"  `person` mediumint(8) unsigned NOT NULL default '0',"
"  `chanid` int(10) unsigned NOT NULL default '0',"
"  `starttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `role` set('actor','director','producer','executive_producer','writer',"
"             'guest_star','host','adapter','presenter','commentator','guest')"
"         NOT NULL default '',"
"  UNIQUE KEY `chanid` (`chanid`,`starttime`,`person`,`role`),"
"  KEY `person` (`person`,`role`)"
");",
"CREATE TABLE IF NOT EXISTS `dtv_multiplex` ("
"  `mplexid` smallint(6) NOT NULL auto_increment,"
"  `sourceid` smallint(6) default NULL,"
"  `transportid` int(11) default NULL,"
"  `networkid` int(11) default NULL,"
"  `frequency` int(11) default NULL,"
"  `inversion` char(1) default 'a',"
"  `symbolrate` int(11) default NULL,"
"  `fec` varchar(10) default 'auto',"
"  `polarity` char(1) default NULL,"
"  `modulation` varchar(10) default 'auto',"
"  `bandwidth` char(1) default 'a',"
"  `lp_code_rate` varchar(10) default 'auto',"
"  `transmission_mode` char(1) default 'a',"
"  `guard_interval` varchar(10) default 'auto',"
"  `visible` smallint(1) NOT NULL default '0',"
"  `constellation` varchar(10) default 'auto',"
"  `hierarchy` varchar(10) default 'auto',"
"  `hp_code_rate` varchar(10) default 'auto',"
"  `sistandard` varchar(10) default 'dvb',"
"  `serviceversion` smallint(6) default '33',"
"  `updatetimestamp` timestamp(14) NOT NULL,"
"  PRIMARY KEY  (`mplexid`)"
");",
"CREATE TABLE IF NOT EXISTS `dtv_privatetypes` ("
"  `sitype` varchar(4) NOT NULL default '',"
"  `networkid` int(11) NOT NULL default '0',"
"  `private_type` varchar(20) NOT NULL default '',"
"  `private_value` varchar(100) NOT NULL default ''"
");",
"CREATE TABLE IF NOT EXISTS `favorites` ("
"  `favid` int(11) unsigned NOT NULL auto_increment,"
"  `userid` int(11) unsigned NOT NULL default '0',"
"  `chanid` int(11) unsigned NOT NULL default '0',"
"  PRIMARY KEY  (`favid`)"
");",
"CREATE TABLE IF NOT EXISTS `housekeeping` ("
"  `tag` varchar(64) NOT NULL default '',"
"  `lastrun` datetime default NULL,"
"  PRIMARY KEY  (`tag`)"
");",
"CREATE TABLE IF NOT EXISTS `inuseprograms` ("
"  `chanid` int(10) unsigned NOT NULL default '0',"
"  `starttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `recusage` varchar(128) NOT NULL default '',"
"  `lastupdatetime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `hostname` varchar(255) NOT NULL default '',"
"  KEY `chanid` (`chanid`,`starttime`)"
");",
"CREATE TABLE IF NOT EXISTS `jobqueue` ("
"  `id` int(11) NOT NULL auto_increment,"
"  `chanid` int(10) NOT NULL default '0',"
"  `starttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `inserttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `type` int(11) NOT NULL default '0',"
"  `cmds` int(11) NOT NULL default '0',"
"  `flags` int(11) NOT NULL default '0',"
"  `status` int(11) NOT NULL default '0',"
"  `statustime` timestamp NOT NULL,"
"  `hostname` varchar(255) NOT NULL default '',"
"  `args` blob NOT NULL,"
"  `comment` varchar(128) NOT NULL default '',"
"  PRIMARY KEY  (`id`),"
"  UNIQUE KEY `chanid` (`chanid`,`starttime`,`type`,`inserttime`)"
");",
"CREATE TABLE IF NOT EXISTS `jumppoints` ("
"  `destination` varchar(128) NOT NULL default '',"
"  `description` varchar(255) default NULL,"
"  `keylist` varchar(32) default NULL,"
"  `hostname` varchar(255) NOT NULL default '',"
"  PRIMARY KEY  (`destination`,`hostname`)"
");",
"CREATE TABLE IF NOT EXISTS `keybindings` ("
"  `context` varchar(32) NOT NULL default '',"
"  `action` varchar(32) NOT NULL default '',"
"  `description` varchar(255) default NULL,"
"  `keylist` varchar(32) default NULL,"
"  `hostname` varchar(255) NOT NULL default '',"
"  PRIMARY KEY  (`context`,`action`,`hostname`)"
");",
"CREATE TABLE IF NOT EXISTS `keyword` ("
"  `phrase` varchar(128) NOT NULL default '',"
"  `searchtype` int(10) unsigned NOT NULL default '3',"
"  UNIQUE KEY `phrase` (`phrase`,`searchtype`)"
");",
"CREATE TABLE IF NOT EXISTS `mythlog` ("
"  `logid` int(10) unsigned NOT NULL auto_increment,"
"  `module` varchar(32) NOT NULL default '',"
"  `priority` int(11) NOT NULL default '0',"
"  `acknowledged` tinyint(1) default '0',"
"  `logdate` datetime default NULL,"
"  `host` varchar(128) default NULL,"
"  `message` varchar(255) NOT NULL default '',"
"  `details` text,"
"  PRIMARY KEY  (`logid`)"
");",
"CREATE TABLE IF NOT EXISTS `networkiconmap` ("
"  `id` int(11) NOT NULL auto_increment,"
"  `network` varchar(20) NOT NULL default '',"
"  `url` varchar(255) NOT NULL default '',"
"  PRIMARY KEY  (`id`),"
"  UNIQUE KEY `network` (`network`)"
");",
"CREATE TABLE IF NOT EXISTS `oldfind` ("
"  `recordid` int(11) NOT NULL default '0',"
"  `findid` int(11) NOT NULL default '0',"
"  PRIMARY KEY  (`recordid`,`findid`)"
");",
"CREATE TABLE IF NOT EXISTS `oldprogram` ("
"  `oldtitle` varchar(128) NOT NULL default '',"
"  `airdate` datetime NOT NULL default '0000-00-00 00:00:00',"
"  PRIMARY KEY  (`oldtitle`)"
");",
"CREATE TABLE IF NOT EXISTS `oldrecorded` ("
"  `chanid` int(10) unsigned NOT NULL default '0',"
"  `starttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `endtime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `title` varchar(128) NOT NULL default '',"
"  `subtitle` varchar(128) NOT NULL default '',"
"  `description` text NOT NULL,"
"  `category` varchar(64) NOT NULL default '',"
"  `seriesid` varchar(12) NOT NULL default '',"
"  `programid` varchar(20) NOT NULL default '',"
"  `findid` int(11) NOT NULL default '0',"
"  `recordid` int(11) NOT NULL default '0',"
"  `station` varchar(20) NOT NULL default '',"
"  `rectype` int(10) unsigned NOT NULL default '0',"
"  `duplicate` tinyint(1) NOT NULL default '0',"
"  `recstatus` int(11) NOT NULL default '0',"
"  `reactivate` smallint(6) NOT NULL default '0',"
"  `generic` tinyint(1) default '0',"
"  PRIMARY KEY  (`station`,`starttime`,`title`),"
"  KEY `endtime` (`endtime`),"
"  KEY `title` (`title`),"
"  KEY `seriesid` (`seriesid`),"
"  KEY `programid` (`programid`),"
"  KEY `recordid` (`recordid`)"
");",
"CREATE TABLE IF NOT EXISTS `people` ("
"  `person` mediumint(8) unsigned NOT NULL auto_increment,"
"  `name` char(128) NOT NULL default '',"
"  PRIMARY KEY  (`person`),"
"  UNIQUE KEY `name` (`name`(41))"
");",
"CREATE TABLE IF NOT EXISTS `pidcache` ("
"  `chanid` smallint(6) NOT NULL default '0',"
"  `pid` int(11) NOT NULL default '-1',"
"  `tableid` int(11) NOT NULL default '-1',"
"  KEY `chanid` (`chanid`)"
");",
"CREATE TABLE IF NOT EXISTS `playgroup` ("
"  `name` varchar(32) NOT NULL default '',"
"  `titlematch` varchar(255) NOT NULL default '',"
"  `skipahead` int(11) NOT NULL default '0',"
"  `skipback` int(11) NOT NULL default '0',"
"  `timestretch` int(11) NOT NULL default '0',"
"  PRIMARY KEY  (`name`)"
");",
"CREATE TABLE IF NOT EXISTS `profilegroups` ("
"  `id` int(10) unsigned NOT NULL auto_increment,"
"  `name` varchar(128) default NULL,"
"  `cardtype` varchar(32) NOT NULL default 'V4L',"
"  `is_default` int(1) default '0',"
"  `hostname` varchar(255) default NULL,"
"  PRIMARY KEY  (`id`),"
"  UNIQUE KEY `name` (`name`,`hostname`)"
");",
"CREATE TABLE IF NOT EXISTS `program` ("
"  `chanid` int(10) unsigned NOT NULL default '0',"
"  `starttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `endtime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `title` varchar(128) NOT NULL default '',"
"  `subtitle` varchar(128) NOT NULL default '',"
"  `description` text NOT NULL,"
"  `category` varchar(64) NOT NULL default '',"
"  `category_type` varchar(64) NOT NULL default '',"
"  `airdate` year(4) NOT NULL default '0000',"
"  `stars` float NOT NULL default '0',"
"  `previouslyshown` tinyint(4) NOT NULL default '0',"
"  `title_pronounce` varchar(128) NOT NULL default '',"
"  `stereo` tinyint(1) NOT NULL default '0',"
"  `subtitled` tinyint(1) NOT NULL default '0',"
"  `hdtv` tinyint(1) NOT NULL default '0',"
"  `closecaptioned` tinyint(1) NOT NULL default '0',"
"  `partnumber` int(11) NOT NULL default '0',"
"  `parttotal` int(11) NOT NULL default '0',"
"  `seriesid` varchar(12) NOT NULL default '',"
"  `originalairdate` date default NULL,"
"  `showtype` varchar(30) NOT NULL default '',"
"  `colorcode` varchar(20) NOT NULL default '',"
"  `syndicatedepisodenumber` varchar(20) NOT NULL default '',"
"  `programid` varchar(20) NOT NULL default '',"
"  `manualid` int(10) unsigned NOT NULL default '0',"
"  `generic` tinyint(1) default '0',"
"  PRIMARY KEY  (`chanid`,`starttime`,`manualid`),"
"  KEY `endtime` (`endtime`),"
"  KEY `title` (`title`),"
"  KEY `title_pronounce` (`title_pronounce`),"
"  KEY `seriesid` (`seriesid`),"
"  KEY `programid` (`programid`),"
"  KEY `id_start_end` (`chanid`,`starttime`,`endtime`)"
");",
"CREATE TABLE IF NOT EXISTS `programgenres` ("
"  `chanid` int(10) unsigned NOT NULL default '0',"
"  `starttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `relevance` char(1) NOT NULL default '',"
"  `genre` char(30) default NULL,"
"  PRIMARY KEY  (`chanid`,`starttime`,`relevance`)"
");",
"CREATE TABLE IF NOT EXISTS `programrating` ("
"  `chanid` int(10) unsigned NOT NULL default '0',"
"  `starttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `system` char(8) NOT NULL default '',"
"  `rating` char(8) NOT NULL default '',"
"  UNIQUE KEY `chanid` (`chanid`,`starttime`,`system`,`rating`),"
"  KEY `starttime` (`starttime`,`system`)"
");",
"CREATE TABLE IF NOT EXISTS `recgrouppassword` ("
"  `recgroup` varchar(32) NOT NULL default '',"
"  `password` varchar(10) NOT NULL default '',"
"  PRIMARY KEY  (`recgroup`),"
"  UNIQUE KEY `recgroup` (`recgroup`)"
");",
"CREATE TABLE IF NOT EXISTS `record` ("
"  `recordid` int(10) unsigned NOT NULL auto_increment,"
"  `type` int(10) unsigned NOT NULL default '0',"
"  `chanid` int(10) unsigned default NULL,"
"  `starttime` time NOT NULL default '00:00:00',"
"  `startdate` date NOT NULL default '0000-00-00',"
"  `endtime` time NOT NULL default '00:00:00',"
"  `enddate` date NOT NULL default '0000-00-00',"
"  `title` varchar(128) NOT NULL default '',"
"  `subtitle` varchar(128) NOT NULL default '',"
"  `description` text NOT NULL,"
"  `category` varchar(64) NOT NULL default '',"
"  `profile` varchar(128) NOT NULL default 'Default',"
"  `recpriority` int(10) NOT NULL default '0',"
"  `autoexpire` int(11) NOT NULL default '0',"
"  `maxepisodes` int(11) NOT NULL default '0',"
"  `maxnewest` int(11) NOT NULL default '0',"
"  `startoffset` int(11) NOT NULL default '0',"
"  `endoffset` int(11) NOT NULL default '0',"
"  `recgroup` varchar(32) NOT NULL default 'Default',"
"  `dupmethod` int(11) NOT NULL default '6',"
"  `dupin` int(11) NOT NULL default '15',"
"  `station` varchar(20) NOT NULL default '',"
"  `seriesid` varchar(12) NOT NULL default '',"
"  `programid` varchar(20) NOT NULL default '',"
"  `search` int(10) unsigned NOT NULL default '0',"
"  `autotranscode` tinyint(1) NOT NULL default '0',"
"  `autocommflag` tinyint(1) NOT NULL default '0',"
"  `autouserjob1` tinyint(1) NOT NULL default '0',"
"  `autouserjob2` tinyint(1) NOT NULL default '0',"
"  `autouserjob3` tinyint(1) NOT NULL default '0',"
"  `autouserjob4` tinyint(1) NOT NULL default '0',"
"  `findday` tinyint(4) NOT NULL default '0',"
"  `findtime` time NOT NULL default '00:00:00',"
"  `findid` int(11) NOT NULL default '0',"
"  `inactive` tinyint(1) NOT NULL default '0',"
"  `parentid` int(11) NOT NULL default '0',"
"  `transcoder` int(11) NOT NULL default '0',"
"  `tsdefault` float NOT NULL default '1',"
"  `playgroup` varchar(32) NOT NULL default 'Default',"
"  PRIMARY KEY  (`recordid`),"
"  KEY `chanid` (`chanid`,`starttime`),"
"  KEY `title` (`title`),"
"  KEY `seriesid` (`seriesid`),"
"  KEY `programid` (`programid`)"
");",
"CREATE TABLE IF NOT EXISTS `recorded` ("
"  `chanid` int(10) unsigned NOT NULL default '0',"
"  `starttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `endtime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `title` varchar(128) NOT NULL default '',"
"  `subtitle` varchar(128) NOT NULL default '',"
"  `description` text NOT NULL,"
"  `category` varchar(64) NOT NULL default '',"
"  `hostname` varchar(255) NOT NULL default '',"
"  `bookmark` varchar(128) default NULL,"
"  `editing` int(10) unsigned NOT NULL default '0',"
"  `cutlist` text,"
"  `autoexpire` int(11) NOT NULL default '0',"
"  `commflagged` int(10) unsigned NOT NULL default '0',"
"  `recgroup` varchar(32) NOT NULL default 'Default',"
"  `recordid` int(11) default NULL,"
"  `seriesid` varchar(12) NOT NULL default '',"
"  `programid` varchar(20) NOT NULL default '',"
"  `lastmodified` timestamp(14) NOT NULL,"
"  `filesize` bigint(20) NOT NULL default '0',"
"  `stars` float NOT NULL default '0',"
"  `previouslyshown` tinyint(1) default '0',"
"  `originalairdate` date default NULL,"
"  `preserve` tinyint(1) NOT NULL default '0',"
"  `findid` int(11) NOT NULL default '0',"
"  `deletepending` tinyint(1) NOT NULL default '0',"
"  `transcoder` int(11) NOT NULL default '0',"
"  `timestretch` float NOT NULL default '1',"
"  `recpriority` int(11) NOT NULL default '0',"
"  `basename` varchar(128) NOT NULL default '',"
"  `progstart` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `progend` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `playgroup` varchar(32) NOT NULL default 'Default',"
"  `profile` varchar(32) NOT NULL default '',"
"  PRIMARY KEY  (`chanid`,`starttime`),"
"  KEY `endtime` (`endtime`),"
"  KEY `seriesid` (`seriesid`),"
"  KEY `programid` (`programid`),"
"  KEY `title` (`title`),"
"  KEY `recordid` (`recordid`)"
");",
"CREATE TABLE IF NOT EXISTS `recordedcredits` ("
"  `person` mediumint(8) unsigned NOT NULL default '0',"
"  `chanid` int(10) unsigned NOT NULL default '0',"
"  `starttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `role` set('actor','director','producer','executive_producer','writer',"
"             'guest_star','host','adapter','presenter','commentator','guest')"
"         NOT NULL default '',"
"  UNIQUE KEY `chanid` (`chanid`,`starttime`,`person`,`role`),"
"  KEY `person` (`person`,`role`)"
");",
"CREATE TABLE IF NOT EXISTS `recordedmarkup` ("
"  `chanid` int(10) unsigned NOT NULL default '0',"
"  `starttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `mark` bigint(20) NOT NULL default '0',"
"  `offset` varchar(32) default NULL,"
"  `type` int(11) NOT NULL default '0',"
"  PRIMARY KEY  (`chanid`,`starttime`,`mark`,`type`)"
");",
"CREATE TABLE IF NOT EXISTS `recordedprogram` ("
"  `chanid` int(10) unsigned NOT NULL default '0',"
"  `starttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `endtime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `title` varchar(128) NOT NULL default '',"
"  `subtitle` varchar(128) NOT NULL default '',"
"  `description` text NOT NULL,"
"  `category` varchar(64) NOT NULL default '',"
"  `category_type` varchar(64) NOT NULL default '',"
"  `airdate` year(4) NOT NULL default '0000',"
"  `stars` float unsigned NOT NULL default '0',"
"  `previouslyshown` tinyint(4) NOT NULL default '0',"
"  `title_pronounce` varchar(128) NOT NULL default '',"
"  `stereo` tinyint(1) NOT NULL default '0',"
"  `subtitled` tinyint(1) NOT NULL default '0',"
"  `hdtv` tinyint(1) NOT NULL default '0',"
"  `closecaptioned` tinyint(1) NOT NULL default '0',"
"  `partnumber` int(11) NOT NULL default '0',"
"  `parttotal` int(11) NOT NULL default '0',"
"  `seriesid` varchar(12) NOT NULL default '',"
"  `originalairdate` date default NULL,"
"  `showtype` varchar(30) NOT NULL default '',"
"  `colorcode` varchar(20) NOT NULL default '',"
"  `syndicatedepisodenumber` varchar(20) NOT NULL default '',"
"  `programid` varchar(20) NOT NULL default '',"
"  `manualid` int(10) unsigned NOT NULL default '0',"
"  `generic` tinyint(1) default '0',"
"  PRIMARY KEY  (`chanid`,`starttime`,`manualid`),"
"  KEY `endtime` (`endtime`),"
"  KEY `title` (`title`),"
"  KEY `title_pronounce` (`title_pronounce`),"
"  KEY `seriesid` (`seriesid`),"
"  KEY `programid` (`programid`),"
"  KEY `id_start_end` (`chanid`,`starttime`,`endtime`)"
");",
"CREATE TABLE IF NOT EXISTS `recordedrating` ("
"  `chanid` int(10) unsigned NOT NULL default '0',"
"  `starttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `system` char(8) NOT NULL default '',"
"  `rating` char(8) NOT NULL default '',"
"  UNIQUE KEY `chanid` (`chanid`,`starttime`,`system`,`rating`),"
"  KEY `starttime` (`starttime`,`system`)"
");",
"CREATE TABLE IF NOT EXISTS `recordingprofiles` ("
"  `id` int(10) unsigned NOT NULL auto_increment,"
"  `name` varchar(128) default NULL,"
"  `videocodec` varchar(128) default NULL,"
"  `audiocodec` varchar(128) default NULL,"
"  `profilegroup` int(10) unsigned NOT NULL default '0',"
"  PRIMARY KEY  (`id`)"
");",
"CREATE TABLE IF NOT EXISTS `recordmatch` ("
"  `recordid` int(10) unsigned default NULL,"
"  `chanid` int(10) unsigned default NULL,"
"  `starttime` datetime default NULL,"
"  `manualid` int(10) unsigned default NULL,"
"  KEY `recordid` (`recordid`)"
");",
"CREATE TABLE IF NOT EXISTS `settings` ("
"  `value` varchar(128) NOT NULL default '',"
"  `data` text,"
"  `hostname` varchar(255) default NULL,"
"  KEY `value` (`value`,`hostname`)"
");",
"CREATE TABLE IF NOT EXISTS `tvchain` ("
"  `chanid` int(10) unsigned NOT NULL default '0',"
"  `starttime` datetime NOT NULL default '0000-00-00 00:00:00',"
"  `chainid` varchar(128) NOT NULL default '',"
"  `chainpos` int(10) NOT NULL default '0',"
"  `discontinuity` tinyint(1) NOT NULL default '0',"
"  `watching` int(10) NOT NULL default '0',"
"  `hostprefix` varchar(128) NOT NULL default '',"
"  `cardtype` varchar(32) NOT NULL default 'V4L',"
"  `input` varchar(32) NOT NULL default '',"
"  `channame` varchar(32) NOT NULL default '',"
"  PRIMARY KEY  (`chanid`,`starttime`)"
");",
"CREATE TABLE IF NOT EXISTS `videobookmarks` ("
"  `filename` varchar(255) NOT NULL default '',"
"  `bookmark` varchar(128) default NULL,"
"  PRIMARY KEY  (`filename`)"
");",
"CREATE TABLE IF NOT EXISTS `videosource` ("
"  `sourceid` int(10) unsigned NOT NULL auto_increment,"
"  `name` varchar(128) NOT NULL default '',"
"  `xmltvgrabber` varchar(128) default NULL,"
"  `userid` varchar(128) NOT NULL default '',"
"  `freqtable` varchar(16) NOT NULL default 'default',"
"  `lineupid` varchar(64) default NULL,"
"  `password` varchar(64) default NULL,"
"  `useeit` smallint(6) NOT NULL default '0',"
"  PRIMARY KEY  (`sourceid`),"
"  UNIQUE KEY `name` (`name`)"
");",
"CREATE TABLE IF NOT EXISTS `xvmc_buffer_settings` ("
"  `id` int(11) NOT NULL auto_increment,"
"  `description` varchar(255) NOT NULL default '',"
"  `osd_num` int(11) NOT NULL default '0',"
"  `osd_res_num` int(11) NOT NULL default '0',"
"  `min_surf` int(11) NOT NULL default '0',"
"  `max_surf` int(11) NOT NULL default '0',"
"  `decode_num` int(11) NOT NULL default '0',"
"  `agressive` int(11) NOT NULL default '1',"
"  PRIMARY KEY  (`id`)"
");",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',9018,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',9018,'guide_fixup','2');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',256,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',257,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',256,'tv_types',"
"                                       '1,150,134,133');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',257,'tv_types',"
"                                       '1,150,134,133');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4100,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4101,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4102,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4103,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4104,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4105,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4106,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4107,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4097,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4098,'sdt_mapping','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4100,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4101,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4102,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4103,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4104,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4105,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4106,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4107,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4097,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4098,'tv_types','1,145,154');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4100,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4101,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4102,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4103,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4104,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4105,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4106,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4107,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4097,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4098,'guide_fixup','1');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',94,'tv_types','1,128');",
"INSERT INTO `dtv_privatetypes` VALUES ('atsc',1793,'guide_fixup','3');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',40999,'guide_fixup','4');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',70,'force_guide_present',"
"                                       'yes');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',70,'guide_ranges',"
"                                       '80,80,96,96');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4112,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4115,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',4116,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',12802,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',12803,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',12829,'channel_numbers','131');",
"INSERT INTO `dtv_privatetypes` VALUES ('dvb',40999,'parse_subtitle_list',"
"                       '1070,1308,1041,1306,1307,1030,1016,1131,1068,1069');",
"INSERT INTO `playgroup` VALUES ('Default','',30,5,100);",
"INSERT INTO `profilegroups` VALUES (1,'Software Encoders (v4l based)','V4L',"
"                                      1,NULL);",
"INSERT INTO `profilegroups` VALUES (2,'MPEG-2 Encoders (PVR-x50, PVR-500)',"
"                                      'MPEG',1,NULL);",
"INSERT INTO `profilegroups` VALUES (3,"
" 'Hardware MJPEG Encoders (Matrox G200-TV, Miro DC10, etc)','MJPEG',1,NULL);",
"INSERT INTO `profilegroups` VALUES (4,'Hardware HDTV','HDTV',1,NULL);",
"INSERT INTO `profilegroups` VALUES (5,'Hardware DVB Encoders','DVB',1,NULL);",
"INSERT INTO `profilegroups` VALUES (6,'Transcoders','TRANSCODE',1,NULL);",
"INSERT INTO `profilegroups` VALUES (7,'FireWire Input','FIREWIRE',1,NULL);",
"INSERT INTO `profilegroups` VALUES (8,"
" 'USB Mpeg-4 Encoder (Plextor ConvertX, etc)','GO7007',1,NULL);",
"INSERT INTO `profilegroups` VALUES (9,'DBOX2 Input','DBOX2',1,NULL);",
"INSERT INTO `recordingprofiles` VALUES (1,'Default',NULL,NULL,1);",
"INSERT INTO `recordingprofiles` VALUES (2,'Live TV',NULL,NULL,1);",
"INSERT INTO `recordingprofiles` VALUES (3,'High Quality',NULL,NULL,1);",
"INSERT INTO `recordingprofiles` VALUES (4,'Low Quality',NULL,NULL,1);",
"INSERT INTO `recordingprofiles` VALUES (5,'Default',NULL,NULL,2);",
"INSERT INTO `recordingprofiles` VALUES (6,'Live TV',NULL,NULL,2);",
"INSERT INTO `recordingprofiles` VALUES (7,'High Quality',NULL,NULL,2);",
"INSERT INTO `recordingprofiles` VALUES (8,'Low Quality',NULL,NULL,2);",
"INSERT INTO `recordingprofiles` VALUES (9,'Default',NULL,NULL,3);",
"INSERT INTO `recordingprofiles` VALUES (10,'Live TV',NULL,NULL,3);",
"INSERT INTO `recordingprofiles` VALUES (11,'High Quality',NULL,NULL,3);",
"INSERT INTO `recordingprofiles` VALUES (12,'Low Quality',NULL,NULL,3);",
"INSERT INTO `recordingprofiles` VALUES (13,'Default',NULL,NULL,4);",
"INSERT INTO `recordingprofiles` VALUES (14,'Live TV',NULL,NULL,4);",
"INSERT INTO `recordingprofiles` VALUES (15,'High Quality',NULL,NULL,4);",
"INSERT INTO `recordingprofiles` VALUES (16,'Low Quality',NULL,NULL,4);",
"INSERT INTO `recordingprofiles` VALUES (17,'Default',NULL,NULL,5);",
"INSERT INTO `recordingprofiles` VALUES (18,'Live TV',NULL,NULL,5);",
"INSERT INTO `recordingprofiles` VALUES (19,'High Quality',NULL,NULL,5);",
"INSERT INTO `recordingprofiles` VALUES (20,'Low Quality',NULL,NULL,5);",
"INSERT INTO `recordingprofiles` VALUES (21,'RTjpeg/MPEG4',NULL,NULL,6);",
"INSERT INTO `recordingprofiles` VALUES (22,'MPEG2',NULL,NULL,6);",
"INSERT INTO `recordingprofiles` VALUES (23,'Default',NULL,NULL,8);",
"INSERT INTO `recordingprofiles` VALUES (24,'Live TV',NULL,NULL,8);",
"INSERT INTO `recordingprofiles` VALUES (25,'High Quality',NULL,NULL,8);",
"INSERT INTO `recordingprofiles` VALUES (26,'Low Quality',NULL,NULL,8);",
"INSERT INTO `recordingprofiles` VALUES (27,'High Quality',NULL,NULL,6);",
"INSERT INTO `recordingprofiles` VALUES (28,'Medium Quality',NULL,NULL,6);",
"INSERT INTO `recordingprofiles` VALUES (29,'Low Quality',NULL,NULL,6);",
"INSERT INTO `settings` VALUES ('mythfilldatabaseLastRunStart',NULL,NULL);",
"INSERT INTO `settings` VALUES ('mythfilldatabaseLastRunEnd',NULL,NULL);",
"INSERT INTO `settings` VALUES ('mythfilldatabaseLastRunStatus',NULL,NULL);",
"INSERT INTO `settings` VALUES ('DataDirectMessage',NULL,NULL);",
"INSERT INTO `settings` VALUES ('HaveRepeats','0',NULL);",
"INSERT INTO `settings` VALUES ('DBSchemaVer','1112',NULL);",
"INSERT INTO `settings` VALUES ('DefaultTranscoder','0',NULL);",
"INSERT INTO `xvmc_buffer_settings` VALUES (1,'Default / nVidia',"
"                                             2,2,8,16,8,1);",
"INSERT INTO `xvmc_buffer_settings` VALUES (2,'VLD (More decode buffers)',"
"                                             2,2,8,16,16,1);",
""
};

    QString dbver = "";
    if (!performActualUpdate(updates, "1112", dbver))
        return false;
    return true;
}

/* vim: set expandtab tabstop=4 shiftwidth=4: */
